﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

using namespace nn;
using namespace nn::fs;
using namespace nn::fs::detail;
using namespace nnt::fs::util;

namespace nnt { namespace fs { namespace util {

    class ExportHelper
    {
    public:
        ExportHelper(nn::fs::SaveDataTransferManagerVersion2& manager, const SaveDataInfo& infoSrc)
            : m_Manager(manager)
            , m_SrcInfo(infoSrc)
        {
            // 一時ディレクトリを作成・マウント
            m_TempDir.Create();
            NNT_EXPECT_RESULT_SUCCESS(MountHost(m_MountName.c_str(), m_TempDir.GetPath().c_str()));
            NN_SDK_LOG("TempDir: %s\n", m_TempDir.GetPath().c_str());
        }

        ~ExportHelper()
        {
            Unmount(m_MountName.c_str());
        }

        void DoExport(int divisionCount)
        {
            // エクスポータを開く
            NNT_EXPECT_RESULT_SUCCESS(m_Manager.OpenSaveDataFullExporter(&m_pExporter, m_SrcInfo.saveDataSpaceId, m_SrcInfo.saveDataId));

            // コミット ID の取得
            NNT_EXPECT_RESULT_SUCCESS(GetSaveDataCommitId(&m_CommitId, m_SrcInfo.saveDataSpaceId, m_SrcInfo.saveDataId));

            // DivisionCount の設定
            m_pExporter->SetDivisionCount(divisionCount);

            // Aad を設定
            {
                nn::fs::ISaveDataDivisionExporter::InitialDataAad aad;
                GenerateAad(&aad);
                m_pExporter->SetExportInitialDataAad(aad);
            }

            // エクスポート
            std::unique_ptr<ISaveDataChunkIterator> iter;
            NNT_EXPECT_RESULT_SUCCESS(m_pExporter->OpenSaveDataDiffChunkIterator(&iter));
            for (; !iter->IsEnd(); iter->Next())
            {
                m_ExportedChunkCount++;
                ExportChunk(iter->GetId());
            }

            {
                nn::fs::ISaveDataDivisionExporter::KeySeed keySeed;
                nn::fs::ISaveDataDivisionExporter::InitialDataMac mac;
                NNT_EXPECT_RESULT_SUCCESS(m_pExporter->FinalizeFullExport(&keySeed, &mac));

                DeleteFile(GetKeySeedFilePath().c_str());
                NNT_EXPECT_RESULT_SUCCESS(CreateAndWriteFileAtOnce(GetKeySeedFilePath().c_str(), keySeed.data, keySeed.Size));

                DeleteFile(GetInitialMacFilePath().c_str());
                NNT_EXPECT_RESULT_SUCCESS(CreateAndWriteFileAtOnce(GetInitialMacFilePath().c_str(), mac.data, mac.Size));
            }
        }

        int GetExportedChunkCount()
        {
            return m_ExportedChunkCount;
        }

        int64_t GetTotalExportedSize()
        {
            return m_TotalExportedSize;
        }

        const SaveDataInfo& GetSrcInfo()
        {
            return m_SrcInfo;
        }

        ISaveDataDivisionExporter::KeySeed GetKeySeed()
        {
            size_t size;
            auto buffer = AllocateBuffer(32);
            ISaveDataDivisionExporter::KeySeed keySeed;
            NNT_EXPECT_RESULT_SUCCESS(ReadFileAtOnce(&size, &buffer, GetKeySeedFilePath().c_str()));
            EXPECT_EQ(size, ISaveDataDivisionExporter::KeySeed::Size);
            memcpy(keySeed.data, buffer.get(), keySeed.Size);
            return keySeed;
        }

        void ReadInitialData(InitialDataVersion2* pOutValue)
        {
            auto initialDataRaw = AllocateBuffer(32);
            size_t initialDataRawSize;

            NNT_EXPECT_RESULT_SUCCESS(ReadFileAtOnce(&initialDataRawSize, &initialDataRaw, GenerateStorageFilePath(SaveDataChunkIdForInitialData).c_str()));
            InitialDataVersion2 initialData = {};
            EXPECT_EQ(sizeof(InitialDataVersion2), initialDataRawSize);
            memcpy(&initialData, initialDataRaw.get(), initialDataRawSize);

            *pOutValue = initialData;
        }

        Result ReadChunkData(size_t* outReadSize, SaveDataChunkId id, int64_t offset, char* buffer, int64_t bufferSize)
        {
            FileHandle handle;
            NNT_EXPECT_RESULT_SUCCESS(OpenFile(&handle, GenerateStorageFilePath(id).c_str(), OpenMode_Read));
            NN_UTIL_SCOPE_EXIT
            {
                CloseFile(handle);
            };

            NNT_EXPECT_RESULT_SUCCESS(ReadFile(outReadSize, handle, offset, buffer, bufferSize, ReadOption()));
            NN_RESULT_SUCCESS;
        }

    private:
        void GenerateAad(nn::fs::detail::InitialDataAad* aad)
        {
            nnt::fs::util::FillBufferWith8BitCount(aad->data, aad->Size, 0);
        }

        String GenerateStorageFileName(SaveDataChunkId id)
        {
            return ToString(static_cast<int64_t>(id));
        }

        String GenerateStorageFilePath(SaveDataChunkId id)
        {
            return m_MountName + ":/" + GenerateStorageFileName(id);
        }

        String GetKeySeedFilePath()
        {
            return m_MountName + ":/keySeed";
        }

        String GetInitialMacFilePath()
        {
            return m_MountName + ":/initialDataMac";
        }

        void ExportChunk(SaveDataChunkId id)
        {
            // サイズを取得
            int64_t size;
            ExportChunkImpl(&size, nullptr, id, true);

            // ファイルを作成
            auto filePath = GenerateStorageFilePath(id);
            DeleteFile(filePath.c_str());
            NNT_EXPECT_RESULT_SUCCESS(CreateFile(filePath.c_str(), size));

            // ファイルを開く
            FileHandle handle;
            NNT_EXPECT_RESULT_SUCCESS(OpenFile(&handle, filePath.c_str(), OpenMode_Write));
            NN_UTIL_SCOPE_EXIT
            {
                CloseFile(handle);
            };

            // エクスポート
            ExportChunkImpl(&size, &handle, id, false);
        }

        void ExportChunkImpl(int64_t* outExportedSize, FileHandle* pHandle, SaveDataChunkId id, bool isDryRun)
        {
            std::unique_ptr<ISaveDataChunkExporter> chunkExporter;
            NNT_EXPECT_RESULT_SUCCESS(m_pExporter->OpenSaveDataChunkExporter(&chunkExporter, id));

            const size_t BufferSize = 1024 * 1024;
            auto buffer = AllocateBuffer(BufferSize);

            int64_t offset = 0;
            *outExportedSize = 0;
            while (true)
            {
                size_t pulledSize;
                NNT_EXPECT_RESULT_SUCCESS(chunkExporter->Pull(&pulledSize, buffer.get(), BufferSize));
                if (pulledSize > 0)
                {
                    if (!isDryRun)
                    {
                        NNT_EXPECT_RESULT_SUCCESS(WriteFile(*pHandle, offset, buffer.get(), pulledSize, WriteOption::MakeValue(WriteOptionFlag_Flush)));
                        offset += pulledSize;
                    }
                    *outExportedSize += pulledSize;
                }
                else
                {
                    break;
                }
            }
        }

    private:
        nn::fs::SaveDataTransferManagerVersion2& m_Manager;
        std::unique_ptr<ISaveDataDivisionExporter> m_pExporter;
        nnt::fs::util::TemporaryHostDirectory m_TempDir;
        String m_MountName = "temp";
        SaveDataCommitId m_CommitId;
        int m_ExportedChunkCount = 0;
        int64_t m_TotalExportedSize = 0;
        const SaveDataInfo& m_SrcInfo;
    };

}}}
