﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

#include "detail/fssrv_SaveDataTransferChunkIterator.h"

using namespace nn;
using namespace nn::fs;
using namespace nn::fs::detail;
using namespace nnt::fs::util;
using namespace nn::fssrv::detail;

namespace {

TEST(IteratorAll, iterate)
{
    bool IncludesInitialArray[] = { true, false };
    int DivisionCountArray[] = { 1, 2, 3, 1024 };

    for (auto includesInitial : IncludesInitialArray)
    {

        for (auto divisionCount : DivisionCountArray)
        {
            SaveDataChunkIteratorAll iter(includesInitial, divisionCount);

            // 4096 ~ 4096 + divisionCount - 1 まで順に列挙されること
            for (int i = 0; i < divisionCount; i++)
            {
                uint32_t id;
                NNT_EXPECT_RESULT_SUCCESS(iter.GetId(sf::Out<uint32_t>(&id)));
                EXPECT_EQ(IdBaseForDivisionIndex + i, id);

                bool isEnd;
                NNT_EXPECT_RESULT_SUCCESS(iter.IsEnd(sf::Out<bool>(&isEnd)));
                EXPECT_FALSE(isEnd);

                NNT_EXPECT_RESULT_SUCCESS(iter.Next());
            }

            if (includesInitial)
            {
                // 未 end
                bool isEnd;
                NNT_EXPECT_RESULT_SUCCESS(iter.IsEnd(sf::Out<bool>(&isEnd)));
                EXPECT_FALSE(isEnd);

                // 最後に InitialData
                {
                    uint32_t id;
                    NNT_EXPECT_RESULT_SUCCESS(iter.GetId(sf::Out<uint32_t>(&id)));
                    EXPECT_EQ(SaveDataChunkIdForInitialData, id);
                    NNT_EXPECT_RESULT_SUCCESS(iter.Next());
                }
            }

            // end
            bool isEnd;
            NNT_EXPECT_RESULT_SUCCESS(iter.IsEnd(sf::Out<bool>(&isEnd)));
            EXPECT_TRUE(isEnd);

            // end 後の next は効果なし
            NNT_EXPECT_RESULT_SUCCESS(iter.Next());
            NNT_EXPECT_RESULT_SUCCESS(iter.IsEnd(sf::Out<bool>(&isEnd)));
            EXPECT_TRUE(isEnd);
        }
    }
}

// SaveDataChunkIteratorDiff が差分箇所に従った id を返すこと
TEST(IteratorDiff, iterateAll)
{
    bool IncludesInitialArray[] = { true, false };

    struct TestParam {
        int divisionCount;
        Vector<int> diffIndexArray;
    } TestParamArray[] = {
        { 1, {}},
        { 1, {0}},
        { 2, {0}},
        { 2, {0, 1} },
        { 2, {1} },
        { 4, {0, 3} },
        { 4, {2} },
        { 8, {1, 2} },
        { 8, {0, 1, 6, 7} },
        { 8, {0, 1, 4, 6, 7} },
        { 128,{ 127 } },
    };


    for (auto includesInitial : IncludesInitialArray)
    {

        for (auto param : TestParamArray)
        {
            InitialDataVersion2Detail::Content initialTheirs;
            InitialDataVersion2Detail::Content initialOurs;

            // 同じハッシュ値で埋める＆差分作る
            FillBufferWith32BitCount(initialTheirs.singleSaveData.chunkDigest, sizeof(initialTheirs.singleSaveData.chunkDigest), 0x1000);
            FillBufferWith32BitCount(initialOurs.singleSaveData.chunkDigest,   sizeof(initialOurs.singleSaveData.chunkDigest),   0x1000);
            for (auto diffIndex : param.diffIndexArray)
            {
                memset(&initialTheirs.singleSaveData.chunkDigest[diffIndex], diffIndex, sizeof(initialTheirs.singleSaveData.chunkDigest[diffIndex]));
            }

            SaveDataChunkIteratorDiff iter(initialTheirs, initialOurs, includesInitial, param.divisionCount);

            for (auto diffIndex : param.diffIndexArray)
            {
                uint32_t id;
                NNT_EXPECT_RESULT_SUCCESS(iter.GetId(sf::Out<uint32_t>(&id)));
                EXPECT_EQ(IdBaseForDivisionIndex + diffIndex, id);

                bool isEnd;
                NNT_EXPECT_RESULT_SUCCESS(iter.IsEnd(sf::Out<bool>(&isEnd)));
                EXPECT_FALSE(isEnd);

                NNT_EXPECT_RESULT_SUCCESS(iter.Next());
            }

            if (includesInitial)
            {
                // 未 end
                bool isEnd;
                NNT_EXPECT_RESULT_SUCCESS(iter.IsEnd(sf::Out<bool>(&isEnd)));
                EXPECT_FALSE(isEnd);

                // 最後に InitialData
                {
                    uint32_t id;
                    NNT_EXPECT_RESULT_SUCCESS(iter.GetId(sf::Out<uint32_t>(&id)));
                    EXPECT_EQ(SaveDataChunkIdForInitialData, id);
                    NNT_EXPECT_RESULT_SUCCESS(iter.Next());
                }
            }

            // end
            bool isEnd;
            NNT_EXPECT_RESULT_SUCCESS(iter.IsEnd(sf::Out<bool>(&isEnd)));
            EXPECT_TRUE(isEnd);

            // end 後の next は効果なし
            NNT_EXPECT_RESULT_SUCCESS(iter.Next());
            NNT_EXPECT_RESULT_SUCCESS(iter.IsEnd(sf::Out<bool>(&isEnd)));
            EXPECT_TRUE(isEnd);
        }
    }
}

}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();

    auto result = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(result);
}

