﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fssystem/fs_BucketTreeBuilder.h>
#include <nn/fssystem/utilTool/fs_SparseStorageBuilder.h>
#include <nnt/nntest.h>
#include <nnt/fsUtil/testFs_util.h>

namespace nn { namespace fssystem { namespace utilTool {

class SparseStorageBuilderTest : public ::testing::Test
{
    typedef nn::fssystem::utilTool::SparseStorageBuilder Builder;

public:
    static void VerifyBuildData(Builder* pBuilder, int endOffset) NN_NOEXCEPT
    {
        // 内部向けの関数なので事前検証はしない
        const auto& rangeArray = pBuilder->GetRangeArray();
        ASSERT_EQ(0, rangeArray.size());

        ASSERT_EQ(pBuilder->GetOriginalStorageSize(), endOffset);
    }

    template< int Length >
    static void VerifyBuildData(Builder* pBuilder, const Builder::Range(&expected)[Length], int endOffset) NN_NOEXCEPT
    {
        // 内部向けの関数なので事前検証はしない
        const auto& rangeArray = pBuilder->GetRangeArray();
        ASSERT_EQ(Length, rangeArray.size());

        for( int i = 0; i < Length; ++i )
        {
            ASSERT_EQ(expected[i].virtualOffset, rangeArray[i].virtualOffset);
            ASSERT_EQ(expected[i].physicalOffset, rangeArray[i].physicalOffset);
            ASSERT_EQ(expected[i].size, rangeArray[i].size);
        }

        ASSERT_EQ(pBuilder->GetOriginalStorageSize(), endOffset);

        std::unique_ptr<char[]> dataBuffer(new char[endOffset]);
        ASSERT_NE(dataBuffer, nullptr);
        nnt::fs::util::FillBufferWithRandomValue(dataBuffer.get(), endOffset);

        nn::fs::MemoryStorage dataStorage(dataBuffer.get(), endOffset);
        pBuilder->SetDataStorage(&dataStorage, 0, endOffset);

        const auto readSize = size_t(pBuilder->QueryDataStorageSize());
        std::unique_ptr<char[]> readBuffer(new char[readSize]);
        ASSERT_NE(readBuffer, nullptr);

        NNT_ASSERT_RESULT_SUCCESS(pBuilder->ReadData(0, readBuffer.get(), readSize));

        for( int i = 0; i < Length; ++i )
        {
            NNT_FS_UTIL_EXPECT_MEMCMPEQ(
                dataBuffer.get() + expected[i].virtualOffset,
                readBuffer.get() + expected[i].physicalOffset,
                size_t(expected[i].size)
            );
        }
    }

    template< int Length >
    static void ImportBuildData(Builder* pBuilder, const Builder::Range(&ranges)[Length], int endOffset) NN_NOEXCEPT
    {
        // 内部向けの関数なので事前検証はしない
        auto& rangeArray = pBuilder->m_RangeArray;

        rangeArray = Builder::RangeArray(Length);
        NN_ABORT_UNLESS(rangeArray.IsValid());

        int entryCount = 0;
        int64_t virtualOffset = 0;
        int64_t physicalOffset = 0;

        for( int i = 0; i < Length; ++i )
        {
            const auto& range = ranges[i];
            NN_ABORT_UNLESS_EQUAL(range.physicalOffset, physicalOffset);

            if( virtualOffset < range.virtualOffset )
            {
                ++entryCount;
            }
            ++entryCount;

            rangeArray.push_back(range);

            virtualOffset = range.virtualOffset + range.size;
            physicalOffset += range.size;
        }

        if( virtualOffset < endOffset )
        {
            ++entryCount;
        }

        pBuilder->m_EntryCount = entryCount;
        pBuilder->m_VirtualStorageSize = endOffset;
        pBuilder->m_PhysicalStorageSize = physicalOffset;
    }
};

}}}

namespace {

typedef nn::fssystem::IndirectStorage IndirectStorage;
typedef nn::fssystem::SparseStorage SparseStorage;
typedef nn::fssystem::utilTool::SparseStorageBuilder SparseStorageBuilder;
typedef nn::fssystem::utilTool::SparseStorageBuilderTest SparseStorageBuilderTest;

class IndirectTableCreator
{
public:
    typedef nn::fssystem::BucketTree BucketTree;
    typedef nn::fssystem::BucketTreeBuilder BucketTreeBuilder;
    typedef nn::fssystem::IndirectStorage IndirectStorage;
    typedef nn::fssystem::IndirectStorage::EntryData EntryData;

public:
    template< int Length >
    IndirectTableCreator(const EntryData(&entries)[Length], int64_t endOffset) NN_NOEXCEPT
        : m_Table()
        , m_StorageBuffer()
        , m_pBaseStorage()
    {
        const auto nodeSize = BucketTreeBuilder::QueryNodeStorageSize(IndirectStorage::NodeSize, sizeof(IndirectStorage::Entry), Length);
        const auto entrySize = BucketTreeBuilder::QueryEntryStorageSize(IndirectStorage::NodeSize, sizeof(IndirectStorage::Entry), Length);

        m_StorageBuffer.reset(new char[size_t(nodeSize + entrySize)]);
        NN_ABORT_UNLESS_NOT_NULL(m_StorageBuffer);

        m_pBaseStorage.reset(new nn::fs::MemoryStorage(m_StorageBuffer.get(), nodeSize + entrySize));
        NN_ABORT_UNLESS_NOT_NULL(m_pBaseStorage);

        nn::fs::SubStorage nodeStorage(m_pBaseStorage.get(), 0, nodeSize);
        nn::fs::SubStorage entryStorage(m_pBaseStorage.get(), nodeSize, entrySize);
        {
            BucketTree::Header header;
            nn::fs::MemoryStorage headerStorage(&header, sizeof(header));

            BucketTreeBuilder builder;
            NN_ABORT_UNLESS_RESULT_SUCCESS(builder.Initialize(
                nnt::fs::util::GetTestLibraryAllocator(),
                nn::fs::SubStorage(&headerStorage, 0, sizeof(header)),
                nodeStorage,
                entryStorage,
                IndirectStorage::NodeSize,
                sizeof(IndirectStorage::Entry),
                Length
            ));

            for( int i = 0; i < Length; ++i )
            {
                IndirectStorage::Entry entry;
                entry.SetVirtualOffset(entries[i].virtualOffset);
                entry.SetPhysicalOffset(entries[i].physicalOffset);
                entry.storageIndex = entries[i].storageIndex;

                NN_ABORT_UNLESS_RESULT_SUCCESS(builder.Write(entry));
            }
            NN_ABORT_UNLESS_RESULT_SUCCESS(builder.Commit(endOffset));
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(m_Table.Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            nodeStorage,
            entryStorage,
            IndirectStorage::NodeSize,
            sizeof(IndirectStorage::Entry),
            Length
        ));
    }

    const BucketTree& Get() const NN_NOEXCEPT
    {
        return m_Table;
    }

private:
    BucketTree m_Table;
    std::unique_ptr<char[]> m_StorageBuffer;
    std::unique_ptr<nn::fs::MemoryStorage> m_pBaseStorage;
};

}

#if !defined(NN_SDK_BUILD_RELEASE)
/**
 * @brief   事前検証の範囲外のテストをします。
 */
TEST(SparseStorageBuilderDeathTest, Precondition)
{
    static const auto BlockSize = SparseStorageBuilder::BlockSizeMin;

    EXPECT_DEATH_IF_SUPPORTED(SparseStorageBuilder(BlockSize / 2, BlockSize * 2), "");
    EXPECT_DEATH_IF_SUPPORTED(SparseStorageBuilder(BlockSize + 1, BlockSize * 2), "");
    EXPECT_DEATH_IF_SUPPORTED(SparseStorageBuilder(BlockSize, BlockSize / 2), "");
    EXPECT_DEATH_IF_SUPPORTED(SparseStorageBuilder(BlockSize, BlockSize + BlockSize / 2), "");

    char buffer[64];
    nn::fs::MemoryStorage baseStorage(buffer, sizeof(buffer));
    nn::fs::SubStorage storage(&baseStorage, 0, sizeof(buffer));
    nn::fssystem::BucketTree table;
    table.Initialize(SparseStorage::NodeSize, 1024);

    EXPECT_DEATH_IF_SUPPORTED(SparseStorageBuilder().Build(-1), "");
    EXPECT_DEATH_IF_SUPPORTED(SparseStorageBuilder().Build(nullptr, storage, storage, 1, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(SparseStorageBuilder().Build(table, -1), "");
    {
        SparseStorageBuilder builder;
        NNT_ASSERT_RESULT_SUCCESS(builder.Build(table, 1024));
        EXPECT_DEATH_IF_SUPPORTED(builder.Build(table, 1024), "");
    }

    EXPECT_DEATH_IF_SUPPORTED(SparseStorageBuilder().Rebuild(table, -1), "");

    EXPECT_DEATH_IF_SUPPORTED(SparseStorageBuilder().WriteHeader(nullptr, 16), "");
    EXPECT_DEATH_IF_SUPPORTED(SparseStorageBuilder().WriteHeader(buffer, 15), "");

    EXPECT_DEATH_IF_SUPPORTED(SparseStorageBuilder().WriteTable(nullptr, storage, storage, storage), "");

    EXPECT_DEATH_IF_SUPPORTED(SparseStorageBuilder().ReadData(-1, nullptr, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(SparseStorageBuilder().ReadData(0, nullptr, 4), "");

    EXPECT_DEATH_IF_SUPPORTED(SparseStorageBuilder().Import(nullptr), "");
}
#endif

/**
 * @brief   関数が想定した値を返すかをテストします。
 */
TEST_F(SparseStorageBuilderTest, ReturnValue)
{
    const char Zero[64] = {};
    char buffer[64];
    const auto pAllocator = nnt::fs::util::GetTestLibraryAllocator();

    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultNotInitialized, SparseStorageBuilder().Build(nn::fssystem::BucketTree(), 0));

    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultNotInitialized, SparseStorageBuilder().Rebuild(nn::fssystem::BucketTree(), 0));
    {
        nn::fssystem::BucketTree table;
        table.Initialize(nn::fssystem::BucketTree::NodeSizeMin, 1024);
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidSize, SparseStorageBuilder().Rebuild(table, 1024));
    }

    // エントリ数 0 の場合
    {
        nn::fs::MemoryStorage baseStorage(buffer, sizeof(buffer));
        nn::fs::SubStorage storage(&baseStorage, 0, sizeof(buffer));

        SparseStorageBuilder builder;
        builder.Build(1024);
        NNT_ASSERT_RESULT_SUCCESS(builder.SetDataStorage(storage));
        NNT_ASSERT_RESULT_SUCCESS(builder.WriteTable(pAllocator, storage, storage, storage));
        NNT_ASSERT_RESULT_SUCCESS(builder.ReadData(0, nullptr, 0));
        NNT_ASSERT_RESULT_SUCCESS(builder.ReadData(0, buffer, sizeof(buffer)));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(buffer, Zero, sizeof(Zero));

        ASSERT_EQ(sizeof(nn::fssystem::BucketTree::Header), builder.QueryTableHeaderStorageSize());
        ASSERT_EQ(0, builder.QueryTableNodeStorageSize());
        ASSERT_EQ(0, builder.QueryTableEntryStorageSize());
        ASSERT_EQ(0, builder.QueryDataStorageSize());

        nn::fs::SubStorage invalidStorage;
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultNotInitialized, builder.SetDataStorage(invalidStorage));
    }

    static const IndirectStorage::EntryData Entries[] =
    {
        {  0 * 1024,   0 * 1024, 0 },
        { 10 * 1024,  20 * 1024, 1 },
        { 30 * 1024,  20 * 1024, 0 },
        { 40 * 1024,  80 * 1024, 1 },
        { 50 * 1024,  50 * 1024, 0 },
        { 80 * 1024, 120 * 1024, 1 },
    };
    static const int IndirectEndOffset = 100 * 1024;
    static const int OriginalEndOffset = 100 * 1024;
    static const int DataStorageSize = 50 * 1024;

    IndirectTableCreator indirectTable(Entries, IndirectEndOffset);

    // エントリ数 > 0 の場合
    {
        SparseStorageBuilder builder;
        NNT_ASSERT_RESULT_SUCCESS(builder.Build(indirectTable.Get(), OriginalEndOffset));

        nn::fssystem::BucketTree::Header header;
        nn::fs::MemoryStorage headerStorage(&header, sizeof(header));

        const auto nodeSize = builder.QueryTableNodeStorageSize();
        const auto entrySize = builder.QueryTableEntryStorageSize();
        ASSERT_EQ(SparseStorage::NodeSize, nodeSize);
        ASSERT_EQ(SparseStorage::NodeSize, entrySize);

        std::unique_ptr<char[]> dataBuffer(new char[OriginalEndOffset]);
        ASSERT_NE(dataBuffer, nullptr);
        nn::fs::MemoryStorage dataStorage(dataBuffer.get(), OriginalEndOffset);

        std::unique_ptr<char[]> tableBuffer(new char[size_t(nodeSize + entrySize)]);
        ASSERT_NE(tableBuffer, nullptr);
        nn::fs::MemoryStorage tableStorage(tableBuffer.get(), nodeSize + entrySize);

        NNT_ASSERT_RESULT_SUCCESS(builder.SetDataStorage(nn::fs::SubStorage(&dataStorage, 0, OriginalEndOffset)));
        ASSERT_EQ(DataStorageSize, builder.QueryDataStorageSize());

        NNT_ASSERT_RESULT_SUCCESS(builder.WriteTable(
            pAllocator,
            nn::fs::SubStorage(&headerStorage, 0, sizeof(header)),
            nn::fs::SubStorage(&tableStorage, 0, nodeSize),
            nn::fs::SubStorage(&tableStorage, nodeSize, entrySize)
        ));
        NNT_ASSERT_RESULT_SUCCESS(builder.ReadData(0, nullptr, 0));
        NNT_ASSERT_RESULT_SUCCESS(builder.ReadData(0, buffer, sizeof(buffer)));
        NNT_ASSERT_RESULT_SUCCESS(builder.ReadData(DataStorageSize, buffer, sizeof(buffer)));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(buffer, Zero, sizeof(Zero));
    }

    SparseStorage sparseStorage;
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultNotInitialized, SparseStorageBuilder().Import(&sparseStorage));

    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultNullptrArgument,
        SparseStorageBuilder().OutputBuildLog(nullptr, &sparseStorage, &sparseStorage, &sparseStorage)
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultNullptrArgument,
        SparseStorageBuilder().OutputBuildLog(pAllocator, nullptr, &sparseStorage, &sparseStorage)
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultNullptrArgument,
        SparseStorageBuilder().OutputBuildLog(pAllocator, &sparseStorage, nullptr, &sparseStorage)
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultNullptrArgument,
        SparseStorageBuilder().OutputBuildLog(pAllocator, &sparseStorage, &sparseStorage, nullptr)
    );
}

/**
 * @brief   Build と Import の動作テストをします。
 */
TEST_F(SparseStorageBuilderTest, BuildAndImport)
{
    static const IndirectStorage::EntryData Entries[] =
    {
        {   0 * 1024,   0 * 1024, 0 },
        {  10 * 1024,  20 * 1024, 1 },
        {  30 * 1024,  61 * 1024, 0 },
        {  40 * 1024,  80 * 1024, 1 },
        {  50 * 1024,  22 * 1024, 0 },
        {  82 * 1024, 120 * 1024, 1 },
        { 100 * 1024,  98 * 1024, 0 },
    };
    static const int IndirectEndOffset = 104 * 1024;
    static const int OriginalEndOffset = 102 * 1024;

    static const int BlockSize = 4 * 1024;
    static const SparseStorageBuilder::Range Expected[] =
    {
        {   0 * 1024,  0 * 1024, 12 * 1024 },
        {  20 * 1024, 12 * 1024, 36 * 1024 },
        {  60 * 1024, 48 * 1024, 12 * 1024 },
        {  96 * 1024, 60 * 1024,  6 * 1024 }, // ストレージ末尾だけ BlockSize の整数倍にならない
    };

    IndirectTableCreator indirectTable(Entries, IndirectEndOffset);

    SparseStorageBuilder builder(BlockSize, BlockSize);
    NNT_ASSERT_RESULT_SUCCESS(builder.Build(indirectTable.Get(), OriginalEndOffset));

    VerifyBuildData(&builder, Expected, OriginalEndOffset);

    const auto nodeSize = builder.QueryTableNodeStorageSize();
    std::unique_ptr<char[]> nodeBuffer(new char[size_t(nodeSize)]);
    ASSERT_NE(nodeBuffer, nullptr);
    nn::fs::MemoryStorage nodeStorage(nodeBuffer.get(), nodeSize);

    const auto entrySize = builder.QueryTableEntryStorageSize();
    std::unique_ptr<char[]> entryBuffer(new char[size_t(entrySize)]);
    ASSERT_NE(entryBuffer, nullptr);
    nn::fs::MemoryStorage entryStorage(entryBuffer.get(), entrySize);

    nn::fssystem::BucketTree::Header header;
    nn::fs::MemoryStorage headerStorage(&header, sizeof(header));

    NNT_ASSERT_RESULT_SUCCESS(builder.WriteTable(
        nnt::fs::util::GetTestLibraryAllocator(),
        nn::fs::SubStorage(&headerStorage, 0, sizeof(header)),
        nn::fs::SubStorage(&nodeStorage, 0, nodeSize),
        nn::fs::SubStorage(&entryStorage, 0, entrySize)
    ));

    SparseStorage storage;
    NNT_ASSERT_RESULT_SUCCESS(storage.Initialize(
        nnt::fs::util::GetTestLibraryAllocator(),
        nn::fs::SubStorage(&nodeStorage, 0, nodeSize),
        nn::fs::SubStorage(&entryStorage, 0, entrySize),
        header.entryCount
    ));

    SparseStorageBuilder importer;
    NNT_ASSERT_RESULT_SUCCESS(importer.Import(&storage));

    const auto& builderRanges = builder.GetRangeArray();
    const auto& importerRanges = importer.GetRangeArray();
    ASSERT_EQ(builderRanges.size(), importerRanges.size());
    ASSERT_TRUE(std::equal(
        builderRanges.begin(),
        builderRanges.end(),
        importerRanges.begin(),
        [](const SparseStorageBuilder::Range& lhs, const SparseStorageBuilder::Range& rhs) NN_NOEXCEPT
        {
            return (lhs.virtualOffset == rhs.virtualOffset)
                && (lhs.physicalOffset == rhs.physicalOffset)
                && (lhs.size == rhs.size);
        }
    ));
}

/**
 * @brief   SparseStorageBuilder の動作テストをします。
 */
TEST_F(SparseStorageBuilderTest, Rebuild)
{
    static const int BlockSize = 4 * 1024;

    static const int OriginalEndOffset = 102 * 1024;
    static const SparseStorageBuilder::Range RangeArray1st[] =
    {
        {   0 * 1024,  0 * 1024, 12 * 1024 },
        {  20 * 1024, 12 * 1024, 36 * 1024 },
        {  60 * 1024, 48 * 1024, 12 * 1024 },
        {  96 * 1024, 60 * 1024,  6 * 1024 },
    };

    static const int IndirectEndOffset = 102 * 1024;
    static const IndirectStorage::EntryData Entries[] =
    {
        {   0 * 1024,   0 * 1024, 0 },
        {  12 * 1024,  10 * 1024, 1 },
        {  32 * 1024,  20 * 1024, 0 },
        {  48 * 1024,  40 * 1024, 1 },
        {  52 * 1024,  44 * 1024, 0 },
        {  72 * 1024,  70 * 1024, 1 },
        {  80 * 1024,  68 * 1024, 0 },
        {  84 * 1024,  80 * 1024, 1 },
        {  96 * 1024,  96 * 1024, 0 },
    };
    static const SparseStorageBuilder::Range RangeArray2nd[] =
    {
        {   0 * 1024,  0 * 1024, 12 * 1024 },
        {  20 * 1024, 12 * 1024, 16 * 1024 },
        {  44 * 1024, 28 * 1024, 20 * 1024 },
        {  68 * 1024, 48 * 1024,  4 * 1024 },
        {  96 * 1024, 52 * 1024,  6 * 1024 },
    };

    static const SparseStorageBuilder::Range Expected[] =
    {
        {   0 * 1024,  0 * 1024, 12 * 1024 },
        {  20 * 1024, 12 * 1024, 16 * 1024 },
        {  44 * 1024, 28 * 1024, 12 * 1024 },
        {  60 * 1024, 40 * 1024,  4 * 1024 },
        {  68 * 1024, 44 * 1024,  4 * 1024 },
        {  96 * 1024, 48 * 1024,  6 * 1024 },
    };

    SparseStorageBuilder builder(BlockSize, BlockSize);
    ImportBuildData(&builder, RangeArray1st, OriginalEndOffset);

    IndirectTableCreator indirectTable(Entries, IndirectEndOffset);
    {
        SparseStorageBuilder builder2nd(BlockSize, BlockSize);
        NNT_ASSERT_RESULT_SUCCESS(builder2nd.Build(indirectTable.Get(), OriginalEndOffset));
        VerifyBuildData(&builder2nd, RangeArray2nd, OriginalEndOffset);
    }

    NNT_ASSERT_RESULT_SUCCESS(builder.Rebuild(indirectTable.Get(), OriginalEndOffset));
    VerifyBuildData(&builder, Expected, OriginalEndOffset);

    // 同じ領域でリビルドしても問題ない
    NNT_ASSERT_RESULT_SUCCESS(builder.Rebuild(indirectTable.Get(), OriginalEndOffset));
    VerifyBuildData(&builder, Expected, OriginalEndOffset);
}

/**
 * @brief   最小削除サイズによってエントリがつながるかテストします。
 */
TEST_F(SparseStorageBuilderTest, MergeEntryByEraseSize)
{
    static const IndirectStorage::EntryData Entries[] =
    {
        {   0 * 1024,   0 * 1024, 0 },
        {  10 * 1024,  20 * 1024, 1 },
        {  30 * 1024,  61 * 1024, 0 },
        {  40 * 1024,  80 * 1024, 1 },
        {  50 * 1024,  22 * 1024, 0 },
        {  82 * 1024, 120 * 1024, 1 },
        { 100 * 1024,  98 * 1024, 0 },
    };
    static const int IndirectEndOffset = 104 * 1024;
    static const int OriginalEndOffset = 102 * 1024;

    static const int BlockSize = 4 * 1024;
    static const int EraseSize = BlockSize * 2;
    static const SparseStorageBuilder::Range Expected[] =
    {
        {   0 * 1024,  0 * 1024, 12 * 1024 },
        {  20 * 1024, 12 * 1024, 52 * 1024 },
        {  96 * 1024, 64 * 1024,  6 * 1024 },
    };

    IndirectTableCreator indirectTable(Entries, IndirectEndOffset);

    SparseStorageBuilder builder(BlockSize, EraseSize);
    NNT_ASSERT_RESULT_SUCCESS(builder.Build(indirectTable.Get(), OriginalEndOffset));

    VerifyBuildData(&builder, Expected, OriginalEndOffset);
}

/**
 * @brief   IndirectStorage の全域が 0 か 1 かのテスト。
 */
TEST_F(SparseStorageBuilderTest, IndirectWholeArea)
{
    static const int EndOffset = 104 * 1024;
    static const int BlockSize = 4 * 1024;
    static const int EraseSize = BlockSize * 4;

    // 全域がオリジナル側
    {
        static const IndirectStorage::EntryData Entries[] =
        {
            { 0, 0, 0 },
        };

        static const SparseStorageBuilder::Range Expected[] =
        {
            { 0, 0, EndOffset },
        };

        IndirectTableCreator indirectTable(Entries, EndOffset);

        SparseStorageBuilder builder(BlockSize, EraseSize);
        NNT_ASSERT_RESULT_SUCCESS(builder.Build(indirectTable.Get(), EndOffset));

        VerifyBuildData(&builder, Expected, EndOffset);
    }

    // 全域がパッチ側
    {
        static const IndirectStorage::EntryData Entries[] =
        {
            { 0, 0, 1 },
        };

        IndirectTableCreator indirectTable(Entries, EndOffset);

        SparseStorageBuilder builder(BlockSize, EraseSize);
        NNT_ASSERT_RESULT_SUCCESS(builder.Build(indirectTable.Get(), EndOffset));

        VerifyBuildData(&builder, EndOffset);
    }
}

/**
 * @brief   空の IndirectStorage を利用してのリビルドをテストします。
 */
TEST_F(SparseStorageBuilderTest, RebuildEmptyStorage)
{
    static const IndirectStorage::EntryData EmptyEntries[] =
    {
        { 0, 0, 1 },
    };

    static const IndirectStorage::EntryData Entries[] =
    {
        {   0 * 1024,   0 * 1024, 0 },
        {  10 * 1024,  20 * 1024, 1 },
        {  30 * 1024,  61 * 1024, 0 },
        {  40 * 1024,  80 * 1024, 1 },
        {  50 * 1024,  22 * 1024, 0 },
        {  82 * 1024, 120 * 1024, 1 },
        { 100 * 1024,  98 * 1024, 0 },
    };
    static const int IndirectEndOffset = 104 * 1024;
    static const int OriginalEndOffset = 102 * 1024;
    static const int BlockSize = 4 * 1024;
    static const int EraseSize = BlockSize * 4;

    IndirectTableCreator indirectTable(Entries, IndirectEndOffset);
    IndirectTableCreator emptyTable(EmptyEntries, IndirectEndOffset);

    // 空でない -> 空にリビルド
    {
        SparseStorageBuilder builder(BlockSize, EraseSize);
        NNT_ASSERT_RESULT_SUCCESS(builder.Build(indirectTable.Get(), OriginalEndOffset));
        NNT_ASSERT_RESULT_SUCCESS(builder.Rebuild(emptyTable.Get(), OriginalEndOffset));
        VerifyBuildData(&builder, OriginalEndOffset);
    }

    // 空 -> 空でないにリビルド
    {
        SparseStorageBuilder builder(BlockSize, EraseSize);
        NNT_ASSERT_RESULT_SUCCESS(builder.Build(emptyTable.Get(), OriginalEndOffset));
        NNT_ASSERT_RESULT_SUCCESS(builder.Rebuild(indirectTable.Get(), OriginalEndOffset));
        VerifyBuildData(&builder, OriginalEndOffset);
    }
}

/**
 * @brief   排他的な領域を参照する IndirectStorage のリビルドテスト。
 */
TEST_F(SparseStorageBuilderTest, RebuildExclusiveRange)
{
    static const int IndirectEndOffset = 104 * 1024;
    static const int OriginalEndOffset = 102 * 1024;
    static const int BlockSize = 4 * 1024;
    static const int EraseSize = BlockSize;

    SparseStorageBuilder builder(BlockSize, EraseSize);
    {
        static const IndirectStorage::EntryData Entries[] =
        {
            {   0 * 1024,   0 * 1024, 0 },
            {  10 * 1024,  20 * 1024, 1 },
            {  30 * 1024,  61 * 1024, 0 },
            {  40 * 1024,  80 * 1024, 1 },
            {  50 * 1024,  22 * 1024, 0 },
            {  78 * 1024, 120 * 1024, 1 },
            { 100 * 1024,  98 * 1024, 0 },
        };

        IndirectTableCreator indirectTable(Entries, IndirectEndOffset);

        NNT_ASSERT_RESULT_SUCCESS(builder.Build(indirectTable.Get(), OriginalEndOffset));
    }
    {
        static const IndirectStorage::EntryData Entries[] =
        {
            {  0 * 1024,   0 * 1024, 1 },
            { 20 * 1024,  14 * 1024, 0 },
            { 26 * 1024,  20 * 1024, 1 },
            { 56 * 1024,  56 * 1024, 0 },
            { 60 * 1024,  60 * 1024, 1 },
            { 74 * 1024,  80 * 1024, 0 },
            { 84 * 1024, 100 * 1024, 1 },
        };

        IndirectTableCreator indirectTable(Entries, IndirectEndOffset);

        NNT_ASSERT_RESULT_SUCCESS(builder.Rebuild(indirectTable.Get(), OriginalEndOffset));
    }
    VerifyBuildData(&builder, OriginalEndOffset);
}
