﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// ビルドスイッチのため、例外的に先頭でインクルードする
#include <nn/fs/detail/fs_ResultHandlingUtilitySuppressRecordingEventOnUnsupportedPlatforms.h>

#include <nn/fs/fs_FileStorage.h>
#include <nn/fssystem/fs_NcaHeader.h>
#include <nn/fssystem/utilTool/fs_IndirectStorageBuilder.h>
#include <nn/fssystem/utilTool/fs_RelocatedIndirectStorageBuilder.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_TinyMt.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/fsUtil/testFs_util_allocator.h>
#include <nnt/nntest.h>

// パッチを準備する関数類
namespace
{
// オリジナルデータ を作成
nn::Result CreateOriginalFile(const char* dstPath, int64_t size, nn::Bit32 seed,
                              void* buffer, size_t bufferSize) NN_NOEXCEPT;

// 指定された場所にランダムデータを挿入し、バージョンアップされたデータを作成
nn::Result CreateNextVersionFile(const char* dstPath, const char* previousPath,
                                 int64_t fragmentInsertionPos, int64_t fragmentInsertionSize,
                                 nn::Bit32 seed,
                                 void* buffer, size_t bufferSize) NN_NOEXCEPT;
// パッチファイルを作成する
nn::Result CreatePatchFile(const char* dstPath,
                           const char* srcPreviousPath, const char* srcNextPath,
                           void* buffer, size_t bufferSize) NN_NOEXCEPT;

// 最適化パッチを作成する
nn::Result CreateOptimizedPatchFile(const char* dstPath, const char* srcPath,
                                    const char* previousPatchPath,
                                    const char* srcPreviousPath, const char* srcNextPath, // 検証用
                                    void* buffer, size_t bufferSize) NN_NOEXCEPT;

// ファイルの差分をブロック比較し、一致ブロック数, 総ブロック数を返す
nn::Result CalculateDifferentBlocks(
    int64_t* pOutSameBlocksCount, int64_t* pOutTotalBlocksCount,
    const char* lhsPath, const char* rhsPath, size_t blockSize) NN_NOEXCEPT;

// IndirectStorage をオープンする
nn::Result OpenIndirectStorage(std::shared_ptr<nn::fs::IStorage>* ppOutOriginalStorage,
                               std::shared_ptr<nn::fs::IStorage>* ppOutBaseStorage,
                               std::shared_ptr<nn::fssystem::IndirectStorage>* ppOutIndirectStorage,
                               nn::fs::FileHandle originalHandle,
                               nn::fs::FileHandle patchHandle) NN_NOEXCEPT;
// IndirectStorage をクローズする
nn::Result CloseIndirectStorage(std::shared_ptr<nn::fs::IStorage> pIndirectStorage);

// ストレージを比較する
nn::Result CompareStorages(
    bool* pOutIsSame, nn::fs::IStorage* pLhsStorage, nn::fs::IStorage* pRhsStorage) NN_NOEXCEPT;

// IndirectStorage を比較する
nn::Result CompareIndirectStorageFile(
    bool* pOutIsSame,
    const char* lhsPatchPath, const char* lhsOriginalPath,
    const char* rhsPatchPath, const char* rhsOriginalPath,
    size_t bufferSize) NN_NOEXCEPT;


// IndirectStorage のパラメータ
const size_t BlockSize = 16;
const size_t RegionSize = 16 * 1024;


} // namespace

namespace
{

// dataSrc からデータを bufferSize ずつとりながらファイルに書き込む
template<class PullNextDataT>
nn::Result WriteBuffered(nn::fs::FileHandle fileHandle, PullNextDataT pullNextData,
                     void* buffer, size_t bufferSize)
{
    NN_ASSERT_NOT_NULL(buffer);
    NN_ASSERT(bufferSize > 0);

    int64_t offset = 0;
    while (NN_STATIC_CONDITION(true))
    {
        size_t pulledSize = 0;
        NN_RESULT_DO(pullNextData(buffer, bufferSize, &pulledSize));
        if (pulledSize <= 0)
        {
            break;
        }

        nn::fs::WriteFile(fileHandle, offset, buffer, pulledSize,
                          nn::fs::WriteOption::MakeValue(0));

        offset += pulledSize;
    }

    NN_RESULT_DO(nn::fs::FlushFile(fileHandle));

    NN_RESULT_SUCCESS;
}

// オリジナルデータ を作成
nn::Result CreateOriginalFile(const char* dstPath, int64_t size, nn::Bit32 seed,
                              void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    // ファイル作成
    nn::fs::DeleteFile(dstPath);
    NN_RESULT_DO(nn::fs::CreateFile(dstPath, 0));

    nn::fs::FileHandle handle;

    // ファイルオープン
    NN_RESULT_DO(nn::fs::OpenFile(
                     &handle, dstPath,
                     nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    {
        nn::util::TinyMt mt;

        mt.Initialize(seed);

        NN_UTIL_SCOPE_EXIT
        {
            mt.Finalize();
        };

        int64_t offset = 0;

        // ファイル書き込み
        NN_RESULT_DO(
            WriteBuffered(
                handle,
                [size, &mt, &offset](void* dstBuffer, size_t bufferSize, size_t* pOutPulledSize)
                NN_NOEXCEPT
                -> nn::Result
                {
                    const int64_t writeSize64 = std::min<int64_t>(size - offset, bufferSize);
                    NN_ASSERT(nn::util::IsIntValueRepresentable<size_t>(writeSize64));
                    const size_t writeSize = static_cast<size_t>(writeSize64);
                    mt.GenerateRandomBytes(dstBuffer, writeSize);
                    offset += writeSize;
                    *pOutPulledSize = writeSize;

                    NN_RESULT_SUCCESS;
                },
                buffer, bufferSize));
    }

    NN_RESULT_SUCCESS;
}

// 指定された場所にランダムデータを挿入し、バージョンアップされたデータを作成
nn::Result CreateNextVersionFile(const char* dstPath, const char* previousPath,
                                 int64_t fragmentInsertionPos, int64_t fragmentInsertionSize,
                                 nn::Bit32 seed,
                                 void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    // ファイル作成
    nn::fs::DeleteFile(dstPath);
    NN_RESULT_DO(nn::fs::CreateFile(dstPath, 0));

    nn::fs::FileHandle dstHandle;
    nn::fs::FileHandle previousHandle;

    // ファイルオープン
    NN_RESULT_DO(nn::fs::OpenFile(
                     &dstHandle, dstPath,
                     nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(dstHandle);
    };
    NN_RESULT_DO(nn::fs::OpenFile(
                     &previousHandle, previousPath,
                     nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(previousHandle);
    };

    {
        nn::util::TinyMt mt;

        mt.Initialize(seed);

        NN_UTIL_SCOPE_EXIT
        {
            mt.Finalize();
        };

        {
            int64_t size;
            NN_RESULT_DO(nn::fs::GetFileSize(&size, previousHandle));
            NN_ASSERT(fragmentInsertionPos >= 0 && fragmentInsertionPos <= size);
        }

        int64_t offset = 0;
        int64_t previousFileOffset = 0;

        const int64_t fragmentInsertionBegin = fragmentInsertionPos;
        const int64_t fragmentInsertionEnd = fragmentInsertionPos + fragmentInsertionSize;

        auto pullNextData =
            [&mt, &offset, &previousFileOffset, previousHandle,
             fragmentInsertionBegin, fragmentInsertionEnd]
            (void* dstBuffer, size_t bufferSize, size_t* pOutPulledSize)
            NN_NOEXCEPT
            -> nn::Result
            {
                if (offset >= fragmentInsertionBegin && offset < fragmentInsertionEnd)
                {
                    // ランダムデータを挿入
                    const int64_t writeSize64 =
                        std::min<int64_t>(fragmentInsertionEnd - offset, bufferSize);

                    NN_ASSERT(nn::util::IsIntValueRepresentable<size_t>(writeSize64));
                    const size_t writeSize = static_cast<size_t>(writeSize64);
                    mt.GenerateRandomBytes(dstBuffer, writeSize);

                    offset += writeSize64;

                    *pOutPulledSize = writeSize;
                }
                else
                {
                    int64_t maxReadSize64;
                    if (offset < fragmentInsertionBegin)
                    {
                        maxReadSize64 = std::min<int64_t>(fragmentInsertionBegin - offset,
                                                          bufferSize);
                    }
                    else
                    {
                        maxReadSize64 = bufferSize;
                    }
                    NN_ASSERT(nn::util::IsIntValueRepresentable<size_t>(maxReadSize64));

                    // 元ファイルのデータを書き込み
                    size_t readSize;
                    NN_RESULT_DO(
                        nn::fs::ReadFile(&readSize, previousHandle, previousFileOffset,
                                         dstBuffer, static_cast<size_t>(maxReadSize64)));

                    previousFileOffset += readSize;

                    offset += readSize;

                    *pOutPulledSize = readSize;
                }

                NN_RESULT_SUCCESS;
            };

        // ファイル書き込み
        NN_RESULT_DO(
            WriteBuffered(
                dstHandle,
                pullNextData,
                buffer, bufferSize));
    }

    NN_RESULT_SUCCESS;
}

// パッチのデータの並びに合わせた SubStorage を作成する
void SetupSubStoragesForTestPatch(
    nn::fs::SubStorage* pOutDataStorage,
    nn::fs::SubStorage* pOutNodeStorage,
    nn::fs::SubStorage* pOutEntryStorage,
    int64_t dataStorageSize,
    int64_t nodeStorageSize,
    int64_t entryStorageSize,
    nn::fs::SubStorage baseStorage)
{
    NN_ASSERT_NOT_NULL(pOutDataStorage);
    NN_ASSERT_NOT_NULL(pOutNodeStorage);
    NN_ASSERT_NOT_NULL(pOutEntryStorage);

    int64_t patchDataOffset = 0;

    *pOutDataStorage = nn::fs::SubStorage(
        &baseStorage, patchDataOffset, dataStorageSize);
    patchDataOffset += dataStorageSize;

    *pOutNodeStorage = nn::fs::SubStorage(
        &baseStorage, patchDataOffset, nodeStorageSize);
    patchDataOffset += nodeStorageSize;

    *pOutEntryStorage = nn::fs::SubStorage(
        &baseStorage, patchDataOffset, entryStorageSize);
    patchDataOffset += entryStorageSize;
}


// パッチファイルを作成する
nn::Result CreatePatchFile(const char* dstPath,
                           const char* srcPreviousPath, const char* srcNextPath,
                           void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    // ファイル作成
    nn::fs::DeleteFile(dstPath);
    NN_RESULT_DO(nn::fs::CreateFile(dstPath, 0));

    // ファイルオープン
    nn::fs::FileHandle dstHandle;
    nn::fs::FileHandle srcPreviousHandle;
    nn::fs::FileHandle srcNextHandle;

    NN_RESULT_DO(nn::fs::OpenFile(
                     &dstHandle, dstPath,
                     nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(dstHandle);
    };

    NN_RESULT_DO(nn::fs::OpenFile(
                     &srcPreviousHandle, srcPreviousPath,
                     nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(srcPreviousHandle);
    };

    NN_RESULT_DO(nn::fs::OpenFile(
                     &srcNextHandle, srcNextPath,
                     nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(srcNextHandle);
    };

    nn::fs::FileHandleStorage srcPreviousStorage(srcPreviousHandle);
    nn::fs::FileHandleStorage srcNextStorage(srcNextHandle);

    {
        nn::fssystem::utilTool::IndirectStorageBuilder builder;

        NN_RESULT_DO(builder.Initialize(&srcPreviousStorage, &srcNextStorage, BlockSize));

        NN_UTIL_SCOPE_EXIT
        {
            builder.Finalize();
        };

        NN_RESULT_DO(builder.Build(BlockSize, RegionSize, 0));

        // ここで作るテスト用パッチファイルの構成:
        // NcaPatchInfo | データ| ノードテーブル | エントリーテーブル
        // の順に格納

        // NcaPatchInfo ヘッダを設定
        nn::fssystem::NcaPatchInfo ncaPatchInfo;
        std::memset(&ncaPatchInfo, 0, sizeof(ncaPatchInfo));
        ncaPatchInfo.indirectOffset = builder.QueryDataStorageSize();
        ncaPatchInfo.indirectSize = builder.QueryTableStorageSize();

        // 書き込み先ファイルのサイズを変更
        {
            const int64_t totalFileSize =
                sizeof(ncaPatchInfo)
                + builder.QueryTableStorageSize() + builder.QueryDataStorageSize();
            NN_RESULT_DO(nn::fs::SetFileSize(dstHandle, totalFileSize));
        }

        // 各種 書き込み対象のStorage を設定
        nn::fs::FileHandleStorage dstStorage(dstHandle);

        nn::fs::MemoryStorage ncaHeaderStorage(&ncaPatchInfo, sizeof(ncaPatchInfo));
        nn::fs::SubStorage headerStorage(
            &ncaHeaderStorage,
            offsetof(struct nn::fssystem::NcaPatchInfo, indirectHeader),
            static_cast<size_t>(nn::fssystem::NcaBucketInfo::HeaderSize));

        nn::fs::SubStorage baseStorage(
            &dstStorage,
            static_cast<int64_t>(sizeof(ncaPatchInfo)),
            builder.QueryTableStorageSize() + builder.QueryDataStorageSize());
        nn::fs::SubStorage dataStorage;
        nn::fs::SubStorage nodeStorage;
        nn::fs::SubStorage entryStorage;

        SetupSubStoragesForTestPatch(
            &dataStorage, &nodeStorage, &entryStorage,
            builder.QueryDataStorageSize(),
            builder.QueryTableNodeStorageSize(),
            builder.QueryTableEntryStorageSize(),
            baseStorage);

        // テーブル書き込み
        NN_RESULT_DO(builder.WriteTable(
            nnt::fs::util::GetTestLibraryAllocator(),
            headerStorage, nodeStorage, entryStorage));

        // ヘッダーを別途書き込み
        NN_RESULT_DO(dstStorage.Write(0, &ncaPatchInfo, sizeof(ncaPatchInfo)));

        // データ書き込み
        const int64_t dataSize = builder.QueryDataStorageSize();
        for (int64_t offset = 0; offset < dataSize;)
        {
            const int64_t readSize64 = std::min<int64_t>(bufferSize, dataSize - offset);
            NN_ASSERT(nn::util::IsIntValueRepresentable<size_t>(readSize64));
            const size_t readSize = static_cast<size_t>(readSize64);
            NN_RESULT_DO(builder.ReadData(offset, buffer, readSize));
            NN_RESULT_DO(dataStorage.Write(offset, buffer, readSize));

            offset += readSize64;
        }
        NN_RESULT_DO(dataStorage.Flush());
    }

    NN_RESULT_SUCCESS;
}

// 最適化パッチを作成する
nn::Result CreateOptimizedPatchFile(const char* dstPath, const char* srcPath,
                                    const char* previousPath,
                                    const char* srcPreviousPath, const char* srcNextPath,
                                    void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    // 書き込み先を作成
    nn::fs::DeleteFile(dstPath);
    NN_RESULT_DO(nn::fs::CreateFile(dstPath, 0));

    // パッチをオープンする
    nn::fs::FileHandle srcFileHandle;
    NN_RESULT_DO(nn::fs::OpenFile(&srcFileHandle, srcPath,
                                  nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(srcFileHandle);
    };

    nn::fs::FileHandle previousFileHandle;
    NN_RESULT_DO(nn::fs::OpenFile(&previousFileHandle, previousPath,
                                  nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(previousFileHandle);
    };

    nn::fs::FileHandle dstFileHandle;
    NN_RESULT_DO(nn::fs::OpenFile(&dstFileHandle, dstPath,
                                  nn::fs::OpenMode_Read | nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(dstFileHandle);
    };

    // src 側の PatchInfo を設定
    nn::fs::FileHandleStorage srcStorage(srcFileHandle);

    int64_t srcStorageSize;
    NN_RESULT_DO(srcStorage.GetSize(&srcStorageSize));

    nn::fssystem::NcaPatchInfo srcNcaPatchInfo;
    NN_RESULT_DO(srcStorage.Read(0, &srcNcaPatchInfo, sizeof(srcNcaPatchInfo)));
    nn::fs::SubStorage srcBaseStorage(&srcStorage, sizeof(srcNcaPatchInfo),
                                      srcStorageSize - sizeof(srcNcaPatchInfo));

    nn::fssystem::utilTool::RelocatedIndirectStorageBuilder::PatchInfo srcPatchInfo;
    NN_RESULT_DO(srcPatchInfo.Initialize(srcNcaPatchInfo, sizeof(srcNcaPatchInfo), &srcBaseStorage));

    // previous 側の PatchInfo を設定
    nn::fs::FileHandleStorage previousStorage(previousFileHandle);

    int64_t previousStorageSize;
    NN_RESULT_DO(previousStorage.GetSize(&previousStorageSize));

    nn::fssystem::NcaPatchInfo previousNcaPatchInfo;
    NN_RESULT_DO(previousStorage.Read(0, &previousNcaPatchInfo, sizeof(previousNcaPatchInfo)));
    nn::fs::SubStorage previousBaseStorage(&previousStorage, sizeof(previousNcaPatchInfo),
                                      previousStorageSize - sizeof(previousNcaPatchInfo));

    nn::fssystem::utilTool::RelocatedIndirectStorageBuilder::PatchInfo previousPatchInfo;
    NN_RESULT_DO(previousPatchInfo.Initialize(previousNcaPatchInfo, sizeof(previousNcaPatchInfo), &previousBaseStorage));

    // 出力用の情報
    const auto OutputDataStorageOffset = srcPatchInfo.GetStorageOffset();

    // 最適化する
    {
        nn::fssystem::utilTool::RelocatedIndirectStorageBuilder builder;
        NN_RESULT_DO(builder.Initialize(previousPatchInfo, srcPatchInfo, OutputDataStorageOffset));

        NN_UTIL_SCOPE_EXIT
        {
            builder.Finalize();
        };

        NN_RESULT_DO(builder.Build(
                         nnt::fs::util::GetTestLibraryAllocator(),
                         BlockSize,
                         BlockSize,
                         RegionSize,
                         0));

        // 書き込む
        // ここで作るテスト用パッチファイルの構成:
        // NcaPatchInfo | データ| ノードテーブル | エントリーテーブル
        // の順に格納

        // NcaPatchInfo ヘッダを設定
        nn::fssystem::NcaPatchInfo ncaPatchInfo;
        std::memset(&ncaPatchInfo, 0, sizeof(ncaPatchInfo));
        ncaPatchInfo.indirectOffset = builder.QueryDataStorageSize();
        ncaPatchInfo.indirectSize = builder.QueryTableStorageSize();

        // 書き込み先ファイルのサイズを変更
        {
            const int64_t totalFileSize =
                sizeof(ncaPatchInfo)
                + builder.QueryTableStorageSize() + builder.QueryDataStorageSize();
            NN_RESULT_DO(nn::fs::SetFileSize(dstFileHandle, totalFileSize));
        }

        // 各種 書き込み対象のStorage を設定
        nn::fs::FileHandleStorage dstStorage(dstFileHandle);

        nn::fs::MemoryStorage ncaHeaderStorage(&ncaPatchInfo, sizeof(ncaPatchInfo));
        nn::fs::SubStorage headerStorage(
            &ncaHeaderStorage,
            offsetof(struct nn::fssystem::NcaPatchInfo, indirectHeader),
            static_cast<size_t>(nn::fssystem::NcaBucketInfo::HeaderSize));

        nn::fs::SubStorage baseStorage(
            &dstStorage,
            static_cast<int64_t>(sizeof(ncaPatchInfo)),
            builder.QueryTableStorageSize() + builder.QueryDataStorageSize());
        nn::fs::SubStorage dataStorage;
        nn::fs::SubStorage nodeStorage;
        nn::fs::SubStorage entryStorage;

        SetupSubStoragesForTestPatch(
            &dataStorage, &nodeStorage, &entryStorage,
            builder.QueryDataStorageSize(),
            builder.QueryTableNodeStorageSize(),
            builder.QueryTableEntryStorageSize(),
            baseStorage);

        // テーブル書き込み
        NN_RESULT_DO(builder.WriteTable(
            nnt::fs::util::GetTestLibraryAllocator(),
            headerStorage, nodeStorage, entryStorage));

        // ヘッダーを別途書き込み
        NN_RESULT_DO(dstStorage.Write(0, &ncaPatchInfo, sizeof(ncaPatchInfo)));

        // データ書き込み
        const int64_t dataSize = builder.QueryDataStorageSize();
        for (int64_t offset = 0; offset < dataSize;)
        {
            const int64_t readSize64 = std::min<int64_t>(bufferSize, dataSize - offset);
            NN_ASSERT(nn::util::IsIntValueRepresentable<size_t>(readSize64));
            const size_t readSize = static_cast<size_t>(readSize64);
            NN_RESULT_DO(builder.ReadData(offset, buffer, readSize));
            NN_RESULT_DO(dataStorage.Write(offset, buffer, readSize));

            offset += readSize64;
        }
        NN_RESULT_DO(dataStorage.Flush());

        // 最適化した後の DataStorage を IndirectStorageBuilder で正常に読み出せる
        {
            nn::fs::FileHandle srcPreviousHandle;
            nn::fs::FileHandle srcNextHandle;

            NN_RESULT_DO(nn::fs::OpenFile(
                             &srcPreviousHandle, srcPreviousPath,
                             nn::fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(srcPreviousHandle);
            };

            NN_RESULT_DO(nn::fs::OpenFile(
                             &srcNextHandle, srcNextPath,
                             nn::fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(srcNextHandle);
            };

            nn::fs::FileHandleStorage srcPreviousStorage(srcPreviousHandle);
            nn::fs::FileHandleStorage srcNextStorage(srcNextHandle);

            nn::fssystem::utilTool::IndirectStorageBuilder builder2;
            NN_RESULT_DO(builder2.Initialize(&srcPreviousStorage, &srcNextStorage, BlockSize));

            NN_UTIL_SCOPE_EXIT
            {
                builder2.Finalize();
            };

            NN_RESULT_DO(builder2.Build(nnt::fs::util::GetTestLibraryAllocator(), headerStorage, nodeStorage, entryStorage));

            if( dataSize != builder2.QueryDataStorageSize() )
            {
                EXPECT_EQ(dataSize, builder2.QueryDataStorageSize());
                return nn::fs::ResultUnexpected();
            }

            nnt::fs::util::Vector<char> lhsBuffer(bufferSize);
            nnt::fs::util::Vector<char> rhsBuffer(bufferSize);

            for( int64_t offset = 0; offset < dataSize; )
            {
                const size_t readSize = static_cast<size_t>(std::min<int64_t>(dataSize - offset, bufferSize));
                NN_RESULT_DO(dataStorage.Read(offset, &lhsBuffer[0], readSize));
                NN_RESULT_DO(builder2.ReadData(offset, &rhsBuffer[0], readSize));
                if(std::memcmp(&lhsBuffer[0], &rhsBuffer[0], readSize) != 0)
                {
                    NN_LOG("offset %lld\n", offset);
                    nnt::fs::util::DumpBufferDiff(&lhsBuffer[0], &rhsBuffer[0], readSize);
                    return nn::fs::ResultUnexpected();
                }
                offset += readSize;
            }
        }
    }

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

// ファイルをブロックで比較する
nn::Result CalculateDifferentBlocks(
    int64_t* pOutSameBlocksCount, int64_t* pOutTotalBlocksCount,
    const char* lhsPath, const char* rhsPath, size_t blockSize) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutSameBlocksCount);
    NN_ASSERT_NOT_NULL(pOutTotalBlocksCount);

    nn::fs::FileHandle lhsHandle;
    nn::fs::FileHandle rhsHandle;

    NN_RESULT_DO(nn::fs::OpenFile(
                     &lhsHandle, lhsPath,
                     nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(lhsHandle);
    };

    NN_RESULT_DO(nn::fs::OpenFile(
                     &rhsHandle, rhsPath,
                     nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(rhsHandle);
    };

    int64_t lhsSize;
    NN_RESULT_DO(nn::fs::GetFileSize(&lhsSize, lhsHandle));
    int64_t rhsSize;
    NN_RESULT_DO(nn::fs::GetFileSize(&rhsSize, rhsHandle));

    const int64_t biggerSize = std::max(lhsSize, rhsSize);
    const int64_t compareSize = std::min(lhsSize, rhsSize);

    nnt::fs::util::Vector<char> lhsBlock(blockSize);
    nnt::fs::util::Vector<char> rhsBlock(blockSize);

    int64_t sameBlocksCount = 0;
    for(int64_t offset = 0; offset < compareSize; )
    {
        const size_t readSize =
            static_cast<size_t>(std::min<int64_t>(compareSize - offset, blockSize));

        NN_RESULT_DO(nn::fs::ReadFile(lhsHandle, offset, &lhsBlock[0], readSize));
        NN_RESULT_DO(nn::fs::ReadFile(rhsHandle, offset, &rhsBlock[0], readSize));

        if (std::memcmp(&lhsBlock[0], &rhsBlock[0], readSize) == 0)
        {
            ++sameBlocksCount;
        }

        offset += readSize;
    }

    *pOutSameBlocksCount = sameBlocksCount;
    *pOutTotalBlocksCount = (biggerSize + blockSize - 1) / blockSize;

    NN_RESULT_SUCCESS;
}

// IndirectStorage をオープンする
nn::Result OpenIndirectStorage(std::shared_ptr<nn::fs::IStorage>* ppOutOriginalStorage,
                               std::shared_ptr<nn::fs::IStorage>* ppOutBaseStorage,
                               std::shared_ptr<nn::fssystem::IndirectStorage>* ppOutIndirectStorage,
                               nn::fs::FileHandle originalHandle,
                               nn::fs::FileHandle patchHandle) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(ppOutIndirectStorage);
    NN_ASSERT_NOT_NULL(ppOutBaseStorage);

    std::shared_ptr<nn::fs::IStorage> pBaseStorage(
        new nn::fs::FileHandleStorage(patchHandle));
    NN_ASSERT_NOT_NULL(pBaseStorage);

    std::shared_ptr<nn::fs::IStorage> pOriginalStorage(
        new nn::fs::FileHandleStorage(originalHandle));
    NN_ASSERT_NOT_NULL(pOriginalStorage);

    int64_t originalStorageSize;
    NN_RESULT_DO(pOriginalStorage->GetSize(&originalStorageSize));

    std::shared_ptr<nn::fssystem::IndirectStorage> pIndirectStorage(
        new nn::fssystem::IndirectStorage());
    NN_ASSERT_NOT_NULL(pIndirectStorage);

    // ヘッダーを読み取り
    nn::fssystem::NcaPatchInfo ncaPatchInfo;
    NN_RESULT_DO(pBaseStorage->Read(0, &ncaPatchInfo, sizeof(ncaPatchInfo)));

    // ヘッダーを除いたストレージを作成
    int64_t baseStorageSize;
    NN_RESULT_DO(pBaseStorage->GetSize(&baseStorageSize));
    nn::fs::SubStorage baseStorageWithoutHeader(pBaseStorage.get(), sizeof(ncaPatchInfo),
                                                baseStorageSize - sizeof(ncaPatchInfo));

    nn::fssystem::utilTool::RelocatedIndirectStorageBuilder::PatchInfo patchInfo;
    NN_RESULT_DO(patchInfo.Initialize(ncaPatchInfo, 0, &baseStorageWithoutHeader));

    // IndirectStorage を初期化
    NN_RESULT_DO(pIndirectStorage->Initialize(
        nnt::fs::util::GetTestLibraryAllocator(),
        patchInfo.GetNodeStorage(),
        patchInfo.GetEntryStorage(),
        patchInfo.GetEntryCount()
    ));

    // Storage 追加
    pIndirectStorage->SetStorage(0, pOriginalStorage, 0, originalStorageSize);
    pIndirectStorage->SetStorage(1, patchInfo.GetDataStorage());

    *ppOutBaseStorage = std::move(pBaseStorage);
    *ppOutIndirectStorage = std::move(pIndirectStorage);
    *ppOutOriginalStorage = std::move(pOriginalStorage);

    NN_RESULT_SUCCESS;
}

nn::Result CloseIndirectStorage(nn::fssystem::IndirectStorage* pIndirectStorage)
{
    NN_ASSERT_NOT_NULL(pIndirectStorage);
    pIndirectStorage->Finalize();
    NN_RESULT_SUCCESS;
}

// ストレージを比較する
nn::Result CompareStorages(
    bool* pOutIsSame, nn::fs::IStorage* pLhsStorage, nn::fs::IStorage* pRhsStorage,
    size_t bufferSize) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutIsSame);
    NN_ASSERT_NOT_NULL(pLhsStorage);
    NN_ASSERT_NOT_NULL(pRhsStorage);

    int64_t lhsSize;
    NN_RESULT_DO(pLhsStorage->GetSize(&lhsSize));
    int64_t rhsSize;
    NN_RESULT_DO(pRhsStorage->GetSize(&rhsSize));

    if (lhsSize != rhsSize)
    {
        NN_LOG("Size %lld != %lld\n", lhsSize, rhsSize);
        *pOutIsSame = false;
        NN_RESULT_SUCCESS;
    }

    const int64_t compareSize = lhsSize;

    nnt::fs::util::Vector<char> lhsBuffer(bufferSize);
    nnt::fs::util::Vector<char> rhsBuffer(bufferSize);

    for(int64_t offset = 0; offset < compareSize; )
    {
        const size_t readSize =
            static_cast<size_t>(std::min<int64_t>(compareSize - offset, bufferSize));

        NN_RESULT_DO(pLhsStorage->Read(offset, &lhsBuffer[0], readSize));
        NN_RESULT_DO(pRhsStorage->Read(offset, &rhsBuffer[0], readSize));

        if (std::memcmp(&lhsBuffer[0], &rhsBuffer[0], readSize) != 0)
        {
            NN_LOG("offset %lld\n", offset);
            nnt::fs::util::DumpBufferDiff(&lhsBuffer[0], &rhsBuffer[0], readSize);
            *pOutIsSame = false;
            NN_RESULT_SUCCESS;
        }

        offset += readSize;
    }

    *pOutIsSame = true;
    NN_RESULT_SUCCESS;
}

// IndirectStorage を比較する
nn::Result CompareIndirectStorageFile(
    bool* pOutIsSame,
    const char* lhsPatchPath, const char* lhsOriginalPath,
    const char* rhsPatchPath, const char* rhsOriginalPath,
    size_t bufferSize) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutIsSame);

    // 各ファイルを開く
    nn::fs::FileHandle lhsOriginalHandle;
    nn::fs::FileHandle lhsPatchHandle;
    nn::fs::FileHandle rhsOriginalHandle;
    nn::fs::FileHandle rhsPatchHandle;

    NN_RESULT_DO(nn::fs::OpenFile(
                     &lhsOriginalHandle, lhsOriginalPath,
                     nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(lhsOriginalHandle);
    };

    NN_RESULT_DO(nn::fs::OpenFile(
                     &lhsPatchHandle, lhsPatchPath,
                     nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(lhsPatchHandle);
    };

    NN_RESULT_DO(nn::fs::OpenFile(
                     &rhsOriginalHandle, rhsOriginalPath,
                     nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(rhsOriginalHandle);
    };

    NN_RESULT_DO(nn::fs::OpenFile(
                     &rhsPatchHandle, rhsPatchPath,
                     nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(rhsPatchHandle);
    };

    // IndirectStorage をオープン
    {
        std::shared_ptr<nn::fs::IStorage> pLhsOriginalStorage;
        std::shared_ptr<nn::fs::IStorage> pLhsBaseStorage;
        std::shared_ptr<nn::fssystem::IndirectStorage> pLhsIndirectStorage;
        NN_RESULT_DO(OpenIndirectStorage(&pLhsOriginalStorage,
                                         &pLhsBaseStorage,
                                         &pLhsIndirectStorage,
                                         lhsOriginalHandle,
                                         lhsPatchHandle));
        NN_UTIL_SCOPE_EXIT
        {
            CloseIndirectStorage(pLhsIndirectStorage.get());
        };

        std::shared_ptr<nn::fs::IStorage> pRhsOriginalStorage;
        std::shared_ptr<nn::fs::IStorage> pRhsBaseStorage;
        std::shared_ptr<nn::fssystem::IndirectStorage> pRhsIndirectStorage;
        NN_RESULT_DO(OpenIndirectStorage(&pRhsOriginalStorage,
                                         &pRhsBaseStorage,
                                         &pRhsIndirectStorage,
                                         rhsOriginalHandle,
                                         rhsPatchHandle));
        NN_UTIL_SCOPE_EXIT
        {
            CloseIndirectStorage(pRhsIndirectStorage.get());
        };

        // ストレージを比較
        NN_RESULT_DO(CompareStorages(pOutIsSame,
                                     pLhsIndirectStorage.get(),
                                     pRhsIndirectStorage.get(),
                                     bufferSize));
    }
    NN_RESULT_SUCCESS;
}

class SeedGenerator
{
public:
    explicit SeedGenerator(nn::Bit32 seed) NN_NOEXCEPT
        : m_Random()
        , m_Seeds()
    {
        m_Random.Initialize(seed);
    }
    ~SeedGenerator() NN_NOEXCEPT
    {
        m_Random.Finalize();
    }

    nn::Bit32 Generate() NN_NOEXCEPT
    {
        for( ; ; )
        {
            const nn::Bit32 seed = static_cast<nn::Bit32>(m_Random.GenerateRandomU32());
            auto it = std::find(m_Seeds.begin(), m_Seeds.end(), seed);
            if( it == m_Seeds.end() )
            {
                m_Seeds.push_back(seed);
                return seed;
            }
        }
    }

private:
    nn::util::TinyMt m_Random;
    nnt::fs::util::Vector<nn::Bit32> m_Seeds;
};

} // namespace

// テストクラス
class RelocatedIndirectStorageBuilderTest : public ::testing::Test, public nnt::fs::util::PrepareWorkDirFixture
{
public:
    RelocatedIndirectStorageBuilderTest() NN_NOEXCEPT
    {
        // Temp フォルダを作成
        CreateWorkRootPath();
        m_TemporaryPath = GetWorkRootPath();
        m_TemporaryPath.append("\\");
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::MountHostRoot());
    }

    virtual ~RelocatedIndirectStorageBuilderTest() NN_NOEXCEPT NN_OVERRIDE
    {
        // 削除
        nn::fs::UnmountHostRoot();
        DeleteWorkRootPath();
    }

    const nnt::fs::util::String& GetTemporaryPath() const NN_NOEXCEPT
    {
        return m_TemporaryPath;
    }

private:
    nnt::fs::util::String m_TemporaryPath;
};

// 事前条件をチェックします
TEST_F(RelocatedIndirectStorageBuilderTest, Precondition)
{
    typedef nn::fssystem::utilTool::RelocatedIndirectStorageBuilder Builder;

    char buffer[4];
    nn::fs::MemoryStorage storage(buffer, sizeof(buffer));
    nn::fs::SubStorage subStorage(&storage, 0, sizeof(buffer));

    Builder::PatchInfo oldPatchInfo(1, 0, subStorage, subStorage, subStorage);
    Builder::PatchInfo newPatchInfo(1, 0, subStorage, subStorage, subStorage);

#if !defined(NN_SDK_BUILD_RELEASE)
    EXPECT_DEATH_IF_SUPPORTED(Builder().Initialize(oldPatchInfo, newPatchInfo, -1), "");
#endif

    const auto pAllocator = nnt::fs::util::GetTestLibraryAllocator();
    {
        Builder builder;
        builder.Initialize(oldPatchInfo, newPatchInfo, 0);
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultNullptrArgument, builder.Build(nullptr, 16, 16, 32, 64));
    }
    {
        Builder builder;
        builder.Initialize(oldPatchInfo, newPatchInfo, 0);
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidSize, builder.Build(pAllocator, 0, 16, 32, 64));
    }
    {
        Builder builder;
        builder.Initialize(oldPatchInfo, newPatchInfo, 0);
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidSize, builder.Build(pAllocator, 3, 16, 32, 64));
    }
    {
        Builder builder;
        builder.Initialize(oldPatchInfo, newPatchInfo, 0);
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidSize, builder.Build(pAllocator, 32, 16, 32, 64));
    }
    {
        Builder builder;
        builder.Initialize(oldPatchInfo, newPatchInfo, 0);
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidSize, builder.Build(pAllocator, 4, 4, 32, 64));
    }
    {
        Builder builder;
        builder.Initialize(oldPatchInfo, newPatchInfo, 0);
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidSize, builder.Build(pAllocator, 8, 9, 32, 64));
    }
    {
        Builder builder;
        builder.Initialize(oldPatchInfo, newPatchInfo, 0);
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidSize, builder.Build(pAllocator, 16, 16, 16, 64));
    }
    {
        Builder builder;
        builder.Initialize(oldPatchInfo, newPatchInfo, 0);
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidSize, builder.Build(pAllocator, 16, 16, 33, 64));
    }
    {
        Builder builder;
        builder.Initialize(oldPatchInfo, newPatchInfo, 0);
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidSize, builder.Build(pAllocator, 16, 16, 32, 63));
    }
}

// Optimized パッチを作成します
TEST_F(RelocatedIndirectStorageBuilderTest, CreateOptimizedPatch)
{
    SeedGenerator seedGenerator(static_cast<nn::Bit32>(nnt::fs::util::GetRandomSeed()));

    const nnt::fs::util::String v0Path = GetTemporaryPath() + nnt::fs::util::String("v0.bin");
    const size_t V0Size = 10 * 1024 * 1024;

    const nnt::fs::util::String v1Path = GetTemporaryPath() + nnt::fs::util::String("v1.bin");
    const size_t V1AddSize = 1 * 1024 * 1024;

    const nnt::fs::util::String v2Path = GetTemporaryPath() + nnt::fs::util::String("v2.bin");
    const size_t V2AddSize = 1 * 1024;

    const nnt::fs::util::String v1PatchPath =
        GetTemporaryPath() + nnt::fs::util::String("v1patch.bin");

    const nnt::fs::util::String v2PatchPath =
        GetTemporaryPath() + nnt::fs::util::String("v2patch.bin");

    const nnt::fs::util::String v2OptimizedPatchPath =
        GetTemporaryPath() + nnt::fs::util::String("v2optpatch.bin");

    const size_t BufferSize = 32 * 1024;
    nnt::fs::util::Vector<char> buffer(BufferSize);

    // v0, v1, v2 rom を作成
    // v0 rom を作成
    NNT_ASSERT_RESULT_SUCCESS(
        CreateOriginalFile(v0Path.c_str(), V0Size, seedGenerator.Generate(),
                           &buffer[0], BufferSize));
    // v1 rom を作成
    NNT_ASSERT_RESULT_SUCCESS(
        CreateNextVersionFile(v1Path.c_str(), v0Path.c_str(), V0Size / 2, V1AddSize,
                              seedGenerator.Generate(), &buffer[0], BufferSize));

    // v2 rom を作成(v1 rom で作られた差分の中央にデータを追加する)
    NNT_ASSERT_RESULT_SUCCESS(
        CreateNextVersionFile(v2Path.c_str(), v1Path.c_str(), V0Size / 2 + V1AddSize / 2, V2AddSize,
                              seedGenerator.Generate(), &buffer[0], BufferSize));

    // v0-v1, v0-v2 パッチを作成
    NNT_ASSERT_RESULT_SUCCESS(
        CreatePatchFile(v1PatchPath.c_str(),
                        v0Path.c_str(), v1Path.c_str(),
                        &buffer[0], BufferSize));
    NNT_ASSERT_RESULT_SUCCESS(
        CreatePatchFile(v2PatchPath.c_str(),
                        v0Path.c_str(), v2Path.c_str(),
                        &buffer[0], BufferSize));

    // v0-v2 Optimized パッチを作成
    NNT_ASSERT_RESULT_SUCCESS(CreateOptimizedPatchFile(
                                  v2OptimizedPatchPath.c_str(),
                                  v2PatchPath.c_str(), v1PatchPath.c_str(),
                                  v0Path.c_str(), v2Path.c_str(),
                                  &buffer[0], BufferSize));

    // Optimized パッチは v0-v1 パッチとの差分が減っていることを確認
    {
        const size_t ComparisonBlockSize = 1 * 1024;
        int64_t notOptimizedSameBlocksCount;
        int64_t notOptimizedTotalBlocksCount;
        NNT_ASSERT_RESULT_SUCCESS(
            CalculateDifferentBlocks(
                &notOptimizedSameBlocksCount, &notOptimizedTotalBlocksCount,
                v1PatchPath.c_str(), v2PatchPath.c_str(), ComparisonBlockSize));

        int64_t optimizedSameBlocksCount;
        int64_t optimizedTotalBlocksCount;
        NNT_ASSERT_RESULT_SUCCESS(
            CalculateDifferentBlocks(
                &optimizedSameBlocksCount, &optimizedTotalBlocksCount,
                v1PatchPath.c_str(), v2OptimizedPatchPath.c_str(), ComparisonBlockSize));

        NN_LOG("(notOptimizedSameBlocksCount, notOptimizedTotalBlocksCount) = (%d, %d)\n",
               notOptimizedSameBlocksCount, notOptimizedTotalBlocksCount);

        NN_LOG("(optimizedSameBlocksCount, optimizedTotalBlocksCount) = (%d, %d)\n",
               optimizedSameBlocksCount, optimizedTotalBlocksCount);

        ASSERT_EQ(notOptimizedTotalBlocksCount, optimizedTotalBlocksCount);
        EXPECT_LT(notOptimizedSameBlocksCount, optimizedSameBlocksCount);
    }


    // Optimized パッチの内容が v0-v2 パッチと変わらないことを確認
    {
        bool isSame = false;
        NNT_ASSERT_RESULT_SUCCESS(
            CompareIndirectStorageFile(
                &isSame,
                v2PatchPath.c_str(), v0Path.c_str(),
                v2OptimizedPatchPath.c_str(), v0Path.c_str(),
                BufferSize));
        EXPECT_TRUE(isSame);
    }
}

// previous 及び current の サイズが 0 のパッチを Optimize するテスト
TEST_F(RelocatedIndirectStorageBuilderTest, ZeroSizeOptimizedPatch)
{
    SeedGenerator seedGenerator(static_cast<nn::Bit32>(nnt::fs::util::GetRandomSeed()));

    const nnt::fs::util::String v0Path = GetTemporaryPath() + nnt::fs::util::String("v0.bin");
    const size_t V0Size = 10 * 1024 * 1024;

    // v0 v1 v2 を同じにする
    const nnt::fs::util::String v1Path = v0Path;
    const nnt::fs::util::String v2Path = v0Path;

    const nnt::fs::util::String v1PatchPath =
        GetTemporaryPath() + nnt::fs::util::String("v1patch.bin");

    const nnt::fs::util::String v2PatchPath =
        GetTemporaryPath() + nnt::fs::util::String("v2patch.bin");

    const nnt::fs::util::String v2OptimizedPatchPath =
        GetTemporaryPath() + nnt::fs::util::String("v2optpatch.bin");

    const size_t BufferSize = 32 * 1024;
    nnt::fs::util::Vector<char> buffer(BufferSize);

    // v0, v1, v2 rom を作成
    // v0 rom を作成
    NNT_ASSERT_RESULT_SUCCESS(
        CreateOriginalFile(v0Path.c_str(), V0Size, seedGenerator.Generate(),
                           &buffer[0], BufferSize));

    // v0-v1, v0-v2 パッチを作成
    NNT_ASSERT_RESULT_SUCCESS(
        CreatePatchFile(v1PatchPath.c_str(),
                        v0Path.c_str(), v1Path.c_str(),
                        &buffer[0], BufferSize));
    NNT_ASSERT_RESULT_SUCCESS(
        CreatePatchFile(v2PatchPath.c_str(),
                        v0Path.c_str(), v2Path.c_str(),
                        &buffer[0], BufferSize));

    // v0-v2 Optimized パッチを作成
    NNT_ASSERT_RESULT_SUCCESS(CreateOptimizedPatchFile(
                                  v2OptimizedPatchPath.c_str(),
                                  v2PatchPath.c_str(), v1PatchPath.c_str(),
                                  v0Path.c_str(), v2Path.c_str(),
                                  &buffer[0], BufferSize));

    // Optimized パッチは v0-v1 パッチとの差分が減っていることを確認
    {
        const size_t ComparisonBlockSize = 1 * 1024;
        int64_t notOptimizedSameBlocksCount;
        int64_t notOptimizedTotalBlocksCount;
        NNT_ASSERT_RESULT_SUCCESS(
            CalculateDifferentBlocks(
                &notOptimizedSameBlocksCount, &notOptimizedTotalBlocksCount,
                v1PatchPath.c_str(), v2PatchPath.c_str(), ComparisonBlockSize));

        int64_t optimizedSameBlocksCount;
        int64_t optimizedTotalBlocksCount;
        NNT_ASSERT_RESULT_SUCCESS(
            CalculateDifferentBlocks(
                &optimizedSameBlocksCount, &optimizedTotalBlocksCount,
                v1PatchPath.c_str(), v2OptimizedPatchPath.c_str(), ComparisonBlockSize));

        NN_LOG("(notOptimizedSameBlocksCount, notOptimizedTotalBlocksCount) = (%d, %d)\n",
               notOptimizedSameBlocksCount, notOptimizedTotalBlocksCount);

        NN_LOG("(optimizedSameBlocksCount, optimizedTotalBlocksCount) = (%d, %d)\n",
               optimizedSameBlocksCount, optimizedTotalBlocksCount);

        ASSERT_EQ(notOptimizedTotalBlocksCount, optimizedTotalBlocksCount);

        // データ領域のサイズは0なので一致するはず
        EXPECT_EQ(optimizedTotalBlocksCount, optimizedSameBlocksCount);
        EXPECT_EQ(notOptimizedSameBlocksCount, optimizedSameBlocksCount);
    }


    // Optimized パッチの内容が v0-v2 パッチと変わらないことを確認
    {
        bool isSame = false;
        NNT_ASSERT_RESULT_SUCCESS(
            CompareIndirectStorageFile(
                &isSame,
                v2PatchPath.c_str(), v0Path.c_str(),
                v2OptimizedPatchPath.c_str(), v0Path.c_str(),
                BufferSize));
        EXPECT_TRUE(isSame);
    }
}

// previous が 0  current が > 0 のサイズのパッチを Optimize するテスト
TEST_F(RelocatedIndirectStorageBuilderTest, IncreaseFromZeroSizeOptimizedPatch)
{
    SeedGenerator seedGenerator(static_cast<nn::Bit32>(nnt::fs::util::GetRandomSeed()));

    const nnt::fs::util::String v0Path = GetTemporaryPath() + nnt::fs::util::String("v0.bin");
    const size_t V0Size = 10 * 1024 * 1024;

    const nnt::fs::util::String v1Path = v0Path;

    const nnt::fs::util::String v2Path = GetTemporaryPath() + nnt::fs::util::String("v2.bin");
    const size_t V2AddSize = 1 * 1024;

    const nnt::fs::util::String v1PatchPath =
        GetTemporaryPath() + nnt::fs::util::String("v1patch.bin");

    const nnt::fs::util::String v2PatchPath =
        GetTemporaryPath() + nnt::fs::util::String("v2patch.bin");

    const nnt::fs::util::String v2OptimizedPatchPath =
        GetTemporaryPath() + nnt::fs::util::String("v2optpatch.bin");

    const size_t BufferSize = 32 * 1024;
    nnt::fs::util::Vector<char> buffer(BufferSize);

    // v0, v2 rom を作成
    // v0 rom を作成
    NNT_ASSERT_RESULT_SUCCESS(
        CreateOriginalFile(v0Path.c_str(), V0Size, seedGenerator.Generate(),
                           &buffer[0], BufferSize));

    // v2 rom を作成(v1 rom で作られた差分の中央にデータを追加する)
    NNT_ASSERT_RESULT_SUCCESS(
        CreateNextVersionFile(v2Path.c_str(), v1Path.c_str(), V0Size / 2, V2AddSize,
                              seedGenerator.Generate(), &buffer[0], BufferSize));

    // v0-v1, v0-v2 パッチを作成
    NNT_ASSERT_RESULT_SUCCESS(
        CreatePatchFile(v1PatchPath.c_str(),
                        v0Path.c_str(), v1Path.c_str(),
                        &buffer[0], BufferSize));
    NNT_ASSERT_RESULT_SUCCESS(
        CreatePatchFile(v2PatchPath.c_str(),
                        v0Path.c_str(), v2Path.c_str(),
                        &buffer[0], BufferSize));

    // v0-v2 Optimized パッチを作成
    NNT_ASSERT_RESULT_SUCCESS(CreateOptimizedPatchFile(
                                  v2OptimizedPatchPath.c_str(),
                                  v2PatchPath.c_str(), v1PatchPath.c_str(),
                                  v0Path.c_str(), v2Path.c_str(),
                                  &buffer[0], BufferSize));

    // Optimized パッチは v0-v1 パッチとの差分が減っていることを確認
    {
        const size_t ComparisonBlockSize = 1 * 1024;
        int64_t notOptimizedSameBlocksCount;
        int64_t notOptimizedTotalBlocksCount;
        NNT_ASSERT_RESULT_SUCCESS(
            CalculateDifferentBlocks(
                &notOptimizedSameBlocksCount, &notOptimizedTotalBlocksCount,
                v1PatchPath.c_str(), v2PatchPath.c_str(), ComparisonBlockSize));

        int64_t optimizedSameBlocksCount;
        int64_t optimizedTotalBlocksCount;
        NNT_ASSERT_RESULT_SUCCESS(
            CalculateDifferentBlocks(
                &optimizedSameBlocksCount, &optimizedTotalBlocksCount,
                v1PatchPath.c_str(), v2OptimizedPatchPath.c_str(), ComparisonBlockSize));

        NN_LOG("(notOptimizedSameBlocksCount, notOptimizedTotalBlocksCount) = (%d, %d)\n",
               notOptimizedSameBlocksCount, notOptimizedTotalBlocksCount);

        NN_LOG("(optimizedSameBlocksCount, optimizedTotalBlocksCount) = (%d, %d)\n",
               optimizedSameBlocksCount, optimizedTotalBlocksCount);

        ASSERT_EQ(notOptimizedTotalBlocksCount, optimizedTotalBlocksCount);
        // 同じはず(全て新規データなので最適化の意味はない)
        EXPECT_EQ(notOptimizedSameBlocksCount, optimizedSameBlocksCount);
    }


    // Optimized パッチの内容が v0-v2 パッチと変わらないことを確認
    {
        bool isSame = false;
        NNT_ASSERT_RESULT_SUCCESS(
            CompareIndirectStorageFile(
                &isSame,
                v2PatchPath.c_str(), v0Path.c_str(),
                v2OptimizedPatchPath.c_str(), v0Path.c_str(),
                BufferSize));
        EXPECT_TRUE(isSame);
    }
}

// previous が > 0 current が 0 のサイズのパッチを Optimized パッチを作成します
TEST_F(RelocatedIndirectStorageBuilderTest, DecreaseToZeroSizeOptimizedPatch)
{
    SeedGenerator seedGenerator(static_cast<nn::Bit32>(nnt::fs::util::GetRandomSeed()));

    const nnt::fs::util::String v0Path = GetTemporaryPath() + nnt::fs::util::String("v0.bin");
    const size_t V0Size = 10 * 1024 * 1024;

    const nnt::fs::util::String v1Path = GetTemporaryPath() + nnt::fs::util::String("v1.bin");
    const size_t V1AddSize = 1 * 1024 * 1024;

    const nnt::fs::util::String v2Path = v0Path;

    const nnt::fs::util::String v1PatchPath =
        GetTemporaryPath() + nnt::fs::util::String("v1patch.bin");

    const nnt::fs::util::String v2PatchPath =
        GetTemporaryPath() + nnt::fs::util::String("v2patch.bin");

    const nnt::fs::util::String v2OptimizedPatchPath =
        GetTemporaryPath() + nnt::fs::util::String("v2optpatch.bin");

    const size_t BufferSize = 32 * 1024;
    nnt::fs::util::Vector<char> buffer(BufferSize);

    // v0, v1 rom を作成
    // v0 rom を作成
    NNT_ASSERT_RESULT_SUCCESS(
        CreateOriginalFile(v0Path.c_str(), V0Size, seedGenerator.Generate(),
                           &buffer[0], BufferSize));
    // v1 rom を作成
    NNT_ASSERT_RESULT_SUCCESS(
        CreateNextVersionFile(v1Path.c_str(), v0Path.c_str(), V0Size / 2, V1AddSize,
                              seedGenerator.Generate(), &buffer[0], BufferSize));

    // v0-v1, v0-v2 パッチを作成
    NNT_ASSERT_RESULT_SUCCESS(
        CreatePatchFile(v1PatchPath.c_str(),
                        v0Path.c_str(), v1Path.c_str(),
                        &buffer[0], BufferSize));
    NNT_ASSERT_RESULT_SUCCESS(
        CreatePatchFile(v2PatchPath.c_str(),
                        v0Path.c_str(), v2Path.c_str(),
                        &buffer[0], BufferSize));

    // v0-v2 Optimized パッチを作成
    NNT_ASSERT_RESULT_SUCCESS(CreateOptimizedPatchFile(
                                  v2OptimizedPatchPath.c_str(),
                                  v2PatchPath.c_str(), v1PatchPath.c_str(),
                                  v0Path.c_str(), v2Path.c_str(),
                                  &buffer[0], BufferSize));

    // Optimized パッチは v0-v1 パッチとの差分が減っていることを確認
    {
        const size_t ComparisonBlockSize = 1 * 1024;
        int64_t notOptimizedSameBlocksCount;
        int64_t notOptimizedTotalBlocksCount;
        NNT_ASSERT_RESULT_SUCCESS(
            CalculateDifferentBlocks(
                &notOptimizedSameBlocksCount, &notOptimizedTotalBlocksCount,
                v1PatchPath.c_str(), v2PatchPath.c_str(), ComparisonBlockSize));

        int64_t optimizedSameBlocksCount;
        int64_t optimizedTotalBlocksCount;
        NNT_ASSERT_RESULT_SUCCESS(
            CalculateDifferentBlocks(
                &optimizedSameBlocksCount, &optimizedTotalBlocksCount,
                v1PatchPath.c_str(), v2OptimizedPatchPath.c_str(), ComparisonBlockSize));

        NN_LOG("(notOptimizedSameBlocksCount, notOptimizedTotalBlocksCount) = (%d, %d)\n",
               notOptimizedSameBlocksCount, notOptimizedTotalBlocksCount);

        NN_LOG("(optimizedSameBlocksCount, optimizedTotalBlocksCount) = (%d, %d)\n",
               optimizedSameBlocksCount, optimizedTotalBlocksCount);

        // データ領域のサイズは 0 なので同じはず
        ASSERT_EQ(notOptimizedTotalBlocksCount, optimizedTotalBlocksCount);
        EXPECT_EQ(notOptimizedSameBlocksCount, optimizedSameBlocksCount);
    }


    // Optimized パッチの内容が v0-v2 パッチと変わらないことを確認
    {
        bool isSame = false;
        NNT_ASSERT_RESULT_SUCCESS(
            CompareIndirectStorageFile(
                &isSame,
                v2PatchPath.c_str(), v0Path.c_str(),
                v2OptimizedPatchPath.c_str(), v0Path.c_str(),
                BufferSize));
        EXPECT_TRUE(isSame);
    }
}

namespace {

typedef nn::fssystem::utilTool::IndirectStorageBuilder IndirectStorageBuilder;
typedef nn::fssystem::utilTool::RelocatedIndirectStorageBuilder RelocatedIndirectStorageBuilder;

template< typename TBuilder >
class PatchInfoBuilderBase
{
private:
    typedef RelocatedIndirectStorageBuilder::PatchInfo PatchInfo;

public:
    virtual ~PatchInfoBuilderBase() NN_NOEXCEPT
    {
        m_Builder.Finalize();
    }

    nn::Result Setup() NN_NOEXCEPT
    {
        NN_ASSERT_EQUAL(
            m_Builder.QueryTableHeaderStorageSize(),
            static_cast<int64_t>(sizeof(nn::fssystem::BucketTree::Header))
        );

        const auto nodeSize = m_Builder.QueryTableNodeStorageSize();
        const auto entrySize = m_Builder.QueryTableEntryStorageSize();

        const auto tableSize = nodeSize + entrySize;
        m_TableBuffer.reset(new char[static_cast<size_t>(tableSize)]);
        NN_RESULT_THROW_UNLESS(
            m_TableBuffer != nullptr, nn::fs::ResultAllocationMemoryFailedNew());
        m_pTableStorage.reset(new nn::fs::MemoryStorage(m_TableBuffer.get(), tableSize));
        NN_RESULT_THROW_UNLESS(
            m_pTableStorage != nullptr, nn::fs::ResultAllocationMemoryFailedNew());

        NN_RESULT_DO(SetupDataStorage());

        nn::fssystem::BucketTree::Header header;
        nn::fs::MemoryStorage headerStorage(&header, sizeof(header));
        nn::fs::SubStorage nodeStorage(m_pTableStorage.get(), 0, nodeSize);
        nn::fs::SubStorage entryStorage(m_pTableStorage.get(), nodeSize, entrySize);

        NN_RESULT_DO(m_Builder.WriteTable(
            nnt::fs::util::GetTestLibraryAllocator(),
            nn::fs::SubStorage(&headerStorage, 0, sizeof(header)),
            nodeStorage,
            entryStorage
        ));

        const auto dataSize = GetBuilder().QueryDataStorageSize();
        m_PatchInfo = RelocatedIndirectStorageBuilder::PatchInfo(
            header.entryCount,
            0,
            nodeStorage,
            entryStorage,
            nn::fs::SubStorage(GetDataStorage(), 0, dataSize)
        );

        NN_RESULT_DO(Verify(&headerStorage));

        NN_RESULT_SUCCESS;
    }

    TBuilder& GetBuilder() NN_NOEXCEPT
    {
        return m_Builder;
    }

    const TBuilder& GetBuilder() const NN_NOEXCEPT
    {
        return m_Builder;
    }

    virtual nn::fs::IStorage* GetDataStorage() NN_NOEXCEPT = 0;

    PatchInfo& GetPatchInfo() NN_NOEXCEPT
    {
        return m_PatchInfo;
    }

    const PatchInfo& GetPatchInfo() const NN_NOEXCEPT
    {
        return m_PatchInfo;
    }

protected:
    virtual nn::Result SetupDataStorage() NN_NOEXCEPT = 0;

    virtual nn::Result Verify(nn::fs::IStorage*) NN_NOEXCEPT
    {
        NN_RESULT_SUCCESS;
    }

private:
    TBuilder m_Builder;
    std::unique_ptr<char[]> m_TableBuffer;
    std::unique_ptr<nn::fs::MemoryStorage> m_pTableStorage;
    PatchInfo m_PatchInfo;
};

template< typename TBuilder >
class PatchInfoBuilder : public PatchInfoBuilderBase<TBuilder>
{
public:
    template< typename TInitializer >
    nn::Result Initialize(TInitializer&& initializer) NN_NOEXCEPT
    {
        NN_RESULT_DO(initializer(&PatchInfoBuilderBase<TBuilder>::GetBuilder()));

        NN_RESULT_DO(PatchInfoBuilderBase<TBuilder>::Setup());

        NN_RESULT_SUCCESS;
    }

    template< typename TInitializer, typename TInitializerForTest >
    nn::Result Initialize(TInitializer&& initializer, TInitializerForTest&& initializerForTest) NN_NOEXCEPT
    {
        NN_RESULT_DO(initializer(&PatchInfoBuilderBase<TBuilder>::GetBuilder()));
        NN_RESULT_DO(initializerForTest(&m_BuilderForTest));

        NN_RESULT_DO(PatchInfoBuilderBase<TBuilder>::Setup());

        NN_RESULT_SUCCESS;
    }

    const char* GetData() const NN_NOEXCEPT
    {
        return m_DataBuffer.get();
    }

    size_t GetDataSize() const NN_NOEXCEPT
    {
        int64_t size;
        m_pDataStorage->GetSize(&size); // 失敗しないので返り値はチェックしない
        return static_cast<size_t>(size);
    }

    virtual nn::fs::IStorage* GetDataStorage() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pDataStorage.get();
    }

protected:
    virtual nn::Result SetupDataStorage() NN_NOEXCEPT NN_OVERRIDE
    {
        const auto dataSize = PatchInfoBuilderBase<TBuilder>::GetBuilder().QueryDataStorageSize();
        m_DataBuffer.reset(new char[static_cast<size_t>(dataSize)]);
        NN_RESULT_THROW_UNLESS(
            m_DataBuffer != nullptr, nn::fs::ResultAllocationMemoryFailedNew());
        m_pDataStorage.reset(new nn::fs::MemoryStorage(m_DataBuffer.get(), dataSize));
        NN_RESULT_THROW_UNLESS(
            m_pDataStorage != nullptr, nn::fs::ResultAllocationMemoryFailedNew());

        NN_RESULT_DO(PatchInfoBuilderBase<TBuilder>::GetBuilder().ReadData(0, m_DataBuffer.get(), static_cast<size_t>(dataSize)));

        NN_RESULT_SUCCESS;
    }

    virtual nn::Result Verify(nn::fs::IStorage* pHeaderStorage) NN_NOEXCEPT
    {
        // 最適化した後の DataStorage を IndirectStorageBuilder で正常に読み出せる
        if( m_BuilderForTest.IsInitialized() )
        {
            auto subHeaderStorage = nn::fs::SubStorage(pHeaderStorage, 0, sizeof(nn::fssystem::BucketTree::Header));
            NN_RESULT_DO(m_BuilderForTest.Build(
                nnt::fs::util::GetTestLibraryAllocator(),
                subHeaderStorage,
                PatchInfoBuilderBase<TBuilder>::GetPatchInfo().GetNodeStorage(),
                PatchInfoBuilderBase<TBuilder>::GetPatchInfo().GetEntryStorage()
            ));

            const auto dataSize = PatchInfoBuilderBase<TBuilder>::GetBuilder().QueryDataStorageSize();
            if( dataSize != m_BuilderForTest.QueryDataStorageSize() )
            {
                EXPECT_EQ(dataSize, m_BuilderForTest.QueryDataStorageSize());
                NN_RESULT_THROW(nn::fs::ResultUnexpected());
            }

            auto testBuffer = std::unique_ptr<char[]>(new char[static_cast<size_t>(dataSize)]);
            NN_RESULT_THROW_UNLESS(
                testBuffer != nullptr, nn::fs::ResultAllocationMemoryFailedNew());
            NN_RESULT_DO(m_BuilderForTest.ReadData(0, testBuffer.get(), static_cast<size_t>(dataSize), m_pDataStorage.get()));

            NNT_FS_UTIL_RESULT_THROW_MEMCMPEQ(
                m_DataBuffer.get(), testBuffer.get(), static_cast<size_t>(dataSize), nn::fs::ResultUnexpected());
        }

        NN_RESULT_SUCCESS;
    }

private:
    nn::fssystem::utilTool::IndirectStorageBuilder m_BuilderForTest;
    std::unique_ptr<char[]> m_DataBuffer;
    std::unique_ptr<nn::fs::MemoryStorage> m_pDataStorage;

};

void ReductionForwardOptimizedPatchImpl(size_t v1RealDataSize) NN_NOEXCEPT
{
    static const size_t TestRegionSize = 16 * 128;
    static const size_t V1DataSize = TestRegionSize * 4;
    static const size_t V2DataSize = TestRegionSize * 3;

    // { 0 }
    char v0Data[TestRegionSize] = {};

    // { 0, 1, 2, 3, ... }
    std::unique_ptr<char[]> v1Data(new char[v1RealDataSize]);
    ASSERT_NE(v1Data, nullptr);
    std::memset(v1Data.get() + TestRegionSize * 0, 0, TestRegionSize);
    std::memset(v1Data.get() + TestRegionSize * 1, 1, TestRegionSize);
    std::memset(v1Data.get() + TestRegionSize * 2, 2, TestRegionSize);
    std::memset(v1Data.get() + TestRegionSize * 3, 3, TestRegionSize);

    size_t v1PatchDataSize = TestRegionSize * 3;
    if( V1DataSize < v1RealDataSize )
    {
        std::memset(v1Data.get() + TestRegionSize * 4, 4, v1RealDataSize - TestRegionSize * 4);

        v1PatchDataSize = v1RealDataSize - TestRegionSize;
    }

    // { 0, 2, 3 }
    std::unique_ptr<char[]> v2Data(new char[V2DataSize]);
    ASSERT_NE(v2Data, nullptr);
    std::memset(v2Data.get() + TestRegionSize * 0, 0, TestRegionSize);
    std::memset(v2Data.get() + TestRegionSize * 1, 2, TestRegionSize);
    std::memset(v2Data.get() + TestRegionSize * 2, 3, TestRegionSize);

    nn::fs::MemoryStorage v0Storage(v0Data, TestRegionSize);
    nn::fs::MemoryStorage v1Storage(v1Data.get(), v1RealDataSize);
    nn::fs::MemoryStorage v2Storage(v2Data.get(), V2DataSize);

    // v0/v1 パッチを作成
    PatchInfoBuilder<IndirectStorageBuilder> v0v1Patch;
    NNT_ASSERT_RESULT_SUCCESS(v0v1Patch.Initialize(
        [&](IndirectStorageBuilder* pBuilder) NN_NOEXCEPT -> nn::Result
        {
            auto result = pBuilder->Initialize(&v0Storage, &v1Storage, BlockSize);
            if( result.IsSuccess() )
            {
                result = pBuilder->Build(
                    BlockSize,
                    BlockSize,
                    TestRegionSize,
                    TestRegionSize,
                    std::numeric_limits<int64_t>::max());
            }
            return result;
        }
    ));

    // { 1, 2, 3, ... }
    ASSERT_EQ(v0v1Patch.GetDataSize(), v1PatchDataSize);
    NNT_FS_UTIL_ASSERT_MEMCMPEQ(
        v0v1Patch.GetData(), v1Data.get() + TestRegionSize, TestRegionSize * 3);
    if( V1DataSize < v1RealDataSize )
    {
        NNT_FS_UTIL_ASSERT_MEMCMPEQ(
            v0v1Patch.GetData() + TestRegionSize * 3,
            v1Data.get() + TestRegionSize * 4,
            TestRegionSize * 3
        );
    }

    // v0/v2 パッチを作成
    PatchInfoBuilder<IndirectStorageBuilder> v0v2Patch;
    NNT_ASSERT_RESULT_SUCCESS(v0v2Patch.Initialize(
        [&](IndirectStorageBuilder* pBuilder) NN_NOEXCEPT -> nn::Result
        {
            auto result = pBuilder->Initialize(&v0Storage, &v2Storage, BlockSize);
            if( result.IsSuccess() )
            {
                result = pBuilder->Build(
                    BlockSize,
                    BlockSize,
                    TestRegionSize,
                    TestRegionSize,
                    std::numeric_limits<int64_t>::max());
            }
            return result;
        }
    ));

    // { 2, 3 }
    ASSERT_EQ(v0v2Patch.GetDataSize(), TestRegionSize * 2);
    NNT_FS_UTIL_ASSERT_MEMCMPEQ(
        v0v2Patch.GetData(), v2Data.get() + TestRegionSize, TestRegionSize * 2);

    // 最適化済みパッチを作成
    PatchInfoBuilder<RelocatedIndirectStorageBuilder> optimizedPatch;
    NNT_ASSERT_RESULT_SUCCESS(optimizedPatch.Initialize(
        [&](RelocatedIndirectStorageBuilder* pBuilder) NN_NOEXCEPT -> nn::Result
        {
            auto result = pBuilder->Initialize(
                v0v1Patch.GetPatchInfo(),
                v0v2Patch.GetPatchInfo(),
                v0v2Patch.GetPatchInfo().GetStorageOffset()
            );
            if( result.IsSuccess() )
            {
                result = pBuilder->Build(
                    nnt::fs::util::GetTestLibraryAllocator(), BlockSize, TestRegionSize);
            }
            return result;
        },
        [&](IndirectStorageBuilder* pBuilderForTest) NN_NOEXCEPT -> nn::Result
        {
            return pBuilderForTest->Initialize(&v0Storage, &v2Storage, BlockSize);
        }
    ));

    // { 1, 2, 3 }
    ASSERT_EQ(optimizedPatch.GetDataSize(), TestRegionSize * 3);
    NNT_FS_UTIL_ASSERT_MEMCMPEQ(
        optimizedPatch.GetData(), v0v1Patch.GetData(), TestRegionSize * 3);
}

}

// previous の先頭を削除した current で意図した最適化済みパッチになるかテスト
TEST_F(RelocatedIndirectStorageBuilderTest, ReductionForwardOptimizedPatch)
{
    ReductionForwardOptimizedPatchImpl(2048 * 4);
}

// previous の先頭を削除した current で意図した最適化済みパッチになるかテスト（100MB 越え）
TEST_F(RelocatedIndirectStorageBuilderTest, ReductionForwardOptimizedPatchLarge)
{
    ReductionForwardOptimizedPatchImpl(128 * 1024 * 1024);
}

// previous の先頭を削除・新規データを追加した current で意図した最適化済みパッチになるかテスト
TEST_F(RelocatedIndirectStorageBuilderTest, RemoveAndAddOptimizedPatch)
{
    static const size_t TestRegionSize = 16 * 128;
    static const size_t V1DataSize = TestRegionSize * 4;
    static const size_t V2DataSize = TestRegionSize * 4;

    // data[0]
    char v0Data[TestRegionSize] = {};

    // data[0, 1, 2, 3]
    std::unique_ptr<char[]> v1Data(new char[V1DataSize]);
    ASSERT_NE(v1Data, nullptr);
    std::memset(v1Data.get() + TestRegionSize * 0, 0, TestRegionSize);
    std::memset(v1Data.get() + TestRegionSize * 1, 1, TestRegionSize);
    std::memset(v1Data.get() + TestRegionSize * 2, 2, TestRegionSize);
    std::memset(v1Data.get() + TestRegionSize * 3, 3, TestRegionSize);

    // data[0, 2, 3, 4]
    std::unique_ptr<char[]> v2Data(new char[V2DataSize]);
    ASSERT_NE(v2Data, nullptr);
    std::memset(v2Data.get() + TestRegionSize * 0, 0, TestRegionSize);
    std::memset(v2Data.get() + TestRegionSize * 1, 2, TestRegionSize);
    std::memset(v2Data.get() + TestRegionSize * 2, 3, TestRegionSize);
    std::memset(v2Data.get() + TestRegionSize * 3, 4, TestRegionSize);

    nn::fs::MemoryStorage v0Storage(v0Data, TestRegionSize);
    nn::fs::MemoryStorage v1Storage(v1Data.get(), V1DataSize);
    nn::fs::MemoryStorage v2Storage(v2Data.get(), V2DataSize);

    // v0/v1 パッチを作成
    PatchInfoBuilder<IndirectStorageBuilder> v0v1Patch;
    NNT_ASSERT_RESULT_SUCCESS(v0v1Patch.Initialize(
        [&](IndirectStorageBuilder* pBuilder) NN_NOEXCEPT -> nn::Result
        {
            auto result = pBuilder->Initialize(&v0Storage, &v1Storage, BlockSize);
            if( result.IsSuccess() )
            {
                result = pBuilder->Build(BlockSize, TestRegionSize, 0);
            }
            return result;
        }
    ));

    // data[1, 2, 3]
    ASSERT_EQ(v0v1Patch.GetDataSize(), V1DataSize - TestRegionSize);
    NNT_FS_UTIL_ASSERT_MEMCMPEQ(
        v0v1Patch.GetData(), v1Data.get() + TestRegionSize, TestRegionSize * 3);

    // v0/v2 パッチを作成
    PatchInfoBuilder<IndirectStorageBuilder> v0v2Patch;
    NNT_ASSERT_RESULT_SUCCESS(v0v2Patch.Initialize(
        [&](IndirectStorageBuilder* pBuilder) NN_NOEXCEPT -> nn::Result
        {
            auto result = pBuilder->Initialize(&v0Storage, &v2Storage, BlockSize);
            if( result.IsSuccess() )
            {
                result = pBuilder->Build(BlockSize, TestRegionSize, 0);
            }
            return result;
        }
    ));

    // data[2, 3, 4]
    ASSERT_EQ(v0v2Patch.GetDataSize(), TestRegionSize * 3);
    NNT_FS_UTIL_ASSERT_MEMCMPEQ(
        v0v2Patch.GetData(), v2Data.get() + TestRegionSize, TestRegionSize * 3);

    // 最適化済みパッチを作成
    PatchInfoBuilder<RelocatedIndirectStorageBuilder> optimizedPatch;
    NNT_ASSERT_RESULT_SUCCESS(optimizedPatch.Initialize(
        [&](RelocatedIndirectStorageBuilder* pBuilder) NN_NOEXCEPT -> nn::Result
        {
            auto result = pBuilder->Initialize(
                v0v1Patch.GetPatchInfo(),
                v0v2Patch.GetPatchInfo(),
                v0v2Patch.GetPatchInfo().GetStorageOffset()
            );
            if( result.IsSuccess() )
            {
                result = pBuilder->Build(
                    nnt::fs::util::GetTestLibraryAllocator(), BlockSize, TestRegionSize);
            }
            return result;
        },
        [&](IndirectStorageBuilder* pBuilderForTest) NN_NOEXCEPT -> nn::Result
        {
            return pBuilderForTest->Initialize(&v0Storage, &v2Storage, BlockSize);
        }
    ));

    // data[1, 2, 3, 4]
    static const size_t PatchDataSize = TestRegionSize * 4;
    std::unique_ptr<char[]> patchData(new char[PatchDataSize]);
    ASSERT_NE(patchData, nullptr);
    std::memset(patchData.get() + TestRegionSize * 0, 1, TestRegionSize);
    std::memset(patchData.get() + TestRegionSize * 1, 2, TestRegionSize);
    std::memset(patchData.get() + TestRegionSize * 2, 3, TestRegionSize);
    std::memset(patchData.get() + TestRegionSize * 3, 4, TestRegionSize);

    ASSERT_EQ(optimizedPatch.GetDataSize(), PatchDataSize);
    NNT_FS_UTIL_ASSERT_MEMCMPEQ(
        optimizedPatch.GetData(), patchData.get(), PatchDataSize);
}

// デフラグして意図した最適化済みパッチになるかテスト
TEST_F(RelocatedIndirectStorageBuilderTest, DefragmentPatch)
{
    static const size_t TestRegionSize = 16 * 128;
    static const size_t V1DataSize = TestRegionSize * 4;
    static const size_t V2DataSize = TestRegionSize * 4;

    // data[0]
    char v0Data[TestRegionSize] = {};

    // data[0, 1, 2, 3]
    std::unique_ptr<char[]> v1Data(new char[V1DataSize]);
    ASSERT_NE(v1Data, nullptr);
    std::memset(v1Data.get() + TestRegionSize * 0, 0, TestRegionSize);
    std::memset(v1Data.get() + TestRegionSize * 1, 1, TestRegionSize);
    std::memset(v1Data.get() + TestRegionSize * 2, 2, TestRegionSize);
    std::memset(v1Data.get() + TestRegionSize * 3, 3, TestRegionSize);

    // data[0, 2, 4, 3]
    std::unique_ptr<char[]> v2Data(new char[V2DataSize]);
    ASSERT_NE(v2Data, nullptr);
    std::memset(v2Data.get() + TestRegionSize * 0, 0, TestRegionSize);
    std::memset(v2Data.get() + TestRegionSize * 1, 2, TestRegionSize);
    std::memset(v2Data.get() + TestRegionSize * 2, 4, TestRegionSize);
    std::memset(v2Data.get() + TestRegionSize * 3, 3, TestRegionSize);

    nn::fs::MemoryStorage v0Storage(v0Data, TestRegionSize);
    nn::fs::MemoryStorage v1Storage(v1Data.get(), V1DataSize);
    nn::fs::MemoryStorage v2Storage(v2Data.get(), V2DataSize);

    // v0/v1 パッチを作成
    PatchInfoBuilder<IndirectStorageBuilder> v0v1Patch;
    NNT_ASSERT_RESULT_SUCCESS(v0v1Patch.Initialize(
        [&](IndirectStorageBuilder* pBuilder) NN_NOEXCEPT -> nn::Result
        {
            auto result = pBuilder->Initialize(&v0Storage, &v1Storage, BlockSize);
            if( result.IsSuccess() )
            {
                result = pBuilder->Build(BlockSize, TestRegionSize, 0);
            }
            return result;
        }
    ));

    // data[1, 2, 3]
    ASSERT_EQ(v0v1Patch.GetDataSize(), V1DataSize - TestRegionSize);
    NNT_FS_UTIL_ASSERT_MEMCMPEQ(
        v0v1Patch.GetData(), v1Data.get() + TestRegionSize, TestRegionSize * 3);

    // v0/v2 パッチを作成
    PatchInfoBuilder<IndirectStorageBuilder> v0v2Patch;
    NNT_ASSERT_RESULT_SUCCESS(v0v2Patch.Initialize(
        [&](IndirectStorageBuilder* pBuilder) NN_NOEXCEPT -> nn::Result
        {
            auto result = pBuilder->Initialize(&v0Storage, &v2Storage, BlockSize);
            if( result.IsSuccess() )
            {
                result = pBuilder->Build(BlockSize, TestRegionSize, 0);
            }
            return result;
        }
    ));

    // data[2, 4, 3]
    ASSERT_EQ(v0v2Patch.GetDataSize(), TestRegionSize * 3);
    NNT_FS_UTIL_ASSERT_MEMCMPEQ(
        v0v2Patch.GetData(), v2Data.get() + TestRegionSize, TestRegionSize * 3);

    const auto pAllocator = nnt::fs::util::GetTestLibraryAllocator();

    // 最適化済みパッチを作成
    {
        PatchInfoBuilder<RelocatedIndirectStorageBuilder> optimizedPatch;
        NNT_ASSERT_RESULT_SUCCESS(optimizedPatch.Initialize(
            [&](RelocatedIndirectStorageBuilder* pBuilder) NN_NOEXCEPT -> nn::Result
            {
                auto result = pBuilder->Initialize(
                    v0v1Patch.GetPatchInfo(),
                    v0v2Patch.GetPatchInfo(),
                    v0v2Patch.GetPatchInfo().GetStorageOffset()
                );
                if( result.IsSuccess() )
                {
                    result = pBuilder->Build(pAllocator, BlockSize, TestRegionSize);
                }
                return result;
            },
            [&](IndirectStorageBuilder* pBuilderForTest) NN_NOEXCEPT -> nn::Result
            {
                return pBuilderForTest->Initialize(&v0Storage, &v2Storage, BlockSize);
            }
        ));

        // data[4, 2, 3]
        static const size_t PatchDataSize = TestRegionSize * 3;
        std::unique_ptr<char[]> patchData(new char[PatchDataSize]);
        ASSERT_NE(patchData, nullptr);
        std::memset(patchData.get() + TestRegionSize * 0, 4, TestRegionSize);
        std::memset(patchData.get() + TestRegionSize * 1, 2, TestRegionSize);
        std::memset(patchData.get() + TestRegionSize * 2, 3, TestRegionSize);

        ASSERT_EQ(optimizedPatch.GetDataSize(), PatchDataSize);
        NNT_FS_UTIL_ASSERT_MEMCMPEQ(
            optimizedPatch.GetData(), patchData.get(), PatchDataSize);
    }

    // デフラグした最適化済みパッチを作成
    {
        PatchInfoBuilder<RelocatedIndirectStorageBuilder> defragmentedPatch;
        NNT_ASSERT_RESULT_SUCCESS(defragmentedPatch.Initialize(
            [&](RelocatedIndirectStorageBuilder* pBuilder) NN_NOEXCEPT -> nn::Result
            {
                auto result = pBuilder->Initialize(
                    v0v1Patch.GetPatchInfo(),
                    v0v2Patch.GetPatchInfo(),
                    v0v2Patch.GetPatchInfo().GetStorageOffset()
                );
                if( result.IsSuccess() )
                {
                    result = pBuilder->Build(pAllocator, BlockSize, BlockSize, TestRegionSize, TestRegionSize * 2);
                }
                return result;
            },
            [&](IndirectStorageBuilder* pBuilderForTest) NN_NOEXCEPT -> nn::Result
            {
                return pBuilderForTest->Initialize(&v0Storage, &v2Storage, BlockSize);
            }
        ));

        // data[2, 4, 3]
        static const size_t PatchDataSize = TestRegionSize * 3;
        std::unique_ptr<char[]> patchData(new char[PatchDataSize]);
        ASSERT_NE(patchData, nullptr);
        std::memset(patchData.get() + TestRegionSize * 0, 2, TestRegionSize);
        std::memset(patchData.get() + TestRegionSize * 1, 4, TestRegionSize);
        std::memset(patchData.get() + TestRegionSize * 2, 3, TestRegionSize);

        ASSERT_EQ(defragmentedPatch.GetDataSize(), PatchDataSize);
        NNT_FS_UTIL_ASSERT_MEMCMPEQ(
            defragmentedPatch.GetData(), patchData.get(), PatchDataSize);
    }
} // NOLINT(impl/function_size)

template<typename TBuilder>
class IndirectDataStorage : public nn::fs::IStorage
{
public:
    explicit IndirectDataStorage() NN_NOEXCEPT
        : m_pBuilder(nullptr)
    {
    }

    void Initialize(TBuilder* pBuilder) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pBuilder);
        m_pBuilder = pBuilder;
    }

    virtual nn::Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(buffer);
        NN_SDK_REQUIRES_NOT_NULL(m_pBuilder);
        NN_SDK_REQUIRES(m_pBuilder->IsBuilt());

        NN_RESULT_DO(m_pBuilder->ReadData(offset, buffer, size));
        NN_RESULT_SUCCESS;
    }

    virtual nn::Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_SUCCESS;
    }

    virtual nn::Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(outValue);
        NN_SDK_REQUIRES_NOT_NULL(m_pBuilder);
        NN_SDK_REQUIRES(m_pBuilder->IsBuilt());

        *outValue = m_pBuilder->QueryDataStorageSize();
        NN_RESULT_SUCCESS;
    }

    int64_t GetSize() NN_NOEXCEPT
    {
        return m_pBuilder->QueryDataStorageSize();
    }

private:
    TBuilder* m_pBuilder;
};

template< typename TBuilder >
class PatchInfoBuilderForLarge : public PatchInfoBuilderBase<TBuilder>
{
public:
    template< typename TInitializer >
    nn::Result Initialize(TInitializer&& initializer) NN_NOEXCEPT
    {
        NN_RESULT_DO(initializer(&PatchInfoBuilderBase<TBuilder>::GetBuilder()));

        NN_RESULT_DO(PatchInfoBuilderBase<TBuilder>::Setup());

        NN_RESULT_SUCCESS;
    }

    virtual nn::fs::IStorage* GetDataStorage() NN_NOEXCEPT NN_OVERRIDE
    {
        return &m_DataStorage;
    }

protected:
    virtual nn::Result SetupDataStorage() NN_NOEXCEPT NN_OVERRIDE
    {
        m_DataStorage.Initialize(&PatchInfoBuilderBase<TBuilder>::GetBuilder());

        NN_RESULT_SUCCESS;
    }

private:
    IndirectDataStorage<TBuilder> m_DataStorage;
};

// 4 GB を超えるデータの最適化パッチを作成する
TEST(RelocatedIndirectStorageBuilderLargeTest, CreateOptimizedPatchLargeHeavy)
{
    static const int BufferSize = 1024 * 1024;

    // v0, v1, v2 のストレージを用意
    static const int64_t V0Size = static_cast<int64_t>(32) * 1024 * 1024 * 1024;
    nnt::fs::util::VirtualMemoryStorage v0Storage;
    v0Storage.Initialize(V0Size);
    NN_UTIL_SCOPE_EXIT
    {
        v0Storage.Finalize();
    };

    static const int64_t V1Size = V0Size + BufferSize;
    nnt::fs::util::VirtualMemoryStorage v1Storage;
    v1Storage.Initialize(V1Size);
    NN_UTIL_SCOPE_EXIT
    {
        v1Storage.Finalize();
    };

    static const int64_t V2Size = V1Size + BufferSize;
    nnt::fs::util::VirtualMemoryStorage v2Storage;
    v2Storage.Initialize(V2Size);
    NN_UTIL_SCOPE_EXIT
    {
        v2Storage.Finalize();
    };


    {
        // 元ストレージに書き込む
        // v0: a 0 b 0
        // v1: a 0 b 0 c
        // v2: a 0 d b 0 c
        auto writeBuffer = nnt::fs::util::AllocateBuffer(BufferSize);
        {
            nnt::fs::util::FillBufferWithRandomValue(writeBuffer.get(), BufferSize);
            v0Storage.Write(0, writeBuffer.get(), BufferSize);
            v1Storage.Write(0, writeBuffer.get(), BufferSize);
            v2Storage.Write(0, writeBuffer.get(), BufferSize);
        }
        {
            const int64_t offset = static_cast<int64_t>(4) * 1024 * 1024 * 1024;

            nnt::fs::util::FillBufferWith32BitCount(writeBuffer.get(), BufferSize, 0);
            v2Storage.Write(offset, writeBuffer.get(), BufferSize);

            nnt::fs::util::FillBufferWithRandomValue(writeBuffer.get(), BufferSize);
            v0Storage.Write(offset, writeBuffer.get(), BufferSize);
            v1Storage.Write(offset, writeBuffer.get(), BufferSize);
            v2Storage.Write(offset + BufferSize, writeBuffer.get(), BufferSize);
        }
        {
            nnt::fs::util::FillBufferWithRandomValue(writeBuffer.get(), BufferSize);
            v1Storage.Write(V1Size - BufferSize, writeBuffer.get(), BufferSize);
            v2Storage.Write(V2Size - BufferSize, writeBuffer.get(), BufferSize);
        }
    }

    // v1, v2 パッチ作成
    PatchInfoBuilderForLarge<nn::fssystem::utilTool::IndirectStorageBuilder> v1Builder;
    NNT_ASSERT_RESULT_SUCCESS(v1Builder.Initialize(
        [&](nn::fssystem::utilTool::IndirectStorageBuilder* pBuilder) NN_NOEXCEPT -> nn::Result
        {
            NN_RESULT_DO(pBuilder->Initialize(&v0Storage, &v1Storage, BlockSize));
            NN_RESULT_DO(pBuilder->Build(BlockSize, RegionSize, 0));

            NN_RESULT_SUCCESS;
        }));

    PatchInfoBuilderForLarge<nn::fssystem::utilTool::IndirectStorageBuilder> v2Builder;
    NNT_ASSERT_RESULT_SUCCESS(v2Builder.Initialize(
        [&](nn::fssystem::utilTool::IndirectStorageBuilder* pBuilder) NN_NOEXCEPT -> nn::Result
        {
            NN_RESULT_DO(pBuilder->Initialize(&v0Storage, &v2Storage, BlockSize));
            NN_RESULT_DO(pBuilder->Build(BlockSize, RegionSize, 0));

            NN_RESULT_SUCCESS;
        }));

    // v2 最適化パッチ作成
    PatchInfoBuilderForLarge<nn::fssystem::utilTool::RelocatedIndirectStorageBuilder> v2OptimizedBuilder;
    NNT_ASSERT_RESULT_SUCCESS(v2OptimizedBuilder.Initialize(
        [&](nn::fssystem::utilTool::RelocatedIndirectStorageBuilder* pBuilder) NN_NOEXCEPT -> nn::Result
        {
            NN_RESULT_DO(pBuilder->Initialize(v1Builder.GetPatchInfo(), v2Builder.GetPatchInfo(), 0));

            NN_RESULT_DO(pBuilder->Build(
                nnt::fs::util::GetTestLibraryAllocator(),
                BlockSize,
                BlockSize,
                RegionSize,
                0));

            NN_RESULT_SUCCESS;
        }));

    const auto patchInfo = v2OptimizedBuilder.GetPatchInfo();

    // v0 に v2 最適化パッチを適用し v2 と同じ内容になるかテスト
    nn::fssystem::IndirectStorage v2OptimizedIndirectStorage;
    NNT_ASSERT_RESULT_SUCCESS(v2OptimizedIndirectStorage.Initialize(
        nnt::fs::util::GetTestLibraryAllocator(),
        patchInfo.GetNodeStorage(),
        patchInfo.GetEntryStorage(),
        patchInfo.GetEntryCount()));
    NN_UTIL_SCOPE_EXIT
    {
        v2OptimizedIndirectStorage.Finalize();
    };

    v2OptimizedIndirectStorage.SetStorage(0, &v0Storage, 0, V0Size);
    v2OptimizedIndirectStorage.SetStorage(1, patchInfo.GetDataStorage());

    bool isSame = false;
    NNT_ASSERT_RESULT_SUCCESS(CompareStorages(&isSame, &v2OptimizedIndirectStorage, &v2Storage, 4 * 1024 * 1024));
    ASSERT_TRUE(isSame);
}
