﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <numeric>
#include <nn/util/util_BytePtr.h>
#include <nn/fssystem/fs_BucketTreeBuilder.h>
#include <nnt/nntest.h>
#include <nnt/fsUtil/testFs_util.h>

namespace nn { namespace fssystem {

/**
 * @brief   BucketTree テストのためのクラスです。
 */
class BucketTreeTest : public ::testing::Test
{
public:
    //! ノードに含まれるエントリの数を取得します。
    static int GetEntryCount(size_t nodeSize, size_t entrySize) NN_NOEXCEPT
    {
        return BucketTree::GetEntryCount(nodeSize, entrySize);
    }

    //! ノードに含まれるオフセットの数を取得します。
    static int GetOffsetCount(size_t nodeSize) NN_NOEXCEPT
    {
        return BucketTree::GetOffsetCount(nodeSize);
    }

    //! エントリセットの数を取得します。
    static int GetEntrySetCount(
                   size_t nodeSize,
                   size_t entrySize,
                   int entryCount
               ) NN_NOEXCEPT
    {
        return BucketTree::GetEntrySetCount(nodeSize, entrySize, entryCount);
    }

    //! L2 のノード数を取得します。
    static int GetNodeL2Count(
                   size_t nodeSize,
                   size_t entrySize,
                   int entryCount
               ) NN_NOEXCEPT
    {
        return BucketTree::GetNodeL2Count(nodeSize, entrySize, entryCount);
    }

    //! ストレージを壊します。
    void DestroyStorage(BucketTree* pTree) NN_NOEXCEPT
    {
        pTree->m_EntryStorage = fs::SubStorage(&pTree->m_EntryStorage, 0, 0);
    }
};

}}

/**
 * @brief   エントリデータです。
 */
struct BucketTreeEntryData
{
    static int64_t GetOffsetStride() NN_NOEXCEPT;
};

/**
 * @brief   IndirectStorage のエントリデータです。
 */
struct IndirectStorageData
{
    char virtualOffset[sizeof(int64_t)];
    char physicalOffset[sizeof(int64_t)];
    int32_t storageIndex;

public:
    //! テストデータを作成します。
    static std::unique_ptr<IndirectStorageData[]> Create(int count) NN_NOEXCEPT;

    //! オフセットを取得します。
    int64_t GetOffset() const NN_NOEXCEPT
    {
        return nn::fssystem::detail::SafeValue::GetInt64(virtualOffset);
    }
};

/**
 * @brief   AesCtrCounterExtendedStorage のエントリデータです。
 */
struct AesCtrCounterExtendedStorageData
{
    char offset[sizeof(int64_t)];
    int32_t keyIndex;
    char counter[16];

public:
    //! テストデータを作成します。
    static std::unique_ptr<AesCtrCounterExtendedStorageData[]> Create(int count) NN_NOEXCEPT;

    //! オフセットを取得します。
    int64_t GetOffset() const NN_NOEXCEPT
    {
        return nn::fssystem::detail::SafeValue::GetInt64(offset);
    }
};

/**
 * @brief   仮想のエントリデータです。
 */
template< size_t Size >
struct UnknownStorageData
{
    char offset[sizeof(int64_t)];
    char data[Size];

public:
    NN_STATIC_ASSERT(0 < Size);

    typedef UnknownStorageData<Size> SelfType;

public:
    //! テストデータを作成します。
    static std::unique_ptr<SelfType[]> Create(int count) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_LESS(0, count);

        int64_t offset = 0;

        std::unique_ptr<SelfType[]> data(new SelfType[count]);
        NN_SDK_ASSERT_NOT_NULL(data.get());

        for( int i = 0; i < count; ++i )
        {
            offset += BucketTreeEntryData::GetOffsetStride();

            SelfType value;
            std::memcpy(value.offset, &offset, sizeof(int64_t));
            std::iota(std::begin(value.data), std::end(value.data), '\x0');

            std::memcpy(&data[i], &value, sizeof(SelfType));
        }

        return data;
    }

    //! オフセットを取得します。
    int64_t GetOffset() const NN_NOEXCEPT
    {
        return nn::fssystem::detail::SafeValue::GetInt64(offset);
    }
};

/**
 * @brief   BucketTree のデータを作成するクラスです。
 */
template< typename T, typename TMemoryStorage = nnt::fs::util::SafeMemoryStorage >
class BucketTreeData
{
    NN_DISALLOW_COPY(BucketTreeData);

public:
    typedef T StorageData;
    typedef std::unique_ptr<StorageData[]> Entry;
    typedef nn::fssystem::BucketTreeBuilder Builder;

public:
    //! コンストラクタです。
    BucketTreeData(size_t nodeSize, int entryCount) NN_NOEXCEPT
        : m_NodeSize(nodeSize)
        , m_EntrySize(sizeof(StorageData))
        , m_EntryCount(entryCount)
    {
        const auto entrySize = sizeof(StorageData);
        const auto headerStorageSize = Builder::QueryHeaderStorageSize();

        m_HeaderStorage.Initialize(headerStorageSize);

        nn::fs::SubStorage headerSubStorage(&m_HeaderStorage, 0, headerStorageSize);

        m_NodeStorageSize =
            Builder::QueryNodeStorageSize(nodeSize, entrySize, entryCount);
        m_NodeStorage.Initialize(m_NodeStorageSize);

        nn::fs::SubStorage nodeSubStorage(&m_NodeStorage, 0, m_NodeStorageSize);

        m_EntryStorageSize =
            Builder::QueryEntryStorageSize(nodeSize, entrySize, entryCount);
        m_EntryStorage.Initialize(m_EntryStorageSize);

        nn::fs::SubStorage entrySubStorage(&m_EntryStorage, 0, m_EntryStorageSize);

        NN_ABORT_UNLESS_RESULT_SUCCESS(
            m_Builder.Initialize(
                nnt::fs::util::GetTestLibraryAllocator(),
                headerSubStorage,
                nodeSubStorage,
                entrySubStorage,
                nodeSize,
                entrySize,
                entryCount
            )
        );

        // 全データ書き出し
        m_Entry = StorageData::Create(entryCount);
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_Builder.Write(m_Entry.get(), entryCount));

        m_EndOffset = m_Entry[entryCount - 1].GetOffset() + BucketTreeEntryData::GetOffsetStride();
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_Builder.Commit(m_EndOffset));
    }

    //! BucketTreeBuilder を取得します。
    const nn::fssystem::BucketTreeBuilder& GetBuilder() const NN_NOEXCEPT
    {
        return m_Builder;
    }

    //! エントリデータを取得します。
    const T* GetEntry() const NN_NOEXCEPT
    {
        return m_Entry.get();
    }

    const T& GetEntry(int index) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_LESS(index, m_EntryCount);
        return m_Entry[index];
    }

    //! エントリ数を取得します。
    const int GetEntryCount() const NN_NOEXCEPT
    {
        return m_EntryCount;
    }

    //! ノードストレージを取得します。
    TMemoryStorage& GetNodeStorage() NN_NOEXCEPT
    {
        return m_NodeStorage;
    }

    //! ノードストレージを取得します。
    const TMemoryStorage& GetNodeStorage() const NN_NOEXCEPT
    {
        return m_NodeStorage;
    }

    //! エントリストレージを取得します。
    TMemoryStorage& GetEntryStorage() NN_NOEXCEPT
    {
        return m_EntryStorage;
    }

    //! エントリストレージを取得します。
    const TMemoryStorage& GetEntryStorage() const NN_NOEXCEPT
    {
        return m_EntryStorage;
    }

    //! 終了オフセットを取得します。
    int64_t GetEndOffset() const NN_NOEXCEPT
    {
        return m_EndOffset;
    }

protected:
    size_t m_NodeSize;
    size_t m_EntrySize;
    int m_EntryCount;
    int64_t m_EndOffset;
    int64_t m_NodeStorageSize;
    int64_t m_EntryStorageSize;
    TMemoryStorage m_HeaderStorage;
    TMemoryStorage m_NodeStorage;
    TMemoryStorage m_EntryStorage;
    nn::fssystem::BucketTreeBuilder m_Builder;
    Entry m_Entry;
};
