﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>
#include <nn/nn_SdkAssert.h>
#include <nn/fssystem/fs_BucketTreeUtility.h>
#include "testFs_Unit_BucketTreeUtil.h"

namespace {

const int64_t DefaultStride = 32 * 1024;

std::mt19937 g_Mt(nnt::fs::util::GetRandomSeed());

}

NN_STATIC_ASSERT(NN_ALIGNOF(IndirectStorageData) == 4);
NN_STATIC_ASSERT(sizeof(IndirectStorageData) == 20);
NN_STATIC_ASSERT(NN_ALIGNOF(AesCtrCounterExtendedStorageData) == 4);
NN_STATIC_ASSERT(sizeof(AesCtrCounterExtendedStorageData) == 28);
NN_STATIC_ASSERT(NN_ALIGNOF(UnknownStorageData<1>) == 1);

/**
 * @brief   オフセットの移動量を取得します。
 */
int64_t BucketTreeEntryData::GetOffsetStride() NN_NOEXCEPT
{
    const auto stride =
        std::uniform_int_distribution<int64_t>(DefaultStride / 8, DefaultStride)(g_Mt);
    return nn::util::align_up(stride, sizeof(int64_t));
}

/**
 * @brief   IndirectStorageData のテストデータを作成します。
 */
std::unique_ptr<IndirectStorageData[]>
    IndirectStorageData::Create(int count) NN_NOEXCEPT
{
    typedef IndirectStorageData StorageData;
    typedef nn::fssystem::detail::SafeValue ValueProxy;

    NN_SDK_REQUIRES_LESS(0, count);

    int64_t virtualOffset = 0;
    int32_t storageIndex = 0;

    std::unique_ptr<StorageData[]> data(new StorageData[count]);
    NN_SDK_ASSERT_NOT_NULL(data.get());

    for( int i = 0; i < count; ++i )
    {
        virtualOffset += BucketTreeEntryData::GetOffsetStride();

        auto physicalOffset =
            std::uniform_int_distribution<int64_t>(0x10000, 0x10000000000)(g_Mt);

        StorageData value;
        ValueProxy::SetInt64(&value.virtualOffset, virtualOffset);
        ValueProxy::SetInt64(&value.physicalOffset, physicalOffset);
        value.storageIndex = storageIndex;

        std::memcpy(&data[i], &value, sizeof(StorageData));

        ++storageIndex;
    }

    return data;
}

/**
 * @brief   AesCtrCounterExtendedStorageData のテストデータを作成します。
 */
std::unique_ptr<AesCtrCounterExtendedStorageData[]>
    AesCtrCounterExtendedStorageData::Create(int count) NN_NOEXCEPT
{
    typedef AesCtrCounterExtendedStorageData StorageData;
    typedef nn::fssystem::detail::SafeValue ValueProxy;

    NN_SDK_REQUIRES_LESS(0, count);

    int64_t offset = 0;
    int32_t keyIndex = 0;

    std::unique_ptr<StorageData[]> data(new StorageData[count]);
    NN_SDK_ASSERT_NOT_NULL(data.get());

    for( int i = 0; i < count; ++i )
    {
        offset += BucketTreeEntryData::GetOffsetStride();

        StorageData value;
        ValueProxy::SetInt64(&value.offset, offset);
        value.keyIndex = keyIndex;
        std::iota(value.counter, value.counter + sizeof(value.counter), '\x0');

        std::memcpy(&data[i], &value, sizeof(StorageData));

        ++keyIndex;
    }

    return data;
}
