﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nn/fssystem/utilTool/fs_BinaryMatch.h>
#include <nn/fssystem/utilTool/fs_BinaryRegionFile.h>

namespace {

typedef nn::fssystem::utilTool::BinaryRegionFileHeader RegionFileHeader;
typedef nn::fssystem::utilTool::BinaryRegionFile RegionFile;
typedef nn::fssystem::utilTool::BinaryRegionArray RegionArray;
typedef nn::fssystem::utilTool::BinaryRegion Region;

}

#if !defined(NN_SDK_BUILD_RELEASE)
/**
 *  @brief  事前条件をテストします。
 */
TEST(BinaryRegionFileDeathTest, Precondition)
{
    nnt::fs::util::Vector<char> buffer(nn::fssystem::NcaFsHeader::Size, 0);

    RegionFileHeader header = {};
    header.Initialize();

    // SetHeader() のチェック
    EXPECT_DEATH_IF_SUPPORTED(header.SetHeader(nullptr, buffer.size()), "");
    EXPECT_DEATH_IF_SUPPORTED(header.SetHeader(buffer.data(), buffer.size() - 1), "");

    // SetData() のチェック
    EXPECT_DEATH_IF_SUPPORTED(header.SetData(-1, buffer.data(), buffer.size()), "");
    EXPECT_DEATH_IF_SUPPORTED(header.SetData(0, nullptr, 1), "");
    EXPECT_DEATH_IF_SUPPORTED(header.SetData(0, buffer.data(), 0), "");

    // GenerateFileName() のチェック
    EXPECT_DEATH_IF_SUPPORTED(header.GenerateFileName(nullptr, 1), "");
    EXPECT_DEATH_IF_SUPPORTED(header.GenerateFileName(buffer.data(), 0), "");
    {
        RegionFileHeader header2 = {};
        EXPECT_DEATH_IF_SUPPORTED(header2.GenerateFileName(buffer.data(), 128), "");
    }

    // SetHeader() のチェック
    EXPECT_DEATH_IF_SUPPORTED(RegionFile().SetHeader(nullptr, buffer.size()), "");
    EXPECT_DEATH_IF_SUPPORTED(RegionFile().SetHeader(buffer.data(), buffer.size() - 1), "");

    // SetData() のチェック
    EXPECT_DEATH_IF_SUPPORTED(RegionFile().SetData(-1, buffer.data(), buffer.size()), "");
    EXPECT_DEATH_IF_SUPPORTED(RegionFile().SetData(0, nullptr, 1), "");
    EXPECT_DEATH_IF_SUPPORTED(RegionFile().SetData(0, buffer.data(), 0), "");

    // CheckRegion() のチェック
    {
        RegionFile regionFile;
        regionFile.SetHeader(buffer.data(), buffer.size());
        regionFile.SetData(0, buffer.data(), buffer.size());
        EXPECT_DEATH_IF_SUPPORTED(regionFile.CheckRegion(nullptr, 1), "");
    }
    {
        RegionFile regionFile;
        regionFile.SetHeader(buffer.data(), buffer.size());
        regionFile.SetData(0, buffer.data(), buffer.size());
        EXPECT_DEATH_IF_SUPPORTED(regionFile.CheckRegion(buffer.data(), 0), "");
    }

    // SetRegion() のチェック
    {
        RegionArray regions;
        RegionFile regionFile;
        regionFile.SetHeader(buffer.data(), buffer.size());
        regionFile.SetData(0, buffer.data(), buffer.size());
        EXPECT_DEATH_IF_SUPPORTED(regionFile.SetRegion(regions), "");
    }
    {
        Region region = {};
        RegionArray regions;
        regions.Reserve(&region, 1);
        RegionFile regionFile;
        regionFile.SetHeader(buffer.data(), buffer.size());
        regionFile.SetData(0, buffer.data(), buffer.size());
        EXPECT_DEATH_IF_SUPPORTED(regionFile.SetRegion(regions), "");
    }

    // GetRegion() のチェック
    {
        RegionFile regionFile;
        regionFile.SetHeader(buffer.data(), buffer.size());
        regionFile.SetData(0, buffer.data(), buffer.size());
        EXPECT_DEATH_IF_SUPPORTED(regionFile.GetRegion(), "");
    }
}
#endif

/**
 *  @brief  リージョンハッシュの設定をテストします。
 */
TEST(BinaryRegionFileTest, CheckRegion)
{
    Region region = {};
    nnt::fs::util::Vector<char> buffer1(nn::fssystem::NcaFsHeader::Size, 0);
    nnt::fs::util::Vector<char> buffer2(nn::fssystem::NcaFsHeader::Size, 1);

    nnt::fs::util::Vector<char> fileBuffer(sizeof(RegionFileHeader) + sizeof(Region), 0);
    {
        RegionFileHeader header;
        header.Initialize();
        header.SetHeader(buffer1.data(), buffer1.size());
        header.SetData(0, buffer1.data(), buffer1.size());
        header.regionCount = 1;

        std::memcpy(fileBuffer.data(), &header, sizeof(header));
        std::memcpy(fileBuffer.data() + sizeof(header), &region, sizeof(region));
    }

    // 同じデータなら CheckRegion() が成功して GetRegion() を呼び出せる
    {
        RegionFile regionFile;
        regionFile.SetHeader(buffer1.data(), buffer1.size());
        regionFile.SetData(0, buffer1.data(), buffer1.size());
        ASSERT_TRUE(regionFile.CheckRegion(fileBuffer.data(), fileBuffer.size()));

        const auto regions = regionFile.GetRegion();
        EXPECT_EQ(1, regions.size());
        EXPECT_EQ(
            reinterpret_cast<uintptr_t>(regions.data()),
            reinterpret_cast<uintptr_t>(fileBuffer.data()) + sizeof(RegionFileHeader)
        );
    }

#if !defined(NN_SDK_BUILD_RELEASE)
    // 同じデータで CheckRegion() 後 SetRegion() を呼び出すと失敗する
    {
        RegionFile regionFile;
        regionFile.SetHeader(buffer1.data(), buffer1.size());
        regionFile.SetData(0, buffer1.data(), buffer1.size());
        ASSERT_TRUE(regionFile.CheckRegion(fileBuffer.data(), fileBuffer.size()));
        EXPECT_DEATH_IF_SUPPORTED(regionFile.SetRegion(RegionArray(&region, 1)), "");
    }
#endif

    // 違うデータなら CheckRegion() が失敗するので SetRegion() を呼び出す
    {
        RegionFile regionFile;
        regionFile.SetHeader(buffer2.data(), buffer2.size());
        regionFile.SetData(0, buffer2.data(), buffer2.size());
        ASSERT_FALSE(regionFile.CheckRegion(fileBuffer.data(), fileBuffer.size()));
        EXPECT_EQ(0, regionFile.GetHeader().regionCount);

        regionFile.SetRegion(RegionArray(&region, 1));
        EXPECT_EQ(1, regionFile.GetHeader().regionCount);
    }

#if !defined(NN_SDK_BUILD_RELEASE)
    // 違うデータで CheckRegion() 後 GetRegion() を呼び出すと失敗する
    {
        RegionFile regionFile;
        regionFile.SetHeader(buffer2.data(), buffer2.size());
        regionFile.SetData(0, buffer2.data(), buffer2.size());
        ASSERT_FALSE(regionFile.CheckRegion(fileBuffer.data(), fileBuffer.size()));
        EXPECT_DEATH_IF_SUPPORTED(regionFile.GetRegion(), "");
    }
#endif
}

/**
 *  @brief  ファイル名生成をテストします。
 */
TEST(BinaryRegionFileTest, GenerateFileName)
{
    nnt::fs::util::Vector<char> buffer(nn::fssystem::NcaFsHeader::Size, 0);
    char fileName1[128];
    {
        RegionFileHeader header;
        header.Initialize();
        header.SetHeader(buffer.data(), buffer.size());
        header.SetData(0, buffer.data(), buffer.size());

        // ヘッダがそのままでないことを確認
        EXPECT_FALSE(std::equal(buffer.begin(), buffer.end(), header.header));

        header.GenerateFileName(fileName1, sizeof(fileName1));
    }

    RegionFile regionFile;
    regionFile.SetHeader(buffer.data(), buffer.size());
    regionFile.SetData(0, buffer.data(), buffer.size());

    char fileName2[128];
    regionFile.GenerateFileName(fileName2, sizeof(fileName2));

    EXPECT_EQ(0, std::strcmp(fileName1, fileName2));
    EXPECT_EQ(68, std::strlen(fileName1));
    EXPECT_EQ(0, std::strcmp(fileName1 + 64, RegionFileHeader::Extension));
}
