﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/mem/mem_StandardAllocator.h>
#include <nn/crypto/crypto_Csrng.h>
#include <nn/crypto/crypto_RsaPssSha256Signer.h>
#include <nn/fs/fs_FileStorage.h>
#include <nn/fs/fs_RomFsFileSystem.h>
#include <nn/fs/fs_SpeedEmulation.h>
#include <nn/fs/fsa/fs_Registrar.h>
#include <nn/fssrv/fssrv_MemoryResourceFromStandardAllocator.h>
#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/fssystem/fs_HostFileSystem.h>
#include <nn/fssystem/fs_PartitionFileSystem.h>
#include <nn/fssystem/fs_NcaFileSystemDriverUtility.h>
#include <nn/fssystem/utilTool/fs_SparseStorageBuilder.h>
#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include "testFs_NcaFileSystemDriverUtil.h"

#if !defined(NNT_FS_NCA_FILE_SYSTEM_DRIVER_WITH_NCA_FILES)

namespace nn { namespace fssystem {

class NcaFileSystemDriverTest
{
public:
    struct HeaderSign
    {
        struct Key
        {
            const void* pModulus;
            size_t modulusSize;
            const void* pPrivateExponent;
            size_t privateExponentSize;

            bool IsValid() const NN_NOEXCEPT
            {
                return (this->pModulus != nullptr) && (this->pPrivateExponent != nullptr);
            }
        };

        Key key1;
        Key key2;
    };
    NN_STATIC_ASSERT(std::is_pod<HeaderSign>::value);

public:
    explicit NcaFileSystemDriverTest(NcaFileSystemDriver* pDriver) NN_NOEXCEPT
        : m_pReader(pDriver->m_pReader)
        , m_pOriginalReader(pDriver->m_pOriginalReader)
        , m_pAllocator(pDriver->m_pAllocator)
        , m_pBufferManager(pDriver->m_pBufferManager)
        , m_pDriver(pDriver)
        , m_PhysicalOffset(0)
    {
    }

    Result CreateSparseTableStorage(std::unique_ptr<fs::IStorage>* pOutStorage, const NcaFsHeaderReader& headerReader, const utilTool::SparseStorageBuilder& storageBuilder) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutStorage);

        const auto dataSize = util::align_up(storageBuilder.QueryDataStorageSize(), 1 << NcaHeader::Log2SectorSize);
        const auto nodeSize = storageBuilder.QueryTableNodeStorageSize();
        const auto entrySize = storageBuilder.QueryTableEntryStorageSize();
        const auto physicalOffset = headerReader.GetSparseInfo().physicalOffset;

        // SparseStorage のテーブル用に暗号化ストレージを敷く
        NcaFileSystemDriver::BaseStorage baseStorage(m_pOriginalReader->GetBodyStorage(), physicalOffset, dataSize + nodeSize + entrySize);
        baseStorage.SetStorageOffset(physicalOffset);
        baseStorage.SetAesCtrUpperIv(headerReader.GetSparseInfo().MakeAesCtrUpperIv(headerReader.GetAesCtrUpperIv()));
        NcaFileSystemDriver driver(m_pOriginalReader, m_pAllocator, m_pBufferManager);
        NN_RESULT_DO(driver.CreateAesCtrStorage(pOutStorage, &baseStorage));
        NN_RESULT_SUCCESS;
    }

    Result SparsifyStorage(bool* outValue, size_t blockSize, size_t fragmentSize, const HeaderSign& headerSign) NN_NOEXCEPT;

private:
    static Result UpdateFsHeader(NcaReader* pReader, const NcaFsHeaderReader& headerReader) NN_NOEXCEPT;
    static Result UpdateHeaderSign1(NcaReader* pReader, const HeaderSign::Key& key) NN_NOEXCEPT;
    static Result UpdateHeaderSign2(NcaReader* pReader, const HeaderSign::Key& key) NN_NOEXCEPT;

    Result SparsifyDataStorage(bool* outValue, int fsIndex, size_t blockSize, size_t fragmentSize) NN_NOEXCEPT;
    Result SparsifyCodeStorage(int fsIndex) NN_NOEXCEPT;

private:
    std::shared_ptr<NcaReader> m_pReader;
    std::shared_ptr<NcaReader> m_pOriginalReader;
    nn::MemoryResource* const m_pAllocator;
    nn::fssystem::IBufferManager* const m_pBufferManager;
    NcaFileSystemDriver* const m_pDriver;
    int64_t m_PhysicalOffset;
};

Result NcaFileSystemDriverTest::UpdateFsHeader(NcaReader* pReader, const NcaFsHeaderReader& headerReader) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pReader);
    NN_SDK_REQUIRES_NOT_NULL(pReader->m_pBodyStorage);

    const auto fsIndex = headerReader.GetFsIndex();
    const auto& fsHeader = headerReader.GetData();

    // ハッシュ書き出し
    {
        auto& hash = pReader->m_Data.fsHeaderHash[fsIndex];

        // キャッシュを更新
        crypto::GenerateSha256Hash(&hash, Hash::Size, &fsHeader, NcaFsHeader::Size);

        // ストレージへ書き出し
        const auto offset = offsetof(NcaHeader, fsHeaderHash) + Hash::Size * fsIndex;
        NN_RESULT_DO(pReader->m_pHeaderStorage->Write(offset, &hash, Hash::Size));
    }
    // ヘッダ書き出し
    {
        const auto offset = NcaHeader::Size + NcaFsHeader::Size * fsIndex;
        NN_RESULT_DO(pReader->m_pHeaderStorage->Write(offset, &fsHeader, NcaFsHeader::Size));
    }

    NN_RESULT_SUCCESS;
}

Result NcaFileSystemDriverTest::UpdateHeaderSign1(NcaReader* pReader, const HeaderSign::Key& key) NN_NOEXCEPT
{
    // その他の事前検証は SignRsa2048PssSha256() に任せる
    NN_SDK_REQUIRES_NOT_NULL(pReader);
    NN_SDK_REQUIRES_NOT_NULL(pReader->m_pBodyStorage);

    char salt[crypto::Rsa2048PssSha256Signer::DefaultSaltSize];
    crypto::GenerateCryptographicallyRandomBytes(salt, sizeof(salt));

    // キャッシュしている署名を更新
    NN_RESULT_THROW_UNLESS(
        crypto::SignRsa2048PssSha256(
            pReader->m_Data.headerSign1,
            NcaHeader::HeaderSignSize,
            key.pModulus,
            key.modulusSize,
            key.pPrivateExponent,
            key.privateExponentSize,
            &pReader->m_Data.signature,
            NcaHeader::Size - NcaHeader::HeaderSignSize * 2,
            salt,
            sizeof(salt)
        ),
        fs::ResultNcaHeaderSignature1VerificationFailed()
    );

    // 署名をストレージへ書き出し
    NN_RESULT_DO(pReader->m_pHeaderStorage->Write(
        offsetof(NcaHeader, headerSign1), pReader->m_Data.headerSign1, NcaHeader::HeaderSignSize));

    NN_RESULT_SUCCESS;
}

Result NcaFileSystemDriverTest::UpdateHeaderSign2(NcaReader* pReader, const HeaderSign::Key& key) NN_NOEXCEPT
{
    // その他の事前検証は SignRsa2048PssSha256() に任せる
    NN_SDK_REQUIRES_NOT_NULL(pReader);
    NN_SDK_REQUIRES_NOT_NULL(pReader->m_pBodyStorage);

    char salt[crypto::Rsa2048PssSha256Signer::DefaultSaltSize];
    crypto::GenerateCryptographicallyRandomBytes(salt, sizeof(salt));

    // キャッシュしている署名を更新
    NN_RESULT_THROW_UNLESS(
        crypto::SignRsa2048PssSha256(
            pReader->m_Data.headerSign2,
            NcaHeader::HeaderSignSize,
            key.pModulus,
            key.modulusSize,
            key.pPrivateExponent,
            key.privateExponentSize,
            &pReader->m_Data.signature,
            NcaHeader::Size - NcaHeader::HeaderSignSize * 2,
            salt,
            sizeof(salt)
        ),
        fs::ResultNcaHeaderSignature2VerificationFailed()
    );

    // 署名をストレージへ書き出し
    NN_RESULT_DO(pReader->m_pHeaderStorage->Write(
        offsetof(NcaHeader, headerSign2), pReader->m_Data.headerSign2, NcaHeader::HeaderSignSize));

    NN_RESULT_SUCCESS;
}

Result NcaFileSystemDriverTest::SparsifyStorage(bool* outValue, size_t blockSize, size_t fragmentSize, const HeaderSign& headerSign) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);

    NN_RESULT_THROW_UNLESS(m_pOriginalReader != nullptr, fs::ResultInvalidArgument());

    static const int IndexProgramCode = 0;
    static const int IndexProgramData = 1;
    static const int IndexManualData = 0;

    *outValue = false;
    m_PhysicalOffset = -1;

    switch( m_pOriginalReader->GetContentType() )
    {
    case NcaHeader::ContentType::Program:
        // 処理単純化のためコードとデータ領域を両方含むものとする
        NN_RESULT_THROW_UNLESS(
            m_pReader->HasFsInfo(IndexProgramCode) && m_pReader->HasFsInfo(IndexProgramData),
            fs::ResultInvalidArgument()
        );
        NN_RESULT_THROW_UNLESS(
            m_pOriginalReader->HasFsInfo(IndexProgramCode) && m_pOriginalReader->HasFsInfo(IndexProgramData),
            fs::ResultInvalidArgument()
        );

        // データは logo -> data -> code と並んでいる
        NN_RESULT_THROW_UNLESS(
            m_pOriginalReader->GetFsOffset(IndexProgramData) < m_pOriginalReader->GetFsOffset(IndexProgramCode),
            fs::ResultInvalidArgument()
        );

        m_PhysicalOffset = m_pOriginalReader->GetFsOffset(IndexProgramData);

        NN_RESULT_DO(SparsifyDataStorage(outValue, IndexProgramData, blockSize, fragmentSize));
        NN_RESULT_DO(SparsifyCodeStorage(IndexProgramCode));
        break;

    case NcaHeader::ContentType::Manual:
        if( m_pOriginalReader->HasFsInfo(IndexManualData) )
        {
            m_PhysicalOffset = m_pOriginalReader->GetFsOffset(IndexManualData);

            // IndirectTable から SparseStorage を作成
            if( m_pReader->HasFsInfo(IndexManualData) )
            {
                NN_RESULT_DO(SparsifyDataStorage(outValue, IndexManualData, blockSize, fragmentSize));
            }
            // 元ロム側をすべて参照しない SparseStorage を作成
            else
            {
                NN_RESULT_DO(SparsifyCodeStorage(IndexManualData));
            }
        }
        break;

        // Program/Manual 以外は何もしない
    default:
        break;
    }

    // ヘッダ署名の更新
    if( 0 <= m_PhysicalOffset )
    {
        if( headerSign.key1.IsValid() )
        {
            NN_RESULT_DO(UpdateHeaderSign1(&*m_pOriginalReader, headerSign.key1));
        }
        if( headerSign.key2.IsValid() )
        {
            NN_RESULT_DO(UpdateHeaderSign2(&*m_pOriginalReader, headerSign.key1));
        }
    }

    NN_RESULT_SUCCESS;
}

Result NcaFileSystemDriverTest::SparsifyDataStorage(bool* outValue, int fsIndex, size_t blockSize, size_t fragmentSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    NN_SDK_REQUIRES_RANGE(fsIndex, 0, NcaHeader::FsCountMax);
    NN_SDK_REQUIRES_LESS_EQUAL(0, m_PhysicalOffset);

    utilTool::SparseStorageBuilder storageBuilder(blockSize, fragmentSize);
    NcaFsHeaderReader headerReader;
    NcaFileSystemDriver::SparseParam sparseParam(&storageBuilder, &headerReader, false, fsIndex, m_PhysicalOffset);

    NN_RESULT_DO(m_pDriver->SparsifyBasicStorage(&sparseParam));
    *outValue = sparseParam.IsStorageSparsified();

    if( headerReader.ExistsSparseLayer() )
    {
        const auto physicalOffset = m_PhysicalOffset;
        const auto dataStorageSize = storageBuilder.QueryDataStorageSize();
        {
            const auto BufferSize = 512 * 1024;
            std::unique_ptr<char[]> buffer(new char[BufferSize]);
            NN_RESULT_THROW_UNLESS(buffer != nullptr, fs::ResultAllocationMemoryFailedNew());

            auto& storage = *m_pOriginalReader->GetBodyStorage();
            const auto writeOffset = physicalOffset;
            int64_t readOffset = 0;

            // 暗号は解かずそのままコピぺ
            while( readOffset < dataStorageSize )
            {
                const auto readSize = static_cast<int>(std::min<int64_t>(dataStorageSize - readOffset, BufferSize));

                NN_RESULT_DO(storageBuilder.ReadData(readOffset, buffer.get(), readSize));
                NN_RESULT_DO(storage.Write(writeOffset + readOffset, buffer.get(), readSize));

                readOffset += readSize;
            }
        }

        const auto dataSize = util::align_up(dataStorageSize, 1 << NcaHeader::Log2SectorSize);
        const auto nodeSize = storageBuilder.QueryTableNodeStorageSize();
        const auto entrySize = storageBuilder.QueryTableEntryStorageSize();

        // SparseStorage のテーブル用に暗号化ストレージを敷く
        std::unique_ptr<fs::IStorage> pEncryptStorage;
        NN_RESULT_DO(CreateSparseTableStorage(&pEncryptStorage, headerReader, storageBuilder));

        BucketTree::Header headerBuffer;
        fs::MemoryStorage headerStorage(&headerBuffer, sizeof(headerBuffer));

        // SparseStorage のテーブル情報を書き出し
        NN_RESULT_DO(storageBuilder.WriteTable(
            m_pAllocator,
            fs::SubStorage(&headerStorage, 0, NcaBucketInfo::HeaderSize),
            fs::SubStorage(pEncryptStorage.get(), dataSize, nodeSize),
            fs::SubStorage(pEncryptStorage.get(), dataSize + nodeSize, entrySize)
        ));

        // nca ヘッダの更新
        NN_RESULT_DO(UpdateFsHeader(&*m_pOriginalReader, headerReader));

        m_PhysicalOffset = physicalOffset + dataSize + nodeSize + entrySize;
    }
    NN_RESULT_SUCCESS;
}

Result NcaFileSystemDriverTest::SparsifyCodeStorage(int fsIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS_EQUAL(0, m_PhysicalOffset);
    NN_SDK_REQUIRES_RANGE(fsIndex, 0, NcaHeader::FsCountMax);

    utilTool::SparseStorageBuilder storageBuilder(utilTool::SparseStorageBuilder::BlockSizeMin, utilTool::SparseStorageBuilder::BlockSizeMin);
    NcaFsHeaderReader originalHeaderReader;
    NcaFileSystemDriver::SparseParam sparseParam(&storageBuilder, &originalHeaderReader, true, fsIndex, m_PhysicalOffset);

    NN_RESULT_DO(m_pDriver->SparsifyEmptyStorage(&sparseParam));

    // nca ヘッダの更新
    NN_RESULT_DO(UpdateFsHeader(&*m_pOriginalReader, originalHeaderReader));

    NN_RESULT_SUCCESS;
}

}}

namespace {

const int BlockSize = nnt::fs::util::NcaReaderBuilder::BlockSize;

int32_t GetExpectAesCtrKeyFlag(const nn::fssystem::NcaReader& ncaReader) NN_NOEXCEPT
{
    if( ncaReader.HasExternalDecryptionKey() )
    {
        return static_cast<int32_t>(nn::fs::AesCtrKeyTypeFlag::ExternalKeyForHardwareAes);
    }
    else if( ncaReader.HasInternalDecryptionKeyForAesHw() )
    {
        return static_cast<int32_t>(nn::fs::AesCtrKeyTypeFlag::InternalKeyForHardwareAes);
    }
    else
    {
        return static_cast<int32_t>(nn::fs::AesCtrKeyTypeFlag::InternalKeyForSoftwareAes);
    }
}

}

// nca ファイルの読み込みをテスト
TEST(NcaFileSystemDriver, ReadNca)
{
    nnt::fs::util::NcaReaderBuilder builder;
    auto ncaReader = builder.Make();
    auto fs0Data = builder.GetFs0Data();
    auto fs1Data = builder.GetFs1Data();

    nn::fssystem::NcaFileSystemDriver ncafs(ncaReader, nnt::fs::util::GetTestLibraryAllocator(), nnt::fs::util::GetBufferManager());
    {
        std::shared_ptr<nn::fs::IStorage> storage;
        NNT_ASSERT_RESULT_SUCCESS(ncafs.OpenStorage(&storage, 0));

        // 署名 2 検証
        NNT_EXPECT_RESULT_SUCCESS(ncaReader->VerifyHeaderSign2(nnt::fs::util::CryptoConfiguration.headerSign1KeyModulus, sizeof(nnt::fs::util::CryptoConfiguration.headerSign1KeyModulus)));

        static char readBuffer[3 * BlockSize];
        NNT_ASSERT_RESULT_SUCCESS(storage->Read(0, readBuffer, fs0Data.size));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(fs0Data.ptr, readBuffer, fs0Data.size);
    }

    {
        std::shared_ptr<nn::fs::IStorage> storage;
        NNT_ASSERT_RESULT_SUCCESS(ncafs.OpenStorage(&storage, 1));

        static char readBuffer[3 * BlockSize];
        NNT_ASSERT_RESULT_SUCCESS(storage->Read(0, readBuffer, 3 * BlockSize));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(fs1Data.ptr, readBuffer, fs1Data.size);
    }
}

// 1st パッチ用 nca ファイルの読み込みをテスト
TEST(NcaFileSystemDriver, ReadNca1stPatch)
{
    nnt::fs::util::NcaReaderBuilder builder(true);
    auto oldReader = builder.Make();
    auto newReader = builder.MakePatch();
    auto fs0Data = builder.GetFs0Data();
    auto newData = builder.GetNewData();

    nn::fssystem::NcaFileSystemDriver ncafs(
        oldReader,
        newReader,
        nnt::fs::util::GetTestLibraryAllocator(),
        nnt::fs::util::GetBufferManager()
    );
    {
        std::shared_ptr<nn::fs::IStorage> storage;
        NNT_ASSERT_RESULT_SUCCESS(ncafs.OpenStorage(&storage, 0));

        // 署名 2 検証
        const void* const ptr = nnt::fs::util::CryptoConfiguration.headerSign1KeyModulus;
        const size_t size = sizeof(nnt::fs::util::CryptoConfiguration.headerSign1KeyModulus);
        NNT_EXPECT_RESULT_SUCCESS(oldReader->VerifyHeaderSign2(ptr, size));

        nnt::fs::util::Vector<char> buffer(3 * BlockSize);
        NNT_ASSERT_RESULT_SUCCESS(storage->Read(0, buffer.data(), fs0Data.size));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(fs0Data.ptr, buffer.data(), fs0Data.size);
    }
    {
        std::shared_ptr<nn::fs::IStorage> storage;
        NNT_ASSERT_RESULT_SUCCESS(ncafs.OpenStorage(&storage, 1));

        nnt::fs::util::Vector<char> buffer(newData.size);
        NNT_ASSERT_RESULT_SUCCESS(storage->Read(0, buffer.data(), buffer.size()));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(newData.ptr, buffer.data(), buffer.size());
    }
}

// SparseStorage を適用した 1st パッチ用 nca ファイルの読み込みをテスト
TEST(NcaFileSystemDriver, ReadNca1stPatchSparseStorage)
{
    nnt::fs::util::NcaReaderBuilder builder(true);
    auto oldReader = builder.Make();
    auto newReader = builder.MakePatch();
    auto fs0Data = builder.GetFs0Data();
    auto newData = builder.GetNewData();

    nn::fssystem::NcaFileSystemDriver ncafs(
        oldReader,
        newReader,
        nnt::fs::util::GetTestLibraryAllocator(),
        nnt::fs::util::GetBufferManager()
    );

    const nn::fssystem::NcaFileSystemDriverTest::HeaderSign HeaderSign =
    {
        {
            nnt::fs::util::CryptoConfiguration.headerSign1KeyModulus,
            nn::fssystem::NcaCryptoConfiguration::Rsa2048KeyModulusSize,
            nnt::fs::util::Rsa2048KeyPrivateExponent,
            nn::fssystem::NcaCryptoConfiguration::Rsa2048KeyPrivateExponentSize,
        },
        {
            nnt::fs::util::CryptoConfiguration.headerSign1KeyModulus,
            nn::fssystem::NcaCryptoConfiguration::Rsa2048KeyModulusSize,
            nnt::fs::util::Rsa2048KeyPrivateExponent,
            nn::fssystem::NcaCryptoConfiguration::Rsa2048KeyPrivateExponentSize,
        },
    };

    // ncafs のスパース化
    bool isDataStorageSparsified = false;
    {
        nn::fssystem::NcaFileSystemDriverTest test(&ncafs);
        NNT_ASSERT_RESULT_SUCCESS(test.SparsifyStorage(&isDataStorageSparsified, 512, 512, HeaderSign));
    }

    // スパース化できているかチェック
    {
        nn::fssystem::NcaFsHeaderReader reader;

        NNT_ASSERT_RESULT_SUCCESS(nn::fssystem::NcaFileSystemDriver::SetupFsHeaderReader(&reader, *oldReader, 0));
        ASSERT_TRUE(reader.ExistsSparseLayer());

        NNT_ASSERT_RESULT_SUCCESS(nn::fssystem::NcaFileSystemDriver::SetupFsHeaderReader(&reader, *oldReader, 1));
        ASSERT_EQ(isDataStorageSparsified, reader.ExistsSparseLayer());
    }

    {
        std::shared_ptr<nn::fs::IStorage> storage;
        NNT_ASSERT_RESULT_SUCCESS(ncafs.OpenStorage(&storage, 0));

        // 署名 2 検証
        const void* const ptr = nnt::fs::util::CryptoConfiguration.headerSign1KeyModulus;
        const size_t size = sizeof(nnt::fs::util::CryptoConfiguration.headerSign1KeyModulus);
        NNT_EXPECT_RESULT_SUCCESS(oldReader->VerifyHeaderSign2(ptr, size));

        nnt::fs::util::Vector<char> buffer(3 * BlockSize);
        NNT_ASSERT_RESULT_SUCCESS(storage->Read(0, buffer.data(), fs0Data.size));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(fs0Data.ptr, buffer.data(), fs0Data.size);
    }
    {
        std::shared_ptr<nn::fs::IStorage> storage;
        NNT_ASSERT_RESULT_SUCCESS(ncafs.OpenStorage(&storage, 1));

        nnt::fs::util::Vector<char> buffer(newData.size);
        NNT_ASSERT_RESULT_SUCCESS(storage->Read(0, buffer.data(), buffer.size()));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(newData.ptr, buffer.data(), buffer.size());
    }
}

TEST(NcaFileSystemDriver, InvalidateCache)
{
    nnt::fs::util::NcaReaderBuilder builder(true);
    auto storageBuffer = builder.GetStorageBuffer();
    auto patchStorageBuffer = builder.GetPatchStorageBuffer();

    nn::fs::MemoryStorage oldBase(storageBuffer.ptr, storageBuffer.size);
    nnt::fs::util::AccessCountWrapperStorage oldCountBase(&oldBase);
    auto oldReader = nn::fssystem::AllocateShared<nn::fssystem::NcaReader>();
    NNT_ASSERT_RESULT_SUCCESS(oldReader->Initialize(&oldCountBase, nnt::fs::util::CryptoConfiguration));

    nn::fs::MemoryStorage newBase(patchStorageBuffer.ptr, patchStorageBuffer.size);
    nnt::fs::util::AccessCountWrapperStorage newCountBase(&newBase);
    auto newReader = nn::fssystem::AllocateShared<nn::fssystem::NcaReader>();
    NNT_ASSERT_RESULT_SUCCESS(newReader->Initialize(&newCountBase, nnt::fs::util::CryptoConfiguration));

    nn::fssystem::NcaFileSystemDriver ncaFs(
        oldReader,
        newReader,
        nnt::fs::util::GetTestLibraryAllocator(),
        nnt::fs::util::GetBufferManager()
    );

    nn::fssystem::NcaFsHeaderReader headerReader;
    std::shared_ptr<nn::fs::IStorage> storage;
    NNT_ASSERT_RESULT_SUCCESS(ncaFs.OpenStorage(&storage, &headerReader, 1));

    int64_t storageSize = 0;
    NNT_ASSERT_RESULT_SUCCESS(storage->GetSize(&storageSize));

    std::mt19937 rng(nnt::fs::util::GetRandomSeed());

    for( auto count = 0; count < 100; ++count )
    {
        char readBuffer[1] = {};
        const auto offset = std::uniform_int_distribution<int64_t>(0, storageSize - 1)(rng);

        NNT_EXPECT_RESULT_SUCCESS(storage->Read(offset, readBuffer, sizeof(readBuffer)));
        EXPECT_GT(oldCountBase.GetReadTimes() + newCountBase.GetReadTimes(), 0);
        oldCountBase.ResetAccessCounter();
        newCountBase.ResetAccessCounter();

        NNT_EXPECT_RESULT_SUCCESS(storage->Read(offset, readBuffer, sizeof(readBuffer)));
        EXPECT_EQ(0, oldCountBase.GetReadTimes() + newCountBase.GetReadTimes());
        NNT_EXPECT_RESULT_SUCCESS(storage->OperateRange(
            nn::fs::OperationId::Invalidate,
            0,
            std::numeric_limits<int64_t>::max()));
        oldCountBase.ResetAccessCounter();
        newCountBase.ResetAccessCounter();

        NNT_EXPECT_RESULT_SUCCESS(storage->Read(offset, readBuffer, sizeof(readBuffer)));
        EXPECT_GT(oldCountBase.GetReadTimes() + newCountBase.GetReadTimes(), 0);
        NNT_EXPECT_RESULT_SUCCESS(storage->OperateRange(
            nn::fs::OperationId::Invalidate,
            0,
            std::numeric_limits<int64_t>::max()));
        oldCountBase.ResetAccessCounter();
        newCountBase.ResetAccessCounter();
    }
}

TEST(NcaFileSystemDriver, OperateRangeQuery)
{
    nn::fs::SpeedEmulationMode mode = nn::fs::SpeedEmulationMode::None;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetSpeedEmulationMode(&mode));
    NN_UTIL_SCOPE_EXIT
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::SetSpeedEmulationMode(mode));
    };

    const auto queryRangeTestFunc = [](std::shared_ptr<nn::fssystem::NcaReader> pNcaReader) NN_NOEXCEPT
    {
        nn::fssystem::NcaFileSystemDriver ncafs(
            pNcaReader,
            nnt::fs::util::GetTestLibraryAllocator(),
            nnt::fs::util::GetBufferManager()
        );

        const auto expectAesCtrKeyFlag = GetExpectAesCtrKeyFlag(*pNcaReader);

        for( int index = 0; index <= 1; ++index )
        {
            std::shared_ptr<nn::fs::IStorage> storage;
            NNT_ASSERT_RESULT_SUCCESS(ncafs.OpenStorage(&storage, index));

            int64_t size = 0;
            NNT_ASSERT_RESULT_SUCCESS(storage->GetSize(&size));

            nn::fs::QueryRangeInfo info;

            NNT_ASSERT_RESULT_SUCCESS(nn::fs::SetSpeedEmulationMode(nn::fs::SpeedEmulationMode::None));
            NNT_ASSERT_RESULT_SUCCESS(
                storage->OperateRange(
                    &info,
                    sizeof(info),
                    nn::fs::OperationId::QueryRange,
                    0,
                    size,
                    nullptr,
                    0));
            EXPECT_EQ(0, info.speedEmulationTypeFlag);
            EXPECT_EQ(expectAesCtrKeyFlag, info.aesCtrKeyTypeFlag);

            NNT_ASSERT_RESULT_SUCCESS(nn::fs::SetSpeedEmulationMode(nn::fs::SpeedEmulationMode::Slower));
            NNT_ASSERT_RESULT_SUCCESS(
                storage->OperateRange(
                    &info,
                    sizeof(info),
                    nn::fs::OperationId::QueryRange,
                    0,
                    size,
                    nullptr,
                    0));
            EXPECT_EQ(0, info.speedEmulationTypeFlag);
            EXPECT_EQ(expectAesCtrKeyFlag, info.aesCtrKeyTypeFlag);
        }
    };

    nnt::fs::util::NcaReaderBuilder builder;
    auto ncaReader = builder.Make();
    {
        queryRangeTestFunc(ncaReader);

        ncaReader->SetExternalDecryptionKey(nnt::fs::util::BodyAesKey[2], sizeof(nnt::fs::util::BodyAesKey[2]));
        queryRangeTestFunc(ncaReader);
    }
}

TEST(NcaFileSystemDriver, OperateRangeQueryPatch)
{
    nn::fs::SpeedEmulationMode mode = nn::fs::SpeedEmulationMode::None;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetSpeedEmulationMode(&mode));
    NN_UTIL_SCOPE_EXIT
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::SetSpeedEmulationMode(mode));
    };

    nnt::fs::util::NcaReaderBuilder builder(true);
    auto oldReader = builder.Make();
    auto newReader = builder.MakePatch();
    auto newData = builder.GetNewData();

    newReader->SetExternalDecryptionKey(nnt::fs::util::BodyAesKey[2], sizeof(nnt::fs::util::BodyAesKey[2]));

    nn::fssystem::NcaFileSystemDriver ncaFs(
        oldReader,
        newReader,
        nnt::fs::util::GetTestLibraryAllocator(),
        nnt::fs::util::GetBufferManager()
    );

    const auto newExpectAesCtrKeyFlag = GetExpectAesCtrKeyFlag(*newReader);
    const auto oldExpectAesCtrKeyFlag = GetExpectAesCtrKeyFlag(*oldReader);

    for( int index = 0; index <= 1; ++index )
    {
        std::shared_ptr<nn::fs::IStorage> storage;
        NNT_ASSERT_RESULT_SUCCESS(ncaFs.OpenStorage(&storage, index));

        int64_t storageSize = 0;
        NNT_ASSERT_RESULT_SUCCESS(storage->GetSize(&storageSize));
        const int64_t querySize = BlockSize;
        for( int64_t offset = 0; offset < storageSize; offset += querySize )
        {
            const auto size = std::min(querySize, storageSize - offset);

            nn::fs::QueryRangeInfo info;

            NNT_ASSERT_RESULT_SUCCESS(nn::fs::SetSpeedEmulationMode(nn::fs::SpeedEmulationMode::None));
            NNT_ASSERT_RESULT_SUCCESS(storage->OperateRange(
                &info,
                sizeof(info),
                nn::fs::OperationId::QueryRange,
                offset,
                size,
                nullptr,
                0));
            EXPECT_EQ(0, info.speedEmulationTypeFlag);
            if( index == 0 )
            {
                EXPECT_EQ(newExpectAesCtrKeyFlag, info.aesCtrKeyTypeFlag) << offset;
            }
            else
            {
                EXPECT_NE(0, info.aesCtrKeyTypeFlag & (newExpectAesCtrKeyFlag | oldExpectAesCtrKeyFlag)) << offset;
            }

            NNT_ASSERT_RESULT_SUCCESS(nn::fs::SetSpeedEmulationMode(nn::fs::SpeedEmulationMode::Slower));
            NNT_ASSERT_RESULT_SUCCESS(storage->OperateRange(
                &info,
                sizeof(info),
                nn::fs::OperationId::QueryRange,
                offset,
                size,
                nullptr,
                0));
            if( index == 0 )
            {
                EXPECT_EQ(0, info.speedEmulationTypeFlag) << offset;
                EXPECT_EQ(newExpectAesCtrKeyFlag, info.aesCtrKeyTypeFlag) << offset;
            }
            else
            {
                EXPECT_EQ(0, info.speedEmulationTypeFlag) << offset;
                EXPECT_TRUE(
                    (info.aesCtrKeyTypeFlag == newExpectAesCtrKeyFlag) || // パッチのみの領域
                    (info.aesCtrKeyTypeFlag == oldExpectAesCtrKeyFlag) || // オリジナルのみの領域
                    (info.aesCtrKeyTypeFlag == (newExpectAesCtrKeyFlag | oldExpectAesCtrKeyFlag)) // パッチとオリジナルがどちらも含まれる領域
                );
            }
        }
    }
}

#if 0
// 指定した NCA ファイルから、指定したパスのファイルと一致するデータが読めること
// TODO: NCA 作成自動化
TEST(NcaFileSystemDriver, CompareBinary)
{
    // パスと NcaFsCountMax は適宜書き換え
    nnt::fs::util::TemporaryHostDirectory hostDirectory;
    hostDirectory.Create();
    const nnt::fs::util::String RootPath(hostDirectory.GetPath());
    const nnt::fs::util::String NcaFileName(RootPath + "/test.nca");
    const nnt::fs::util::String NcaCodePath(RootPath + "/code");
    const nnt::fs::util::String NcaRomPath(RootPath + "/rom");
    const int NcaFsCountMax = 2; // NcaReader から取得できない

    const int ncaBufferPoolSize = 4 * 1024 * 1024;
    static char ncaBufferPool[ncaBufferPoolSize];
    nn::mem::StandardAllocator stdAllocator(ncaBufferPool, ncaBufferPoolSize);
    nn::fssrv::MemoryResourceFromStandardAllocator allocator(&stdAllocator);

    nn::fssystem::HostFileSystem hostFs;
    hostFs.Initialize("");
    std::unique_ptr<nn::fs::fsa::IFile> file;
    NNT_ASSERT_RESULT_SUCCESS(hostFs.OpenFile(&file, NcaFileName.c_str(), nn::fs::OpenMode_Read));
    nn::fs::FileStorage storage(file.get());

    auto ncaReader = nn::fssystem::AllocateShared<nn::fssystem::NcaReader>();
    NNT_ASSERT_RESULT_SUCCESS(ncaReader->Initialize(&storage, nnt::fs::util::CryptoConfiguration));
    int ncaFsCount = ncaReader->GetFsCount();

    nn::fssystem::NcaFsHeaderReader ncaFsHeaderReader[NcaFsCountMax];
    for (int i = 0; i < ncaFsCount; i++)
    {
        nn::fssystem::Hash fsHeaderHash;
        ncaReader->GetFsHeaderHash(&fsHeaderHash, i);
        NNT_ASSERT_RESULT_SUCCESS(ncaFsHeaderReader[i].Initialize(*ncaReader, i));
    }

    nn::fssystem::NcaFileSystemDriver ncafs(ncaReader, &allocator, nnt::fs::util::GetBufferManager());

    for (int i = 0; i < ncaFsCount; i++)
    {
        std::shared_ptr<nn::fs::IStorage> dataStorage;
        NNT_ASSERT_RESULT_SUCCESS(ncafs.OpenStorage(&dataStorage, i));
        std::unique_ptr<nn::fssystem::HostFileSystem> ansFs(new nn::fssystem::HostFileSystem());

        size_t cacheBufferSize = 1024 * 1024;
        std::unique_ptr<char[]> cacheBuffer(new char[cacheBufferSize]);

        nn::fssystem::NcaFsHeader::FsType fsType = ncaFsHeaderReader[i].GetFsType();
        if (fsType == nn::fssystem::NcaFsHeader::FsType::RomFs)
        {
            NN_LOG("Read FS %d as RomFs\n", i);
            ansFs->Initialize(NcaRomPath.c_str());

            std::unique_ptr<nn::fs::RomFsFileSystem> romFs(new nn::fs::RomFsFileSystem());
            NNT_ASSERT_RESULT_SUCCESS(romFs->Initialize(dataStorage.get(), cacheBuffer.get(), cacheBufferSize, false));
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::fsa::Register("test", std::move(romFs)));
        }
        else if (fsType == nn::fssystem::NcaFsHeader::FsType::PartitionFs)
        {
            NN_LOG("Read FS %d as PartitionFs\n", i);
            ansFs->Initialize(NcaCodePath.c_str());

            std::unique_ptr<nn::fssystem::PartitionFileSystem> partFs(new nn::fssystem::PartitionFileSystem());
            NNT_ASSERT_RESULT_SUCCESS(partFs->Initialize(dataStorage.get()));
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::fsa::Register("test", std::move(partFs)));
        }
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::fsa::Unregister("test");
        };

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::fsa::Register("answer", std::move(ansFs)));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::fsa::Unregister("answer");
        };

        NNT_FS_ASSERT_NO_FATAL_FAILURE(nnt::fs::util::CompareDirectoryRecursive("test:/", "answer:/"));
    }
}
#endif // 0

#else // !defined(NNT_FS_NCA_FILE_SYSTEM_DRIVER_WITH_NCA_FILES)

namespace {

const int StorageSize = 4 * 1024 * 1024;
char g_StorageBuffer[StorageSize];

const int PatchStorageSize = StorageSize + 1024 * 1024;
char g_PatchStorageBuffer[PatchStorageSize];

const char* g_TestApplicationNcaFilePath = nullptr;
const char* g_TestPatchNcaFilePath = nullptr;

void LoadApplicationNca() NN_NOEXCEPT
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHostRoot());
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::UnmountHostRoot();
    };

    nn::fs::FileHandle file;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&file, g_TestApplicationNcaFilePath, nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(file);
    };

    int64_t size;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&size, file));
    ASSERT_LE(size, StorageSize);

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(file, 0, g_StorageBuffer, static_cast<size_t>(size)));
}

void LoadPatchNca() NN_NOEXCEPT
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHostRoot());
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::UnmountHostRoot();
    };

    nn::fs::FileHandle file;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&file, g_TestPatchNcaFilePath, nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(file);
    };

    int64_t size;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&size, file));
    ASSERT_LE(size, PatchStorageSize);

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(file, 0, g_PatchStorageBuffer, static_cast<size_t>(size)));
}

} // namespace

TEST(NcaFileSystemDriver, ReadHierarchicalIntegrityHashAplicationNca)
{
    static const int64_t BufferSize = 3 * 512 * 1024;

    ASSERT_NO_FATAL_FAILURE(LoadApplicationNca());
    nn::fs::MemoryStorage base(g_StorageBuffer, StorageSize);

    auto ncaReader = nn::fssystem::AllocateShared<nn::fssystem::NcaReader>();
    NNT_ASSERT_RESULT_SUCCESS(ncaReader->Initialize(&base, nnt::fs::util::CryptoConfiguration));

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    nn::fssystem::NcaFileSystemDriver ncafs(
        ncaReader,
        nnt::fs::util::GetTestLibraryAllocator(),
        nnt::fs::util::GetBufferManager()
    );
    {
        std::shared_ptr<nn::fs::IStorage> storage;
        NNT_ASSERT_RESULT_SUCCESS(ncafs.OpenStorage(&storage, 0));

        int64_t sizeStorage;
        NNT_ASSERT_RESULT_SUCCESS(storage->GetSize(&sizeStorage));
        ASSERT_LE(BufferSize, sizeStorage);

        nnt::fs::util::Vector<char> buffer(BufferSize);

        static const auto LoopCount = 100;
        for( int i = 0; i < LoopCount; ++i )
        {
            auto size = std::uniform_int_distribution<size_t>(0, buffer.size())(mt);
            auto offset = std::uniform_int_distribution<int64_t>(0, sizeStorage - static_cast<int64_t>(size))(mt);

            NNT_ASSERT_RESULT_SUCCESS(storage->Read(offset, buffer.data(), size));
        }
    }
}

TEST(NcaFileSystemDriver, ReadHierarchicalIntegrityHashPatchNca)
{
    static const int64_t BufferSize = 3 * 512 * 1024;

    ASSERT_NO_FATAL_FAILURE(LoadApplicationNca());
    nn::fs::MemoryStorage oldBase(g_StorageBuffer, StorageSize);
    auto oldReader = nn::fssystem::AllocateShared<nn::fssystem::NcaReader>();
    NNT_ASSERT_RESULT_SUCCESS(oldReader->Initialize(&oldBase, nnt::fs::util::CryptoConfiguration));

    ASSERT_NO_FATAL_FAILURE(LoadPatchNca());
    nn::fs::MemoryStorage newBase(g_PatchStorageBuffer, PatchStorageSize);
    auto newReader = nn::fssystem::AllocateShared<nn::fssystem::NcaReader>();
    NNT_ASSERT_RESULT_SUCCESS(newReader->Initialize(&newBase, nnt::fs::util::CryptoConfiguration));

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    nn::fssystem::NcaFileSystemDriver ncafs(
        oldReader,
        newReader,
        nnt::fs::util::GetTestLibraryAllocator(),
        nnt::fs::util::GetBufferManager()
    );
    {
        std::shared_ptr<nn::fs::IStorage> storage;
        NNT_ASSERT_RESULT_SUCCESS(ncafs.OpenStorage(&storage, 0));

        int64_t sizeStorage;
        NNT_ASSERT_RESULT_SUCCESS(storage->GetSize(&sizeStorage));
        ASSERT_LE(BufferSize, sizeStorage);

        nnt::fs::util::Vector<char> buffer(BufferSize);

        static const auto LoopCount = 100;
        for( int i = 0; i < LoopCount; ++i )
        {
            auto size = std::uniform_int_distribution<size_t>(0, buffer.size())(mt);
            auto offset = std::uniform_int_distribution<int64_t>(0, sizeStorage - static_cast<int64_t>(size))(mt);

            NNT_ASSERT_RESULT_SUCCESS(storage->Read(offset, buffer.data(), size));
        }
    }
}

TEST(NcaFileSystemDriver, RandomAllocationFailureForAplication)
{
    static const int64_t BufferSize = 3 * 512 * 1024;

    ASSERT_NO_FATAL_FAILURE(LoadApplicationNca());
    nn::fs::MemoryStorage base(g_StorageBuffer, StorageSize);

    auto ncaReader = nn::fssystem::AllocateShared<nn::fssystem::NcaReader>();
    NNT_ASSERT_RESULT_SUCCESS(ncaReader->Initialize(&base, nnt::fs::util::CryptoConfiguration));

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    nnt::fs::util::RandomFailureBufferManager randomFailureBufferManager(nnt::fs::util::GetBufferManager(), &mt);

    nn::fssystem::NcaFileSystemDriver ncafs(
        ncaReader,
        nnt::fs::util::GetTestLibraryAllocator(),
        &randomFailureBufferManager
    );
    {
        std::shared_ptr<nn::fs::IStorage> storage;
        NNT_ASSERT_RESULT_SUCCESS(ncafs.OpenStorage(&storage, 0));

        int64_t sizeStorage;
        NNT_ASSERT_RESULT_SUCCESS(storage->GetSize(&sizeStorage));
        ASSERT_LE(BufferSize, sizeStorage);

        nnt::fs::util::Vector<char> buffer(BufferSize);

        for( ; ; )
        {
            auto size = std::uniform_int_distribution<size_t>(0, buffer.size())(mt);
            auto offset = std::uniform_int_distribution<int64_t>(0, sizeStorage - static_cast<int64_t>(size))(mt);

            randomFailureBufferManager.SetDivider(10);
            auto result = storage->Read(offset, buffer.data(), size);
            if( result.IsFailure() )
            {
                NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultBufferAllocationFailed, result);
                break;
            }
        }
    }
}

TEST(NcaFileSystemDriver, RandomAllocationFailureForPatch)
{
    static const int64_t BufferSize = 3 * 512 * 1024;

    ASSERT_NO_FATAL_FAILURE(LoadApplicationNca());
    nn::fs::MemoryStorage oldBase(g_StorageBuffer, StorageSize);
    auto oldReader = nn::fssystem::AllocateShared<nn::fssystem::NcaReader>();
    NNT_ASSERT_RESULT_SUCCESS(oldReader->Initialize(&oldBase, nnt::fs::util::CryptoConfiguration));

    ASSERT_NO_FATAL_FAILURE(LoadPatchNca());
    nn::fs::MemoryStorage newBase(g_PatchStorageBuffer, PatchStorageSize);
    auto newReader = nn::fssystem::AllocateShared<nn::fssystem::NcaReader>();
    NNT_ASSERT_RESULT_SUCCESS(newReader->Initialize(&newBase, nnt::fs::util::CryptoConfiguration));

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    nnt::fs::util::RandomFailureBufferManager randomFailureBufferManager(nnt::fs::util::GetBufferManager(), &mt);

    nn::fssystem::NcaFileSystemDriver ncafs(
        oldReader,
        newReader,
        nnt::fs::util::GetTestLibraryAllocator(),
        &randomFailureBufferManager
    );
    {
        std::shared_ptr<nn::fs::IStorage> storage;
        NNT_ASSERT_RESULT_SUCCESS(ncafs.OpenStorage(&storage, 0));

        int64_t sizeStorage;
        NNT_ASSERT_RESULT_SUCCESS(storage->GetSize(&sizeStorage));
        ASSERT_LE(BufferSize, sizeStorage);

        nnt::fs::util::Vector<char> buffer(BufferSize);

        for( ; ; )
        {
            auto size = std::uniform_int_distribution<size_t>(0, buffer.size())(mt);
            auto offset = std::uniform_int_distribution<int64_t>(0, sizeStorage - static_cast<int64_t>(size))(mt);

            randomFailureBufferManager.SetDivider(10);
            auto result = storage->Read(offset, buffer.data(), size);
            if( result.IsFailure() )
            {
                NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultBufferAllocationFailed, result);
                break;
            }
        }
    }
}

#endif // !defined(NNT_FS_NCA_FILE_SYSTEM_DRIVER_WITH_NCA_FILES)

namespace {

void* Allocate(size_t size)
{
    return malloc(size);
}

void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    return free(p);
}

} // namespace

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

#if defined(NNT_FS_NCA_FILE_SYSTEM_DRIVER_WITH_NCA_FILES)
    if( argc <= 2 )
    {
        NN_LOG("Usage: <application nca file path> <patch nca file path>\n");
        nnt::Exit(1);
    }
    g_TestApplicationNcaFilePath = argv[1];
    g_TestPatchNcaFilePath = argv[2];
#endif // defined(NNT_FS_NCA_FILE_SYSTEM_DRIVER_WITH_NCA_FILES)

    static const size_t HeapSize = 64 * 1024 * 1024;
    static char s_HeapStack[HeapSize];

    nnt::fs::util::InitializeTestLibraryHeap(s_HeapStack, HeapSize);
    nn::fs::SetAllocator(Allocate, Deallocate);

    // TORIAEZU: nn::fssystem::InitializeAllocator() の設定のため
    nn::fs::MountHostRoot();
    nn::fs::UnmountHostRoot();

    int result = RUN_ALL_TESTS();

    nnt::Exit(result);
}
