﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <mutex>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/os.h>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

#include <nn/fssrv/fscreator/fssrv_HostFileSystemCreator.h>
#include <nn/fssrv/fscreator/fssrv_SubDirectoryFileSystemCreator.h>
#include <nn/fssrv/fscreator/fssrv_BuiltInStorageFileSystemCreatorHostFs.h>
#include <nn/fssrv/fscreator/fssrv_SdCardProxyFileSystemCreatorHostFs.h>
#include <nn/fssrv/fssrv_IFileSystemCreator.h>

#include <nn/fssystem/fs_AllocatorUtility.h>

#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_StorageId.h>
#include <nn/ncm/ncm_ContentStorage.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>

#include "detail/fssrv_FileSystemProxyCoreImpl.h"
#include <nn/fs/fsa/fs_IDirectory.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fssystem/fs_NcaFileSystemDriver.h>
#include <nn/fssystem/save/fs_JournalIntegritySaveDataFileSystemDriver.h>

#include "testFs_Unit_DummyFileSystem.h"
#include <nnt/fsUtil/testFs_util_function.h>

#include "detail/fssrv_ProgramRegistryManager.cpp"
#include "detail/fssrv_LocationResolverSet.cpp"

using namespace nn::fs;
using namespace nn::fs::detail;
using namespace nn::fssystem;
using namespace nn::fssrv;
using namespace nn::fssrv::detail;

#include <nnt/fsUtil/testFs_util_GlobalNewDeleteChecker.Impl.h>

namespace {
    const int HeapSize = 512 * 1024;
    char g_HeapStack[HeapSize];
    nn::lmem::HeapHandle g_HeapHandle = nullptr;
    bool g_HeapAllocateAvalilable = true;
    nn::os::Mutex g_HeapMutex(false);
    nnt::fs::util::TemporaryHostDirectory g_HostDirectory;
    char g_RootDirPath[256];

void InitializeMyHeap()
{
    std::lock_guard<nn::os::Mutex> scopedLock(g_HeapMutex);
    if( g_HeapHandle == nullptr )
    {
        g_HeapHandle = nn::lmem::CreateExpHeap(g_HeapStack, HeapSize, nn::lmem::CreationOption_DebugFill);
        NN_ASSERT_NOT_NULL(g_HeapHandle);
    }
}

void SetAllocateAvailable(bool value)
{
    std::lock_guard<nn::os::Mutex> scopedLock(g_HeapMutex);
    g_HeapAllocateAvalilable = value;
}

void* MyAllocate(size_t size)
{
    std::lock_guard<nn::os::Mutex> scopedLock(g_HeapMutex);
    if (!g_HeapAllocateAvalilable)
    {
        return nullptr;
    }
    return nn::lmem::AllocateFromExpHeap(g_HeapHandle, size);
}

void MyDeallocate(void* p, size_t size)
{
    std::lock_guard<nn::os::Mutex> scopedLock(g_HeapMutex);
    NN_UNUSED(size);
    return nn::lmem::FreeToExpHeap(g_HeapHandle, p);
}

void GenerateRandom(void* pHash, size_t sizeHash) NN_NOEXCEPT
{
    NN_ABORT();
    memset(pHash, 0, sizeHash);
}

}


typedef nnt::fs::util::CheckGlobalNewDeleteFlagTestFixture FileSystemProxyBasic;

class DeathTest : public nnt::fs::util::CheckGlobalNewDeleteFlagTestFixture
{
protected:
    DeathTest()
    {
        CheckGlobalNewDeleteFlagTestFixture::DisableCheck();
    }
};

// Win32 版の LR で new が呼ばれるので Global New Check はしない
class FileSystemProxyCoreImplPostConditionTest : public ::testing::Test
{
protected:
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::ncm::Initialize();
        NNT_EXPECT_RESULT_SUCCESS(nn::ncm::OpenContentMetaDatabase(&metaDb, nn::ncm::StorageId::BuildInSystem));
        NNT_EXPECT_RESULT_SUCCESS(nn::ncm::OpenContentStorage(&storage, nn::ncm::StorageId::BuildInSystem));
    }
    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::ncm::Finalize();
    }
private:
    nn::ncm::ContentMetaDatabase metaDb;
    nn::ncm::ContentStorage storage;
};

class FileSystemProxyCoreImplRaceConditionOpenAndClose : public nnt::fs::util::CheckGlobalNewDeleteFlagTestFixture
{
public:
    static const int ThreadCount = 2;

    struct ThreadParams
    {
        FileSystemProxyCoreImpl* fileSystemProxy;
    };

    static void ThreadFunction(void* arg)
    {
        const int loopNum = 10000;
        ThreadParams* params = static_cast<ThreadParams*>(arg);
        for (int i = 0; i < loopNum; i++)
        {
            std::shared_ptr<fsa::IFileSystem> fileSystem;
            NNT_EXPECT_RESULT_SUCCESS(params->fileSystemProxy->OpenHostFileSystem(&fileSystem, g_RootDirPath));
        }
    }
};

typedef DeathTest ProgramRegistryManagerPreConditionDeathTest;
typedef nnt::fs::util::CheckGlobalNewDeleteFlagTestFixture ProgramRegistryManagerPostConditionTest;
typedef nnt::fs::util::CheckGlobalNewDeleteFlagTestFixture ProgramRegistryManagerAllocationMemoryFailed;
typedef DeathTest ProgramRegistryManagerPointerArgumentNullDeathTest;

TEST(FileSystemProxyBasic, OpenBisFs)
{
    nn::fssrv::fscreator::HostFileSystemCreator hostFsCreator(true);
    nn::fssrv::fscreator::SubDirectoryFileSystemCreator subDirFsCreator;
    nn::fssrv::fscreator::BuiltInStorageFileSystemCreatorHostFs bisFsCreator(&hostFsCreator);
    nn::fssrv::fscreator::SdCardProxyFileSystemCreatorHostFs sdProxyFsCreator(&hostFsCreator);
    nn::fssrv::fscreator::FileSystemCreatorInterfaces fsCreatorInterfaces =
    {
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        &hostFsCreator,
        nullptr,
        &subDirFsCreator,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        &bisFsCreator,
        &sdProxyFsCreator,
    };
    FileSystemProxyCoreImpl fsp(&fsCreatorInterfaces, nullptr, GenerateRandom);

    std::shared_ptr<fsa::IFileSystem> fileSystem;
    NNT_ASSERT_RESULT_SUCCESS(fsp.OpenBisFileSystem(&fileSystem, "", nn::fs::BisPartitionId::System));
    fileSystem->DeleteFile("/test.dat");
    NNT_ASSERT_RESULT_SUCCESS(fileSystem->CreateFile("/test.dat", 1024));
    NNT_ASSERT_RESULT_SUCCESS(fileSystem->DeleteFile("/test.dat"));
}

TEST(FileSystemProxyBasic, OpenHostFs)
{
    nn::fssrv::fscreator::HostFileSystemCreator hostFsCreator(true);
    nn::fssrv::fscreator::SubDirectoryFileSystemCreator subDirFsCreator;
    nn::fssrv::fscreator::BuiltInStorageFileSystemCreatorHostFs bisFsCreator(&hostFsCreator);
    nn::fssrv::fscreator::SdCardProxyFileSystemCreatorHostFs sdProxyFsCreator(&hostFsCreator);
    nn::fssrv::fscreator::FileSystemCreatorInterfaces fsCreatorInterfaces =
    {
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        &hostFsCreator,
        nullptr,
        &subDirFsCreator,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        &bisFsCreator,
        &sdProxyFsCreator,
    };
    FileSystemProxyCoreImpl fsp(&fsCreatorInterfaces, nullptr, GenerateRandom);

    std::shared_ptr<fsa::IFileSystem> fileSystem;
    NNT_ASSERT_RESULT_SUCCESS(fsp.OpenHostFileSystem(&fileSystem, g_RootDirPath));
    fileSystem->DeleteFile("/test.dat");
    NNT_ASSERT_RESULT_SUCCESS(fileSystem->CreateFile("/test.dat", 1024));
    NNT_ASSERT_RESULT_SUCCESS(fileSystem->DeleteFile("/test.dat"));
}

TEST(FileSystemProxyBasic, OpenHostFsRoot)
{
    nn::fssrv::fscreator::HostFileSystemCreator hostFsCreator(true);
    nn::fssrv::fscreator::SubDirectoryFileSystemCreator subDirFsCreator;
    nn::fssrv::fscreator::BuiltInStorageFileSystemCreatorHostFs bisFsCreator(&hostFsCreator);
    nn::fssrv::fscreator::SdCardProxyFileSystemCreatorHostFs sdProxyFsCreator(&hostFsCreator);
    nn::fssrv::fscreator::FileSystemCreatorInterfaces fsCreatorInterfaces =
    {
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        &hostFsCreator,
        nullptr,
        &subDirFsCreator,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        &bisFsCreator,
        &sdProxyFsCreator,
    };
    FileSystemProxyCoreImpl fsp(&fsCreatorInterfaces, nullptr, GenerateRandom);

    std::shared_ptr<fsa::IFileSystem> fileSystem;
    NNT_ASSERT_RESULT_SUCCESS(fsp.OpenHostFileSystem(&fileSystem, ""));
    char testFilePath[256 + 9];
    memset(testFilePath, 0x00, sizeof(testFilePath));
    strncat(testFilePath, g_RootDirPath + 1, sizeof(g_RootDirPath) - 1);
    strncat(testFilePath, "/test.dat", 1 + 4 + 1 + 3);
    fileSystem->DeleteFile(testFilePath);
    NNT_ASSERT_RESULT_SUCCESS(fileSystem->CreateFile(testFilePath, 1024));
    NNT_ASSERT_RESULT_SUCCESS(fileSystem->DeleteFile(testFilePath));
}

TEST(FileSystemProxyFailed, OpenHostFs)
{
    nn::fssrv::fscreator::HostFileSystemCreator hostFsCreator(true);
    nn::fssrv::fscreator::SubDirectoryFileSystemCreator subDirFsCreator;
    nn::fssrv::fscreator::BuiltInStorageFileSystemCreatorHostFs bisFsCreator(&hostFsCreator);
    nn::fssrv::fscreator::SdCardProxyFileSystemCreatorHostFs sdProxyFsCreator(&hostFsCreator);
    nn::fssrv::fscreator::FileSystemCreatorInterfaces fsCreatorInterfaces =
    {
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        &hostFsCreator,
        nullptr,
        &subDirFsCreator,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        &bisFsCreator,
        &sdProxyFsCreator,
    };
    FileSystemProxyCoreImpl fsp(&fsCreatorInterfaces, nullptr, GenerateRandom);

    std::shared_ptr<fsa::IFileSystem> fileSystem;
    char testDirPath[256 + 5];
    memset(testDirPath, 0x00, sizeof(testDirPath));
    strncat(testDirPath, g_RootDirPath + 1, sizeof(g_RootDirPath) - 1);
    strncat(testDirPath, "/test", 1 + 4);
    NNT_EXPECT_RESULT_FAILURE(ResultPathNotFound, fsp.OpenHostFileSystem(&fileSystem, testDirPath));
}

// Todo: パス解釈テスト全般要追加
// LR で登録した Program のパスを OpenFileSystemByCurrentProcess で取得できる

// TODO: テスト整理
// OpenFileSystem/CloseFileSystem を複数スレッドから実行する
TEST_F(FileSystemProxyCoreImplRaceConditionOpenAndClose, OpenFileSystemCloseFileSystem)
{
    const size_t StackSize = 64 * 1024;
    static NN_ALIGNAS(4096) uint8_t stack[ThreadCount][StackSize];

    nn::os::ThreadType threads[ThreadCount];
    ThreadParams params[ThreadCount];

    nn::fssrv::fscreator::HostFileSystemCreator hostFsCreator(true);
    nn::fssrv::fscreator::SubDirectoryFileSystemCreator subDirFsCreator;
    nn::fssrv::fscreator::BuiltInStorageFileSystemCreatorHostFs bisFsCreator(&hostFsCreator);
    nn::fssrv::fscreator::SdCardProxyFileSystemCreatorHostFs sdProxyFsCreator(&hostFsCreator);
    nn::fssrv::fscreator::FileSystemCreatorInterfaces fsCreatorInterfaces =
    {
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        &hostFsCreator,
        nullptr,
        &subDirFsCreator,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        nullptr,
        &bisFsCreator,
        &sdProxyFsCreator,
    };
    FileSystemProxyCoreImpl fsp(&fsCreatorInterfaces, nullptr, GenerateRandom);

    for (int i = 0; i < ThreadCount; i++)
    {
        params[i].fileSystemProxy = &fsp;

        NNT_EXPECT_RESULT_SUCCESS(nn::os::CreateThread(&threads[i], ThreadFunction, &params[i], stack[i], StackSize, nn::os::DefaultThreadPriority));
        nn::os::StartThread(&threads[i]);
    }

    for (auto& t : threads)
    {
        nn::os::WaitThread(&t);
        nn::os::DestroyThread(&t);
    }
}

#if 0
TEST_F(FileSystemProxyCoreImplRaceConditionSetBisRoot, SetBisRootForHost)
{
}
#endif

TEST_F(ProgramRegistryManagerPreConditionDeathTest, RegisterProgram)
{
    ProgramRegistryManager prm;
    NNT_EXPECT_RESULT_SUCCESS(prm.RegisterProgram(0x1234, 0x1, 0, nullptr, 0, nullptr, 0));

    // ABORT を無くす修正により落ちなくなったので、コメントアウトしています。
//    EXPECT_DEATH(prm.RegisterProgram(0x1234, 0x1, 0, nullptr, 0, nullptr, 0), "");
}

TEST_F(ProgramRegistryManagerPreConditionDeathTest, UnregisterProgram)
{
    ProgramRegistryManager prm;
    EXPECT_DEATH(prm.UnregisterProgram(0x1234), "");
}

#if 0 // Win 版ではテスト不可
TEST_F(ProgramRegistryManagerPostConditionTest, RegisterProgramUnregisterProgramGetProgramInfo)
{
    ProgramRegistryManager prm;
    NNT_EXPECT_RESULT_SUCCESS(prm.RegisterProgram(0x1234, 0x1, 0, nullptr, 0, nullptr, 0));
    ProgramInfo* pProgramInfo;
    NNT_EXPECT_RESULT_SUCCESS(prm.GetProgramInfo(&pProgramInfo, 0x1234));
    EXPECT_EQ(pProgramInfo->GetProgramIdValue(), 0x1);
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultProgramInfoNotFound, prm.GetProgramInfo(&pProgramInfo, 0x1235));
    prm.UnregisterProgram(0x1234);
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultProgramInfoNotFound, prm.GetProgramInfo(&pProgramInfo, 0x1234));
}
#endif

TEST_F(ProgramRegistryManagerAllocationMemoryFailed, RegisterProgram)
{
    ProgramRegistryManager prm;
    SetAllocateAvailable(false);
    NNT_EXPECT_RESULT_FAILURE(ResultAllocationMemoryFailed, prm.RegisterProgram(0x1234, 0x1, 0, nullptr, 0, nullptr, 0));
    SetAllocateAvailable(true);
}

TEST_F(ProgramRegistryManagerPointerArgumentNullDeathTest, GetProgramInfo)
{
    ProgramRegistryManager prm;
    NNT_EXPECT_RESULT_SUCCESS(prm.RegisterProgram(0x1234, 0x1, 0, nullptr, 0, nullptr, 0));
    EXPECT_DEATH(prm.GetProgramInfo(nullptr, 0x1234), "");
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    InitializeMyHeap();
    nn::fs::SetAllocator(MyAllocate, MyDeallocate);
    nn::fssystem::InitializeAllocator(MyAllocate, MyDeallocate);

    g_HostDirectory.Create();
    memset(g_RootDirPath, 0x00, sizeof(g_RootDirPath));
    g_RootDirPath[0] = '/';
    strncat(g_RootDirPath, g_HostDirectory.GetPath(). c_str(), sizeof(g_RootDirPath) - 1);

    auto ret = RUN_ALL_TESTS();

    g_HostDirectory.Delete();

    nnt::Exit(ret);
}
