﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <numeric>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/fssystem/fs_AsynchronousAccess.h>
#include <nn/fssystem/fs_StorageFile.h>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>

namespace {
    class MemoryFile : public nn::fs::fsa::IFile, public nn::fs::detail::Newable
    {
        NN_DISALLOW_COPY(MemoryFile);

    public:
        MemoryFile() NN_NOEXCEPT
            : m_Memory()
        {
        }

        virtual ~MemoryFile() NN_NOEXCEPT NN_OVERRIDE
        {
        }

        virtual nn::Result DoRead(
            size_t* outValue,
            int64_t offset,
            void* buffer,
            size_t size,
            const nn::fs::ReadOption& option) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_SDK_REQUIRES_NOT_NULL(outValue);
            NN_UNUSED(option);
            if( static_cast<int64_t>(m_Memory.size()) < offset )
            {
                NN_RESULT_THROW(nn::fs::ResultOutOfRange());
            }
            *outValue = offset + size < m_Memory.size()
                ? size : static_cast<size_t>(m_Memory.size() - offset);
            std::copy(
                std::next(m_Memory.begin(), static_cast<size_t>(offset)),
                std::next(m_Memory.begin(), static_cast<size_t>(offset + *outValue)),
                reinterpret_cast<char*>(buffer));
            NN_RESULT_SUCCESS;
        }

        virtual nn::Result DoWrite(
            int64_t offset,
            const void* buffer,
            size_t size,
            const nn::fs::WriteOption& option) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(option);
            if( static_cast<int64_t>(m_Memory.size()) < offset )
            {
                NN_RESULT_THROW(nn::fs::ResultOutOfRange());
            }
            if( m_Memory.size() < offset + size )
            {
                m_Memory.resize(static_cast<size_t>(offset + size));
            }
            std::copy(
                reinterpret_cast<const char*>(buffer),
                reinterpret_cast<const char*>(buffer) + size,
                std::next(m_Memory.begin(), static_cast<size_t>(offset)));
            NN_RESULT_SUCCESS;
        }

        virtual nn::Result DoFlush() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_SUCCESS;
        }

        virtual nn::Result DoSetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            m_Memory.resize(static_cast<size_t>(size));
            NN_RESULT_SUCCESS;
        }

        virtual nn::Result DoGetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_SDK_REQUIRES_NOT_NULL(outValue);
            *outValue = static_cast<int64_t>(m_Memory.size());
            NN_RESULT_SUCCESS;
        }

        virtual nn::Result DoOperateRange(
            void*, size_t, nn::fs::OperationId, int64_t, int64_t, const void*, size_t
        ) NN_NOEXCEPT NN_OVERRIDE
        {
            return nn::fs::ResultUnsupportedOperation();
        }

    private:
        nnt::fs::util::Vector<char> m_Memory;
    };
}

template<typename FileType>
class FileTest : public ::testing::Test
{
protected:
    void CreateFile(std::unique_ptr<nn::fs::fsa::IFile>* outValue, nn::fs::OpenMode mode) NN_NOEXCEPT;

    virtual void CreateBaseFile(std::unique_ptr<nn::fs::fsa::IFile>* outValue, nn::fs::OpenMode mode) NN_NOEXCEPT
    {
        NN_UNUSED(mode);
        std::unique_ptr<nn::fs::fsa::IFile> pFile(new MemoryFile);
        *outValue = std::move(pFile);
    }
};

namespace {
    const auto PooledThreadCount = 3;
    const auto PooledThreadStackSize = 32 * 1024;
    nn::fssystem::PooledThread g_PooledThreads[PooledThreadCount] = {};
    NN_OS_ALIGNAS_THREAD_STACK
        char g_PooledThreadStack[PooledThreadCount * PooledThreadStackSize] = {};
    nn::fssystem::ThreadPool g_ThreadPool(g_PooledThreads, PooledThreadCount);
}

template<>
void FileTest<nn::fssystem::AsynchronousAccessFile>::CreateFile(
    std::unique_ptr<nn::fs::fsa::IFile>* outValue,
    nn::fs::OpenMode mode) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        g_ThreadPool.Initialize(g_PooledThreadStack, PooledThreadCount * PooledThreadStackSize));
    std::unique_ptr<nn::fs::fsa::IFile> pBaseFile;
    CreateBaseFile(&pBaseFile, mode);
    std::unique_ptr<nn::fs::fsa::IFile> pFile(
        new nn::fssystem::AsynchronousAccessFile(std::move(pBaseFile), mode, &g_ThreadPool));
    *outValue = std::move(pFile);
}

// 4 GB を超えるオフセットのテストフィクスチャ
template<typename FileType>
class LargeFileTest : public FileTest<FileType>
{
protected:
    static const size_t AccessSize = 1024;

protected:
    virtual void CreateBaseFile(std::unique_ptr<nn::fs::fsa::IFile>* outValue, nn::fs::OpenMode mode) NN_NOEXCEPT
    {
        const int64_t fileSize = nnt::fs::util::LargeOffsetMax + AccessSize;
        std::unique_ptr<nn::fs::fsa::IFile> pFile(new VirtualMemoryFile(fileSize, mode));
        *outValue = std::move(pFile);
    }

private:
    class VirtualMemoryFile : public nn::fs::fsa::IFile
    {
    public:
        VirtualMemoryFile(int64_t size, nn::fs::OpenMode mode) NN_NOEXCEPT
            : m_BaseStorage(size)
            , m_pStorageFile(new nn::fssystem::StorageFile(&m_BaseStorage, mode))
        {
        }

        virtual ~VirtualMemoryFile() NN_NOEXCEPT NN_OVERRIDE
        {
            // m_BaseStorage より前に解放するように明示
            m_pStorageFile.reset();
        }

        virtual nn::Result DoRead(size_t* outValue, int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(m_pStorageFile->Read(outValue, offset, buffer, size, option));
            NN_RESULT_SUCCESS;
        }

        virtual nn::Result DoWrite(int64_t offset, const void* buffer, size_t size, const nn::fs::WriteOption& option) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(m_pStorageFile->Write(offset, buffer, size, option));
            NN_RESULT_SUCCESS;
        }

        virtual nn::Result DoFlush() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(m_pStorageFile->Flush());
            NN_RESULT_SUCCESS;
        }

        virtual nn::Result DoSetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(m_pStorageFile->SetSize(size));
            NN_RESULT_SUCCESS;
        }

        virtual nn::Result DoGetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(m_pStorageFile->GetSize(outValue));
            NN_RESULT_SUCCESS;
        }

        virtual nn::Result DoOperateRange(
            void* outBuffer,
            size_t outBufferSize,
            nn::fs::OperationId operationId,
            int64_t offset,
            int64_t size,
            const void* inBuffer,
            size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(m_pStorageFile->OperateRange(
                outBuffer, outBufferSize, operationId, offset, size, inBuffer, inBufferSize));
            NN_RESULT_SUCCESS;
        }

    private:
        nnt::fs::util::VirtualMemoryStorage m_BaseStorage;
        std::unique_ptr<nn::fssystem::StorageFile> m_pStorageFile;
    };
};

TYPED_TEST_CASE(FileTest, nn::fssystem::AsynchronousAccessFile);
TYPED_TEST_CASE(LargeFileTest, nn::fssystem::AsynchronousAccessFile);

TYPED_TEST(FileTest, Random)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    std::unique_ptr<nn::fs::fsa::IFile> pFile;
    this->CreateFile(
        &pFile,
        static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read |
                                      nn::fs::OpenMode_Write |
                                      nn::fs::OpenMode_AllowAppend));

    static const auto FileSizeMin = 1;
    static const auto FileSizeMax = 1024 * 1024;
    static const auto LoopCountMax = 100;

    nnt::fs::util::Vector<char> readBuffer(FileSizeMax);
    nnt::fs::util::Vector<char> writeBuffer(FileSizeMax);

    std::iota(writeBuffer.begin(), writeBuffer.end(), '\x0');

    for( auto loopCount = 0; loopCount < LoopCountMax; ++loopCount )
    {
        const auto fileSize = std::uniform_int_distribution<int64_t>(FileSizeMin, FileSizeMax)(mt);
        NNT_ASSERT_RESULT_SUCCESS(pFile->SetSize(fileSize));

        // 範囲内アクセス
        {
            const auto offset = std::uniform_int_distribution<int64_t>(0, fileSize)(mt);
            const auto accessibleSize = static_cast<size_t>(fileSize - offset);
            const auto size = std::uniform_int_distribution<size_t>(0, accessibleSize)(mt);
            size_t readSize = 0;
            NNT_ASSERT_RESULT_SUCCESS(
                pFile->Write(offset, writeBuffer.data(), size, nn::fs::WriteOption()));
            NNT_ASSERT_RESULT_SUCCESS(
                pFile->Read(&readSize, offset, readBuffer.data(), size, nn::fs::ReadOption()));
            EXPECT_EQ(size, readSize);
            EXPECT_TRUE(
                std::equal(
                    writeBuffer.begin(),
                    std::next(writeBuffer.begin(), size),
                    readBuffer.begin()));
        }

        // 部分的範囲外アクセス
        {
            const auto offset = std::uniform_int_distribution<int64_t>(0, fileSize)(mt);
            const auto accessibleSize = static_cast<size_t>(fileSize - offset);
            const auto size = static_cast<size_t>(
                std::uniform_int_distribution<size_t>(0, accessibleSize)(mt) + fileSize);
            size_t readSize = 0;
            NNT_ASSERT_RESULT_SUCCESS(
                pFile->Write(offset, writeBuffer.data(), accessibleSize, nn::fs::WriteOption()));
            NNT_ASSERT_RESULT_SUCCESS(
                pFile->Read(&readSize, offset, readBuffer.data(), size, nn::fs::ReadOption()));
            EXPECT_EQ(accessibleSize, readSize);
            EXPECT_TRUE(std::equal(
                writeBuffer.begin(),
                std::next(writeBuffer.begin(), accessibleSize),
                readBuffer.begin()));
        }

        // 範囲外アクセス
        {
            const auto offset = std::uniform_int_distribution<int64_t>(1, fileSize)(mt) + fileSize;
            const auto size
                = std::uniform_int_distribution<size_t>(1, static_cast<size_t>(fileSize))(mt);
            size_t readSize = 0;
            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultOutOfRange,
                pFile->Read(&readSize, offset, readBuffer.data(), size, nn::fs::ReadOption()));
        }

        // 終端アクセス
        {
            const auto offset = fileSize;
            const auto size
                = std::uniform_int_distribution<size_t>(1, static_cast<size_t>(fileSize))(mt);
            size_t readSize = 0;
            NNT_ASSERT_RESULT_SUCCESS(
                pFile->Read(&readSize, offset, readBuffer.data(), size, nn::fs::ReadOption()));
            EXPECT_EQ(0, readSize);
        }
    }
}

// 4 GB を超えるオフセットのテスト
TYPED_TEST(LargeFileTest, ReadWriteLargeOffset)
{
    {
        std::unique_ptr<nn::fs::fsa::IFile> pFile;
        this->CreateFile(
            &pFile,
            static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read |
                                          nn::fs::OpenMode_Write));

        nnt::fs::util::TestFileAccessWithLargeOffset(pFile.get(), this->AccessSize);
        nnt::fs::util::TestFileAccessNotAllowAppend(pFile.get(), this->AccessSize);
    }
    {
        std::unique_ptr<nn::fs::fsa::IFile> pFile;
        this->CreateFile(
            &pFile,
            static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read |
                                          nn::fs::OpenMode_Write |
                                          nn::fs::OpenMode_AllowAppend));

        nnt::fs::util::TestFileAccessAllowAppend(pFile.get(), this->AccessSize);
    }
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);

    nn::fs::MountHostRoot();

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
