﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/fssystem/fs_AsynchronousAccess.h>
#include <nn/fssystem/fs_ServiceContext.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/nnt_Argument.h>

namespace {
    const int FileSize = 32;
    const auto PooledThreadCount = 3;
    const auto PooledThreadStackSize = 32 * 1024;
    nn::fssystem::PooledThread g_PooledThreads[PooledThreadCount] = {};
    NN_OS_ALIGNAS_THREAD_STACK
        char g_PooledThreadStack[PooledThreadCount * PooledThreadStackSize] = {};
    nn::fssystem::ThreadPool g_ThreadPool(g_PooledThreads, PooledThreadCount);
}

class AsynchronousAccessFileTest : public ::testing::Test, public ::testing::WithParamInterface<nn::fs::PriorityRaw>
{
protected:
    AsynchronousAccessFileTest() NN_NOEXCEPT
    {
        if( GetParam() == static_cast<nn::fs::PriorityRaw>(-1) )
        {
            nn::fssystem::UnregisterServiceContext();
        }
        else
        {
            nn::fssystem::RegisterServiceContext(&m_Context);
            m_Context.SetPriority(GetParam());
        }
    }

    virtual ~AsynchronousAccessFileTest() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::fssystem::UnregisterServiceContext();
    }

private:
    nn::fssystem::ServiceContext m_Context;
};

nn::Result OpenAsynchronousAccessFile(
    std::unique_ptr<nn::fssystem::AsynchronousAccessFile>* outAsynchronousFile,
    nnt::fs::util::RecordingAccessRangeFile** pOutRecordingAccessRangeFile,
    nn::fs::OpenMode mode,
    bool isWithAccessRangeRecorder) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        g_ThreadPool.Initialize(g_PooledThreadStack, PooledThreadCount * PooledThreadStackSize));
    std::unique_ptr<nn::fs::fsa::IFile> pBaseFile(new nnt::fs::util::MemoryFile);
    if( isWithAccessRangeRecorder )
    {
        *pOutRecordingAccessRangeFile = new nnt::fs::util::RecordingAccessRangeFile(std::move(pBaseFile));
        pBaseFile.reset(*pOutRecordingAccessRangeFile);
    }
    outAsynchronousFile->reset(new nn::fssystem::AsynchronousAccessFile(std::move(pBaseFile), mode, &g_ThreadPool));
    NN_RESULT_SUCCESS;
}

nn::Result OpenAsynchronousAccessFile(std::unique_ptr<nn::fs::fsa::IFile>* outValue, nn::fs::OpenMode mode) NN_NOEXCEPT
{
    std::unique_ptr<nn::fssystem::AsynchronousAccessFile> asynchronousFile;
    nnt::fs::util::RecordingAccessRangeFile* pAccessRangeRecorder;
    NN_RESULT_DO(OpenAsynchronousAccessFile(&asynchronousFile, &pAccessRangeRecorder, mode, false));
    *outValue = std::move(asynchronousFile);
    NN_RESULT_SUCCESS;
}

//! @brief 書き込みモードで開いたファイルを読み込み、ResultInvalidOperationForOpenMode が返却されることを確認する
TEST_P(AsynchronousAccessFileTest, InvalidOpenModeReadFile)
{
    std::unique_ptr<nn::fs::fsa::IFile> file;
    char buffer[FileSize + 1] = {0};
    size_t readSize = 0;
    NNT_EXPECT_RESULT_SUCCESS(OpenAsynchronousAccessFile(&file, nn::fs::OpenMode_Write));
    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultInvalidOperationForOpenMode,
        file->Read(&readSize, 0, buffer, FileSize, nn::fs::ReadOption()));
}

//! @brief 読み込みモードで開いたファイルに書き込み、ResultInvalidOperationForOpenMode が返却されることを確認する
TEST_P(AsynchronousAccessFileTest, InvalidOpenModeWriteFile)
{
    std::unique_ptr<nn::fs::fsa::IFile> file;
    char buffer[FileSize + 1] = {0};
    NNT_EXPECT_RESULT_SUCCESS(OpenAsynchronousAccessFile(&file, nn::fs::OpenMode_Read));
    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultInvalidOperationForOpenMode,
        file->Write(0, buffer, FileSize, nn::fs::WriteOption()));
}

//! @brief 読み込みモードで開いたファイルを SetSize し、ResultSuccess が返却されることを確認する
TEST_P(AsynchronousAccessFileTest, InvalidOpenModeSetFileSize)
{
    std::unique_ptr<nn::fs::fsa::IFile> file;
    NNT_EXPECT_RESULT_SUCCESS(OpenAsynchronousAccessFile(&file, nn::fs::OpenMode_Read));
    NNT_EXPECT_RESULT_SUCCESS(file->SetSize(0));
    NNT_EXPECT_RESULT_SUCCESS(file->SetSize(FileSize + 1));
}

//! @brief 読み込みモードで開いたファイルを Flush し、ResultSuccess が返却されることを確認する
TEST_P(AsynchronousAccessFileTest, InvalidOpenModeFlushFile)
{
    std::unique_ptr<nn::fs::fsa::IFile> file;
    NNT_EXPECT_RESULT_SUCCESS(OpenAsynchronousAccessFile(&file, nn::fs::OpenMode_Read));
    NNT_EXPECT_RESULT_SUCCESS(file->Flush());
}

TEST_P(AsynchronousAccessFileTest, AlignmentAndSplitSizeCheckForReading)
{
    std::unique_ptr<nn::fssystem::AsynchronousAccessFile> pFile;
    nnt::fs::util::RecordingAccessRangeFile* pRecorder;
    NNT_EXPECT_RESULT_SUCCESS(OpenAsynchronousAccessFile(&pFile, &pRecorder, static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write), true));

    NNT_ASSERT_RESULT_SUCCESS(pFile->SetSize(2 * 1024 * 1024));

    auto priority = GetParam();
    nnt::fs::util::AsynchronousAccessAlignmentAndSplitSizeChecker checker(
        priority,
        pRecorder->GetRecorder(),
        pFile.get()
    );
    checker.CheckForReading();
}

TEST_P(AsynchronousAccessFileTest, AlignmentAndSplitSizeCheckForWriting)
{
    std::unique_ptr<nn::fssystem::AsynchronousAccessFile> pFile;
    nnt::fs::util::RecordingAccessRangeFile* pRecorder;
    NNT_EXPECT_RESULT_SUCCESS(OpenAsynchronousAccessFile(&pFile, &pRecorder, nn::fs::OpenMode_Write, true));

    NNT_ASSERT_RESULT_SUCCESS(pFile->SetSize(2 * 1024 * 1024));

    auto priority = GetParam();
    nnt::fs::util::AsynchronousAccessAlignmentAndSplitSizeChecker checker(
        priority,
        pRecorder->GetRecorder(),
        pFile.get()
    );
    checker.CheckForWriting();
}

nn::fs::PriorityRaw priorityParam[] = {
    static_cast<nn::fs::PriorityRaw>(-1),
    nn::fs::PriorityRaw_Background,
    nn::fs::PriorityRaw_Low,
    nn::fs::PriorityRaw_Normal,
    nn::fs::PriorityRaw_Realtime
};

INSTANTIATE_TEST_CASE_P(WithPriority,
                        AsynchronousAccessFileTest,
                        ::testing::ValuesIn(priorityParam));
