﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstdlib>
#include <mutex>
#include <random>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/os/os_Argument.h>
#include <nn/os/os_Debug.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_Tick.h>
#include <nn/os/os_Thread.h>

#include <nn/fs.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/detail/fs_LruFileDataCacheSystem.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>


#define LRU_CACHE_SYSTEM_TEST_INFO_LOG(...)
#define LRU_CACHE_SYSTEM_TEST_ERROR_LOG(...)  NN_LOG(__VA_ARGS__)

namespace
{

struct ThreadInfo
{
    static const size_t ThreadStackSize = 32768;
    NN_OS_ALIGNAS_THREAD_STACK nn::Bit8 threadStack[ThreadStackSize];
    nn::os::ThreadType threadType;
};
const int MaxThreadInfoCount = 9;
ThreadInfo g_ThreadInfo[MaxThreadInfoCount];

// ----------------------------------------------------------------------------

const size_t StorageSize = 64 * 1024 * 1024;
const int StorageCount = 3;
nn::Bit8* g_pStorage[StorageCount];

struct InitializeStorageArg
{
    int64_t seed;
    int id;
};

void InitializeStorage(void* theArgs)
{
    InitializeStorageArg* args = static_cast<InitializeStorageArg*>(theArgs);

    std::mt19937 mt(static_cast<unsigned int>(args->seed));
    std::uniform_int_distribution<int> dist(0, 255);

    int id = args->id;

    g_pStorage[id] = new nn::Bit8[StorageSize];
    NN_ABORT_UNLESS_NOT_NULL(g_pStorage[id]);
    for (int i = 0; i < StorageSize; i++)
    {
        g_pStorage[id][i] = static_cast<nn::Bit8>(dist(mt));
    }
}

// ----------------------------------------------------------------------------

class VirtualStorage : public nn::fs::IStorage
{
private:
    void* m_pStorage;
    size_t m_StorageSize;
public:
    explicit VirtualStorage(void* pStorage, size_t storageSize) NN_NOEXCEPT
        : m_pStorage(pStorage)
        , m_StorageSize(storageSize)
    {
        if (NN_STATIC_CONDITION(sizeof(size_t) == 8))
        {
            NN_ABORT_UNLESS(storageSize <= INT64_MAX);
        }
    }
    virtual nn::Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_ABORT_UNLESS(offset >= 0);

        if (offset >= static_cast<int64_t>(m_StorageSize))
        {
            NN_RESULT_THROW(nn::fs::ResultOutOfRange());
        }

        std::memcpy(buffer, reinterpret_cast<nn::Bit8*>(m_pStorage) + offset, std::min(size, m_StorageSize - static_cast<size_t>(offset)));
        NN_RESULT_SUCCESS;
    }
    virtual nn::Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_ABORT("not implemented");
    }
    virtual nn::Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        *outValue = static_cast<int64_t>(m_StorageSize);
        NN_RESULT_SUCCESS;
    }
};

class VirtualOccasionallyBrokenStorage : public nn::fs::IStorage
{
private:
    std::mt19937 m_Mt;
    std::uniform_int_distribution<int> m_Distribution;
    nn::os::Mutex m_Lock;
    nn::fs::IStorage* m_pStorage;
public:
    typedef nn::fs::ResultNotImplemented ErrorResult;

    explicit VirtualOccasionallyBrokenStorage(nn::fs::IStorage* pStorage) NN_NOEXCEPT
        : m_Distribution(0, 1000)
        , m_Lock(false)
        , m_pStorage(pStorage)
    {
    }
    virtual nn::Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        int random;
        {
            std::lock_guard<nn::os::Mutex> lk(m_Lock);
            random = m_Distribution(m_Mt);
        }
        if (random == 0)
        {
            return ErrorResult();
        }
        return m_pStorage->Read(offset, buffer, size);
    }
    virtual nn::Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pStorage->Flush();
    }
    virtual nn::Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pStorage->GetSize(outValue);
    }
};

// ----------------------------------------------------------------------------

struct ReadTestArg
{
    nn::fs::IStorage* pStorage;
    nn::fs::detail::LruFileDataCacheSystem* pLruCacheSystem;
    nn::Bit8* pRawStorage;

    int threadId;
    int trialCount;
    nn::TimeSpan trialTime;
    int64_t seed;
    size_t stride;

    bool success;
};

void SequentialReadStorageTest(void* theArgs)
{
    ReadTestArg* args = static_cast<ReadTestArg*>(theArgs);
    args->success = true;

    std::mt19937 mt(static_cast<unsigned int>(args->seed));
    std::uniform_int_distribution<size_t> dist(0, args->pLruCacheSystem->GetCacheSize());

    auto getStride = [&]() -> size_t
    {
        if (args->stride == 0)
        {
            return dist(mt);
        }
        else
        {
            return args->stride;
        }
    };

    LRU_CACHE_SYSTEM_TEST_INFO_LOG("sequential read test: seed = %lld, stride = %zd %s\n", args->seed, args->stride, args->stride == 0 ? "(random)" : "");

    int64_t offset = 0;
    for (int trial = 0; trial < args->trialCount; trial++)
    {
        size_t stride = getStride();
        size_t size = stride;
        if (size > (StorageSize - offset))
        {
            size = StorageSize - static_cast<size_t>(offset);
        }

        nn::Bit8* pFileBuffer = new nn::Bit8[size + 16];
        NN_ABORT_UNLESS_NOT_NULL(pFileBuffer);
        NN_UTIL_SCOPE_EXIT
        {
            delete[] pFileBuffer;
        };
        std::memset(pFileBuffer, 0xcd, 8);
        std::memset(pFileBuffer + 8 + size, 0xcd, 8);

        nn::fs::detail::FileDataCacheAccessResult cacheAccessResult;
        nn::Result result = args->pLruCacheSystem->Read(args->pStorage, offset, pFileBuffer + 8, size, 0, StorageSize, &cacheAccessResult);

        if (result.IsSuccess())
        {
            if (!nnt::fs::util::IsFilledWithValue(pFileBuffer, 8, 0xcd))
            {
                LRU_CACHE_SYSTEM_TEST_ERROR_LOG("memory corruption (head), seed = %lld, stride = %zd, trial = %d\n", args->seed, args->stride, trial);
                args->success = false;
            }
            if (std::memcmp(pFileBuffer + 8, args->pRawStorage + offset, size) != 0)
            {
                LRU_CACHE_SYSTEM_TEST_ERROR_LOG("compare failed, seed = %lld, stride = %zd, trial = %d\n", args->seed, args->stride, trial);
                args->success = false;
            }
            if (!nnt::fs::util::IsFilledWithValue(pFileBuffer + 8 + size, 8, 0xcd))
            {
                LRU_CACHE_SYSTEM_TEST_ERROR_LOG("memory corruption (tail), seed = %lld, stride = %zd, trial = %d\n", args->seed, args->stride, trial);
                args->success = false;
            }
        }
        else if (VirtualOccasionallyBrokenStorage::ErrorResult::Includes(result))
        {
            // 何もしない
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }

        if (!args->success)
        {
            break;
        }

        offset = (offset + stride) % StorageSize;
    }
}

void RandomReadStorageTest(void* theArgs)
{
    ReadTestArg* args = static_cast<ReadTestArg*>(theArgs);
    args->success = true;

    std::mt19937 mt(static_cast<unsigned int>(args->seed));
    std::uniform_int_distribution<int64_t> dist(0, StorageSize - 1);

    nn::os::Tick startTick = nn::os::GetSystemTick();

    LRU_CACHE_SYSTEM_TEST_INFO_LOG("random read test: seed = %lld\n", args->seed);

    for (int trial = 0; trial < args->trialCount || args->trialCount == 0; trial++)
    {
        int64_t offset = dist(mt);
        size_t size = static_cast<size_t>(dist(mt));
        while (size > (StorageSize - offset))
        {
            size = static_cast<size_t>(dist(mt));
        }

        nn::Bit8* pFileBuffer = new nn::Bit8[size + 16];
        NN_ABORT_UNLESS_NOT_NULL(pFileBuffer);
        NN_UTIL_SCOPE_EXIT
        {
            delete[] pFileBuffer;
        };
        std::memset(pFileBuffer, 0xcd, 8);
        std::memset(pFileBuffer + 8 + size, 0xcd, 8);

        nn::fs::detail::FileDataCacheAccessResult cacheAccessResult;
        nn::Result result = args->pLruCacheSystem->Read(args->pStorage, offset, pFileBuffer + 8, size, 0, StorageSize, &cacheAccessResult);

        if (result.IsSuccess())
        {
            if (!nnt::fs::util::IsFilledWithValue(pFileBuffer, 8, 0xcd))
            {
                LRU_CACHE_SYSTEM_TEST_ERROR_LOG("memory corruption (head), seed = %lld, trial = %d\n", args->seed, trial);
                args->success = false;
            }
            if (std::memcmp(pFileBuffer + 8, args->pRawStorage + offset, size) != 0)
            {
                LRU_CACHE_SYSTEM_TEST_ERROR_LOG("compare failed, seed = %lld, trial = %d\n", args->seed, trial);
                args->success = false;
            }
            if (!nnt::fs::util::IsFilledWithValue(pFileBuffer + 8 + size, 8, 0xcd))
            {
                LRU_CACHE_SYSTEM_TEST_ERROR_LOG("memory corruption (tail), seed = %lld, trial = %d\n", args->seed, trial);
                args->success = false;
            }
        }
        else if (VirtualOccasionallyBrokenStorage::ErrorResult::Includes(result))
        {
            // 何もしない
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }

        if (!args->success)
        {
            break;
        }

        if (args->trialTime != nn::TimeSpan(0))
        {
            nn::os::Tick currentTick = nn::os::GetSystemTick();
            nn::TimeSpan elapsedTime = nn::os::ConvertToTimeSpan(currentTick - startTick);
            if (elapsedTime > args->trialTime)
            {
                break;
            }
        }
    }
}

// ----------------------------------------------------------------------------

const size_t MinAcceptableCacheMemorySize = 24 * 1024;  // FS API の事前条件とは関係なく LruFileDataCacheSystem の最小値

size_t GetMaxAllocatableCacheMemorySize() NN_NOEXCEPT
{
    const size_t memorySizeUsedByTest = StorageSize * StorageCount + StorageSize * 2;

#if defined(NN_BUILD_TARGET_PLATFORM_WIN)

    // TORIAEZU: 適当なサイズを設定
    const size_t maxAllocatableMemorySize = 1 * 1024 * 1024 * 1024 - memorySizeUsedByTest;

#elif defined(NN_BUILD_TARGET_PLATFORM_NX)

    nn::os::MemoryInfo memoryInfo;
    nn::os::QueryMemoryInfo(&memoryInfo);
    // 利用可能なメモリからテストで使われる分を引く
    NN_ABORT_UNLESS(memoryInfo.allocatedMemoryHeapSize > memorySizeUsedByTest);
    const size_t maxAllocatableMemorySize = memoryInfo.allocatedMemoryHeapSize - memorySizeUsedByTest;

#else
#error "unknown platform"
#endif

    // 二分探索で実際に割り当てられそうなメモリサイズを探す
    size_t upperLimit = maxAllocatableMemorySize;
    size_t lowerLimit = 0;

    while (upperLimit - lowerLimit > 1 * 1024 * 1024)  // 厳密な値は要らないので適当なところで切り上げる
    {
        size_t testSize = lowerLimit + (upperLimit - lowerLimit) / 2;
        void* p = std::malloc(testSize);

        if (p)
        {
            std::free(p);
            lowerLimit = testSize;
        }
        else
        {
            upperLimit = testSize;
        }
    }

    const size_t availableMemorySize = lowerLimit;
    NN_ABORT_UNLESS(availableMemorySize > 0);

    return availableMemorySize;
}

}  // namespace unnamed


TEST(LruFileDataCacheSystem, SingleThreadSequentialReadIdenticalStride)
{
    VirtualStorage virtualStorage(g_pStorage[0], StorageSize);
    VirtualOccasionallyBrokenStorage virtualOccasionallyBrokenStorage(&virtualStorage);

    const size_t cacheSize = 8 * 1024 * 1024;
    void* pCacheBuffer = std::malloc(cacheSize);
    NN_ABORT_UNLESS_NOT_NULL(pCacheBuffer);
    NN_UTIL_SCOPE_EXIT
    {
        std::free(pCacheBuffer);
    };
    {
        nn::fs::detail::LruFileDataCacheSystem lruCacheSystem;
        NN_ABORT_UNLESS_RESULT_SUCCESS(lruCacheSystem.Initialize(pCacheBuffer, cacheSize));

        ReadTestArg args;
        args.pLruCacheSystem = &lruCacheSystem;
        args.pStorage = &virtualOccasionallyBrokenStorage;
        args.pRawStorage = g_pStorage[0];
        args.success = false;
        args.threadId = 0;
        args.trialCount = 262144;
        args.trialTime = nn::TimeSpan(0);
        args.seed = 0;
        args.stride = 16 * 1024;

        SequentialReadStorageTest(&args);
        EXPECT_TRUE(args.success);
    }
}

TEST(LruFileDataCacheSystem, SingleThreadSequentialReadRandomStride)
{
    VirtualStorage virtualStorage(g_pStorage[0], StorageSize);
    VirtualOccasionallyBrokenStorage virtualOccasionallyBrokenStorage(&virtualStorage);

    const size_t cacheSize = 8 * 1024 * 1024;
    void* pCacheBuffer = std::malloc(cacheSize);
    NN_ABORT_UNLESS_NOT_NULL(pCacheBuffer);
    NN_UTIL_SCOPE_EXIT
    {
        std::free(pCacheBuffer);
    };
    {
        nn::fs::detail::LruFileDataCacheSystem lruCacheSystem;
        NN_ABORT_UNLESS_RESULT_SUCCESS(lruCacheSystem.Initialize(pCacheBuffer, cacheSize));

        ReadTestArg args;
        args.pLruCacheSystem = &lruCacheSystem;
        args.pStorage = &virtualOccasionallyBrokenStorage;
        args.pRawStorage = g_pStorage[0];
        args.success = false;
        args.threadId = 0;
        args.trialCount = 1024;
        args.trialTime = nn::TimeSpan(0);
        args.seed = nn::os::GetSystemTick().GetInt64Value();
        args.stride = 0;

        SequentialReadStorageTest(&args);
        EXPECT_TRUE(args.success);
    }
}

TEST(LruFileDataCacheSystem, SingleThreadRandomRead)
{
    VirtualStorage virtualStorage(g_pStorage[0], StorageSize);
    VirtualOccasionallyBrokenStorage virtualOccasionallyBrokenStorage(&virtualStorage);

    const size_t cacheSize = 8 * 1024 * 1024;
    void* pCacheBuffer = std::malloc(cacheSize);
    NN_ABORT_UNLESS_NOT_NULL(pCacheBuffer);
    NN_UTIL_SCOPE_EXIT
    {
        std::free(pCacheBuffer);
    };
    {
        nn::fs::detail::LruFileDataCacheSystem lruCacheSystem;
        NN_ABORT_UNLESS_RESULT_SUCCESS(lruCacheSystem.Initialize(pCacheBuffer, cacheSize));

        ReadTestArg args;
        args.pLruCacheSystem = &lruCacheSystem;
        args.pStorage = &virtualOccasionallyBrokenStorage;
        args.pRawStorage = g_pStorage[0];
        args.success = false;
        args.threadId = 0;
        args.trialCount = 1024;
        args.trialTime = nn::TimeSpan(0);
        args.seed = nn::os::GetSystemTick().GetInt64Value();
        args.stride = 0;

        RandomReadStorageTest(&args);
        EXPECT_TRUE(args.success);
    }
}

TEST(LruFileDataCacheSystem, MultiThreadSequentialReadIdenticalStride)
{
    const int ThreadCount = 3;

    VirtualStorage virtualStorage(g_pStorage[0], StorageSize);
    VirtualOccasionallyBrokenStorage virtualOccasionallyBrokenStorage(&virtualStorage);

    const size_t cacheSize = 8 * 1024 * 1024;
    void* pCacheBuffer = std::malloc(cacheSize);
    NN_ABORT_UNLESS_NOT_NULL(pCacheBuffer);
    NN_UTIL_SCOPE_EXIT
    {
        std::free(pCacheBuffer);
    };
    {
        nn::fs::detail::LruFileDataCacheSystem lruCacheSystem;
        NN_ABORT_UNLESS_RESULT_SUCCESS(lruCacheSystem.Initialize(pCacheBuffer, cacheSize));

        ReadTestArg args[ThreadCount];
        for (int i = 0; i < ThreadCount; i++)
        {
            args[i].pLruCacheSystem = &lruCacheSystem;
            args[i].pStorage = &virtualOccasionallyBrokenStorage;
            args[i].pRawStorage = g_pStorage[0];
            args[i].success = false;
            args[i].threadId = i;
            args[i].trialCount = 65536;
            args[i].trialTime = nn::TimeSpan(0);
            args[i].seed = 0;
            args[i].stride = 16 * 1024;
        }

        for (int i = 0; i < ThreadCount; i++)
        {
            nn::os::CreateThread(
                &g_ThreadInfo[i].threadType,
                SequentialReadStorageTest,
                &args[i],
                g_ThreadInfo[i].threadStack,
                g_ThreadInfo[i].ThreadStackSize,
                nn::os::DefaultThreadPriority,
                i);
            nn::os::StartThread(&g_ThreadInfo[i].threadType);
        }
        for (int i = 0; i < ThreadCount; i++)
        {
            nn::os::WaitThread(&g_ThreadInfo[i].threadType);
            nn::os::DestroyThread(&g_ThreadInfo[i].threadType);
        }
        for (int i = 0; i < ThreadCount; i++)
        {
            EXPECT_TRUE(args[i].success);
        }
    }
}

TEST(LruFileDataCacheSystem, MultiThreadRandomReadRandomSeed)
{
    const int ThreadCount = 3;

    VirtualStorage virtualStorage(g_pStorage[0], StorageSize);
    VirtualOccasionallyBrokenStorage virtualOccasionallyBrokenStorage(&virtualStorage);

    const size_t cacheSize = 8 * 1024 * 1024;
    void* pCacheBuffer = std::malloc(cacheSize);
    NN_ABORT_UNLESS_NOT_NULL(pCacheBuffer);
    NN_UTIL_SCOPE_EXIT
    {
        std::free(pCacheBuffer);
    };
    {
        nn::fs::detail::LruFileDataCacheSystem lruCacheSystem;
        NN_ABORT_UNLESS_RESULT_SUCCESS(lruCacheSystem.Initialize(pCacheBuffer, cacheSize));

        ReadTestArg args[ThreadCount];
        for (int i = 0; i < ThreadCount; i++)
        {
            args[i].pLruCacheSystem = &lruCacheSystem;
            args[i].pStorage = &virtualOccasionallyBrokenStorage;
            args[i].pRawStorage = g_pStorage[0];
            args[i].success = false;
            args[i].threadId = i;
            args[i].trialCount = 256;
            args[i].trialTime = nn::TimeSpan(0);
            args[i].seed = nn::os::GetSystemTick().GetInt64Value();
            // Tick の値を変えたいのでちょっと待つ
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
        }

        for (int i = 0; i < ThreadCount; i++)
        {
            nn::os::CreateThread(
                &g_ThreadInfo[i].threadType,
                RandomReadStorageTest,
                &args[i],
                g_ThreadInfo[i].threadStack,
                g_ThreadInfo[i].ThreadStackSize,
                nn::os::DefaultThreadPriority,
                i);
            nn::os::StartThread(&g_ThreadInfo[i].threadType);
        }
        for (int i = 0; i < ThreadCount; i++)
        {
            nn::os::WaitThread(&g_ThreadInfo[i].threadType);
            nn::os::DestroyThread(&g_ThreadInfo[i].threadType);
        }
        for (int i = 0; i < ThreadCount; i++)
        {
            EXPECT_TRUE(args[i].success);
        }
    }
}

TEST(LruFileDataCacheSystem, MultiThreadPrioritizedSequentialReadIdenticalStride)
{
    const int CoreCount = 3;
    const int ThreadCountPerCore = 3;

    VirtualStorage virtualStorage(g_pStorage[0], StorageSize);
    VirtualOccasionallyBrokenStorage virtualOccasionallyBrokenStorage(&virtualStorage);

    const size_t cacheSize = 8 * 1024 * 1024;
    void* pCacheBuffer = std::malloc(cacheSize);
    NN_ABORT_UNLESS_NOT_NULL(pCacheBuffer);
    NN_UTIL_SCOPE_EXIT
    {
        std::free(pCacheBuffer);
    };
    {
        nn::fs::detail::LruFileDataCacheSystem lruCacheSystem;
        NN_ABORT_UNLESS_RESULT_SUCCESS(lruCacheSystem.Initialize(pCacheBuffer, cacheSize));

        ReadTestArg args[CoreCount * ThreadCountPerCore];
        for (int core = 0; core < CoreCount; core++)
        {
            for (int thread = 0; thread < ThreadCountPerCore; thread++)
            {
                int n = core * ThreadCountPerCore + thread;

                args[n].pLruCacheSystem = &lruCacheSystem;
                args[n].pStorage = &virtualOccasionallyBrokenStorage;
                args[n].pRawStorage = g_pStorage[0];
                args[n].success = false;
                args[n].threadId = n;
                args[n].trialCount = 65536;
                args[n].trialTime = nn::TimeSpan(0);
                args[n].seed = 0;
                args[n].stride = 16 * 1024;
            }
        }

        for (int core = 0; core < CoreCount; core++)
        {
            for (int thread = 0; thread < ThreadCountPerCore; thread++)
            {
                int n = core * ThreadCountPerCore + thread;

                nn::os::CreateThread(
                    &g_ThreadInfo[n].threadType,
                    SequentialReadStorageTest,
                    &args[n],
                    g_ThreadInfo[n].threadStack,
                    g_ThreadInfo[n].ThreadStackSize,
                    nn::os::DefaultThreadPriority - n,
                    core);
                nn::os::StartThread(&g_ThreadInfo[n].threadType);
            }
        }
        for (int i = 0; i < CoreCount * ThreadCountPerCore; i++)
        {
            nn::os::WaitThread(&g_ThreadInfo[i].threadType);
            nn::os::DestroyThread(&g_ThreadInfo[i].threadType);
        }
        for (int i = 0; i < CoreCount * ThreadCountPerCore; i++)
        {
            EXPECT_TRUE(args[i].success);
        }
    }
}

TEST(LruFileDataCacheSystem, MultiThreadPrioritizedSequentialReadRandomStride)
{
    const int CoreCount = 3;
    const int ThreadCountPerCore = 3;

    VirtualStorage virtualStorage(g_pStorage[0], StorageSize);
    VirtualOccasionallyBrokenStorage virtualOccasionallyBrokenStorage(&virtualStorage);

    const size_t cacheSize = 8 * 1024 * 1024;
    void* pCacheBuffer = std::malloc(cacheSize);
    NN_ABORT_UNLESS_NOT_NULL(pCacheBuffer);
    NN_UTIL_SCOPE_EXIT
    {
        std::free(pCacheBuffer);
    };
    {
        nn::fs::detail::LruFileDataCacheSystem lruCacheSystem;
        NN_ABORT_UNLESS_RESULT_SUCCESS(lruCacheSystem.Initialize(pCacheBuffer, cacheSize));

        ReadTestArg args[CoreCount * ThreadCountPerCore];
        for (int core = 0; core < CoreCount; core++)
        {
            for (int thread = 0; thread < ThreadCountPerCore; thread++)
            {
                int n = core * ThreadCountPerCore + thread;

                args[n].pLruCacheSystem = &lruCacheSystem;
                args[n].pStorage = &virtualOccasionallyBrokenStorage;
                args[n].pRawStorage = g_pStorage[0];
                args[n].success = false;
                args[n].threadId = n;
                args[n].trialCount = 256;
                args[n].trialTime = nn::TimeSpan(0);
                args[n].seed = nn::os::GetSystemTick().GetInt64Value();
                args[n].stride = 0;
                // Tick の値を変えたいのでちょっと待つ
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
            }
        }

        for (int core = 0; core < CoreCount; core++)
        {
            for (int thread = 0; thread < ThreadCountPerCore; thread++)
            {
                int n = core * ThreadCountPerCore + thread;

                nn::os::CreateThread(
                    &g_ThreadInfo[n].threadType,
                    SequentialReadStorageTest,
                    &args[n],
                    g_ThreadInfo[n].threadStack,
                    g_ThreadInfo[n].ThreadStackSize,
                    nn::os::DefaultThreadPriority - n,
                    core);
                nn::os::StartThread(&g_ThreadInfo[n].threadType);
            }
        }
        for (int i = 0; i < CoreCount * ThreadCountPerCore; i++)
        {
            nn::os::WaitThread(&g_ThreadInfo[i].threadType);
            nn::os::DestroyThread(&g_ThreadInfo[i].threadType);
        }
        for (int i = 0; i < CoreCount * ThreadCountPerCore; i++)
        {
            EXPECT_TRUE(args[i].success);
        }
    }
}

TEST(LruFileDataCacheSystem, MultiThreadPrioritizedRandomReadIdenticalStride)
{
    const int CoreCount = 3;
    const int ThreadCountPerCore = 3;

    VirtualStorage virtualStorage(g_pStorage[0], StorageSize);
    VirtualOccasionallyBrokenStorage virtualOccasionallyBrokenStorage(&virtualStorage);

    const size_t cacheSize = 8 * 1024 * 1024;
    void* pCacheBuffer = std::malloc(cacheSize);
    NN_ABORT_UNLESS_NOT_NULL(pCacheBuffer);
    NN_UTIL_SCOPE_EXIT
    {
        std::free(pCacheBuffer);
    };
    {
        nn::fs::detail::LruFileDataCacheSystem lruCacheSystem;
        NN_ABORT_UNLESS_RESULT_SUCCESS(lruCacheSystem.Initialize(pCacheBuffer, cacheSize));

        ReadTestArg args[CoreCount * ThreadCountPerCore];
        int64_t seed = nn::os::GetSystemTick().GetInt64Value();
        for (int core = 0; core < CoreCount; core++)
        {
            for (int thread = 0; thread < ThreadCountPerCore; thread++)
            {
                int n = core * ThreadCountPerCore + thread;

                args[n].pLruCacheSystem = &lruCacheSystem;
                args[n].pStorage = &virtualOccasionallyBrokenStorage;
                args[n].pRawStorage = g_pStorage[0];
                args[n].success = false;
                args[n].threadId = n;
                args[n].trialCount = 256;
                args[n].trialTime = nn::TimeSpan(0);
                args[n].seed = seed;
            }
        }

        for (int core = 0; core < CoreCount; core++)
        {
            for (int thread = 0; thread < ThreadCountPerCore; thread++)
            {
                int n = core * ThreadCountPerCore + thread;

                nn::os::CreateThread(
                    &g_ThreadInfo[n].threadType,
                    RandomReadStorageTest,
                    &args[n],
                    g_ThreadInfo[n].threadStack,
                    g_ThreadInfo[n].ThreadStackSize,
                    nn::os::DefaultThreadPriority - n,
                    core);
                nn::os::StartThread(&g_ThreadInfo[n].threadType);
            }
        }
        for (int i = 0; i < CoreCount * ThreadCountPerCore; i++)
        {
            nn::os::WaitThread(&g_ThreadInfo[i].threadType);
            nn::os::DestroyThread(&g_ThreadInfo[i].threadType);
        }
        for (int i = 0; i < CoreCount * ThreadCountPerCore; i++)
        {
            EXPECT_TRUE(args[i].success);
        }
    }
}

TEST(LruFileDataCacheSystem, MultiThreadPrioritizedRandomReadRandomStride)
{
    const int CoreCount = 3;
    const int ThreadCountPerCore = 3;

    VirtualStorage virtualStorage(g_pStorage[0], StorageSize);
    VirtualOccasionallyBrokenStorage virtualOccasionallyBrokenStorage(&virtualStorage);

    const size_t cacheSize = 8 * 1024 * 1024;
    void* pCacheBuffer = std::malloc(cacheSize);
    NN_ABORT_UNLESS_NOT_NULL(pCacheBuffer);
    NN_UTIL_SCOPE_EXIT
    {
        std::free(pCacheBuffer);
    };
    {
        nn::fs::detail::LruFileDataCacheSystem lruCacheSystem;
        NN_ABORT_UNLESS_RESULT_SUCCESS(lruCacheSystem.Initialize(pCacheBuffer, cacheSize));

        ReadTestArg args[CoreCount * ThreadCountPerCore];
        for (int core = 0; core < CoreCount; core++)
        {
            for (int thread = 0; thread < ThreadCountPerCore; thread++)
            {
                int n = core * ThreadCountPerCore + thread;

                args[n].pLruCacheSystem = &lruCacheSystem;
                args[n].pStorage = &virtualOccasionallyBrokenStorage;
                args[n].pRawStorage = g_pStorage[0];
                args[n].success = false;
                args[n].threadId = n;
                args[n].trialCount = 256;
                args[n].trialTime = nn::TimeSpan(0);
                args[n].seed = nn::os::GetSystemTick().GetInt64Value();
                // Tick の値を変えたいのでちょっと待つ
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
            }
        }

        for (int core = 0; core < CoreCount; core++)
        {
            for (int thread = 0; thread < ThreadCountPerCore; thread++)
            {
                int n = core * ThreadCountPerCore + thread;

                nn::os::CreateThread(
                    &g_ThreadInfo[n].threadType,
                    RandomReadStorageTest,
                    &args[n],
                    g_ThreadInfo[n].threadStack,
                    g_ThreadInfo[n].ThreadStackSize,
                    nn::os::DefaultThreadPriority - n,
                    core);
                nn::os::StartThread(&g_ThreadInfo[n].threadType);
            }
        }
        for (int i = 0; i < CoreCount * ThreadCountPerCore; i++)
        {
            nn::os::WaitThread(&g_ThreadInfo[i].threadType);
            nn::os::DestroyThread(&g_ThreadInfo[i].threadType);
        }
        for (int i = 0; i < CoreCount * ThreadCountPerCore; i++)
        {
            EXPECT_TRUE(args[i].success);
        }
    }
}

TEST(LruFileDataCacheSystem, TooSmallMemorySizeResult)
{
    const size_t cacheSize = 1024;
    void* pCacheBuffer = std::malloc(cacheSize);
    NN_ABORT_UNLESS_NOT_NULL(pCacheBuffer);
    NN_UTIL_SCOPE_EXIT
    {
        std::free(pCacheBuffer);
    };

    nn::fs::detail::LruFileDataCacheSystem lruCacheSystem;
    nn::Result result = lruCacheSystem.Initialize(pCacheBuffer, cacheSize);
    EXPECT_TRUE(result.IsFailure());
    EXPECT_TRUE(nn::fs::ResultFileDataCacheMemorySizeTooSmall::Includes(result));
}

TEST(LruFileDataCacheSystem, SmallSizeMemoryRead)
{
    VirtualStorage virtualStorage(g_pStorage[0], StorageSize);
    VirtualOccasionallyBrokenStorage virtualOccasionallyBrokenStorage(&virtualStorage);

    const size_t cacheSize = MinAcceptableCacheMemorySize;
    void* pCacheBuffer = std::malloc(cacheSize);
    NN_ABORT_UNLESS_NOT_NULL(pCacheBuffer);
    NN_UTIL_SCOPE_EXIT
    {
        std::free(pCacheBuffer);
    };
    {
        nn::fs::detail::LruFileDataCacheSystem lruCacheSystem;
        NN_ABORT_UNLESS_RESULT_SUCCESS(lruCacheSystem.Initialize(pCacheBuffer, cacheSize));

        ReadTestArg args;
        args.pLruCacheSystem = &lruCacheSystem;
        args.pStorage = &virtualOccasionallyBrokenStorage;
        args.pRawStorage = g_pStorage[0];
        args.success = false;
        args.threadId = 0;
        args.trialCount = 1024;
        args.trialTime = nn::TimeSpan(0);
        args.seed = nn::os::GetSystemTick().GetInt64Value();
        args.stride = 0;

        RandomReadStorageTest(&args);
        EXPECT_TRUE(args.success);
    }
}

TEST(LruFileDataCacheSystem, LargeSizeMemoryRead)
{
    VirtualStorage virtualStorage(g_pStorage[0], StorageSize);
    VirtualOccasionallyBrokenStorage virtualOccasionallyBrokenStorage(&virtualStorage);

    const size_t cacheSize = GetMaxAllocatableCacheMemorySize();
    void* pCacheBuffer = std::malloc(cacheSize);
    NN_ABORT_UNLESS_NOT_NULL(pCacheBuffer);
    NN_UTIL_SCOPE_EXIT
    {
        std::free(pCacheBuffer);
    };
    {
        nn::fs::detail::LruFileDataCacheSystem lruCacheSystem;
        NN_ABORT_UNLESS_RESULT_SUCCESS(lruCacheSystem.Initialize(pCacheBuffer, cacheSize));

        ReadTestArg args;
        args.pLruCacheSystem = &lruCacheSystem;
        args.pStorage = &virtualOccasionallyBrokenStorage;
        args.pRawStorage = g_pStorage[0];
        args.success = false;
        args.threadId = 0;
        args.trialCount = 1024;
        args.trialTime = nn::TimeSpan(0);
        args.seed = nn::os::GetSystemTick().GetInt64Value();
        args.stride = 0;

        RandomReadStorageTest(&args);
        EXPECT_TRUE(args.success);
    }
}

TEST(LruFileDataCacheSystem, RandomSizeMemoryRead)
{
    VirtualStorage virtualStorage(g_pStorage[0], StorageSize);
    VirtualOccasionallyBrokenStorage virtualOccasionallyBrokenStorage(&virtualStorage);

    const int64_t seed = nn::os::GetSystemTick().GetInt64Value();
    std::mt19937 mt(static_cast<unsigned int>(seed));
    std::uniform_int_distribution<size_t> dist(MinAcceptableCacheMemorySize, GetMaxAllocatableCacheMemorySize());

    const int trialCount = 32;
    for (int trial = 0; trial < trialCount; trial++)
    {
        const size_t cacheSize = dist(mt);
        void* pCacheBuffer = std::malloc(cacheSize);
        NN_ABORT_UNLESS_NOT_NULL(pCacheBuffer);
        NN_UTIL_SCOPE_EXIT
        {
            std::free(pCacheBuffer);
        };
        {
            nn::fs::detail::LruFileDataCacheSystem lruCacheSystem;
            NN_ABORT_UNLESS_RESULT_SUCCESS(lruCacheSystem.Initialize(pCacheBuffer, cacheSize));

            ReadTestArg args;
            args.pLruCacheSystem = &lruCacheSystem;
            args.pStorage = &virtualOccasionallyBrokenStorage;
            args.pRawStorage = g_pStorage[0];
            args.success = false;
            args.threadId = 0;
            args.trialCount = 32;
            args.trialTime = nn::TimeSpan(0);
            args.seed = 0;
            args.stride = 0;

            RandomReadStorageTest(&args);
            if (!args.success)
            {
                LRU_CACHE_SYSTEM_TEST_ERROR_LOG("cache size seed = %lld, trial = %d\n", seed, trial);
            }
            EXPECT_TRUE(args.success);
        }
    }
}

TEST(LruFileDataCacheSystem, SynchronousPurge)
{
    const int ThreadCount = 3;

    VirtualStorage* virtualStorages = static_cast<VirtualStorage*>(std::malloc(sizeof(VirtualStorage) * ThreadCount));
    VirtualOccasionallyBrokenStorage* virtualOccasionallyBrokenStorages
        = static_cast<VirtualOccasionallyBrokenStorage*>(std::malloc(sizeof(VirtualOccasionallyBrokenStorage) * ThreadCount));
    NN_ABORT_UNLESS_NOT_NULL(virtualStorages);
    NN_ABORT_UNLESS_NOT_NULL(virtualOccasionallyBrokenStorages);
    NN_UTIL_SCOPE_EXIT
    {
        std::free(virtualOccasionallyBrokenStorages);
        std::free(virtualStorages);
    };
    for (int i = 0; i < ThreadCount; i++)
    {
        new (&virtualStorages[i]) VirtualStorage(g_pStorage[i], StorageSize);
        new (&virtualOccasionallyBrokenStorages[i]) VirtualOccasionallyBrokenStorage(&virtualStorages[i]);
    }

    const size_t cacheSize = 8 * 1024 * 1024;
    void* pCacheBuffer = std::malloc(cacheSize);
    NN_ABORT_UNLESS_NOT_NULL(pCacheBuffer);
    NN_UTIL_SCOPE_EXIT
    {
        std::free(pCacheBuffer);
    };
    {
        nn::fs::detail::LruFileDataCacheSystem lruCacheSystem;
        NN_ABORT_UNLESS_RESULT_SUCCESS(lruCacheSystem.Initialize(pCacheBuffer, cacheSize));

        std::mt19937 mt;
        std::uniform_int_distribution<int> dist(0, (1 << ThreadCount) - 1);

        for (int trial = 0; trial < 16; trial++)
        {
            ReadTestArg args[ThreadCount];
            for (int i = 0; i < ThreadCount; i++)
            {
                args[i].pLruCacheSystem = &lruCacheSystem;
                args[i].pStorage = &virtualOccasionallyBrokenStorages[i];
                args[i].pRawStorage = g_pStorage[i];
                args[i].success = false;
                args[i].threadId = i;
                args[i].trialCount = 16;
                args[i].trialTime = nn::TimeSpan(0);
                args[i].seed = nn::os::GetSystemTick().GetInt64Value();
                // Tick の値を変えたいのでちょっと待つ
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
            }

            for (int i = 0; i < ThreadCount; i++)
            {
                nn::os::CreateThread(
                    &g_ThreadInfo[i].threadType,
                    RandomReadStorageTest,
                    &args[i],
                    g_ThreadInfo[i].threadStack,
                    g_ThreadInfo[i].ThreadStackSize,
                    nn::os::DefaultThreadPriority,
                    i);
                nn::os::StartThread(&g_ThreadInfo[i].threadType);
            }
            for (int i = 0; i < ThreadCount; i++)
            {
                nn::os::WaitThread(&g_ThreadInfo[i].threadType);
                nn::os::DestroyThread(&g_ThreadInfo[i].threadType);
            }
            for (int i = 0; i < ThreadCount; i++)
            {
                EXPECT_TRUE(args[i].success);
            }

            int rand = dist(mt);
            for (int i = 0; i < ThreadCount; i++)
            {
                if (rand & (1 << i))
                {
                    lruCacheSystem.Purge(&virtualOccasionallyBrokenStorages[i]);
                }
            }
        }
    }
}

namespace {

struct RandomPurgeArg
{
    static const int ThreadCount = 3;

    nn::TimeSpan trialTime;
    nn::fs::detail::LruFileDataCacheSystem* pLruFileDataCacheSystem;
    nn::fs::IStorage* pStorages[ThreadCount - 1];
};
void RandomPurgeThreadFunc(void* theArgs)
{
    RandomPurgeArg* args = static_cast<RandomPurgeArg*>(theArgs);

    std::mt19937 mt;
    std::uniform_int_distribution<int> distWait(0, 1000);
    std::uniform_int_distribution<int> distThread(0, (1 << (RandomPurgeArg::ThreadCount - 1)) - 1);

    nn::os::Tick startTick = nn::os::GetSystemTick();
    nn::os::Tick endTick = startTick + nn::os::ConvertToTick(args->trialTime);

    while (endTick > nn::os::GetSystemTick())
    {
        int64_t waitMilliSeconds = distWait(mt);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(waitMilliSeconds));

        int rand = distThread(mt);
        for (int i = 0; i < RandomPurgeArg::ThreadCount - 1; i++)
        {
            if (rand & (1 << i))
            {
                args->pLruFileDataCacheSystem->Purge(args->pStorages[i]);
            }
        }
    }
}

}  // namespace unnamed

TEST(LruFileDataCacheSystem, AsynchronousPurge)
{
    const int ThreadCount = RandomPurgeArg::ThreadCount;

    VirtualStorage* virtualStorages = static_cast<VirtualStorage*>(std::malloc(sizeof(VirtualStorage) * (ThreadCount - 1)));
    VirtualOccasionallyBrokenStorage* virtualOccasionallyBrokenStorages
        = static_cast<VirtualOccasionallyBrokenStorage*>(std::malloc(sizeof(VirtualOccasionallyBrokenStorage) * (ThreadCount - 1)));
    NN_ABORT_UNLESS_NOT_NULL(virtualStorages);
    NN_ABORT_UNLESS_NOT_NULL(virtualOccasionallyBrokenStorages);
    NN_UTIL_SCOPE_EXIT
    {
        std::free(virtualOccasionallyBrokenStorages);
        std::free(virtualStorages);
    };
    for (int i = 0; i < ThreadCount - 1; i++)
    {
        new (&virtualStorages[i]) VirtualStorage(g_pStorage[i], StorageSize);
        new (&virtualOccasionallyBrokenStorages[i]) VirtualOccasionallyBrokenStorage(&virtualStorages[i]);
    }

    const size_t cacheSize = 8 * 1024 * 1024;
    void* pCacheBuffer = std::malloc(cacheSize);
    NN_ABORT_UNLESS_NOT_NULL(pCacheBuffer);
    NN_UTIL_SCOPE_EXIT
    {
        std::free(pCacheBuffer);
    };
    {
        nn::fs::detail::LruFileDataCacheSystem lruCacheSystem;
        NN_ABORT_UNLESS_RESULT_SUCCESS(lruCacheSystem.Initialize(pCacheBuffer, cacheSize));

        nn::TimeSpan trialTime = nn::TimeSpan::FromSeconds(20);

        ReadTestArg args[ThreadCount - 1];
        for (int i = 0; i < ThreadCount - 1; i++)
        {
            args[i].pLruCacheSystem = &lruCacheSystem;
            args[i].pStorage = &virtualOccasionallyBrokenStorages[i];
            args[i].pRawStorage = g_pStorage[i];
            args[i].success = false;
            args[i].threadId = i;
            args[i].trialCount = 0;
            args[i].trialTime = trialTime;
            args[i].seed = i;
        }

        RandomPurgeArg purgeArgs;
        purgeArgs.pLruFileDataCacheSystem = &lruCacheSystem;
        for (int i = 0; i < ThreadCount - 1; i++)
        {
            purgeArgs.pStorages[i] = &virtualOccasionallyBrokenStorages[i];
        }
        purgeArgs.trialTime = trialTime;

        for (int i = 0; i < ThreadCount - 1; i++)
        {
            nn::os::CreateThread(
                &g_ThreadInfo[i].threadType,
                RandomReadStorageTest,
                &args[i],
                g_ThreadInfo[i].threadStack,
                g_ThreadInfo[i].ThreadStackSize,
                nn::os::DefaultThreadPriority,
                i);
            nn::os::StartThread(&g_ThreadInfo[i].threadType);
        }
        nn::os::CreateThread(
            &g_ThreadInfo[ThreadCount - 1].threadType,
            RandomPurgeThreadFunc,
            &purgeArgs,
            g_ThreadInfo[ThreadCount - 1].threadStack,
            g_ThreadInfo[ThreadCount - 1].ThreadStackSize,
            nn::os::DefaultThreadPriority,
            ThreadCount - 1);
        nn::os::StartThread(&g_ThreadInfo[ThreadCount - 1].threadType);

        for (int i = 0; i < ThreadCount; i++)
        {
            nn::os::WaitThread(&g_ThreadInfo[i].threadType);
            nn::os::DestroyThread(&g_ThreadInfo[i].threadType);
        }

        for (int i = 0; i < ThreadCount - 1; i++)
        {
            EXPECT_TRUE(args[i].success);
        }
    }
}

// ----------------------------------------------------------------------------

TEST(LruFileDataCacheSystem, ReadaheadAppliedOnSequentialReading)
{
    auto checkReadahead = [](int64_t startReadOffset, size_t readBufferSize) -> bool
    {
        VirtualStorage virtualStorage(g_pStorage[0], StorageSize);

        const size_t cacheSize = 8 * 1024 * 1024;
        void* pCacheBuffer = std::malloc(cacheSize);
        NN_UTIL_SCOPE_EXIT
        {
            std::free(pCacheBuffer);
        };
        {
            nn::fs::detail::LruFileDataCacheSystem lruCacheSystem;
            NN_ABORT_UNLESS_RESULT_SUCCESS(lruCacheSystem.Initialize(pCacheBuffer, cacheSize));

            void* pReadBuffer = std::malloc(readBufferSize);
            NN_UTIL_SCOPE_EXIT
            {
                std::free(pReadBuffer);
            };

            int64_t readOffset = startReadOffset;
            int64_t readaheadOffset = 0;
            bool firstTimeReading = true;
            while (readOffset < StorageSize)
            {
                size_t readSize = std::min(readBufferSize, StorageSize - static_cast<size_t>(readOffset));

                nn::fs::detail::FileDataCacheAccessResult cacheAccessResult;
                NN_ABORT_UNLESS_RESULT_SUCCESS(lruCacheSystem.Read(&virtualStorage, readOffset, pReadBuffer, readSize, 0, StorageSize, &cacheAccessResult));

                readOffset += static_cast<int64_t>(readSize);
                for (int i = 0; i < cacheAccessResult.GetCacheFetchedRegionCount(); i++)
                {
                    const nn::fs::detail::BufferRegion& fetchedRegion = cacheAccessResult.GetCacheFetchedRegion(i);
                    if (readaheadOffset < fetchedRegion.GetEndOffset())
                    {
                        readaheadOffset = fetchedRegion.GetEndOffset();
                    }
                }

                if (readaheadOffset < readOffset)
                {
                    // 読み込みが先読み済みの地点を追い越しているのはおかしい
                    return false;
                }
                if ((readaheadOffset - readOffset) < nn::fs::detail::LruFileDataCacheSystem::PageSize)
                {
                    // 読み込みが先読み済み地点の近くまで到達した
                    // 初回読み込みの場合はシーケンシャルリードか判定できないので、先読みされていなくても無視する
                    if (firstTimeReading)
                    {
                    }
                    // 読み込みがストレージの終端に到達している場合も先読みできないので問題ない
                    else if (readOffset == StorageSize)
                    {
                    }
                    // キャッシュミスが発生していない場合は、次回以降先読みが行われることを確認できれば問題ない
                    else if (cacheAccessResult.GetCacheFetchedRegionCount() == 0)
                    {
                    }
                    // いずれも満たしていなければ、期待通りに先読みが行われていないと判断する
                    else
                    {
                        return false;
                    }
                }

                firstTimeReading = false;
            }
        }
        return true;
    };

    EXPECT_TRUE(checkReadahead(0, nn::fs::detail::LruFileDataCacheSystem::PageSize));
    EXPECT_TRUE(checkReadahead(1, nn::fs::detail::LruFileDataCacheSystem::PageSize));
    EXPECT_TRUE(checkReadahead(0, nn::fs::detail::LruFileDataCacheSystem::MaxReadAheadSize));
    EXPECT_TRUE(checkReadahead(1, nn::fs::detail::LruFileDataCacheSystem::MaxReadAheadSize));
}

// ----------------------------------------------------------------------------

extern "C" void nnMain()
{
    int     argc = nn::os::GetHostArgc();
    char**  argv = nn::os::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    {
        InitializeStorageArg args[StorageCount];
        for (int i = 0; i < StorageCount; i++)
        {
            args[i].id = i;
            args[i].seed = nn::os::GetSystemTick().GetInt64Value();
            // Tick の値を変えたいのでちょっと待つ
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
        }

        NN_STATIC_ASSERT(StorageCount <= MaxThreadInfoCount);
        for (int i = 0; i < StorageCount; i++)
        {
            nn::os::CreateThread(
                &g_ThreadInfo[i].threadType,
                InitializeStorage,
                &args[i],
                g_ThreadInfo[i].threadStack,
                g_ThreadInfo[i].ThreadStackSize,
                nn::os::DefaultThreadPriority);
            nn::os::StartThread(&g_ThreadInfo[i].threadType);
        }
        for (int i = 0; i < StorageCount; i++)
        {
            nn::os::WaitThread(&g_ThreadInfo[i].threadType);
            nn::os::DestroyThread(&g_ThreadInfo[i].threadType);
        }
    }

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
