﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// ビルドスイッチのため、例外的に先頭でインクルードする
#include <nn/fs/detail/fs_ResultHandlingUtilitySuppressRecordingEventOnUnsupportedPlatforms.h>

#include <cstring>

#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>

#include <nn/fs/detail/fs_Newable.h>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fs/fs_FileStorage.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_ImageDirectory.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fsa/fs_Registrar.h>
#include <nn/fat/fat_FatFileSystem.h>
#include <nn/fssystem/fs_WriteThroughCacheStorage.h>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>

#include "prfile2/pf_api.h"

namespace nn { namespace fs { namespace detail {

#if defined(NN_BUILD_CONFIG_OS_WIN)
class FatFileSystemWithCache : public ::testing::Test
{
protected:
    static const int  FileCount = 1024;
    static const int  DirectoryCount = 256;
    static const char MountName[];
    static const char DirectoryPath[];

    char g_Path[FileCount][nnt::fs::util::NameMaxLength];

    void SetupPathTable(char(*pPath)[nnt::fs::util::NameMaxLength], int count, const bool isRandom) NN_NOEXCEPT
    {
        if( isRandom )
        {
            std::unique_ptr<int[]> pathCount(new int[count]);
            nnt::fs::util::CreateRandomArray(pathCount.get(), count, 1);
            for( int i = 0; i < count; ++i )
            {
                nn::util::SNPrintf(pPath[i], nnt::fs::util::NameMaxLength, "%s%04d", DirectoryPath, pathCount[i]);
            }
        }
        else
        {
            for( int i = 0; i < count; ++i)
            {
                nn::util::SNPrintf(pPath[i], nnt::fs::util::NameMaxLength, "%s%04d", DirectoryPath, i);
            }
        }

        if( count == FileCount )
        {
            nnt::fs::util::String path;
            for( int i = 0; i < count; ++i )
            {
                path = pPath[i];
                path.append(".file");
                nn::util::SNPrintf(pPath[i], nnt::fs::util::NameMaxLength, "%s", path.c_str());
            }
        }
    }

    nnt::fs::util::String GenerateMountName(int index) NN_NOEXCEPT
    {
        char tmp[32];
        nn::util::SNPrintf(tmp, sizeof(tmp), "%d", index);
        nnt::fs::util::String mountName = "fat";
        mountName += tmp;
        return mountName;
    }
};

const char FatFileSystemWithCache::MountName[] = "test";
const char FatFileSystemWithCache::DirectoryPath[] = "test:/testdir/";

TEST_F(FatFileSystemWithCache, CreateImageSample)
{
    const int BufferSize = 8 * 1024 * 1024;

    nnt::fs::util::AccessCountedMemoryStorage storageMem;
    storageMem.Initialize(BufferSize);

    // FatFs を生成
    std::unique_ptr<nn::fat::FatFileSystem> fatFs(new nn::fat::FatFileSystem());
    ASSERT_NE(fatFs, nullptr);

    // キャッシュバッファを確保
    size_t cacheBufferSize = nn::fat::FatFileSystem::GetCacheBufferSize();
    std::unique_ptr<char[]> cacheBuffer(new char[cacheBufferSize]);

    // ライトスルーキャッシュを被せる
    nn::fssystem::WriteThroughCacheStorage storageCached;
    const size_t CacheBlockSize = 128;
    const size_t CacheBlockCount = 64;
    storageCached.Initialize(
        nnt::fs::util::GetTestLibraryAllocator(),
        nn::fs::SubStorage(&storageMem, 0, BufferSize),
        CacheBlockSize,
        CacheBlockCount
    );

    // キャッシュバッファ・IStorage を割り当て
    NNT_ASSERT_RESULT_SUCCESS(fatFs->Initialize(&storageCached, cacheBuffer.get(), cacheBufferSize));

    // FAT にフォーマット
    NNT_ASSERT_RESULT_SUCCESS(fatFs->Format());

    // fatfs 内部のマウント処理
    NNT_ASSERT_RESULT_SUCCESS(fatFs->Mount());

    // fsa に登録 (MountXxx() に相当)
    NNT_ASSERT_RESULT_SUCCESS(fsa::Register("fat", std::move(fatFs)));

    NN_UTIL_SCOPE_EXIT
    {
        Unmount("fat");
    };

    {
        // ディレクトリ・ファイルを作成する等
        storageMem.ResetAccessCounter();
        NNT_EXPECT_RESULT_SUCCESS(CreateDirectory("fat:/directory"));
        EXPECT_GE(3, storageMem.GetReadTimes());
        EXPECT_EQ(4, storageMem.GetWriteTimes());

        storageMem.ResetAccessCounter();
        NNT_EXPECT_RESULT_SUCCESS(CreateFile("fat:/directory/file", 1024));
        EXPECT_EQ(1, storageMem.GetReadTimes());
        EXPECT_EQ(3, storageMem.GetWriteTimes());

        // long file name
        storageMem.ResetAccessCounter();
        NNT_EXPECT_RESULT_SUCCESS(CreateFile("fat:/directory/longfilename________________________________", 1024));
        EXPECT_GE(1, storageMem.GetReadTimes());
        EXPECT_EQ(3, storageMem.GetWriteTimes());

        // 8.3 大文字エントリ名
        storageMem.ResetAccessCounter();
        NNT_EXPECT_RESULT_SUCCESS(CreateFile("fat:/directory/SHORTF", 1024));
        EXPECT_EQ(0, storageMem.GetReadTimes());
        EXPECT_EQ(3, storageMem.GetWriteTimes());

        storageMem.ResetAccessCounter();
        NNT_EXPECT_RESULT_SUCCESS(CreateDirectory("fat:/directory/SHORTD"));
        EXPECT_EQ(0, storageMem.GetReadTimes());
        EXPECT_EQ(4, storageMem.GetWriteTimes());

        FileHandle handle;

        storageMem.ResetAccessCounter();
        NNT_EXPECT_RESULT_SUCCESS(CreateFile("fat:/directory/file2", 0));
        EXPECT_EQ(0, storageMem.GetReadTimes());
        EXPECT_EQ(1, storageMem.GetWriteTimes());

        storageMem.ResetAccessCounter();
        NNT_EXPECT_RESULT_SUCCESS(OpenFile(&handle, "fat:/directory/file2", static_cast<OpenMode>(OpenMode_Read | OpenMode_Write | OpenMode_AllowAppend)));
        EXPECT_EQ(0, storageMem.GetReadTimes());
        EXPECT_EQ(0, storageMem.GetWriteTimes());

        const int fileSize = 1024 * 1024;
        std::unique_ptr<char[]> writeBuffer(new char[fileSize]);
        ASSERT_NE(writeBuffer, nullptr);

        storageMem.ResetAccessCounter();
        memset(writeBuffer.get(), 'A', fileSize);
        NNT_EXPECT_RESULT_SUCCESS(WriteFile(handle,        0, writeBuffer.get(), fileSize, WriteOption()));
        EXPECT_EQ(0, storageMem.GetReadTimes());
        EXPECT_EQ(5, storageMem.GetWriteTimes());

        storageMem.ResetAccessCounter();
        memset(writeBuffer.get(), 'B', fileSize);
        NNT_EXPECT_RESULT_SUCCESS(WriteFile(handle, fileSize, writeBuffer.get(), fileSize, WriteOption()));
        EXPECT_EQ(0, storageMem.GetReadTimes());
        EXPECT_EQ(7, storageMem.GetWriteTimes());

        storageMem.ResetAccessCounter();
        NNT_EXPECT_RESULT_SUCCESS(FlushFile(handle));
        EXPECT_EQ(0, storageMem.GetReadTimes());
        EXPECT_EQ(0, storageMem.GetWriteTimes());

        storageMem.ResetAccessCounter();
        CloseFile(handle);
        EXPECT_EQ(0, storageMem.GetReadTimes());
        EXPECT_EQ(0, storageMem.GetWriteTimes());

        // 作成したエントリを列挙
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::DumpDirectoryRecursive("fat:/"));
    }
}

// FatSafe 有効版のマウント
TEST_F(FatFileSystemWithCache, MountWithFatSafe)
{
    const int64_t Giga = 1024 * 1024 * 1024;
    const int64_t StorageSize = 2 * Giga; // must be FAT32
    nnt::fs::util::VirtualMemoryStorage baseStorage(StorageSize);

    nnt::fs::util::AccessCountWrapperStorage storageMem(&baseStorage);

    nn::fat::FatFileSystem::SetAllocatorForFat(nnt::fs::util::GetTestLibraryAllocator());

    // FatFs を生成
    std::unique_ptr<nn::fat::FatFileSystem> fatFs(new nn::fat::FatFileSystem());
    ASSERT_NE(fatFs, nullptr);
    nn::fat::FatAttribute attrs = {true, false};
    std::unique_ptr<nn::fat::FatErrorInfoSetter> pFatErrorInfoSetter(new nn::fat::FatErrorInfoSetter());
    ASSERT_NE(pFatErrorInfoSetter, nullptr);

    // キャッシュバッファを確保
    size_t cacheBufferSize = nn::fat::FatFileSystem::GetCacheBufferSize();
    std::unique_ptr<char[]> cacheBuffer(new char[cacheBufferSize]);

    // ライトスルーキャッシュを被せる
    nn::fssystem::WriteThroughCacheStorage storageCached;
    static const size_t CacheBlockSize = 128;
    static const size_t CacheBlockCount = 64;
    storageCached.Initialize(
        nnt::fs::util::GetTestLibraryAllocator(),
        nn::fs::SubStorage(&storageMem, 0, StorageSize),
        CacheBlockSize,
        CacheBlockCount
    );

    NNT_ASSERT_RESULT_SUCCESS(
        fatFs->Initialize(
            &storageCached,
            cacheBuffer.get(),
            cacheBufferSize,
            &attrs,
            std::move(pFatErrorInfoSetter),
            ResultInvalidFatFormat(),
            ResultUsableSpaceNotEnough()
        )
    );
    NNT_ASSERT_RESULT_SUCCESS(fatFs->Format());
    NNT_ASSERT_RESULT_SUCCESS(fatFs->Mount());
    NNT_ASSERT_RESULT_SUCCESS(fsa::Register("fat", std::move(fatFs)));

    NN_UTIL_SCOPE_EXIT
    {
        Unmount("fat");
    };

    {
        storageMem.ResetAccessCounter();
        NNT_EXPECT_RESULT_SUCCESS(CreateDirectory("fat:/directory"));
        EXPECT_GE(3, storageMem.GetReadTimes());
        EXPECT_EQ(9, storageMem.GetWriteTimes());

        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::DumpDirectoryRecursive("fat:/"));
    }
}

// 複数の別インスタンス（別ストレージ）が互いに影響しないこと
TEST_F(FatFileSystemWithCache, MultiInstanceOrthogonality)
{
    const int StorageSize = 1 * 1024 * 1024;
    const int BufferSize  = 1 * 1024 * 1024;
    const int DriveNum = 5;
    const size_t CacheBlockSize = 128;
    const size_t CacheBlockCount = 64;

    // ストレージを複数作成
    std::unique_ptr<nn::fat::FatFileSystem>                     pInstance[DriveNum];
    std::unique_ptr<nnt::fs::util::AccessCountedMemoryStorage>  storage[DriveNum];
    std::unique_ptr<char>                                       cacheBuffer[DriveNum];
    std::unique_ptr<char[]>                                     writeThroughCacheBuffer[DriveNum];
    std::unique_ptr<nn::fssystem::WriteThroughCacheStorage>     storageCached[DriveNum];

    for( int i = 0; i < DriveNum; ++i )
    {
        // ストレージ
        storage[i].reset(new nnt::fs::util::AccessCountedMemoryStorage());
        storage[i]->Initialize(StorageSize);
        storageCached[i].reset(new nn::fssystem::WriteThroughCacheStorage());

        // バッファ
        cacheBuffer[i].reset(new char[BufferSize]);

        // fatfs 生成
        auto& pFatFs = pInstance[i];
        pFatFs.reset(new nn::fat::FatFileSystem());
        ASSERT_NE(pFatFs, nullptr);
        size_t cacheBufferSize = nn::fat::FatFileSystem::GetCacheBufferSize();
        ASSERT_GE(static_cast<size_t>(BufferSize), cacheBufferSize);

        // ライトスルーキャッシュを被せる
        storageCached[i]->Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            nn::fs::SubStorage(storage[i].get(), 0, StorageSize),
            CacheBlockSize,
            CacheBlockCount
        );

        // フォーマット
        NNT_ASSERT_RESULT_SUCCESS(pFatFs->Initialize(storageCached[i].get(), cacheBuffer[i].get(), cacheBufferSize));
        NNT_ASSERT_RESULT_SUCCESS(pFatFs->Format());
    }

    NN_UTIL_SCOPE_EXIT
    {
        for( int i = 0; i < DriveNum; ++i )
        {
            Unmount(GenerateMountName(i).c_str());
        }
    };

    // それぞれマウント
    for( int i = 0; i < DriveNum; ++i )
    {
        auto& pFatFs = pInstance[i];

        NNT_ASSERT_RESULT_SUCCESS(pFatFs->Mount());

        auto mountName = GenerateMountName(i);
        NNT_ASSERT_RESULT_SUCCESS(fsa::Register(mountName.c_str(), std::move(pFatFs)));

        // ファイル作成
        auto filePath = mountName + ":/file_" + mountName;
        storage[i]->ResetAccessCounter();
        NNT_EXPECT_RESULT_SUCCESS(CreateFile(filePath.c_str(), i));
        EXPECT_GE(3, storage[i]->GetReadTimes());
        EXPECT_GE(3, storage[i]->GetWriteTimes());

        // 作ったファイルのみが存在することを確認
        // ディレクトリは無いはず
        NNT_EXPECT_RESULT_SUCCESS(
            nnt::fs::util::IterateDirectoryRecursive(
                (mountName + ":/").c_str(),
                [](const char* path, const DirectoryEntry& entry) NN_NOEXCEPT
                {
                    NN_UNUSED(entry);
                    NN_UNUSED(path);
                    NN_RESULT_THROW(nn::fs::ResultDataCorrupted());
                },
                nullptr,
                [&](const char* path, const DirectoryEntry& entry) NN_NOEXCEPT -> Result
                {
                    NN_UNUSED(entry);
                    if( strncmp(path, filePath.c_str(), nn::fs::EntryNameLengthMax) == 0 )
                    {
                        NN_RESULT_SUCCESS;
                    }
                    else
                    {
                        // ファイルは filePath のみのはず
                        NN_RESULT_THROW(nn::fs::ResultDataCorrupted());
                    }
                }
            )
        );

        // (目視確認用)
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::DumpDirectoryRecursive((mountName + ":/").c_str()));
    }
}

TEST_F(FatFileSystemWithCache, ReadOnlyEntry)
{
    // メモリ上に構築する版
    const int BufferSize = 1 * 1024 * 1024;
    nnt::fs::util::AccessCountedMemoryStorage baseStorage;
    baseStorage.Initialize(BufferSize);

    size_t cacheBufferSize = nn::fat::FatFileSystem::GetCacheBufferSize();
    std::unique_ptr<char> cacheBuffer(new char[cacheBufferSize]);

    // ライトスルーキャッシュを被せる
    nn::fssystem::WriteThroughCacheStorage storageCached;
    static const size_t CacheBlockSize = 128;
    static const size_t CacheBlockCount = 64;
    storageCached.Initialize(
        nnt::fs::util::GetTestLibraryAllocator(),
        nn::fs::SubStorage(&baseStorage, 0, BufferSize),
        CacheBlockSize,
        CacheBlockCount
    );

    {
        std::unique_ptr<nn::fat::FatFileSystem> fatFs(new nn::fat::FatFileSystem());
        ASSERT_NE(fatFs, nullptr);
        NNT_ASSERT_RESULT_SUCCESS(fatFs->Initialize(&storageCached, cacheBuffer.get(), cacheBufferSize));
        NNT_ASSERT_RESULT_SUCCESS(fatFs->Format());
        NNT_ASSERT_RESULT_SUCCESS(fatFs->Mount());
        NNT_ASSERT_RESULT_SUCCESS(fsa::Register("fat", std::move(fatFs)));
        NN_UTIL_SCOPE_EXIT
        {
            Unmount("fat");
        };

        baseStorage.ResetAccessCounter();
        NNT_ASSERT_RESULT_SUCCESS(CreateDirectory("fat:/readonlydir"));
        EXPECT_GE(3, baseStorage.GetReadTimes());
        EXPECT_EQ(4, baseStorage.GetWriteTimes());

        baseStorage.ResetAccessCounter();
        NNT_ASSERT_RESULT_SUCCESS(CreateFile("fat:/readonlyfile", 1024));
        EXPECT_EQ(0, baseStorage.GetReadTimes());
        EXPECT_EQ(3, baseStorage.GetWriteTimes());

        baseStorage.ResetAccessCounter();
        NNT_ASSERT_RESULT_SUCCESS(CreateFile("fat:/readonlydir/file", 1024));
        EXPECT_EQ(1, baseStorage.GetReadTimes());
        EXPECT_EQ(3, baseStorage.GetWriteTimes());

        baseStorage.ResetAccessCounter();
        ASSERT_EQ(nne::prfile2::pf_chmod("A:\\readonlyfile", ATTR_RDONLY), 0);
        EXPECT_EQ(0, baseStorage.GetReadTimes());
        EXPECT_EQ(1, baseStorage.GetWriteTimes());

        baseStorage.ResetAccessCounter();
        ASSERT_EQ(nne::prfile2::pf_chdmod("A:\\readonlydir", ATTR_RDONLY), 0);
        EXPECT_EQ(0, baseStorage.GetReadTimes());
        EXPECT_EQ(1, baseStorage.GetWriteTimes());

        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::DumpDirectoryRecursive("fat:/"));

        FileHandle file;

        baseStorage.ResetAccessCounter();
        NNT_EXPECT_RESULT_FAILURE(ResultTargetLocked, OpenFile(&file, "fat:/readonlyfile", OpenMode_Read | OpenMode_Write)); // Todo: Result 修正
        EXPECT_EQ(0, baseStorage.GetReadTimes());
        EXPECT_EQ(0, baseStorage.GetWriteTimes());

        baseStorage.ResetAccessCounter();
        NNT_ASSERT_RESULT_SUCCESS(OpenFile(&file, "fat:/readonlyfile", OpenMode_Read));
        EXPECT_EQ(0, baseStorage.GetReadTimes());
        EXPECT_EQ(0, baseStorage.GetWriteTimes());

        {
            baseStorage.ResetAccessCounter();
            char readBuffer[1024];
            NNT_EXPECT_RESULT_SUCCESS(ReadFile(file, 0, readBuffer, 1024));
            EXPECT_EQ(1, baseStorage.GetReadTimes());
            EXPECT_EQ(0, baseStorage.GetWriteTimes());
        }
        CloseFile(file);
    }
}

class FatFileSystemWithCacheAccessCount : public ::testing::Test
{
protected:
    static const int  FileCount = 1024;
    static const int  DirectoryCount = 256;
    static const char MountName0[];
    static const char MountName1[];
    static const char DirectoryPath[];
    static const char DirectoryPathCached[];

protected:
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        m_StorageMem.Initialize(BufferSize);
        m_StorageMemCached.Initialize(BufferSize);

        // FatFs を生成
        m_FatFs.reset(new nn::fat::FatFileSystem());
        ASSERT_NE(m_FatFs, nullptr);
        std::unique_ptr<nn::fat::FatErrorInfoSetter> pFatErrorInfoSetter(new nn::fat::FatErrorInfoSetter());
        ASSERT_NE(pFatErrorInfoSetter, nullptr);
        m_FatFsCached.reset(new nn::fat::FatFileSystem());
        ASSERT_NE(m_FatFsCached, nullptr);
        std::unique_ptr<nn::fat::FatErrorInfoSetter> pFatErrorInfoSetterCached(new nn::fat::FatErrorInfoSetter());
        ASSERT_NE(pFatErrorInfoSetterCached, nullptr);

        nn::fat::FatAttribute attrs = {true, false};

        // キャッシュバッファを確保
        size_t cacheBufferSize = nn::fat::FatFileSystem::GetCacheBufferSize();
        m_CacheBuffer.reset(new char[cacheBufferSize]);
        m_CacheBufferCached.reset(new char[cacheBufferSize]);

        // ライトスルーキャッシュを被せる
        m_StorageCached.Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            nn::fs::SubStorage(&m_StorageMemCached, 0, BufferSize),
            CacheBlockSize,
            CacheBlockCount
        );

        // キャッシュバッファ・IStorage を割り当て
        NNT_ASSERT_RESULT_SUCCESS(
            m_FatFs->Initialize(
                &m_StorageMem,
                m_CacheBuffer.get(),
                cacheBufferSize,
                &attrs,
                std::move(pFatErrorInfoSetter),
                ResultInvalidFatFormat(),
                ResultUsableSpaceNotEnough()
            )
        );
        NNT_ASSERT_RESULT_SUCCESS(
            m_FatFsCached->Initialize(
                &m_StorageCached,
                m_CacheBufferCached.get(),
                cacheBufferSize,
                &attrs,
                std::move(pFatErrorInfoSetterCached),
                ResultInvalidFatFormat(),
                ResultUsableSpaceNotEnough()
            )
        );

        // FAT にフォーマット
        NNT_ASSERT_RESULT_SUCCESS(m_FatFs->Format());
        NNT_ASSERT_RESULT_SUCCESS(m_FatFsCached->Format());

        // fatfs 内部のマウント処理
        NNT_ASSERT_RESULT_SUCCESS(m_FatFs->Mount());
        NNT_ASSERT_RESULT_SUCCESS(m_FatFsCached->Mount());

        // fsa に登録 (MountXxx() に相当)
        NNT_ASSERT_RESULT_SUCCESS(fsa::Register(MountName0, std::move(m_FatFs)));
        NNT_ASSERT_RESULT_SUCCESS(fsa::Register(MountName1, std::move(m_FatFsCached)));
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        Unmount(MountName0);
        Unmount(MountName1);
    }

    void SetupPathTable(int count, const bool isRandom) NN_NOEXCEPT
    {
        if( isRandom )
        {
            std::unique_ptr<int[]> pathCount(new int[count]);
            nnt::fs::util::CreateRandomArray(pathCount.get(), count, 1);
            for( int i = 0; i < count; ++i )
            {
                nn::util::SNPrintf(g_Path[i], nnt::fs::util::NameMaxLength, "%s%04d", DirectoryPath, pathCount[i]);
                nn::util::SNPrintf(g_PathCached[i], nnt::fs::util::NameMaxLength, "%s%04d", DirectoryPathCached, pathCount[i]);
            }
        }
        else
        {
            for( int i = 0; i < count; ++i )
            {
                nn::util::SNPrintf(g_Path[i], nnt::fs::util::NameMaxLength, "%s%04d", DirectoryPath, i);
                nn::util::SNPrintf(g_PathCached[i], nnt::fs::util::NameMaxLength, "%s%04d", DirectoryPathCached, i);
            }
        }

        if( count == FileCount )
        {
            nnt::fs::util::String path;
            for( int i = 0; i < count; ++i )
            {
                path = g_Path[i];
                path.append(".file");
                nn::util::SNPrintf(g_Path[i], nnt::fs::util::NameMaxLength, "%s", path.c_str());

                path = g_PathCached[i];
                path.append(".file");
                nn::util::SNPrintf(g_PathCached[i], nnt::fs::util::NameMaxLength, "%s", path.c_str());
            }
        }
    }

protected:
    nnt::fs::util::AccessCountedMemoryStorage m_StorageMem;
    nnt::fs::util::AccessCountedMemoryStorage m_StorageMemCached;

    char g_Path[FileCount][nnt::fs::util::NameMaxLength];
    char g_PathCached[FileCount][nnt::fs::util::NameMaxLength];

private:
    static const int BufferSize = 64 * 1024 * 1024;
    static const size_t CacheBlockSize = 1024;
    static const size_t CacheBlockCount = 8;

    std::unique_ptr<nn::fat::FatFileSystem> m_FatFs;
    std::unique_ptr<char[]> m_CacheBuffer;

    std::unique_ptr<nn::fat::FatFileSystem> m_FatFsCached;
    std::unique_ptr<char[]> m_CacheBufferCached;
    nn::fssystem::WriteThroughCacheStorage m_StorageCached;
};

const char FatFileSystemWithCacheAccessCount::MountName0[] = "test";
const char FatFileSystemWithCacheAccessCount::MountName1[] = "cached";
const char FatFileSystemWithCacheAccessCount::DirectoryPath[] = "test:/testdir/";
const char FatFileSystemWithCacheAccessCount::DirectoryPathCached[] = "cached:/testdir/";

// ファイルを作成する
TEST_F(FatFileSystemWithCacheAccessCount, CreateFile)
{
    NN_LOG("\nCreate File Performance Test\n");

    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(DirectoryPath));
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(DirectoryPathCached));

    nnt::fs::util::TimeCount timeCountCreate;
    nnt::fs::util::TimeCount timeCountCreateCached;

    m_StorageMem.ResetAccessCounter();
    m_StorageMemCached.ResetAccessCounter();

    SetupPathTable(FileCount, true);
    for( int i = 0; i < FileCount; ++i )
    {
        timeCountCreate.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(CreateFile(g_Path[i], 0));
        timeCountCreate.StopTime();

        timeCountCreateCached.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(CreateFile(g_PathCached[i], 0));
        timeCountCreateCached.StopTime();
    }

    timeCountCreate.ViewTime(timeCountCreate.GetTotalTime());
    NN_LOG(" Read=%d, Write=%d\n", m_StorageMem.GetReadTimes(), m_StorageMem.GetWriteTimes());
    timeCountCreateCached.ViewTime(timeCountCreateCached.GetTotalTime());
    NN_LOG(" Read=%d, Write=%d [Cached]\n", m_StorageMemCached.GetReadTimes(), m_StorageMemCached.GetWriteTimes());

    ASSERT_LE(m_StorageMemCached.GetReadTimes(), m_StorageMem.GetReadTimes());
    ASSERT_GE(m_StorageMemCached.GetWriteTimes(), m_StorageMem.GetWriteTimes());
}

// ファイルを作成した順に Open/Close する
TEST_F(FatFileSystemWithCacheAccessCount, OpenCloseFilenSequencial)
{
    NN_LOG("\nSequencial Open/Close File Performance Test\n");

    (void)DeleteDirectoryRecursively(DirectoryPath);
    (void)DeleteDirectoryRecursively(DirectoryPathCached);
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(DirectoryPath));
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(DirectoryPathCached));
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateTestFile(nullptr, g_Path, DirectoryPath, FileCount, 0));
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateTestFile(nullptr, g_PathCached, DirectoryPathCached, FileCount, 0));

    nnt::fs::util::TimeCount timeCountSequencialOpen;
    nnt::fs::util::TimeCount timeCountSequencialOpenCached;
    nnt::fs::util::TimeCount timeCountSequencialClose;
    nnt::fs::util::TimeCount timeCountSequencialCloseCached;

    m_StorageMem.ResetAccessCounter();
    m_StorageMemCached.ResetAccessCounter();

    for( int i = 0; i < FileCount; ++i )
    {
        FileHandle handle;
        FileHandle handleCached;

        timeCountSequencialOpen.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(OpenFile(&handle, g_Path[i], OpenMode_Read | OpenMode_Write | OpenMode_AllowAppend));
        timeCountSequencialOpen.StopTime();

        timeCountSequencialOpenCached.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(OpenFile(&handleCached, g_PathCached[i], OpenMode_Read | OpenMode_Write | OpenMode_AllowAppend));
        timeCountSequencialOpenCached.StopTime();

        timeCountSequencialClose.StartTime();
        CloseFile(handle);
        timeCountSequencialClose.StopTime();

        timeCountSequencialCloseCached.StartTime();
        CloseFile(handleCached);
        timeCountSequencialCloseCached.StopTime();
    }

    NN_LOG(" Open  Files\n");
    timeCountSequencialOpen.ViewTime(timeCountSequencialOpen.GetTotalTime());
    NN_LOG(" Close Files\n");
    timeCountSequencialClose.ViewTime(timeCountSequencialClose.GetTotalTime());
    NN_LOG(" Read=%d, Write=%d\n", m_StorageMem.GetReadTimes(), m_StorageMem.GetWriteTimes());

    NN_LOG(" Open  Files [Cached]\n");
    timeCountSequencialOpenCached.ViewTime(timeCountSequencialOpenCached.GetTotalTime());
    NN_LOG(" Close Files [Cached]\n");
    timeCountSequencialCloseCached.ViewTime(timeCountSequencialCloseCached.GetTotalTime());
    NN_LOG(" Read=%d, Write=%d [Cached]\n", m_StorageMemCached.GetReadTimes(), m_StorageMemCached.GetWriteTimes());

    ASSERT_LE(m_StorageMemCached.GetReadTimes(), m_StorageMem.GetReadTimes());
    ASSERT_GE(m_StorageMemCached.GetWriteTimes(), m_StorageMem.GetWriteTimes());
}

// ファイルをランダムな順に Open/Close する
TEST_F(FatFileSystemWithCacheAccessCount, OpenCloseFilenRandom)
{
    NN_LOG("\nRandom Open/Close File Performance Test\n");

    (void)DeleteDirectoryRecursively(DirectoryPath);
    (void)DeleteDirectoryRecursively(DirectoryPathCached);
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(DirectoryPath));
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(DirectoryPathCached));
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateTestFile(nullptr, g_Path, DirectoryPath, FileCount, 0));
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateTestFile(nullptr, g_PathCached, DirectoryPathCached, FileCount, 0));

    nnt::fs::util::TimeCount timeCountRandomOpen;
    nnt::fs::util::TimeCount timeCountRandomOpenCached;
    nnt::fs::util::TimeCount timeCountRandomClose;
    nnt::fs::util::TimeCount timeCountRandomCloseCached;

    m_StorageMem.ResetAccessCounter();
    m_StorageMemCached.ResetAccessCounter();

    SetupPathTable(FileCount, true);

    for( int i = 0; i < FileCount; ++i )
    {
        FileHandle handle;
        FileHandle handleCached;

        timeCountRandomOpen.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(OpenFile(&handle, g_Path[i], OpenMode_Read | OpenMode_Write | OpenMode_AllowAppend));
        timeCountRandomOpen.StopTime();

        timeCountRandomOpenCached.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(OpenFile(&handleCached, g_PathCached[i], OpenMode_Read | OpenMode_Write | OpenMode_AllowAppend));
        timeCountRandomOpenCached.StopTime();

        timeCountRandomClose.StartTime();
        CloseFile(handle);
        timeCountRandomClose.StopTime();

        timeCountRandomCloseCached.StartTime();
        CloseFile(handleCached);
        timeCountRandomCloseCached.StopTime();
    }

    NN_LOG(" Open  Files\n");
    timeCountRandomOpen.ViewTime(timeCountRandomOpen.GetTotalTime());
    NN_LOG(" Close Files\n");
    timeCountRandomClose.ViewTime(timeCountRandomClose.GetTotalTime());
    NN_LOG(" Read=%d, Write=%d\n", m_StorageMem.GetReadTimes(), m_StorageMem.GetWriteTimes());

    NN_LOG(" Open  Files [Cached]\n");
    timeCountRandomOpenCached.ViewTime(timeCountRandomOpenCached.GetTotalTime());
    NN_LOG(" Close Files [Cached]\n");
    timeCountRandomCloseCached.ViewTime(timeCountRandomCloseCached.GetTotalTime());
    NN_LOG(" Read=%d, Write=%d [Cached]\n", m_StorageMemCached.GetReadTimes(), m_StorageMemCached.GetWriteTimes());

    ASSERT_LE(m_StorageMemCached.GetReadTimes(), m_StorageMem.GetReadTimes());
    ASSERT_GE(m_StorageMemCached.GetWriteTimes(), m_StorageMem.GetWriteTimes());
}

// ファイルを削除する
TEST_F(FatFileSystemWithCacheAccessCount, DeleteFile)
{
    NN_LOG("\nDelete File Performance Test\n");

    (void)DeleteDirectoryRecursively(DirectoryPath);
    (void)DeleteDirectoryRecursively(DirectoryPathCached);
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(DirectoryPath));
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(DirectoryPathCached));
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateTestFile(nullptr, g_Path, DirectoryPath, FileCount, 0));
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateTestFile(nullptr, g_PathCached, DirectoryPathCached, FileCount, 0));

    nnt::fs::util::TimeCount timeCountDelete;
    nnt::fs::util::TimeCount timeCountDeleteCached;

    m_StorageMem.ResetAccessCounter();
    m_StorageMemCached.ResetAccessCounter();

    SetupPathTable(FileCount, false);
    for( int i = 0; i < FileCount; ++i )
    {
        timeCountDelete.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(DeleteFile(g_Path[i]));
        timeCountDelete.StopTime();

        timeCountDeleteCached.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(DeleteFile(g_PathCached[i]));
        timeCountDeleteCached.StopTime();
    }

    timeCountDelete.ViewTime(timeCountDelete.GetTotalTime());
    NN_LOG(" Read=%d, Write=%d\n", m_StorageMem.GetReadTimes(), m_StorageMem.GetWriteTimes());
    timeCountDeleteCached.ViewTime(timeCountDeleteCached.GetTotalTime());
    NN_LOG(" Read=%d, Write=%d [Cached]\n", m_StorageMemCached.GetReadTimes(), m_StorageMemCached.GetWriteTimes());

    ASSERT_LE(m_StorageMemCached.GetReadTimes(), m_StorageMem.GetReadTimes());
    ASSERT_GE(m_StorageMemCached.GetWriteTimes(), m_StorageMem.GetWriteTimes());
}

// ディレクトリを作成する
TEST_F(FatFileSystemWithCacheAccessCount, CreateDirectory)
{
    NN_LOG("\nCreate Directory Performance Test\n");

    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(DirectoryPath));
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(DirectoryPathCached));

    nnt::fs::util::TimeCount timeCountCreate;
    nnt::fs::util::TimeCount timeCountCreateCached;

    m_StorageMem.ResetAccessCounter();
    m_StorageMemCached.ResetAccessCounter();

    SetupPathTable(DirectoryCount, false);
    for( int i = 0; i < DirectoryCount; ++i )
    {
        timeCountCreate.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(g_Path[i]));
        timeCountCreate.StopTime();

        timeCountCreateCached.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(g_PathCached[i]));
        timeCountCreateCached.StopTime();
    }

    timeCountCreate.ViewTime(timeCountCreate.GetTotalTime());
    NN_LOG(" Read=%d, Write=%d\n", m_StorageMem.GetReadTimes(), m_StorageMem.GetWriteTimes());
    timeCountCreateCached.ViewTime(timeCountCreateCached.GetTotalTime());
    NN_LOG(" Read=%d, Write=%d\n [Cached]", m_StorageMemCached.GetReadTimes(), m_StorageMemCached.GetWriteTimes());

    ASSERT_LE(m_StorageMemCached.GetReadTimes(), m_StorageMem.GetReadTimes());
    ASSERT_GE(m_StorageMemCached.GetWriteTimes(), m_StorageMem.GetWriteTimes());
}

// ディレクトリを作成した順に Open/Close する
TEST_F(FatFileSystemWithCacheAccessCount, OpenCloseDirectorySequencial)
{
    NN_LOG("\nSequencial Open/Close Directory Performance Test");

    (void)DeleteDirectoryRecursively(DirectoryPath);
    (void)DeleteDirectoryRecursively(DirectoryPathCached);
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(DirectoryPath));
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(DirectoryPathCached));
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateTestDirectory(nullptr, g_Path, DirectoryPath, DirectoryCount, nnt::fs::util::DIRTYPE_FLAT, false, 0));
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateTestDirectory(nullptr, g_PathCached, DirectoryPathCached, DirectoryCount, nnt::fs::util::DIRTYPE_FLAT, false, 0));

    nnt::fs::util::TimeCount timeCountSequencialOpen;
    nnt::fs::util::TimeCount timeCountSequencialOpenCached;
    nnt::fs::util::TimeCount timeCountSequencialClose;
    nnt::fs::util::TimeCount timeCountSequencialCloseCached;

    m_StorageMem.ResetAccessCounter();
    m_StorageMemCached.ResetAccessCounter();

    for( int i = 0; i < DirectoryCount; ++i )
    {
        DirectoryHandle directoryHandle;
        DirectoryHandle directoryHandleCached;

        timeCountSequencialOpen.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(OpenDirectory(&directoryHandle, g_Path[i], OpenDirectoryMode_All));
        timeCountSequencialOpen.StopTime();

        timeCountSequencialOpenCached.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(OpenDirectory(&directoryHandleCached, g_PathCached[i], OpenDirectoryMode_All));
        timeCountSequencialOpenCached.StopTime();

        timeCountSequencialClose.StartTime();
        CloseDirectory(directoryHandle);
        timeCountSequencialClose.StopTime();

        timeCountSequencialCloseCached.StartTime();
        CloseDirectory(directoryHandleCached);
        timeCountSequencialCloseCached.StopTime();
    }

    NN_LOG(" Open Directory\n");
    timeCountSequencialOpen.ViewTime(timeCountSequencialOpen.GetTotalTime());
    NN_LOG(" Close Directory\n");
    timeCountSequencialClose.ViewTime(timeCountSequencialClose.GetTotalTime());
    NN_LOG(" Read=%d, Write=%d\n", m_StorageMem.GetReadTimes(), m_StorageMem.GetWriteTimes());

    NN_LOG(" Open Directory [Cached]\n");
    timeCountSequencialOpenCached.ViewTime(timeCountSequencialOpenCached.GetTotalTime());
    NN_LOG(" Close Directory [Cached]\n");
    timeCountSequencialCloseCached.ViewTime(timeCountSequencialCloseCached.GetTotalTime());
    NN_LOG(" Read=%d, Write=%d [Cached]\n", m_StorageMemCached.GetReadTimes(), m_StorageMemCached.GetWriteTimes());

    ASSERT_LE(m_StorageMemCached.GetReadTimes(), m_StorageMem.GetReadTimes());
    ASSERT_GE(m_StorageMemCached.GetWriteTimes(), m_StorageMem.GetWriteTimes());
}

// ディレクトリをランダムな順に Open/Close する
TEST_F(FatFileSystemWithCacheAccessCount, OpenCloseDirectoryRandom)
{
    NN_LOG("\nRandom Open/Close Directory Performance Test");

    (void)DeleteDirectoryRecursively(DirectoryPath);
    (void)DeleteDirectoryRecursively(DirectoryPathCached);
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(DirectoryPath));
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(DirectoryPathCached));
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateTestDirectory(nullptr, g_Path, DirectoryPath, DirectoryCount, nnt::fs::util::DIRTYPE_FLAT, false, 0));
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateTestDirectory(nullptr, g_PathCached, DirectoryPathCached, DirectoryCount, nnt::fs::util::DIRTYPE_FLAT, false, 0));

    nnt::fs::util::TimeCount timeCountRandomOpen;
    nnt::fs::util::TimeCount timeCountRandomOpenCached;
    nnt::fs::util::TimeCount timeCountRandomClose;
    nnt::fs::util::TimeCount timeCountRandomCloseCached;

    m_StorageMem.ResetAccessCounter();
    m_StorageMemCached.ResetAccessCounter();

    SetupPathTable(DirectoryCount, true);
    for( int i = 0; i < DirectoryCount; ++i )
    {
        DirectoryHandle directoryHandle;
        DirectoryHandle directoryHandleCached;

        timeCountRandomOpen.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(OpenDirectory(&directoryHandle, g_Path[i], OpenDirectoryMode_All));
        timeCountRandomOpen.StopTime();

        timeCountRandomOpenCached.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(OpenDirectory(&directoryHandleCached, g_PathCached[i], OpenDirectoryMode_All));
        timeCountRandomOpenCached.StopTime();

        timeCountRandomClose.StartTime();
        CloseDirectory(directoryHandle);
        timeCountRandomClose.StopTime();

        timeCountRandomCloseCached.StartTime();
        CloseDirectory(directoryHandleCached);
        timeCountRandomCloseCached.StopTime();
    }

    NN_LOG(" Open Directory\n");
    timeCountRandomOpen.ViewTime(timeCountRandomOpen.GetTotalTime());
    NN_LOG(" Close Directory\n");
    timeCountRandomClose.ViewTime(timeCountRandomClose.GetTotalTime());
    NN_LOG(" Read=%d, Write=%d\n", m_StorageMem.GetReadTimes(), m_StorageMem.GetWriteTimes());

    NN_LOG(" Open Directory\n");
    timeCountRandomOpenCached.ViewTime(timeCountRandomOpenCached.GetTotalTime());
    NN_LOG(" Close Directory\n");
    timeCountRandomCloseCached.ViewTime(timeCountRandomCloseCached.GetTotalTime());
    NN_LOG(" Read=%d, Write=%d\n", m_StorageMemCached.GetReadTimes(), m_StorageMemCached.GetWriteTimes());

    ASSERT_LE(m_StorageMemCached.GetReadTimes(), m_StorageMem.GetReadTimes());
    ASSERT_GE(m_StorageMemCached.GetWriteTimes(), m_StorageMem.GetWriteTimes());
}

// ディレクトリを削除する時間を計測
TEST_F(FatFileSystemWithCacheAccessCount, DeleteDirectory)
{
    NN_LOG("\nDelete Directory Performance Test\n");

    (void)DeleteDirectoryRecursively(DirectoryPath);
    (void)DeleteDirectoryRecursively(DirectoryPathCached);
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(DirectoryPath));
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(DirectoryPathCached));
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateTestDirectory(nullptr, g_Path, DirectoryPath, DirectoryCount, nnt::fs::util::DIRTYPE_FLAT, false, 0));
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateTestDirectory(nullptr, g_PathCached, DirectoryPathCached, DirectoryCount, nnt::fs::util::DIRTYPE_FLAT, false, 0));

    nnt::fs::util::TimeCount timeCountDelete;
    nnt::fs::util::TimeCount timeCountDeleteCached;

    m_StorageMem.ResetAccessCounter();
    m_StorageMemCached.ResetAccessCounter();

    SetupPathTable(DirectoryCount, false);

    for( int i = 0; i < DirectoryCount; ++i )
    {
        timeCountDelete.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(DeleteDirectory(g_Path[i]));
        timeCountDelete.StopTime();

        timeCountDeleteCached.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(DeleteDirectory(g_PathCached[i]));
        timeCountDeleteCached.StopTime();
    }

    timeCountDelete.ViewTime(timeCountDelete.GetTotalTime());
    NN_LOG(" Read=%d, Write=%d\n", m_StorageMem.GetReadTimes(), m_StorageMem.GetWriteTimes());
    timeCountDeleteCached.ViewTime(timeCountDeleteCached.GetTotalTime());
    NN_LOG(" Read=%d, Write=%d [Cached]\n", m_StorageMemCached.GetReadTimes(), m_StorageMemCached.GetWriteTimes());

    ASSERT_LE(m_StorageMemCached.GetReadTimes(), m_StorageMem.GetReadTimes());
    ASSERT_GE(m_StorageMemCached.GetWriteTimes(), m_StorageMem.GetWriteTimes());
}

// ファイルの作成、リネーム、削除
TEST_F(FatFileSystemWithCacheAccessCount, CreateRenameDeleteFile)
{
    SetupPathTable(FileCount, true);

    (void)DeleteDirectoryRecursively(DirectoryPath);
    (void)DeleteDirectoryRecursively(DirectoryPathCached);
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(DirectoryPath));
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(DirectoryPathCached));

    nnt::fs::util::TimeCount timeCreateCount;
    nnt::fs::util::TimeCount timeCreateCountCached;

    m_StorageMem.ResetAccessCounter();
    m_StorageMemCached.ResetAccessCounter();

    for( int i = 0; i < FileCount; ++i )
    {
        timeCreateCount.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(CreateFile(g_Path[i], 0));
        timeCreateCount.StopTime();

        timeCreateCountCached.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(CreateFile(g_PathCached[i], 0));
        timeCreateCountCached.StopTime();
    }

    NN_LOG(" CreateFile\n");
    timeCreateCount.ViewSimpleTime(timeCreateCount.GetTotalTime());
    NN_LOG(" Read=%d, Write=%d\n", m_StorageMem.GetReadTimes(), m_StorageMem.GetWriteTimes());
    NN_LOG(" CreateFile [Cached]\n");
    timeCreateCountCached.ViewSimpleTime(timeCreateCountCached.GetTotalTime());
    NN_LOG(" Read=%d, Write=%d [Cached]\n", m_StorageMemCached.GetReadTimes(), m_StorageMemCached.GetWriteTimes());

    ASSERT_LE(m_StorageMemCached.GetReadTimes(), m_StorageMem.GetReadTimes());
    ASSERT_GE(m_StorageMemCached.GetWriteTimes(), m_StorageMem.GetWriteTimes());

    nnt::fs::util::TimeCount timeRenameCount;
    nnt::fs::util::TimeCount timeRenameCountCached;

    m_StorageMem.ResetAccessCounter();
    m_StorageMemCached.ResetAccessCounter();

    for( int i = 0; i < FileCount; ++i )
    {
        char newPath[nnt::fs::util::NameMaxLength];
        nn::util::SNPrintf(newPath, nnt::fs::util::NameMaxLength, "%s_", g_Path[i]);
        char newPathCached[nnt::fs::util::NameMaxLength];
        nn::util::SNPrintf(newPathCached, nnt::fs::util::NameMaxLength, "%s_", g_PathCached[i]);

        timeRenameCount.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(RenameFile(g_Path[i], newPath));
        timeRenameCount.StopTime();

        timeRenameCountCached.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(RenameFile(g_PathCached[i], newPathCached));
        timeRenameCountCached.StopTime();

        std::strncpy(g_Path[i], newPath, nnt::fs::util::NameMaxLength);
        std::strncpy(g_PathCached[i], newPathCached, nnt::fs::util::NameMaxLength);
    }

    NN_LOG(" RenameFile\n");
    timeRenameCount.ViewSimpleTime(timeRenameCount.GetTotalTime());
    NN_LOG(" Read=%d, Write=%d\n", m_StorageMem.GetReadTimes(), m_StorageMem.GetWriteTimes());
    NN_LOG(" RenameFile [Cached]\n");
    timeRenameCountCached.ViewSimpleTime(timeRenameCountCached.GetTotalTime());
    NN_LOG(" Read=%d, Write=%d [Cached]\n", m_StorageMemCached.GetReadTimes(), m_StorageMemCached.GetWriteTimes());

    ASSERT_LE(m_StorageMemCached.GetReadTimes(), m_StorageMem.GetReadTimes());
    ASSERT_GE(m_StorageMemCached.GetWriteTimes(), m_StorageMem.GetWriteTimes());

    nnt::fs::util::TimeCount timeDeleteCount;
    nnt::fs::util::TimeCount timeDeleteCountCached;

    m_StorageMem.ResetAccessCounter();
    m_StorageMemCached.ResetAccessCounter();

    for( int i = 0; i < FileCount; ++i )
    {
        timeDeleteCount.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(DeleteFile(g_Path[i]));
        timeDeleteCount.StopTime();

        timeDeleteCountCached.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(DeleteFile(g_PathCached[i]));
        timeDeleteCountCached.StopTime();
    }

    NN_LOG(" DeleteFile\n");
    timeDeleteCount.ViewSimpleTime(timeDeleteCount.GetTotalTime());
    NN_LOG(" Read=%d, Write=%d\n", m_StorageMem.GetReadTimes(), m_StorageMem.GetWriteTimes());
    NN_LOG(" DeleteFile [Cached]\n");
    timeDeleteCountCached.ViewSimpleTime(timeDeleteCountCached.GetTotalTime());
    NN_LOG(" Read=%d, Write=%d [Cached]\n", m_StorageMemCached.GetReadTimes(), m_StorageMemCached.GetWriteTimes());

    ASSERT_LE(m_StorageMemCached.GetReadTimes(), m_StorageMem.GetReadTimes());
    ASSERT_GE(m_StorageMemCached.GetWriteTimes(), m_StorageMem.GetWriteTimes());
}

// ディレクトリの作成、リネーム、削除
TEST_F(FatFileSystemWithCacheAccessCount, CreateRenameDeleteDirectory)
{
    SetupPathTable(DirectoryCount, true);

    (void)DeleteDirectoryRecursively(DirectoryPath);
    (void)DeleteDirectoryRecursively(DirectoryPathCached);
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(DirectoryPath));
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(DirectoryPathCached));

    nnt::fs::util::TimeCount timeCreateCount;
    nnt::fs::util::TimeCount timeCreateCountCached;

    m_StorageMem.ResetAccessCounter();
    m_StorageMemCached.ResetAccessCounter();

    for( int i = 0; i < DirectoryCount; ++i )
    {
        timeCreateCount.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(g_Path[i]));
        timeCreateCount.StopTime();

        timeCreateCountCached.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(g_PathCached[i]));
        timeCreateCountCached.StopTime();
    }

    NN_LOG(" CreateDirectory\n");
    timeCreateCount.ViewSimpleTime(timeCreateCount.GetTotalTime());
    NN_LOG(" Read=%d, Write=%d\n", m_StorageMem.GetReadTimes(), m_StorageMem.GetWriteTimes());
    NN_LOG(" CreateDirectory [Cached]\n");
    timeCreateCountCached.ViewSimpleTime(timeCreateCountCached.GetTotalTime());
    NN_LOG(" Read=%d, Write=%d [Cached]\n", m_StorageMemCached.GetReadTimes(), m_StorageMemCached.GetWriteTimes());

    ASSERT_LE(m_StorageMemCached.GetReadTimes(), m_StorageMem.GetReadTimes());
    ASSERT_GE(m_StorageMemCached.GetWriteTimes(), m_StorageMem.GetWriteTimes());

    nnt::fs::util::TimeCount timeRenameCount;
    nnt::fs::util::TimeCount timeRenameCountCached;

    m_StorageMem.ResetAccessCounter();
    m_StorageMemCached.ResetAccessCounter();

    for( int i = 0; i < DirectoryCount; ++i )
    {
        char newPath[nnt::fs::util::NameMaxLength];
        nn::util::SNPrintf(newPath, nnt::fs::util::NameMaxLength, "%s_", g_Path[i]);
        char newPathCached[nnt::fs::util::NameMaxLength];
        nn::util::SNPrintf(newPathCached, nnt::fs::util::NameMaxLength, "%s_", g_PathCached[i]);

        timeRenameCount.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(RenameDirectory(g_Path[i], newPath));
        timeRenameCount.StopTime();

        timeRenameCountCached.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(RenameDirectory(g_PathCached[i], newPathCached));
        timeRenameCountCached.StopTime();

        strncpy(g_Path[i], newPath, nnt::fs::util::NameMaxLength);
        strncpy(g_PathCached[i], newPathCached, nnt::fs::util::NameMaxLength);
    }

    NN_LOG(" RenameDirectory\n");
    timeRenameCount.ViewSimpleTime(timeRenameCount.GetTotalTime());
    NN_LOG(" Read=%d, Write=%d\n", m_StorageMem.GetReadTimes(), m_StorageMem.GetWriteTimes());
    NN_LOG(" RenameDirectory [Cached]\n");
    timeRenameCountCached.ViewSimpleTime(timeRenameCountCached.GetTotalTime());
    NN_LOG(" Read=%d, Write=%d [Cached]\n", m_StorageMemCached.GetReadTimes(), m_StorageMemCached.GetWriteTimes());

    ASSERT_LE(m_StorageMemCached.GetReadTimes(), m_StorageMem.GetReadTimes());
    ASSERT_GE(m_StorageMemCached.GetWriteTimes(), m_StorageMem.GetWriteTimes());

    nnt::fs::util::TimeCount timeDeleteCount;
    nnt::fs::util::TimeCount timeDeleteCountCached;

    m_StorageMem.ResetAccessCounter();
    m_StorageMemCached.ResetAccessCounter();

    for( int i = 0; i < DirectoryCount; ++i )
    {
        timeDeleteCount.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(DeleteDirectory(g_Path[i]));
        timeDeleteCount.StopTime();

        timeDeleteCountCached.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(DeleteDirectory(g_PathCached[i]));
        timeDeleteCountCached.StopTime();
    }

    NN_LOG("DeleteDirectory\n");
    timeDeleteCount.ViewSimpleTime(timeDeleteCount.GetTotalTime());
    NN_LOG("Read=%d, Write=%d\n", m_StorageMem.GetReadTimes(), m_StorageMem.GetWriteTimes());
    NN_LOG("DeleteDirectory [Cached]\n");
    timeDeleteCountCached.ViewSimpleTime(timeDeleteCountCached.GetTotalTime());
    NN_LOG("Read=%d, Write=%d [Cached]\n", m_StorageMemCached.GetReadTimes(), m_StorageMemCached.GetWriteTimes());

    ASSERT_LE(m_StorageMemCached.GetReadTimes(), m_StorageMem.GetReadTimes());
    ASSERT_GE(m_StorageMemCached.GetWriteTimes(), m_StorageMem.GetWriteTimes());
}

// エントリタイプを読み込む
TEST_F(FatFileSystemWithCacheAccessCount, GetEntryType)
{
    SetupPathTable(DirectoryCount, true);

    (void)DeleteDirectoryRecursively(DirectoryPath);
    (void)DeleteDirectoryRecursively(DirectoryPathCached);
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(DirectoryPath));
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(DirectoryPathCached));
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateTestFile(nullptr, g_Path, DirectoryPath, FileCount, 0));
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateTestFile(nullptr, g_PathCached, DirectoryPathCached, FileCount, 0));

    nnt::fs::util::TimeCount timeCount;
    nnt::fs::util::TimeCount timeCountCached;

    m_StorageMem.ResetAccessCounter();
    m_StorageMemCached.ResetAccessCounter();

    for( int i = 0; i < FileCount; ++i )
    {
        nn::fs::DirectoryEntryType entryType;

        timeCount.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(GetEntryType(&entryType, g_Path[i]));
        timeCount.StopTime();

        timeCountCached.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(GetEntryType(&entryType, g_PathCached[i]));
        timeCountCached.StopTime();
    }

    NN_LOG("GetEntryType\n");
    timeCount.ViewSimpleTime(timeCount.GetTotalTime());
    NN_LOG("Read=%d, Write=%d\n", m_StorageMem.GetReadTimes(), m_StorageMem.GetWriteTimes());
    NN_LOG("GetEntryType [Cached]\n");
    timeCountCached.ViewSimpleTime(timeCountCached.GetTotalTime());
    NN_LOG("Read=%d, Write=%d [Cached]\n", m_StorageMemCached.GetReadTimes(), m_StorageMemCached.GetWriteTimes());

    ASSERT_LE(m_StorageMemCached.GetReadTimes(), m_StorageMem.GetReadTimes());
    ASSERT_GE(m_StorageMemCached.GetWriteTimes(), m_StorageMem.GetWriteTimes());
}

TEST_F(FatFileSystemWithCacheAccessCount, ReadDirectory)
{
    SetupPathTable(DirectoryCount, true);

    (void)DeleteDirectoryRecursively(DirectoryPath);
    (void)DeleteDirectoryRecursively(DirectoryPathCached);
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(DirectoryPath));
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(DirectoryPathCached));
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateTestFile(nullptr, g_Path, DirectoryPath, FileCount, 0));
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateTestFile(nullptr, g_PathCached, DirectoryPathCached, FileCount, 0));

    nnt::fs::util::TimeCount timeCount;
    nnt::fs::util::TimeCount timeCountCached;

    for( int i = 1; i <= DirectoryCount; i *= 2 )
    {
        m_StorageMem.ResetAccessCounter();
        m_StorageMemCached.ResetAccessCounter();
        timeCount.ResetTime();
        timeCountCached.ResetTime();

        DirectoryHandle directoryHandle;
        DirectoryHandle directoryHandleCached;

        NNT_ASSERT_RESULT_SUCCESS(
            OpenDirectory(
                &directoryHandle,
                DirectoryPath,
                OpenDirectoryMode_File
            )
        );
        NNT_ASSERT_RESULT_SUCCESS(
            OpenDirectory(
                &directoryHandleCached,
                DirectoryPathCached,
                OpenDirectoryMode_File
            )
        );

        while( NN_STATIC_CONDITION(true) )
        {
            int64_t readNum = 0;
            std::unique_ptr<DirectoryEntry[]> dirEntry(new DirectoryEntry[i]);

            timeCount.StartTime();
            NNT_EXPECT_RESULT_SUCCESS(ReadDirectory(&readNum, dirEntry.get(), directoryHandle, i));
            timeCount.StopTime();

            timeCountCached.StartTime();
            NNT_EXPECT_RESULT_SUCCESS(ReadDirectory(&readNum, dirEntry.get(), directoryHandleCached, i));
            timeCountCached.StopTime();

            if( readNum == 0 )
            {
                break;
            }
        }
        CloseDirectory(directoryHandle);
        CloseDirectory(directoryHandleCached);

        NN_LOG("ReadDirectory\n");
        timeCount.ViewSimpleTime(timeCount.GetTotalTime());
        NN_LOG("Read=%d, Write=%d\n", m_StorageMem.GetReadTimes(), m_StorageMem.GetWriteTimes());
        NN_LOG("ReadDirectory [Cached]\n");
        timeCountCached.ViewSimpleTime(timeCountCached.GetTotalTime());
        NN_LOG("Read=%d, Write=%d\n", m_StorageMemCached.GetReadTimes(), m_StorageMemCached.GetWriteTimes());

        ASSERT_LE(m_StorageMemCached.GetReadTimes(), m_StorageMem.GetReadTimes());
        ASSERT_GE(m_StorageMemCached.GetWriteTimes(), m_StorageMem.GetWriteTimes());
    }
}

// ファイルの読み込み
TEST_F(FatFileSystemWithCacheAccessCount, ReadFile)
{
    FileHandle handle;
    int64_t offset;
    std::unique_ptr<char[]> buffer(new char[8 * 1024]);

    (void)DeleteFile("test:/file");
    NNT_ASSERT_RESULT_SUCCESS(CreateFile("test:/file", 512 * 1024));

    NNT_ASSERT_RESULT_SUCCESS(OpenFile(&handle, "test:/file", OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        CloseFile(handle);
    };

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    nnt::fs::util::TimeCount timeCount;
    for( size_t size = 4; size < 8 * 1024; size *= 2 )
    {
        timeCount.ResetTime();
        m_StorageMem.ResetAccessCounter();

        for( int i = 0; i < 100; ++i )
        {
            offset = std::uniform_int_distribution<int64_t>(0, static_cast<int64_t>(size - 1))(mt);

            timeCount.StartTime();
            NNT_ASSERT_RESULT_SUCCESS(ReadFile(handle, offset, buffer.get(), size));
            timeCount.StopTime();
        }

        NN_LOG("ReadFile (Size=%zu) : ", size);
        timeCount.ViewSimpleTime(timeCount.GetTotalTime());

        NN_LOG("Read=%d, Write=%d\n", m_StorageMem.GetReadTimes(), m_StorageMem.GetWriteTimes());
    }
}

// ファイルの書き込み
TEST_F(FatFileSystemWithCacheAccessCount, WriteFile)
{
    const int64_t MaxSize = 8 * 1024;
    FileHandle handle;
    FileHandle handleCached;
    int64_t offset;
    std::unique_ptr<char[]> buffer(new char[MaxSize]);

    (void)DeleteFile("test:/file");
    (void)DeleteFile("cached:/file");
    NNT_ASSERT_RESULT_SUCCESS(CreateFile("test:/file", 512 * 1024));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile("cached:/file", 512 * 1024));

    NNT_ASSERT_RESULT_SUCCESS(OpenFile(&handle, "test:/file", OpenMode_Read | OpenMode_Write));
    NN_UTIL_SCOPE_EXIT
    {
        CloseFile(handle);
    };
    NNT_ASSERT_RESULT_SUCCESS(OpenFile(&handleCached, "cached:/file", OpenMode_Read | OpenMode_Write));
    NN_UTIL_SCOPE_EXIT
    {
        CloseFile(handleCached);
    };

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    nnt::fs::util::TimeCount timeCount;
    nnt::fs::util::TimeCount timeCountCached;

    for( size_t size = 4; size < MaxSize; size *= 2 )
    {
        timeCount.ResetTime();
        timeCountCached.ResetTime();

        m_StorageMem.ResetAccessCounter();
        m_StorageMemCached.ResetAccessCounter();

        for( int i = 0; i < 300; ++i )
        {
            offset = std::uniform_int_distribution<int64_t>(0, static_cast<int64_t>(512 * 1024 - size - 1))(mt);

            timeCount.StartTime();
            NNT_ASSERT_RESULT_SUCCESS(
                WriteFile(
                    handle,
                    offset,
                    buffer.get(),
                    size,
                    WriteOption::MakeValue(0)
                )
            );
            NNT_ASSERT_RESULT_SUCCESS(FlushFile(handle));
            timeCount.StopTime();

            timeCountCached.StartTime();
            NNT_ASSERT_RESULT_SUCCESS(
                WriteFile(
                    handleCached,
                    offset,
                    buffer.get(),
                    size,
                    WriteOption::MakeValue(0)
                )
            );
            NNT_ASSERT_RESULT_SUCCESS(FlushFile(handleCached));
            timeCountCached.StopTime();

            timeCount.StartTime();
            NNT_ASSERT_RESULT_SUCCESS(ReadFile(handle, offset, buffer.get(), size));
            timeCount.StopTime();

            timeCountCached.StartTime();
            NNT_ASSERT_RESULT_SUCCESS(ReadFile(handleCached, offset, buffer.get(), size));
            timeCountCached.StopTime();
        }
        timeCount.ViewSimpleTime(timeCount.GetTotalTime());
        timeCountCached.ViewSimpleTime(timeCountCached.GetTotalTime());
    }
}

// ファイルの作成、書き込み、削除
TEST_F(FatFileSystemWithCacheAccessCount, CreateWriteDeleteFile)
{
    const int64_t MaxSize = 8 * 1024;
    std::unique_ptr<char[]> buffer(new char[MaxSize]);

    SetupPathTable(FileCount, false);

    (void)DeleteDirectoryRecursively(DirectoryPath);
    (void)DeleteDirectoryRecursively(DirectoryPathCached);
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(DirectoryPath));
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(DirectoryPathCached));

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    m_StorageMem.ResetAccessCounter();
    m_StorageMemCached.ResetAccessCounter();
    for( int i = 0; i < FileCount; ++i )
    {
        NNT_ASSERT_RESULT_SUCCESS(CreateFile(g_Path[i], 0));
        NNT_ASSERT_RESULT_SUCCESS(CreateFile(g_PathCached[i], 0));
    }
    NN_LOG(" Read=%d, Write=%d\n", m_StorageMem.GetReadTimes(), m_StorageMem.GetWriteTimes());
    NN_LOG(" Read=%d, Write=%d [Cached]\n", m_StorageMemCached.GetReadTimes(), m_StorageMemCached.GetWriteTimes());

    m_StorageMem.ResetAccessCounter();
    m_StorageMemCached.ResetAccessCounter();

    for( int i = 0; i < FileCount; ++i )
    {
        nn::fs::DirectoryEntryType entryType;
        NNT_ASSERT_RESULT_SUCCESS(GetEntryType(&entryType, g_Path[i]));
        NNT_ASSERT_RESULT_SUCCESS(GetEntryType(&entryType, g_PathCached[i]));
    }
    NN_LOG(" Read=%d, Write=%d\n", m_StorageMem.GetReadTimes(), m_StorageMem.GetWriteTimes());
    NN_LOG(" Read=%d, Write=%d [Cached]\n", m_StorageMemCached.GetReadTimes(), m_StorageMemCached.GetWriteTimes());

    {
        nnt::fs::util::TimeCount timeCount;
        nnt::fs::util::TimeCount timeCountCached;
        FileHandle handle;
        FileHandle handleCached;
        int64_t offset;
        (void)DeleteFile("test:/file");
        (void)DeleteFile("cached:/file");
        NNT_ASSERT_RESULT_SUCCESS(CreateFile("test:/file", 512 * 1024));
        NNT_ASSERT_RESULT_SUCCESS(CreateFile("cached:/file", 512 * 1024));
        NNT_ASSERT_RESULT_SUCCESS(OpenFile(&handle, "test:/file", OpenMode_Read | OpenMode_Write));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };
        NNT_ASSERT_RESULT_SUCCESS(OpenFile(&handleCached, "cached:/file", OpenMode_Read | OpenMode_Write));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handleCached);
        };
        for( size_t size = 4; size < MaxSize; size *= 2 )
        {
            timeCount.ResetTime();
            timeCountCached.ResetTime();

            m_StorageMem.ResetAccessCounter();
            m_StorageMemCached.ResetAccessCounter();

            for( int i = 0; i < 300; ++i )
            {
                offset = std::uniform_int_distribution<int64_t>(0, static_cast<int64_t>(512 * 1024 - size - 1))(mt);

                timeCount.StartTime();
                NNT_ASSERT_RESULT_SUCCESS(
                    WriteFile(
                        handle,
                        offset,
                        buffer.get(),
                        size,
                        WriteOption::MakeValue(0)
                    )
                );
                NNT_ASSERT_RESULT_SUCCESS(FlushFile(handle));
                timeCount.StopTime();

                timeCountCached.StartTime();
                NNT_ASSERT_RESULT_SUCCESS(
                    WriteFile(
                        handleCached,
                        offset,
                        buffer.get(),
                        size,
                        WriteOption::MakeValue(0)
                    )
                );
                NNT_ASSERT_RESULT_SUCCESS(FlushFile(handleCached));
                timeCountCached.StopTime();

                timeCount.StartTime();
                NNT_ASSERT_RESULT_SUCCESS(ReadFile(handle, offset, buffer.get(), size));
                timeCount.StopTime();

                timeCountCached.StartTime();
                NNT_ASSERT_RESULT_SUCCESS(ReadFile(handleCached, offset, buffer.get(), size));
                timeCountCached.StopTime();
            }

            timeCount.ViewSimpleTime(timeCount.GetTotalTime());
            timeCountCached.ViewSimpleTime(timeCountCached.GetTotalTime());
        }
    }

    SetupPathTable(FileCount, true);

    m_StorageMem.ResetAccessCounter();
    m_StorageMemCached.ResetAccessCounter();
    for( int i = 0; i < FileCount; ++i )
    {
        NNT_ASSERT_RESULT_SUCCESS(DeleteFile(g_Path[i]));
        NNT_ASSERT_RESULT_SUCCESS(DeleteFile(g_PathCached[i]));
    }
    NN_LOG(" Read=%d, Write=%d\n", m_StorageMem.GetReadTimes(), m_StorageMem.GetWriteTimes());
    NN_LOG(" Read=%d, Write=%d [Cached]\n", m_StorageMemCached.GetReadTimes(), m_StorageMemCached.GetWriteTimes());
} // NOLINT(impl/function_size)
#endif

}}}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);

    static const size_t BufferPoolSize = 8 * 1024 * 1024;
    static NN_ALIGNAS(4096) char s_BufferPool[BufferPoolSize];
    nn::fssystem::InitializeBufferPool(s_BufferPool, BufferPoolSize);

    ::testing::InitGoogleTest(&argc, argv);
    int result = RUN_ALL_TESTS();

    nnt::Exit(result);
}
