﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>

#include <nn/fssystem/dbm/fs_AllocationTable.h>
#include <nn/fssystem/dbm/fs_AllocationTableStorage.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>

//! AllocationTableStorage に必要なバッファとテーブルを初期化して準備するクラスです。
template<typename TStorage>
class AllocationTableStorageSetupTemplate
{
public:
    //! コンストラクタでブロック数とストレージのサイズを指定して初期化します。
    AllocationTableStorageSetupTemplate(uint32_t countBlocks, int64_t sizeBody) NN_NOEXCEPT
    : m_TableStore(nn::fssystem::dbm::AllocationTable::QuerySize(countBlocks)),
      m_BodyStore(sizeBody),
      m_TableSubStorage(
          &m_TableStore, 0, nn::fssystem::dbm::AllocationTable::QuerySize(countBlocks)
      ),
      m_BodySubStorage(&m_BodyStore, 0, sizeBody),
      m_AllocationTable()
    {
        // ストレージ実体を生成し、初期化します。
        nn::Result result = m_AllocationTable.Format(m_TableSubStorage, countBlocks);
        NN_ASSERT(result.IsSuccess());
        m_AllocationTable.Initialize(m_TableSubStorage, countBlocks);
    }

public:
    TStorage& GetBodyStore() NN_NOEXCEPT
    {
        return m_BodyStore;
    }

    nn::fs::SubStorage& GetBodySubStorage() NN_NOEXCEPT
    {
        return m_BodySubStorage;
    }

    nn::fssystem::dbm::AllocationTable& GetAllocationTable() NN_NOEXCEPT
    {
        return m_AllocationTable;
    }

private:
    TStorage m_TableStore;
    TStorage m_BodyStore;
    nn::fs::SubStorage m_TableSubStorage;
    nn::fs::SubStorage m_BodySubStorage;
    nn::fssystem::dbm::AllocationTable m_AllocationTable;
};
typedef AllocationTableStorageSetupTemplate<nnt::fs::util::SafeMemoryStorage> AllocationTableStorageSetup;

class AllocationTableStorageTest : public ::testing::Test
{
public:
    static nn::Result SeekTo(
                          const nn::fssystem::dbm::AllocationTableStorage& storage,
                          uint32_t* outIndex,
                          uint32_t* outBlockCount,
                          int64_t* outOffset,
                          int64_t offset
                      ) NN_NOEXCEPT
    {
        return storage.SeekTo(outIndex, outBlockCount, outOffset, offset);
    }
};

//! 初期化前のブロックサイズ取得テストです。
#if !defined(NN_SDK_BUILD_RELEASE)
TEST(AllocationTableStorageDeathTest, DeathTestGetBlockSize)
{
    // ストレージを作成します。
    nn::fssystem::dbm::AllocationTableStorage storage;

    // ストレージを初期化する前だとブロックサイズの取得に失敗します。
    EXPECT_DEATH_IF_SUPPORTED(storage.GetBlockSize(1), "");
}
#endif

//! ブロックサイズ取得のテストです。
TEST_F(AllocationTableStorageTest, TestGetBlockSize)
{
    // 確保するサイズを定義します。
    static const uint32_t CountAllocTable = 16;
    static const int SizeBody = 16;
    const size_t sizeBlock = sizeof(uint32_t);

    // テーブルとストレージを初期化します。
    AllocationTableStorageSetup tableStorageSetup(CountAllocTable, SizeBody * sizeBlock);

    // ストレージ管理領域を確保します。
    nn::fssystem::dbm::AllocationTable& allocationTable = tableStorageSetup.GetAllocationTable();
    uint32_t index;
    NNT_ASSERT_RESULT_SUCCESS(allocationTable.Allocate(&index, 1));

    // ストレージを作成します。
    nn::fssystem::dbm::AllocationTableStorage storage;

    // ストレージを初期化します。
    NNT_ASSERT_RESULT_SUCCESS(
        storage.Initialize(
            &allocationTable,
            index,
            sizeBlock,
            tableStorageSetup.GetBodySubStorage()
        )
    );

    // ブロックの個数を変えながらそのサイズを取得します。
    for( int i = 0; i < 1000; ++i )
    {
        ASSERT_EQ(sizeBlock * i, storage.GetBlockSize(i));
    }
}

//! 断片化したストレージの読み書きをテストします。
TEST_F(AllocationTableStorageTest, TestReadWriteToFragmentation)
{
    // 確保するサイズを定義します。
    static const uint32_t CountAllocationTable = 16;
    static const int SizeBody = 16;

    // テーブルとストレージを初期化します。
    AllocationTableStorageSetup tableStorageSetup(
        CountAllocationTable, SizeBody * sizeof(uint32_t)
    );
    nnt::fs::util::SafeMemoryStorage& bodyStore = tableStorageSetup.GetBodyStore();

    // body のバッファを添字で初期化します。
    uint32_t* bufStorageBuffer = static_cast<uint32_t*>(bodyStore.GetBuffer());

    for( int i = 0; i < SizeBody; ++i )
    {
        bufStorageBuffer[i] = i;
    }

    nn::fssystem::dbm::AllocationTable& allocationTable = tableStorageSetup.GetAllocationTable();

    // 分断化したテーブルを作成します。
    uint32_t nIndex[CountAllocationTable];

    // 1 ブロックずつ確保し、偶数番目のインデックスのみ解放します。
    for( uint32_t& index : nIndex )
    {
        NNT_ASSERT_RESULT_SUCCESS(allocationTable.Allocate(&index, 1));
    }
    for( int i = 0; i < CountAllocationTable; i += 2 )
    {
        NNT_ASSERT_RESULT_SUCCESS(allocationTable.Free(nIndex[i]));
    }
    // 全体の半分のブロック数を確保します。
    // 解放した偶数番目のインデックスが確保されます。
    NNT_ASSERT_RESULT_SUCCESS(allocationTable.Allocate(&nIndex[0], CountAllocationTable / 2));

    nnt::fs::util::Vector<uint32_t> bufTemporary(SizeBody, 0);

    // ストレージを作成します。
    nn::fssystem::dbm::AllocationTableStorage storage;
    NNT_ASSERT_RESULT_SUCCESS(
        storage.Initialize(
            &allocationTable,
            nIndex[0],
            sizeof(uint32_t),
            tableStorageSetup.GetBodySubStorage()
        )
    );

    // テーブルにしたがってアクセスします。
    nn::Result result = storage.Read(
        0, &bufTemporary[0], (CountAllocationTable / 2) * sizeof(uint32_t)
    );
    NNT_ASSERT_RESULT_SUCCESS(result);

    // テーブルと突合せします。これと一緒になるはず。
    {
        int i = 0;
        uint32_t indexNext;
        uint32_t index = nIndex[0];
        uint32_t nextSize = 0;
        do
        {
            // テーブルから確保した領域をひとつ取得します。
            NNT_ASSERT_RESULT_SUCCESS(allocationTable.ReadNext(&indexNext, &nextSize, index));
            // ストレージからアクセスした内容と一致するかテストします。
            // 最初に body のバッファを添字で初期化しているため index と一致します。
            ASSERT_EQ(index, bufTemporary[i]);
            ++i;
            index = indexNext;

        } while( allocationTable.IndexEnd != index );
    }

    // テーブルに従ってアクセスします。
    for( int i = 0; i < CountAllocationTable / 2; ++i )
    {
        // バッファを書き換えます。
        bufTemporary[i] = 2313 * i;
    }
    // ストレージに書き換えたバッファをコピーします。
    result = storage.Write(0, &bufTemporary[0], (CountAllocationTable / 2) * sizeof(uint32_t));
    NNT_ASSERT_RESULT_SUCCESS(result);

    NNT_ASSERT_RESULT_SUCCESS(storage.Flush());

    // ストレージからバッファに読み込みます。
    // 出力用引数が書き換わることをテストするために一度クリアします。
    std::fill(bufTemporary.begin(), bufTemporary.end(), 0);
    result = storage.Read(0, &bufTemporary[0], (CountAllocationTable / 2) * sizeof(uint32_t));
    NNT_ASSERT_RESULT_SUCCESS(result);
    for( int i = 0; i < CountAllocationTable / 2; ++i )
    {
        // 読み込んだバッファをテストします。
        ASSERT_EQ(static_cast<uint32_t>(2313 * i), bufTemporary[i]);
    }

    storage.Finalize();
    NNT_ASSERT_RESULT_SUCCESS(allocationTable.Free(nIndex[0]));
}

//! ストレージの割り当て領域を増やして読み書きをテストします。
TEST_F(AllocationTableStorageTest, TestReadWriteAllocateBlock)
{
    // 確保するサイズを定義します。
    static const uint32_t CountAllocationTable = 16;
    static const int SizeBody = 16;

    // テーブルとストレージを初期化します。
    AllocationTableStorageSetup tableStorageSetup(
        CountAllocationTable, SizeBody * sizeof(uint32_t)
    );
    nnt::fs::util::SafeMemoryStorage& bodyStore = tableStorageSetup.GetBodyStore();

    // body のバッファを添字で初期化します。
    uint32_t* bufStorageBuffer = static_cast<uint32_t*>(bodyStore.GetBuffer());

    for( int i = 0; i < SizeBody; ++i )
    {
        bufStorageBuffer[i] = i;
    }

    nn::fssystem::dbm::AllocationTable& allocationTable = tableStorageSetup.GetAllocationTable();

    // 分断化したテーブルを作成します。
    uint32_t nIndex[CountAllocationTable];

    // 1 ブロック割り当てます。
    NNT_ASSERT_RESULT_SUCCESS(allocationTable.Allocate(&nIndex[0], 1));

    nnt::fs::util::Vector<uint32_t> bufTemporary(SizeBody);

    // ストレージを作成します。
    nn::fssystem::dbm::AllocationTableStorage storage;
    NNT_ASSERT_RESULT_SUCCESS(
        storage.Initialize(
            &allocationTable,
            nIndex[0],
            sizeof(uint32_t),
            tableStorageSetup.GetBodySubStorage()
        )
    );
    // 1 ブロックずつ 3 ブロック割り当てます。（合計 4 ブロック）
    NNT_ASSERT_RESULT_SUCCESS(storage.AllocateBlock(1));
    NNT_ASSERT_RESULT_SUCCESS(storage.AllocateBlock(1));
    NNT_ASSERT_RESULT_SUCCESS(storage.AllocateBlock(1));

    // 16バイトのデータを読み書きします
    // テーブルに従ってアクセスします
    for( int i = 0; i < 4; ++i )
    {
        // バッファに書き込みます。
        bufTemporary[i] = i * 0x11111111;
    }
    // バッファをストレージにコピーします。
    NNT_ASSERT_RESULT_SUCCESS(storage.Write(0, &bufTemporary[0], 4 * sizeof(uint32_t)));
    // ストレージからバッファに読み込みます。
    // 出力用引数が書き換わることをテストするために一度クリアします。
    std::fill(bufTemporary.begin(), bufTemporary.end(), 0);
    NNT_ASSERT_RESULT_SUCCESS(storage.Read(0, &bufTemporary[0], 4 * sizeof(uint32_t)));
    for( int i = 0; i < 4; ++i )
    {
        // 読み込んだバッファをテストします。
        ASSERT_EQ(static_cast<uint32_t>(0x11111111 * i), bufTemporary[i]);
    }
}

//! AllocationTableStorage への以下のランダムな操作をテストします。
//!     - データ領域の確保と追加
//!     - データ領域の解放
//!     - 確保したデータ領域への読み書き
TEST_F(AllocationTableStorageTest, TestRandom)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    static const int MinBlockSize = 1;

    for( int i = 0; i < 20; ++i )
    {
        // ブロックの数と 1 ブロックあたりのサイズをランダムに決定します。
        const uint32_t countBlocks = std::uniform_int_distribution<uint32_t>(1, 1000)(mt);
        const int sizeBlock = 1 << (MinBlockSize + std::uniform_int_distribution<int>(0, 6)(mt));

        // テーブルとストレージを初期化します。
        AllocationTableStorageSetup tableStorageSetup(countBlocks, sizeBlock * countBlocks);
        nn::fssystem::dbm::AllocationTable& allocationTable
            = tableStorageSetup.GetAllocationTable();

        // 確保したデータ領域の情報を保存する配列を用意します。
        static const int CountTestIndex = 10;
        // 確保した領域の先頭インデックス
        uint32_t allocatedIndexies[CountTestIndex] = {0};
        // 確保した領域のブロック数
        uint32_t allocatedSizes[CountTestIndex] = {0};
        // アロケーションテーブルストレージ
        nn::fssystem::dbm::AllocationTableStorage storage[CountTestIndex];
        // アロケーションテーブルストレージをマウントしたか
        bool bMounted[CountTestIndex] = {false};
        // 確保されていないブロックの残り数
        uint32_t countLeft = countBlocks;
        for( int j = 0; j < 100; ++j )
        {
            // 操作するストレージをランダムに決定します。
            int index = std::uniform_int_distribution<int>(0, CountTestIndex - 1)(mt);
            // ランダムに操作を行います。
            switch( std::uniform_int_distribution<>(0, 5)(mt) )
            {
            case 0:
            case 1:
            case 2:
            case 3:
                // 確保、追加のテストを行います。
                // 空いているブロックからランダムな個数を確保します。
                if( countLeft > 0 )
                {
                    const uint32_t counts
                        = std::uniform_int_distribution<uint32_t>(0, countLeft - 1)(mt);
                    if( counts > 0 )
                    {
                        countLeft -= counts;
                        if( bMounted[index] )
                        {
                            // 既にマウントされている場合
                            // 追加確保のため、一旦アンマウントします。
                            ASSERT_TRUE(allocatedSizes[index] > 0);
                            storage[index].Finalize();
                            bMounted[index] = false;

                            // 新規にデータ領域を確保します。
                            uint32_t indexAdd;
                            NNT_ASSERT_RESULT_SUCCESS(
                                allocationTable.Allocate(&indexAdd, counts)
                            );
                            // 新規に確保した領域と既に確保した領域を連結します。
                            NNT_ASSERT_RESULT_SUCCESS(
                                allocationTable.Concat(
                                    allocatedIndexies[index],
                                    indexAdd
                                )
                            );
                            // 確保したブロック数を増やします。
                            allocatedSizes[index] += counts;
                        }
                        else
                        {
                            // 新規確保します。
                            ASSERT_EQ(0, allocatedSizes[index]);
                            allocatedSizes[index] = counts;
                            NNT_ASSERT_RESULT_SUCCESS(
                                allocationTable.Allocate(&allocatedIndexies[index], counts)
                            );
                        }
                        // マウントします。
                        NNT_ASSERT_RESULT_SUCCESS(
                            storage[index].Initialize(
                                &allocationTable,
                                allocatedIndexies[index],
                                static_cast<uint32_t>(sizeBlock),
                                tableStorageSetup.GetBodySubStorage()
                            )
                        );
                        bMounted[index] = true;
                    }
                }
                break;

            case 4:
                // 解放のテストを行います。
                if( allocatedSizes[index] > 0 )
                {
                    if( bMounted[index] )
                    {
                        // アンマウントします。
                        ASSERT_TRUE(allocatedSizes[index] > 0);
                        storage[index].Finalize();
                        bMounted[index] = false;
                    }
                    // データ領域を解放します。
                    NNT_ASSERT_RESULT_SUCCESS(
                        allocationTable.Free(allocatedIndexies[index])
                    );
                    countLeft += allocatedSizes[index];
                    allocatedSizes[index] = 0;
                    allocatedIndexies[index] = 0;
                }
                break;

            case 5:
                // ベリファイテストを行います。
                {
                    // 各ストレージがマウントされていれば全領域を書き込みます。
                    for( index = 0; index < CountTestIndex; ++index )
                    {
                        if( bMounted[index] )
                        {
                            nnt::fs::util::Vector<char> testBuf(
                                sizeBlock * allocatedSizes[index], static_cast<char>(index)
                            );
                            NNT_ASSERT_RESULT_SUCCESS(
                                storage[index].Write(
                                    0,
                                    &testBuf[0],
                                    sizeBlock * allocatedSizes[index]
                                )
                            );
                        }
                    }
                    // ストレージからデータを読み込み、書き込んだ内容と一致するかテストします。
                    for( index = 0; index < CountTestIndex; ++index )
                    {
                        if( bMounted[index] )
                        {
                            // 書き込んだバッファと同じものを作ります。
                            const size_t sizeArray = sizeBlock * allocatedSizes[index];
                            const nnt::fs::util::Vector<char> testBuf1(sizeArray, static_cast<char>(index));
                            nnt::fs::util::Vector<char> testBuf2(sizeArray, static_cast<char>(index + 1));
                            // ストレージから読み込みます。
                            NNT_ASSERT_RESULT_SUCCESS(
                                storage[index].Read(
                                    0,
                                    &testBuf2[0],
                                    sizeArray
                                )
                            );
                            // 読み込んだデータと書き込んだバッファが一致するかテストします。
                            ASSERT_EQ(0, std::memcmp(&testBuf1[0], &testBuf2[0], sizeArray));
                        }
                    }
                }
                break;

            default: NN_UNEXPECTED_DEFAULT;
            }
        }
        for( int j = 0; j < CountTestIndex; ++j )
        {
            if( bMounted[j] )
            {
                // このストレージで簡易テスト
                const int size = sizeBlock * allocatedSizes[j];
                for( int k = size; k > 0; --k )
                {
                    // ストレージに 1 バイト書き込みます。
                    NNT_ASSERT_RESULT_SUCCESS(storage[j].Write(k - 1, &k, 1));
                    // ストレージから 1 バイト読み込みます。
                    uint32_t l = 0;
                    NNT_ASSERT_RESULT_SUCCESS(storage[j].Read(k - 1, &l, 1));
                    // 書き込んだ値と読み込んだ値が一致するかをテストします。
                    ASSERT_EQ(k & 255, l);
                }
                for( int k = size; k > 0; --k )
                {
                    // ストレージから 1 バイト読み込みます。
                    uint32_t l = 0;
                    NNT_ASSERT_RESULT_SUCCESS(storage[j].Read(k - 1, &l, 1));
                    // 書き込んだ値と読み込んだ値が一致するかをテストします。
                    ASSERT_EQ(k & 255, l);
                }
                break;
            }
        }
    }
} // NOLINT(impl/function_size)

//! AllocationTableStorage の割り当て領域追加をテストします。
TEST_F(AllocationTableStorageTest, TestAllocateBlock)
{
    // 10 ブロック分の AllocationTable 作成
    static const uint32_t CountBlocks = 10;
    for( int i = 4; i < 8; ++i )
    {
        for( int j = 4; j < 8; ++j )
        {
            // AllocationTableStorage のエントリー（ブロック）のサイズ
            const int sizeEntry = 1 << j;

            // テーブルとストレージを初期化します。
            AllocationTableStorageSetup tableStorageSetup(
                                            CountBlocks, sizeEntry * (CountBlocks + 2)
                                        );
            nn::fssystem::dbm::AllocationTable& allocationTable
                = tableStorageSetup.GetAllocationTable();

            // マウントします。
            nn::fssystem::dbm::AllocationTableStorage storage;
            NNT_ASSERT_RESULT_SUCCESS(
                storage.Initialize(
                    &allocationTable,
                    0,                                    // 管理用ブロック数
                    static_cast<uint32_t>(sizeEntry),     // ブロックのサイズ
                    tableStorageSetup.GetBodySubStorage() // 実データストレージ
                )
            );

            // ストレージの割り当て領域を増やします。

            // テーブルは 10 ブロックなので 11 ブロック追加はできません。（領域不足）
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultAllocationTableFull,
                storage.AllocateBlock(11)
            );

            // 0 ブロック割り当てに 5 ブロック追加し、計 5 ブロック
            NNT_ASSERT_RESULT_SUCCESS(storage.AllocateBlock(5));

            // 5 ブロック割り当てに 1 ブロック追加し、計 6 ブロック
            NNT_ASSERT_RESULT_SUCCESS(storage.AllocateBlock(1));

            // 6 ブロック割り当てに 5 ブロック追加はできません。（領域不足）
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultAllocationTableFull,
                storage.AllocateBlock(5)
            );

            // 6 ブロック割り当てに 4 ブロック追加し、計 10 ブロック
            NNT_ASSERT_RESULT_SUCCESS(storage.AllocateBlock(4));

            // 10 ブロック割り当てに 1 ブロック追加はできません。（領域不足）
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultAllocationTableFull,
                storage.AllocateBlock(1)
            );
        }
    }
}

//! アロケーションテーブルストレージのシークをテストします。

//! 断片化していないストレージのシークをテストします。
TEST_F(AllocationTableStorageTest, TestSeekNoFragmentation)
{
    // AllocationTable のブロック数は 10 とします。
    // ブロックサイズは 64 とします。
    static const uint32_t CountBlocks = 10;
    static const uint32_t SizeEntry = 1 << 6;
    static const int64_t StorageOffset = 16;

    // テーブルとストレージを初期化します。
    AllocationTableStorageSetup tableStorageSetup(
        CountBlocks, SizeEntry * CountBlocks + StorageOffset
    );

    nn::fssystem::dbm::AllocationTableStorage storage;

    // マウントします。
    NNT_ASSERT_RESULT_SUCCESS(
        storage.Initialize(
            &tableStorageSetup.GetAllocationTable(),
            0,                                       // 管理用ブロック数
            SizeEntry,                               // ブロックのサイズ
            tableStorageSetup.GetBodySubStorage()    // 実データストレージ
        )
    );

    // データ領域は分断されていないので、どのオフセット位置も先頭ブロックに含まれます。
    uint32_t index;         // ブロックインデックス
    uint32_t blockCount;    // 該当ブロック数
    int64_t offset;         // ブロック内での位置

    // 先頭からの位置が 0 なら、index は 0、blockCount は 10、offset は 0
    NNT_ASSERT_RESULT_SUCCESS(SeekTo(storage, &index, &blockCount, &offset, 0));
    ASSERT_EQ(0, index);
    ASSERT_EQ(10, blockCount);
    ASSERT_EQ(0, offset);

    // 先頭からの位置が 639 なら、index は 0、blockCount は 10、offset は 639
    NNT_ASSERT_RESULT_SUCCESS(SeekTo(storage, &index, &blockCount, &offset, 639));
    ASSERT_EQ(0, index);
    ASSERT_EQ(10, blockCount);
    ASSERT_EQ(639, offset);

    // 先頭からの位置が 192 なら、index は 0、blockCount は 10、offset は 192
    NNT_ASSERT_RESULT_SUCCESS(SeekTo(storage, &index, &blockCount, &offset, 192));
    ASSERT_EQ(0, index);
    ASSERT_EQ(10, blockCount);
    ASSERT_EQ(192, offset);
}

//! AllocationTableStorage 以外にも Allocate されている場合のシークをテストします。
TEST_F(AllocationTableStorageTest, TestSeekSimpleAllocation)
{
    // AllocationTable のブロック数は 10 とします。
    // ブロックサイズは 64 とします。
    static const uint32_t CountBlocks = 10;
    static const uint32_t SizeEntry = 1 << 6;
    static const int64_t StorageOffset = 16;

    // テーブルとストレージを初期化します。
    AllocationTableStorageSetup tableStorageSetup(
        CountBlocks, SizeEntry * CountBlocks + StorageOffset
    );
    nn::fssystem::dbm::AllocationTable& allocationTable = tableStorageSetup.GetAllocationTable();

    // ストレージに割り当てるブロックを [2] => [3] => [5-6] => [7] とします。
    // 先にデータ領域を 2 個割り当てておきます。
    uint32_t dummyIndex;
    // ブロック [0-1]
    NNT_ASSERT_RESULT_SUCCESS(allocationTable.Allocate(&dummyIndex, 2));
    // ブロック [2]
    NNT_ASSERT_RESULT_SUCCESS(allocationTable.Allocate(&dummyIndex, 1));

    // マウントします。
    nn::fssystem::dbm::AllocationTableStorage storage;
    NNT_ASSERT_RESULT_SUCCESS(
        storage.Initialize(
            &allocationTable,
            dummyIndex,                           // 開始ブロックのインデックス
            SizeEntry,                            // ブロックのサイズ
            tableStorageSetup.GetBodySubStorage() // 実データストレージ
        )
    );

    // これでブロック [2] が storage に割り当てられます。

    // ブロック [3]   storage に割り当て
    NNT_ASSERT_RESULT_SUCCESS(storage.AllocateBlock(1));
    // ブロック [4]
    NNT_ASSERT_RESULT_SUCCESS(allocationTable.Allocate(&dummyIndex, 1));
    // ブロック [5-6] storage に割り当て
    NNT_ASSERT_RESULT_SUCCESS(storage.AllocateBlock(2));
    // ブロック [7]   storage に割り当て
    NNT_ASSERT_RESULT_SUCCESS(storage.AllocateBlock(1));
    // ブロック [8-9]
    NNT_ASSERT_RESULT_SUCCESS(allocationTable.Allocate(&dummyIndex, 2));

    // 11 個目が割り当てられないことを確認
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAllocationTableFull,
        allocationTable.Allocate(&dummyIndex, 1)
    );

    //  先頭からの位置 (x) → index   blockCount  offset
    //    0～ 63              2       1           x       (0～63)
    //   64～127              3       1           x - 64  (0～63)
    //  128～255              5       2           x - 128 (0～127)
    //  256～320              7       1           x - 256 (0～64)
    uint32_t index;
    uint32_t blockCount;
    int64_t offset;

    // 先頭からの位置が 0 なら、index は 2、blockCount は 1、offset は 0
    NNT_ASSERT_RESULT_SUCCESS(SeekTo(storage, &index, &blockCount, &offset, 0));
    ASSERT_EQ(2, index);
    ASSERT_EQ(1, blockCount);
    ASSERT_EQ(0, offset);

    // 先頭からの位置が 255 なら、index は 5、blockCount は 2、offset は 127
    NNT_ASSERT_RESULT_SUCCESS(SeekTo(storage, &index, &blockCount, &offset, 255));
    ASSERT_EQ(5, index);
    ASSERT_EQ(2, blockCount);
    ASSERT_EQ(127, offset);

    // 先頭からの位置が 256 なら、index は 7、blockCount は 1、offset は 0
    NNT_ASSERT_RESULT_SUCCESS(SeekTo(storage, &index, &blockCount, &offset, 256));
    ASSERT_EQ(7, index);
    ASSERT_EQ(1, blockCount);
    ASSERT_EQ(0, offset);

    // 先頭からの位置が 319 なら、index は 7、blockCount は 1、offset は 63
    NNT_ASSERT_RESULT_SUCCESS(SeekTo(storage, &index, &blockCount, &offset, 319));
    ASSERT_EQ(7, index);
    ASSERT_EQ(1, blockCount);
    ASSERT_EQ(63, offset);

    // 先頭からの位置が 320 なら範囲外
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultInvalidOffset,
        SeekTo(storage, &index, &blockCount, &offset, 320)
    );

    // 先頭からの位置が 321 なら範囲外
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultInvalidOffset,
        SeekTo(storage, &index, &blockCount, &offset, 321)
    );

    // 先頭からの位置が 255 なら、index は 5、blockCount は 2、offset は 127
    NNT_ASSERT_RESULT_SUCCESS(SeekTo(storage, &index, &blockCount, &offset, 255));
    ASSERT_EQ(5, index);
    ASSERT_EQ(2, blockCount);
    ASSERT_EQ(127, offset);

    // 先頭からの位置が 64 なら、index は 3、blockCount は 1、offset は 0
    NNT_ASSERT_RESULT_SUCCESS(SeekTo(storage, &index, &blockCount, &offset, 64));
    ASSERT_EQ(3, index);
    ASSERT_EQ(1, blockCount);
    ASSERT_EQ(0, offset);

    // 先頭からの位置が 63 なら、index は 2、blockCount は 1、offset は 63
    NNT_ASSERT_RESULT_SUCCESS(SeekTo(storage, &index, &blockCount, &offset, 63));
    ASSERT_EQ(2, index);
    ASSERT_EQ(1, blockCount);
    ASSERT_EQ(63, offset);
}

//! 割り当てたブロックが前後する状態のシークをテストします。
TEST_F(AllocationTableStorageTest, TestSeekComplexAllocation)
{
    // AllocationTable のブロック数は 10 とします。
    // ブロックサイズは 64 とします。
    static const uint32_t CountBlocks = 10;
    static const uint32_t SizeEntry = 1 << 6;
    static const int64_t StorageOffset = 16;

    // テーブルとストレージを初期化します。
    AllocationTableStorageSetup tableStorageSetup(
        CountBlocks, SizeEntry * CountBlocks + StorageOffset
    );
    nn::fssystem::dbm::AllocationTable& allocationTable = tableStorageSetup.GetAllocationTable();

    // ストレージに割り当てるブロックを [1-3] => [5-6] => [0] => [8-9] とします。
    uint32_t dummyIndex;
    uint32_t indexZero;
    // [0]
    NNT_ASSERT_RESULT_SUCCESS(allocationTable.Allocate(&indexZero, 1));
    // [1-3] storage に割り当て
    NNT_ASSERT_RESULT_SUCCESS(allocationTable.Allocate(&dummyIndex, 3));

    // マウントします。
    nn::fssystem::dbm::AllocationTableStorage storage;
    NNT_ASSERT_RESULT_SUCCESS(
        storage.Initialize(
            &allocationTable,
            dummyIndex,                           // 管理用ブロック数
            SizeEntry,                            // ブロックのサイズ
            tableStorageSetup.GetBodySubStorage() // 実データストレージ
        )
    );

    // [4]
    NNT_ASSERT_RESULT_SUCCESS(allocationTable.Allocate(&dummyIndex, 1));
    // [5-6] storage に割り当て
    NNT_ASSERT_RESULT_SUCCESS(storage.AllocateBlock(2));
    // [0]   解放
    NNT_ASSERT_RESULT_SUCCESS(allocationTable.Free(indexZero));
    // [0]   storage に割り当て
    NNT_ASSERT_RESULT_SUCCESS(storage.AllocateBlock(1));
    // [7]
    NNT_ASSERT_RESULT_SUCCESS(allocationTable.Allocate(&dummyIndex, 1));
    // [8-9] storage に割り当て
    NNT_ASSERT_RESULT_SUCCESS(storage.AllocateBlock(2));

    //  先頭からの位置 (x) → index   blockCount  offset
    //    0～191              1       3           x       (0～191)
    //  192～319              5       2           x - 192 (0～127)
    //  320～383              0       1           x - 320 (0～63)
    //  384～512              8       2           x - 384 (0～128)
    uint32_t index;
    uint32_t blockCount;
    int64_t offset;

    // 先頭からの位置が 500 なら、index は 8、blockCount は 2、offset は 116
    NNT_ASSERT_RESULT_SUCCESS(SeekTo(storage, &index, &blockCount, &offset, 500));
    ASSERT_EQ(8, index);
    ASSERT_EQ(2, blockCount);
    ASSERT_EQ(116, offset);

    // 先頭からの位置が 1 なら、index は 1、blockCount は 3、offset は 1
    NNT_ASSERT_RESULT_SUCCESS(SeekTo(storage, &index, &blockCount, &offset, 1));
    ASSERT_EQ(1, index);
    ASSERT_EQ(3, blockCount);
    ASSERT_EQ(1, offset);

    // 先頭からの位置が 400 なら、index は 8、blockCount は 2、offset は 16
    NNT_ASSERT_RESULT_SUCCESS(SeekTo(storage, &index, &blockCount, &offset, 400));
    ASSERT_EQ(8, index);
    ASSERT_EQ(2, blockCount);
    ASSERT_EQ(16, offset);

    // 先頭からの位置が 350 なら、index は 0、blockCount は 1、offset は 30
    NNT_ASSERT_RESULT_SUCCESS(SeekTo(storage, &index, &blockCount, &offset, 350));
    ASSERT_EQ(0, index);
    ASSERT_EQ(1, blockCount);
    ASSERT_EQ(30, offset);

    // 先頭からの位置が 300 なら、index は 5、blockCount は 2、offset は 108
    NNT_ASSERT_RESULT_SUCCESS(SeekTo(storage, &index, &blockCount, &offset, 300));
    ASSERT_EQ(5, index);
    ASSERT_EQ(2, blockCount);
    ASSERT_EQ(108, offset);

    // 先頭からの位置が 250 なら、index は 5、blockCount は 2、offset は 58
    NNT_ASSERT_RESULT_SUCCESS(SeekTo(storage, &index, &blockCount, &offset, 250));
    ASSERT_EQ(5, index);
    ASSERT_EQ(2, blockCount);
    ASSERT_EQ(58, offset);

    // 先頭からの位置が 200 なら、index は 5、blockCount は 2、offset は 8
    NNT_ASSERT_RESULT_SUCCESS(SeekTo(storage, &index, &blockCount, &offset, 200));
    ASSERT_EQ(5, index);
    ASSERT_EQ(2, blockCount);
    ASSERT_EQ(8, offset);

    // 先頭からの位置が 500 なら、index は 8、blockCount は 2、offset は 116
    NNT_ASSERT_RESULT_SUCCESS(SeekTo(storage, &index, &blockCount, &offset, 500));
    ASSERT_EQ(8, index);
    ASSERT_EQ(2, blockCount);
    ASSERT_EQ(116, offset);

    // 先頭からの位置が 100 なら、index は 1、blockCount は 3、offset は 100
    NNT_ASSERT_RESULT_SUCCESS(SeekTo(storage, &index, &blockCount, &offset, 100));
    ASSERT_EQ(1, index);
    ASSERT_EQ(3, blockCount);
    ASSERT_EQ(100, offset);
}

//! 様々なブロックサイズでマウントしたアロケーションテーブルストレージに書き込むテストを行います。
TEST_F(AllocationTableStorageTest, TestWrite)
{
    // 1 ブロックぶんの AllocationTable 作成
    static const uint32_t CountBlocks = 1 << 10;

    // AllocationTable のブロックのサイズ
    static const size_t SizeBlock = 1 << 10;

    // Write するためのデータ作成
    nnt::fs::util::Vector<char> writeData(SizeBlock);

    for( int i = 4; i <= 7; ++i )
    {
        // AllocationTableStorage のエントリーのサイズ
        // 16 ～ 128 で変動させます。
        const uint32_t sizeEntry = 1 << i;

        // テーブルとストレージを初期化します。
        AllocationTableStorageSetup tableStorageSetup(CountBlocks, sizeEntry * CountBlocks);

        // テーブルの領域全体をデータ領域に割り当てます。
        uint32_t storageIndex;
        nn::fssystem::dbm::AllocationTable& allocationTable
            = tableStorageSetup.GetAllocationTable();
        NNT_ASSERT_RESULT_SUCCESS(allocationTable.Allocate(&storageIndex, CountBlocks));

        // マウントします。
        nn::fssystem::dbm::AllocationTableStorage storage;
        NNT_ASSERT_RESULT_SUCCESS(
            storage.Initialize(
                &allocationTable,
                storageIndex,                         // 管理用ブロック数
                sizeEntry,                            // ブロックのサイズ
                tableStorageSetup.GetBodySubStorage() // 実データストレージ
            )
        );
        // オフセットは 8 きざみで変動させます。
        for( uint32_t offset = 0; offset <= SizeBlock; offset += 8 )
        {
            // 書き込むサイズを 8 きざみで変動させます。
            for( uint32_t writeSize = 0; writeSize <= SizeBlock - offset; writeSize += 8 )
            {
                // 書き込みます。
                NNT_ASSERT_RESULT_SUCCESS(storage.Write(offset, &writeData[0], writeSize));
            }
        }
    }
}

//! 範囲外まで Read/Write するテストを行います。
TEST_F(AllocationTableStorageTest, OverLoad)
{
    static const uint32_t CountAllocationTable = 31;
    static const int BodyBlockSize = 64;
    static const int64_t AllocationTableOffset = 32;
    static const int64_t AllocationTableStorageOffset = 1024;

    // メモリストレージを準備します。
    const int64_t allocationTableSize
        = nn::fssystem::dbm::AllocationTable::QuerySize(CountAllocationTable)
        + AllocationTableOffset;
    nnt::fs::util::SafeMemoryStorage tableStore(static_cast<size_t>(allocationTableSize));
    nnt::fs::util::SafeMemoryStorage bodyStore(
        CountAllocationTable * BodyBlockSize + AllocationTableStorageOffset
    );

    // テーブル用 SubStorage を作成します。
    nn::fs::SubStorage tableSubStorage(
        &tableStore, AllocationTableOffset, allocationTableSize
    );

    // AllocationTable を作成します。
    nn::fssystem::dbm::AllocationTable allocationTable;
    NNT_ASSERT_RESULT_SUCCESS(
        nn::fssystem::dbm::AllocationTable::Format(tableSubStorage, CountAllocationTable)
    );
    allocationTable.Initialize(tableSubStorage, CountAllocationTable);
    // AllocationTableStorage 初期化時に割り当てる領域として 4 ブロック確保
    uint32_t manageIndex;
    NNT_ASSERT_RESULT_SUCCESS(allocationTable.Allocate(&manageIndex, 4));

    // AllocationTableStorage と、そのデータの SubStorage を作成します。
    nn::fs::SubStorage bodySubStorage(
        &bodyStore, AllocationTableStorageOffset, CountAllocationTable * BodyBlockSize
    );
    // 4 ブロック確保した状態で初期化
    nn::fssystem::dbm::AllocationTableStorage storage;
    NNT_ASSERT_RESULT_SUCCESS(
        storage.Initialize(
            &allocationTable,
            manageIndex,
            BodyBlockSize,
            bodySubStorage
        )
    );

    // さらに 10 ブロック確保（合計 14 ブロック ）
    NNT_ASSERT_RESULT_SUCCESS(storage.AllocateBlock(10));

    int64_t maxSize = 0;
    NNT_ASSERT_RESULT_SUCCESS(storage.GetSize(&maxSize));
    ASSERT_LT(0, maxSize);
    nnt::fs::util::Vector<char> buffer(2048);
    // 確保したサイズより大きいサイズを読み込もうとして失敗します。
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidOffset, storage.Read(0, &(buffer[0]), 2048));
    // 64 バイト目から最後まで読み込みます。
    NNT_ASSERT_RESULT_SUCCESS(storage.Read(64, &(buffer[0]), size_t(maxSize - 64)));
    // 64 バイト目から最後 + 1 まで読み込もうとして失敗します。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultInvalidOffset,
        storage.Read(64, &(buffer[0]), size_t(maxSize - 63))
    );

    // 確保したサイズより大きいサイズを書き込もうとして失敗します。
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidOffset, storage.Write(0, &(buffer[0]), 2048));
    // 64 バイト目から最後まで書き込みます。
    NNT_ASSERT_RESULT_SUCCESS(storage.Write(64, &(buffer[0]), size_t(maxSize - 64)));
    // 64 バイト目から最後 + 1 まで書き込もうとして失敗します。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultInvalidOffset,
        storage.Write(64, &(buffer[0]), size_t(maxSize - 63))
    );
}

// 4 GB を超えるオフセットのアクセステストです。
TEST(AllocationTableStorageLargeTest, ReadWriteLargeOffset)
{
    static const size_t AccessSize = 1024;
    AllocationTableStorageSetupTemplate<nnt::fs::util::VirtualMemoryStorage> tableStorageSetup(
        8 * 1024 * 1024,
        nnt::fs::util::LargeOffsetMax + AccessSize);

    nn::fssystem::dbm::AllocationTableStorage storage;
    NNT_ASSERT_RESULT_SUCCESS(storage.Initialize(
        &tableStorageSetup.GetAllocationTable(),
        0,
        16 * 1024,
        tableStorageSetup.GetBodySubStorage()));

    nnt::fs::util::TestStorageAccessWithLargeOffset(&storage, AccessSize);
}
