﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <random>
#include <mutex>
#include <numeric>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nn/fssystem/buffers/fs_FileSystemBuddyHeap.h>

#include <nnt/nntest.h>
#include <nnt/fsUtil/testFs_util.h>

namespace {

/*!
* @brief   バディヒープアロケータクラス
*/
class BuddyHeapAllocator
{
public:
    /*!
    * @brief       初期化をしないコンストラクタです。
    */
    BuddyHeapAllocator() NN_NOEXCEPT
    : m_pHeap(nullptr)
    {
    }

    /*!
    * @brief       アロケータを初期化します。
    *
    * @param[in]   pHeap   バディヒープ
    *
    * @pre
    *              - pHeap != nullptr
    */
    explicit BuddyHeapAllocator(nn::fssystem::FileSystemBuddyHeap* pHeap) NN_NOEXCEPT
                : m_pHeap(nullptr)
    {
        Initialize(pHeap);
    }

    /*!
    * @brief       アロケータを初期化します。
    *
    * @param[in]   pHeap   バディヒープ
    *
    * @pre
    *              - アロケータが未初期化である
    *              - pHeap != nullptr
    */
    void Initialize(nn::fssystem::FileSystemBuddyHeap* pHeap) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_pHeap == nullptr);
        NN_SDK_REQUIRES_NOT_NULL(pHeap);
        m_pHeap = pHeap;
    }

    /*!
    * @brief       アロケータにセットされているヒープを返します。
    *
    * @return      アロケータにセットされているヒープ
    */
    nn::fssystem::FileSystemBuddyHeap* GetHeap() NN_NOEXCEPT
    {
        return m_pHeap;
    }

    /*!
    * @brief       アロケータにセットされているヒープを返します。
    *
    * @return      アロケータにセットされているヒープ
    */
    const nn::fssystem::FileSystemBuddyHeap* GetHeap() const NN_NOEXCEPT
    {
        return m_pHeap;
    }

    /*!
    * @brief       指定したサイズでメモリ領域を確保します。
    *
    * @param[in]   size    確保するメモリのサイズ
    *
    * @return      確保したメモリ領域の先頭へのポインタ
    * @retval      nullptr 確保に失敗した
    *
    * @pre
    *              - アロケータが初期化済みである
    *              - size が GetHeap()->GetOrderMax() 以下のオーダーで確保できる
    */
    void* Allocate(size_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pHeap);
        const auto order = m_pHeap->GetOrderFromBytes(size);
        NN_SDK_REQUIRES_GREATER_EQUAL(order, 0);
        const auto ptr = m_pHeap->AllocateByOrder(order);
        if( nullptr == ptr )
        {
            // 空きメモリ不足
            return nullptr;
        }
        return ptr;
    }

    /*!
    * @brief       メモリ領域を解放します。
    *
    * @param[in]   ptr     確保されているメモリ領域の先頭へのポインタ
    * @param[in]   size    確保されているメモリ領域のサイズ
    *
    * @pre
    *              - アロケータが初期化済みである
    *              - size が ptr の確保時に渡した値に等しい
    */
    void Free(void* ptr, size_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pHeap);
        const auto order = m_pHeap->GetOrderFromBytes(size);
        NN_SDK_REQUIRES_GREATER_EQUAL(order, 0);
        m_pHeap->Free(ptr, order);
    }

    /*!
    * @brief       オーダーをバイト数に変換します。
    *
    * @param[in]   order    変換するオーダー
    *
    * @return      バイト数
    *
    * @pre
    *              - アロケータが初期化済みである
    */
    size_t GetBytesFromOrder(int order) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pHeap);
        return m_pHeap->GetBytesFromOrder(order);
    }

private:
    nn::fssystem::FileSystemBuddyHeap* m_pHeap; //!< バディヒープ
};

} // namespace

//! 順番に確保、解放します。
void Allocate(
         nn::fssystem::FileSystemBuddyHeap* pHeap,
         size_t heapSize,
         const nnt::fs::util::SafeMemoryStorage& buf,
         std::mt19937* pMt,
         int loopCountMax
     ) NN_NOEXCEPT
{
    ASSERT_NE(pHeap, nullptr);
    ASSERT_NE(pMt, nullptr);

    const auto totalFreeSize = pHeap->GetTotalFreeSize();

    BuddyHeapAllocator allocator(pHeap);

    // 1 ページに確保する最大サイズと確保するページ数
    const auto allocateSize = std::min(heapSize, pHeap->GetPageSizeMax());
    const auto pageCount = static_cast<int>(heapSize / allocateSize);

    // 確保した領域を保存する配列
    nnt::fs::util::Vector<char*> ptrs(pageCount);
    nnt::fs::util::Vector<size_t> sizes(pageCount);

    for( auto loopCount = 0; loopCount < loopCountMax; ++loopCount )
    {
        ASSERT_EQ(totalFreeSize, pHeap->GetTotalFreeSize());

        // 順番にデータを確保します。
        for( auto pageIndex = 0; pageIndex < pageCount; ++pageIndex )
        {
            const auto size
                = std::uniform_int_distribution<size_t>(1, allocateSize - 1)(*pMt);
            ptrs[pageIndex] = static_cast<char*>(allocator.Allocate(size));
            sizes[pageIndex] = size;
            ASSERT_NE(nullptr, ptrs[pageIndex]);

            // バッファ破壊確認のためデータをFillしておきます。
            std::memset(ptrs[pageIndex], (pageIndex & 0xff), size);
        }
        buf.CheckValid();

        // バッファが破損していないか確認します。
        for( auto pageIndex = 0; pageIndex < pageCount; ++pageIndex )
        {
            const char truth = pageIndex & 0xff;
            const auto size = sizes[pageIndex];
            // 先頭と末尾を確認すれば必要十分。
            EXPECT_EQ(truth, ptrs[pageIndex][0]);
            EXPECT_EQ(truth, ptrs[pageIndex][size - 1]);
        }
        buf.CheckValid();

        // 順番にデータを解放します。
        for( auto pageIndex = 0; pageIndex < pageCount; ++pageIndex )
        {
            allocator.Free(ptrs[pageIndex], sizes[pageIndex]);
            ptrs[pageIndex] = nullptr;
        }
        buf.CheckValid();
    }
}

//! 最大まで確保して、順番に解放します。
void AllocateMax(
         nn::fssystem::FileSystemBuddyHeap* pHeap,
         size_t heapSize,
         size_t blockSize,
         const nnt::fs::util::SafeMemoryStorage& buf,
         int loopCountMax
     ) NN_NOEXCEPT
{
    ASSERT_NE(pHeap, nullptr);

    const auto totalFreeSize = pHeap->GetTotalFreeSize();

    BuddyHeapAllocator allocator(pHeap);
    const auto pageCount = static_cast<int>(heapSize / blockSize);

    nnt::fs::util::Vector<char*> ptrs(pageCount);

    for( auto loopCount = 0; loopCount < loopCountMax; ++loopCount )
    {
        ASSERT_EQ(totalFreeSize, pHeap->GetTotalFreeSize());

        // 順番にページを確保します。
        for( auto pageIndex = 0; pageIndex < pageCount; ++pageIndex )
        {
            ptrs[pageIndex] = static_cast<char*>(allocator.Allocate(blockSize));
            ASSERT_NE(nullptr, ptrs[pageIndex]);

            // バッファ破壊確認のためデータをFillしておきます。
            std::memset(ptrs[pageIndex], (pageIndex & 0xff), blockSize);
        }
        buf.CheckValid();

        // これ以上は確保できません。
        {
            const auto ptr = allocator.Allocate(blockSize);
            ASSERT_EQ(nullptr, ptr);
            buf.CheckValid();
        }

        // バッファが破損していないか確認します。
        for( auto pageIndex = 0; pageIndex < pageCount; ++pageIndex )
        {
            const char truth = (pageIndex & 0xff);
            // 先頭と末尾を確認すれば必要十分
            EXPECT_EQ(truth, ptrs[pageIndex][0]);
            EXPECT_EQ(truth, ptrs[pageIndex][blockSize - 1]);
        }
        buf.CheckValid();

        // 順番にデータを解放します。
        for( auto& ptr : ptrs )
        {
            allocator.Free(ptr, blockSize);
            ptr = nullptr;
        }
        buf.CheckValid();
    }
}

//! メモリ領域を確保して順番に解放するテストです。
TEST(FileSystemBuddyHeapTest, Allocation)
{
    typedef nn::fssystem::FileSystemBuddyHeap BuddyHeap;

    std::mt19937 mt(1);

    static const auto LoopCountMax = 256;
    static const auto InnerLoopCountMax = 8;

    BuddyHeap heap;

    for( auto loopCount = 0; loopCount < LoopCountMax; ++loopCount )
    {
        NN_LOG(".");

        // ブロックサイズをランダムに決定します。
        const auto blockSizeExponent = std::uniform_int_distribution<>(8, 12)(mt);
        const auto blockSize = sizeof(char) << blockSizeExponent;

        // 最大ページサイズをランダムに決定します。（2^16 を超えない範囲）
        const auto orderMax = std::uniform_int_distribution<>(2, 16 - blockSizeExponent)(mt);
        const auto pageSizeMax = (static_cast<size_t>(1) << orderMax) * blockSize;

        // バディヒープのサイズをランダムに決定します。（最大ページサイズ以上になるように）
        // まず最大ブロック数を求めます。
        const auto blockCountMax = pageSizeMax / sizeof(void*);
        // 最大サイズのページの最大個数を求めます。
        const auto largestPageCountMax
            = static_cast<int>((blockCountMax * blockSize + (pageSizeMax - 1)) / pageSizeMax);
        // バディヒープのサイズを決定します。
        const auto buddySize
            = pageSizeMax * (std::uniform_int_distribution<>(1, largestPageCountMax - 1)(mt))
                + std::uniform_int_distribution<size_t>(0, pageSizeMax - 1)(mt);

        // バッファを確保します。（アライメント分を余分に確保）
        nnt::fs::util::SafeMemoryStorage buf(buddySize + BuddyHeap::BufferAlignment);
        const uintptr_t addr = reinterpret_cast<uintptr_t>(buf.GetBuffer());

        // バディヒープを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(
            heap.Initialize(
                nn::util::align_up(addr, BuddyHeap::BufferAlignment),
                buddySize,
                blockSize,
                orderMax
            )
        );
        buf.CheckValid();

        // 初期化直後の空き容量を記憶しておき、確保と解放後に元に戻るかチェックします。
        const auto totalFreeSize = heap.GetTotalFreeSize();

        // 適当なサイズで確保と解放を順次繰り返します。
        Allocate(&heap, buddySize, buf, &mt, InnerLoopCountMax);

        // 計算上の最大ページ数まで確保を試行します。
        // ここまでの処理でフリーリストがクリアな状態になっていないと失敗します。
        AllocateMax(&heap, buddySize, blockSize, buf, InnerLoopCountMax);

        // 解放されて空き容量が初期化直後と同じ量になっているかチェックします。
        ASSERT_EQ(totalFreeSize, heap.GetTotalFreeSize());

        heap.Finalize();
        buf.CheckValid();
    }

    NN_LOG("\n");
}

//! メモリを確保してランダムな順番で解放します。
void AllocateRandom(
         nn::fssystem::FileSystemBuddyHeap* pHeap,
         size_t heapSize,
         const nnt::fs::util::SafeMemoryStorage& buf,
         std::mt19937* pMt,
         int loopCountMax
     ) NN_NOEXCEPT
{
    ASSERT_NE(pHeap, nullptr);

    const auto totalFreeSize = pHeap->GetTotalFreeSize();

    BuddyHeapAllocator allocator(pHeap);
    const auto allocateSize = std::min(heapSize, pHeap->GetPageSizeMax());
    const auto pageCount = static_cast<int>(heapSize / allocateSize);

    nnt::fs::util::Vector<char*> ptrs(pageCount);
    nnt::fs::util::Vector<size_t> sizes(pageCount);

    // ソートされた順番の配列とランダムな順番の配列を用意します。
    // （ソートされた配列は、毎回 iota を呼ばないために保持します。）
    nnt::fs::util::Vector<int> sortedIdx(pageCount);
    nnt::fs::util::Vector<int> shuffledIdx(pageCount);
    std::iota(sortedIdx.begin(), sortedIdx.end(), 0);

    for( auto loopCount = 0; loopCount < loopCountMax; ++loopCount )
    {
        ASSERT_EQ(totalFreeSize, pHeap->GetTotalFreeSize());

        // 解放順番がランダムになるように適当に入れ替えます。
        shuffledIdx = sortedIdx;
        std::shuffle(shuffledIdx.begin(), shuffledIdx.end(), *pMt);

        // 順番にデータを確保します。
        for( auto pageIndex = 0; pageIndex < pageCount; ++pageIndex )
        {
            const auto sizeMax = pageIndex + 1 == pageCount
                ? nn::util::floorp2(heapSize - (pageCount - 1) * allocateSize)
                : allocateSize;
            const auto size = std::uniform_int_distribution<size_t>(1, sizeMax)(*pMt);
            ptrs[pageIndex] = reinterpret_cast<char*>(allocator.Allocate(size));
            sizes[pageIndex] = size;

            // バッファ破壊確認のためデータをFillしておきます。
            std::memset(ptrs[pageIndex], (pageIndex & 0xff), size);
        }
        buf.CheckValid();

        // バッファが破損していないか確認します。
        for( auto pageIndex = 0; pageIndex < pageCount; ++pageIndex )
        {
            const char truth = pageIndex & 0xff;
            const auto size = sizes[pageIndex];
            // 先頭と末尾を確認すれば必要十分
            ASSERT_EQ(truth, ptrs[pageIndex][0]);
            ASSERT_EQ(truth, ptrs[pageIndex][size - 1]);
        }
        buf.CheckValid();

        // 事前に生成したランダムな順番でデータを解放します。
        for( const auto idx : shuffledIdx )
        {
            allocator.Free(ptrs[idx], sizes[idx]);
            ptrs[idx] = nullptr;
        }
        buf.CheckValid();
    }
}

//! メモリを最大まで確保して、ランダムな順番で解放します。
void AllocateMaxRandom(
         nn::fssystem::FileSystemBuddyHeap* pHeap,
         size_t heapSize,
         size_t blockSize,
         const nnt::fs::util::SafeMemoryStorage& buf,
         std::mt19937* pMt,
         int loopCountMax
     ) NN_NOEXCEPT
{
    ASSERT_NE(pHeap, nullptr);

    const auto totalFreeSize = pHeap->GetTotalFreeSize();

    BuddyHeapAllocator allocator(pHeap);

    const auto pageCount = static_cast<int>(heapSize / blockSize);

    nnt::fs::util::Vector<char*> ptrs(pageCount);

    // ソートされた順番の配列とランダムな順番の配列を用意します。
    // （ソートされた配列は、毎回 iota を呼ばないために保持します。）
    nnt::fs::util::Vector<int> sortedIdx(pageCount);
    nnt::fs::util::Vector<int> shuffledIdx(pageCount);
    std::iota(sortedIdx.begin(), sortedIdx.end(), 0);

    for( auto loopCount = 0; loopCount < loopCountMax; ++loopCount )
    {
        ASSERT_EQ(totalFreeSize, pHeap->GetTotalFreeSize());

        // 解放順番がランダムになるように適当に入れ替えます。
        shuffledIdx = sortedIdx;
        std::shuffle(shuffledIdx.begin(), shuffledIdx.end(), *pMt);

        // 順番にページを確保します。
        for( auto pageIndex = 0; pageIndex < pageCount; ++pageIndex )
        {
            ptrs[pageIndex] = static_cast<char*>(allocator.Allocate(blockSize));
            ASSERT_NE(nullptr, ptrs[pageIndex]);

            // バッファ破壊確認のためデータをFillしておきます。
            std::memset(ptrs[pageIndex], (pageIndex & 0xff), blockSize);
        }
        buf.CheckValid();

        // これ以上は確保できません。
        const auto ptr = static_cast<char*>(allocator.Allocate(blockSize));
        ASSERT_EQ(nullptr, ptr);
        buf.CheckValid();

        // バッファが破損していないか確認します。
        for( auto pageIndex = 0; pageIndex < pageCount; ++pageIndex )
        {
            const char truth = (pageIndex & 0xff);
            // 先頭と末尾を確認すれば必要十分
            ASSERT_EQ(truth, ptrs[pageIndex][0]);
            ASSERT_EQ(truth, ptrs[pageIndex][blockSize - 1]);
        }
        buf.CheckValid();

        // 事前に生成したランダムな順番でデータを解放します。
        for( const auto idx : shuffledIdx )
        {
            allocator.Free(ptrs[idx], blockSize);
            ptrs[idx] = nullptr;
        }
        buf.CheckValid();
    }
}

//! 順番に確保、ランダムに解放するテストです。
TEST(FileSystemBuddyHeapTest, Random)
{
    typedef nn::fssystem::FileSystemBuddyHeap BuddyHeap;

    std::mt19937 mt(1);

    static const auto LoopCountMax = 128;
    static const auto InnerLoopCountMax = 8;

    BuddyHeap heap;

    for( auto loopCount = 0; loopCount < LoopCountMax; ++loopCount )
    {
        NN_LOG(".");

        // ブロックサイズをランダムに決定します。
        const auto blockSizeExponent = std::uniform_int_distribution<>(8, 12)(mt);
        const auto blockSize = sizeof(char) << blockSizeExponent;

        // ページサイズをランダムに決定します。（2^18 を超えない範囲）
        const auto orderMax
            = std::uniform_int_distribution<>(4, 18 - static_cast<int>(blockSizeExponent))(mt);
        const auto pageSizeMax = (static_cast<size_t>(1) << orderMax) * blockSize;

        // バディヒープのサイズをランダムに決定します。（最大ページサイズ以上になるように）
        const auto blockCountMax = std::max(1, static_cast<int>(pageSizeMax / 16));
        const auto maxPageCountMax
            = static_cast<int>((blockCountMax * blockSize + (pageSizeMax - 1)) / pageSizeMax);
        const auto buddySize
            = pageSizeMax * (std::uniform_int_distribution<>(1, maxPageCountMax - 1)(mt))
                + std::uniform_int_distribution<size_t>(0, pageSizeMax - 1)(mt);

        // バッファを確保します。（アライメント分を余分に確保）
        nnt::fs::util::SafeMemoryStorage buf(buddySize + BuddyHeap::BufferAlignment);
        const uintptr_t addr = reinterpret_cast<uintptr_t>(buf.GetBuffer());

        // バディヒープを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(
            heap.Initialize(
                nn::util::align_up(addr, BuddyHeap::BufferAlignment),
                buddySize,
                blockSize,
                orderMax
            )
        );
        buf.CheckValid();

        // 初期化直後の空き容量を記憶しておき、確保と解放後に元に戻るかチェックします。
        const auto totalFreeSize = heap.GetTotalFreeSize();

        // 適当なサイズで確保と解放をランダムに繰り返します。
        AllocateRandom(&heap, buddySize, buf, &mt, InnerLoopCountMax);

        // 計算上の最大ページ数まで確保を試行し、ランダムに解放します。
        // ここまでの処理でフリーリストがクリアな状態になっていないと失敗します。
        AllocateMaxRandom(&heap, buddySize, blockSize, buf, &mt, InnerLoopCountMax);

        // 解放されて空き容量が初期化直後と同じ量になっているかチェックします。
        ASSERT_EQ(totalFreeSize, heap.GetTotalFreeSize());

        heap.Finalize();
        buf.CheckValid();
    }

    NN_LOG("\n");
}

//! 最大ページサイズがヒープサイズより大きい時のページ確保失敗のテスト
TEST(FileSystemBuddyHeapTest, Failure)
{
    typedef nn::fssystem::FileSystemBuddyHeap BuddyHeap;

    std::mt19937 mt(1);

    static const auto LoopCountMax = 32;

    BuddyHeap heap;

    for( auto loopCount = 0; loopCount < LoopCountMax; ++loopCount )
    {
        NN_LOG(".");

        // ブロックサイズをランダムに決定します。
        const auto blockSizeExponent = std::uniform_int_distribution<>(8, 12)(mt);
        const auto blockSize = sizeof(char) << blockSizeExponent;

        // ページサイズをランダムに決定します。（2^18 を超えない範囲）
        const auto orderMax
            = std::uniform_int_distribution<>(4, 18 - static_cast<int>(blockSizeExponent))(mt);

        // バディヒープのサイズをランダムに決定します。（最大ページサイズ未満になるように）
        const auto pageSizeMax = (static_cast<size_t>(1) << orderMax) * blockSize;
        const auto buddySize
            = std::uniform_int_distribution<size_t>(pageSizeMax / 2, pageSizeMax - 1)(mt);

        // バッファを確保します。（アライメント分を余分に確保）
        nnt::fs::util::SafeMemoryStorage buf(buddySize + BuddyHeap::BufferAlignment);
        const uintptr_t addr = reinterpret_cast<uintptr_t>(buf.GetBuffer());

        // バディヒープを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(
            heap.Initialize(
                nn::util::align_up(addr, BuddyHeap::BufferAlignment),
                buddySize,
                blockSize,
                orderMax
            )
        );
        buf.CheckValid();

        // ヒープサイズがページサイズ以下になるオーダーを指定すれば確保に成功する
        for( auto order = 0; order < orderMax; ++order )
        {
            const auto ptr = heap.AllocateByOrder(order);
            ASSERT_NE(nullptr, ptr);
            heap.Free(ptr, order);
        }

        // ヒープサイズよりページサイズが大きくなるオーダーを指定すると確保に失敗する
        const auto ptr = heap.AllocateByOrder(orderMax);
        ASSERT_EQ(nullptr, ptr);

        heap.Finalize();
        buf.CheckValid();
    }

    NN_LOG("\n");
}

//! オーダーからブロック数を求める事前条件違反のテストです。
TEST(FileSystemBuddyHeapDeathTest, PreconditionGetBlockCountFromOrder)
{
    typedef nn::fssystem::FileSystemBuddyHeap BuddyHeap;

    // オーダーが負の値
    EXPECT_DEATH_IF_SUPPORTED(BuddyHeap::GetBlockCountFromOrder(-1), "");

    // オーダーが上限を超えている
    EXPECT_DEATH_IF_SUPPORTED(
        BuddyHeap::GetBlockCountFromOrder(BuddyHeap::OrderUpperLimit),
        ""
    );

    // 成功する呼び出し（最小値、最大値）
    BuddyHeap::GetBlockCountFromOrder(0);
    BuddyHeap::GetBlockCountFromOrder(BuddyHeap::OrderUpperLimit - 1);
}

//! 割当可能な最大オーダー取得の事前条件違反のテストです。
TEST(FileSystemBuddyHeapDeathTest, PreconditionQueryOrderMax)
{
    typedef nn::fssystem::FileSystemBuddyHeap BuddyHeap;

    for( auto bufferSizeExponent = 14; bufferSizeExponent <= 18; bufferSizeExponent += 2 )
    {
        for( auto blockSizeExponent = bufferSizeExponent / 2;
            blockSizeExponent <= (bufferSizeExponent + 4) / 2;
            blockSizeExponent += 2 )
        {
            const auto bufferSize = sizeof(char) << bufferSizeExponent;
            const auto blockSize = sizeof(char) << blockSizeExponent;

            // ブロックサイズが大きすぎる
            EXPECT_DEATH_IF_SUPPORTED(BuddyHeap::QueryOrderMax(bufferSize, bufferSize + 1), "");

            // ブロックサイズが 2 の累乗数ではない
            EXPECT_DEATH_IF_SUPPORTED(BuddyHeap::QueryOrderMax(bufferSize, blockSize - 1), "");
            EXPECT_DEATH_IF_SUPPORTED(BuddyHeap::QueryOrderMax(bufferSize, blockSize + 1), "");

            // ブロックサイズが小さすぎる
            EXPECT_DEATH_IF_SUPPORTED(
                BuddyHeap::QueryOrderMax(BuddyHeap::BlockSizeMin / 2, BuddyHeap::BlockSizeMin / 2),
                ""
            );

            // 成功する呼び出し
            BuddyHeap::QueryOrderMax(bufferSize, bufferSize);
        }
    }
}

//! 未初期化時の事前条件違反のテストです。
TEST(FileSystemBuddyHeapDeathTest, PreconditionUninitialized)
{
    nn::fssystem::FileSystemBuddyHeap heap;

    // 初期化が必要なメンバ関数の未初期化時の初呼び出し
    EXPECT_DEATH_IF_SUPPORTED(heap.Finalize(), "");
    EXPECT_DEATH_IF_SUPPORTED(heap.AllocateByOrder(0), "");
    EXPECT_DEATH_IF_SUPPORTED(heap.Free(nullptr, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(heap.GetOrderFromBytes(0), "");
    EXPECT_DEATH_IF_SUPPORTED(heap.GetBytesFromOrder(0), "");
    EXPECT_DEATH_IF_SUPPORTED(heap.GetOrderMax(), "");
    EXPECT_DEATH_IF_SUPPORTED(heap.GetBlockSize(), "");
    EXPECT_DEATH_IF_SUPPORTED(heap.GetPageBlockCountMax(), "");
    EXPECT_DEATH_IF_SUPPORTED(heap.GetPageSizeMax(), "");
    EXPECT_DEATH_IF_SUPPORTED(heap.GetTotalFreeSize(), "");
    EXPECT_DEATH_IF_SUPPORTED(heap.GetAllocatableSizeMax(), "");
    EXPECT_DEATH_IF_SUPPORTED(heap.Dump(), "");
}

//! 初期化の事前条件違反のテストです。
TEST(FileSystemBuddyHeapDeathTest, PreconditionInitialize)
{
    typedef nn::fssystem::FileSystemBuddyHeap BuddyHeap;

    for( auto bufferSizeExponent = 14; bufferSizeExponent <= 18; bufferSizeExponent += 2 )
    {
        for( auto blockSizeExponent = bufferSizeExponent / 2;
             blockSizeExponent <= (bufferSizeExponent + 4) / 2;
             blockSizeExponent += 2 )
        {
            for( auto orderMax = 1; orderMax <= 16; orderMax *= 4 )
            {
                const auto bufferSize = sizeof(char) << bufferSizeExponent;
                const auto blockSize = sizeof(char) << blockSizeExponent;

                nnt::fs::util::Vector<char> buffer(bufferSize);
                const auto bufferData = reinterpret_cast<uintptr_t>(buffer.data());
                nn::fssystem::FileSystemBuddyHeap heap;

                // バッファのアドレスが 0
                EXPECT_DEATH_IF_SUPPORTED(
                    heap.Initialize(0, bufferSize, blockSize, orderMax),
                    ""
                );

                // バッファアドレスが不正
                EXPECT_DEATH_IF_SUPPORTED(
                    heap.Initialize(bufferData - 1, bufferSize, blockSize, orderMax),
                    ""
                );
                EXPECT_DEATH_IF_SUPPORTED(
                    heap.Initialize(bufferData + 1, bufferSize, blockSize, orderMax),
                    ""
                );

                // ブロックサイズが大きすぎる
                EXPECT_DEATH_IF_SUPPORTED(
                    heap.Initialize(bufferData, bufferSize, bufferSize + 1, orderMax),
                    ""
                );

                // ブロックサイズが 2 の累乗数ではない
                EXPECT_DEATH_IF_SUPPORTED(
                    heap.Initialize(bufferData, bufferSize, blockSize - 1, orderMax),
                    ""
                );
                EXPECT_DEATH_IF_SUPPORTED(
                    heap.Initialize(bufferData, bufferSize, blockSize + 1, orderMax),
                    ""
                );

                // 最大ページ数オーダーが負の値
                EXPECT_DEATH_IF_SUPPORTED(
                    heap.Initialize(bufferData, bufferSize, blockSize, -1),
                    ""
                );

                // 最大ページ数オーダーが上限を超えている
                EXPECT_DEATH_IF_SUPPORTED(
                    heap.Initialize(bufferData, bufferSize, blockSize, BuddyHeap::OrderUpperLimit),
                    ""
                );

                // 成功する呼び出し
                NNT_ASSERT_RESULT_SUCCESS(
                    heap.Initialize(bufferData, bufferSize, blockSize, orderMax)
                );

                heap.Finalize();
            }
        }
    }
}

//! ページサイズ計算の事前条件違反のテストです。
TEST(FileSystemBuddyHeapDeathTest, PreconditionGetBytesFromOrder)
{
    for( auto bufferSizeExponent = 14; bufferSizeExponent <= 18; bufferSizeExponent += 2 )
    {
        for( auto blockSizeExponent = bufferSizeExponent / 2;
             blockSizeExponent <= (bufferSizeExponent + 4) / 2;
             blockSizeExponent += 2 )
        {
            for( auto orderMax = 1; orderMax <= 16; orderMax *= 4 )
            {
                const auto bufferSize = sizeof(char) << bufferSizeExponent;
                const auto blockSize = sizeof(char) << blockSizeExponent;

                nnt::fs::util::Vector<char> buffer(bufferSize);
                const auto bufferData = reinterpret_cast<uintptr_t>(buffer.data());
                nn::fssystem::FileSystemBuddyHeap heap;

                NNT_ASSERT_RESULT_SUCCESS(
                    heap.Initialize(bufferData, bufferSize, blockSize, orderMax)
                );

                // オーダーが負の値
                EXPECT_DEATH_IF_SUPPORTED(heap.GetBytesFromOrder(-1), "");

                // オーダーが上限を超えている
                EXPECT_DEATH_IF_SUPPORTED(heap.GetBytesFromOrder(heap.GetOrderMax() + 1), "");

                // 成功する呼び出し（最小値）
                heap.GetOrderFromBytes(0);

                heap.Finalize();
            }
        }
    }
}

//! ページの確保と解放の事前条件違反のテストです。
TEST(FileSystemBuddyHeapDeathTest, PreconditionAllocate)
{
    for( auto bufferSizeExponent = 14; bufferSizeExponent <= 18; bufferSizeExponent += 2 )
    {
        for( auto blockSizeExponent = bufferSizeExponent / 2;
             blockSizeExponent <= (bufferSizeExponent + 4) / 2;
             blockSizeExponent += 2 )
        {
            for( auto orderMax = 1; orderMax <= 16; orderMax *= 4 )
            {
                const auto bufferSize = sizeof(char) << bufferSizeExponent;
                const auto blockSize = sizeof(char) << blockSizeExponent;

                nnt::fs::util::Vector<char> buffer(bufferSize);
                const auto bufferData = reinterpret_cast<uintptr_t>(buffer.data());
                nn::fssystem::FileSystemBuddyHeap heap;

                NNT_ASSERT_RESULT_SUCCESS(
                    heap.Initialize(bufferData, bufferSize, blockSize, orderMax)
                );

                // ページ確保
                // オーダーが負の値
                EXPECT_DEATH_IF_SUPPORTED(heap.AllocateByOrder(-1), "");
                // オーダーが上限を超えている
                EXPECT_DEATH_IF_SUPPORTED(heap.AllocateByOrder(heap.GetOrderMax() + 1), "");

                // 成功する呼び出し
                const auto order = std::min(bufferSizeExponent - blockSizeExponent, orderMax);
                const auto ptr = heap.AllocateByOrder(order);
                ASSERT_NE(nullptr, ptr);

                // ページ解放
                // オーダーが負の値
                ASSERT_DEATH_IF_SUPPORTED(heap.Free(ptr, -1), "");
                // オーダーが上限を超えている
                ASSERT_DEATH_IF_SUPPORTED(heap.Free(ptr, heap.GetOrderMax() + 1), "");

                // 成功する呼び出し
                heap.Free(ptr, order);

                // 二重に解放
                EXPECT_DEATH_IF_SUPPORTED(heap.Free(ptr, order), "");

                heap.Finalize();
            }
        }
    }
}

