﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>

#include <memory>
#include <nnt/nntest.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/util/util_BitUtil.h>

#include <nnt/base/testBase_Exit.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/nnt_Argument.h>

#include <nn/crypto.h>
#include <nn/fat/fat_FatFileSystem.h>
#include <nn/fs/fsa/fs_IDirectory.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fssystem/fs_AesXtsFile.h>
#include <nn/fssystem/fs_AesXtsFileSystem.h>
#include <nn/fssystem/fs_AsynchronousAccess.h>
#include <nn/fssystem/fs_PathOnExecutionDirectory.h>
#include <nn/fssystem/fs_ConcatenationFileSystem.h>

#include "detail/fssrv_DeviceBuffer.h"
#include "shim/fs_Library.h"

namespace nnt { namespace fs { namespace detail {

    nn::fs::fsa::IFile* GetBaseFile(nn::fssystem::AesXtsFile* pAesXtsFile) NN_NOEXCEPT
    {
        return pAesXtsFile->m_pBaseFile.get();
    }

}}}

namespace {

void GenerateRandomForAesXtsFileSystemForWin(void* pData, size_t size) NN_NOEXCEPT
{
    const auto MacSize = nn::crypto::HmacSha256Generator::MacSize;
    NN_ABORT_UNLESS_LESS_EQUAL(size, MacSize);

    static auto s_IsInitialized = false;
    static auto s_Now = nn::os::GetSystemTick().GetInt64Value();
    static const char s_Key[] = "AesXtsFileSystem";
    if (s_IsInitialized)
    {
        ++s_Now;
    }
    else
    {
        s_IsInitialized = true;
    }
    char data[nn::crypto::HmacSha256Generator::MacSize];
    nn::crypto::GenerateHmacSha256Mac(
        &data, sizeof(data),
        &s_Now, sizeof(s_Now),
        s_Key, sizeof(s_Key)
    );
    std::memcpy(pData, data, size);
}

class MockFile : public nn::fs::fsa::IFile, public nn::fs::detail::Newable
{
public:
    MockFile(std::unique_ptr<nn::fs::fsa::IFile>&& pBaseFile, bool isWriteFailure, bool isGetSizeFailure) NN_NOEXCEPT
        : m_pBaseFile(std::move(pBaseFile)),
          m_IsWriteFailure(isWriteFailure),
          m_IsGetSizeFailure(isGetSizeFailure)
    {
    }

private:
    virtual nn::Result DoRead(size_t* outValue, int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result DoWrite(int64_t offset, const void* buffer, size_t size, const nn::fs::WriteOption& option) NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result DoFlush() NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result DoSetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result DoGetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result DoOperateRange(
        void* outBuffer,
        size_t outBufferSize,
        nn::fs::OperationId operationID,
        int64_t offset,
        int64_t size,
        const void* inBuffer,
        size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE;

private:
    std::unique_ptr<nn::fs::fsa::IFile> m_pBaseFile;
    bool m_IsWriteFailure;
    bool m_IsGetSizeFailure;
};

nn::Result MockFile::DoRead(size_t* outValue, int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT
{
    return m_pBaseFile->Read(outValue, offset, buffer, size, option);
}

nn::Result MockFile::DoWrite(int64_t offset, const void* buffer, size_t size, const nn::fs::WriteOption& option) NN_NOEXCEPT
{
    if (m_IsWriteFailure)
    {
        return nn::fs::ResultInternal();
    }
    return m_pBaseFile->Write(offset, buffer, size, option);
}

nn::Result MockFile::DoFlush() NN_NOEXCEPT
{
    return m_pBaseFile->Flush();
}

nn::Result MockFile::DoSetSize(int64_t size) NN_NOEXCEPT
{
    return m_pBaseFile->SetSize(size);
}

nn::Result MockFile::DoGetSize(int64_t* outValue) NN_NOEXCEPT
{
    if (m_IsGetSizeFailure)
    {
        return nn::fs::ResultInternal();
    }
    return m_pBaseFile->GetSize(outValue);
}

nn::Result MockFile::DoOperateRange(
    void*, size_t, nn::fs::OperationId, int64_t, int64_t, const void*, size_t) NN_NOEXCEPT
{
    return nn::fs::ResultUnsupportedOperation();
}

class MockFileSystem : public nn::fs::fsa::IFileSystem, public nn::fs::detail::Newable
{
public:
    explicit MockFileSystem(nn::fs::fsa::IFileSystem* pBaseFileSystem) NN_NOEXCEPT
        : m_pBaseFileSystem(pBaseFileSystem)
    {
        Reset();
    }

public:
    void Reset() NN_NOEXCEPT
    {
        m_WriteFileFailureName[0] = '\0';
        m_IsGetFileSizeFailure = false;
    }

    void SetWriteFileFailureName(const char* name) NN_NOEXCEPT
    {
        std::strncpy(m_WriteFileFailureName, name, nn::fs::EntryNameLengthMax);
        m_WriteFileFailureName[nn::fs::EntryNameLengthMax] = '\0';
    }

    void SetGetFileSizeFailure(bool isFailure) NN_NOEXCEPT
    {
        m_IsGetFileSizeFailure = isFailure;
    }

private:
    virtual nn::Result DoCreateFile(const char* path, int64_t size, int option) NN_NOEXCEPT;
    virtual nn::Result DoDeleteFile(const char* path) NN_NOEXCEPT;
    virtual nn::Result DoCreateDirectory(const char* path) NN_NOEXCEPT;
    virtual nn::Result DoDeleteDirectory(const char* path) NN_NOEXCEPT;
    virtual nn::Result DoDeleteDirectoryRecursively(const char* path) NN_NOEXCEPT;
    virtual nn::Result DoCleanDirectoryRecursively(const char* path) NN_NOEXCEPT;
    virtual nn::Result DoRenameFile(const char* currentPath, const char* newPath) NN_NOEXCEPT;
    virtual nn::Result DoRenameDirectory(const char* currentPath, const char* newPath) NN_NOEXCEPT;
    virtual nn::Result DoGetEntryType(nn::fs::DirectoryEntryType* outValue, const char* path) NN_NOEXCEPT;
    virtual nn::Result DoOpenFile(std::unique_ptr<nn::fs::fsa::IFile>* outValue, const char* path, nn::fs::OpenMode mode) NN_NOEXCEPT;
    virtual nn::Result DoOpenDirectory(std::unique_ptr<nn::fs::fsa::IDirectory>* outValue, const char* path, nn::fs::OpenDirectoryMode mode) NN_NOEXCEPT;
    virtual nn::Result DoCommit() NN_NOEXCEPT;

private:
    nn::fs::fsa::IFileSystem* m_pBaseFileSystem;
    char m_WriteFileFailureName[nn::fs::EntryNameLengthMax + 1];
    bool m_IsGetFileSizeFailure;
};

nn::Result MockFileSystem::DoCreateFile(const char* path, int64_t size, int option) NN_NOEXCEPT
{
    return m_pBaseFileSystem->CreateFile(path, size, option);
}

nn::Result MockFileSystem::DoDeleteFile(const char* path) NN_NOEXCEPT
{
    return m_pBaseFileSystem->DeleteFile(path);
}

nn::Result MockFileSystem::DoCreateDirectory(const char* path) NN_NOEXCEPT
{
    return m_pBaseFileSystem->CreateDirectory(path);
}

nn::Result MockFileSystem::DoDeleteDirectory(const char* path) NN_NOEXCEPT
{
    return m_pBaseFileSystem->DeleteDirectory(path);
}

nn::Result MockFileSystem::DoDeleteDirectoryRecursively(const char* path) NN_NOEXCEPT
{
    return m_pBaseFileSystem->DeleteDirectoryRecursively(path);
}

nn::Result MockFileSystem::DoCleanDirectoryRecursively(const char* path) NN_NOEXCEPT
{
    return m_pBaseFileSystem->CleanDirectoryRecursively(path);
}

nn::Result MockFileSystem::DoRenameFile(const char* currentPath, const char* newPath) NN_NOEXCEPT
{
    return m_pBaseFileSystem->RenameFile(currentPath, newPath);
}

nn::Result MockFileSystem::DoRenameDirectory(const char* currentPath, const char* newPath) NN_NOEXCEPT
{
    return m_pBaseFileSystem->RenameDirectory(currentPath, newPath);
}

nn::Result MockFileSystem::DoGetEntryType(nn::fs::DirectoryEntryType* outValue, const char* path) NN_NOEXCEPT
{
    return m_pBaseFileSystem->GetEntryType(outValue, path);
}

nn::Result MockFileSystem::DoOpenFile(std::unique_ptr<nn::fs::fsa::IFile>* outValue, const char* path, nn::fs::OpenMode mode) NN_NOEXCEPT
{
    if ((mode & nn::fs::OpenMode::OpenMode_Write) != 0)
    {
        if (std::strcmp(m_WriteFileFailureName, path) == 0)
        {
            std::unique_ptr<nn::fs::fsa::IFile> pValue;
            NN_RESULT_DO(m_pBaseFileSystem->OpenFile(&pValue, path, mode));
            outValue->reset(new MockFile(std::move(pValue), true, false));
            NN_RESULT_SUCCESS;
        }
        else if (m_IsGetFileSizeFailure)
        {
            std::unique_ptr<nn::fs::fsa::IFile> pValue;
            NN_RESULT_DO(m_pBaseFileSystem->OpenFile(&pValue, path, mode));
            outValue->reset(new MockFile(std::move(pValue), false, true));
            NN_RESULT_SUCCESS;
        }
    }

    // デフォルト
    return m_pBaseFileSystem->OpenFile(outValue, path, mode);
}

nn::Result MockFileSystem::DoOpenDirectory(std::unique_ptr<nn::fs::fsa::IDirectory>* outValue, const char* path, nn::fs::OpenDirectoryMode mode) NN_NOEXCEPT
{
    return m_pBaseFileSystem->OpenDirectory(outValue, path, mode);
}

nn::Result MockFileSystem::DoCommit() NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

class AesXtsFileSystemTestBase
{
public:
    AesXtsFileSystemTestBase()
        : m_pFileSystem()
        , m_pOriginalFileSystem()
        , m_FatCacheBuffer()
        , m_MemoryStorageBuffer()
        , m_MemoryStorage()
    {
    }

    virtual ~AesXtsFileSystemTestBase()
    {
        m_pFileSystem.reset();
        m_pMockFileSystem.reset();
        m_pOriginalFileSystem.reset();

        m_FatCacheBuffer.reset();
        m_MemoryStorage.reset();
        m_MemoryStorageBuffer.reset();
    }

protected:

    void SetUpImpl(int xtsBlockSize) NN_NOEXCEPT
    {
        // 元となる FatFileSystem
        {
            static const size_t storageSize = 512 * 1024;
            m_MemoryStorageBuffer.reset(new char[storageSize]);

            m_MemoryStorage.reset(new nn::fs::MemoryStorage(m_MemoryStorageBuffer.get(), storageSize));
            ASSERT_NE(nullptr, m_MemoryStorage.get());

            size_t cacheBufferSize = nn::fat::FatFileSystem::GetCacheBufferSize();
            m_FatCacheBuffer.reset(new char[cacheBufferSize]);
            ASSERT_NE(nullptr, m_FatCacheBuffer.get());

            memset(m_MemoryStorageBuffer.get(), 0xCD, storageSize);
            std::unique_ptr<nn::fat::FatFileSystem> fatFs(new nn::fat::FatFileSystem());
            ASSERT_NE(nullptr, fatFs.get());

            NNT_ASSERT_RESULT_SUCCESS(fatFs->Initialize(m_MemoryStorage.get(), m_FatCacheBuffer.get(),
                                                        cacheBufferSize));
            NNT_ASSERT_RESULT_SUCCESS(fatFs->Format());
            NNT_ASSERT_RESULT_SUCCESS(fatFs->Mount());
            m_pOriginalFileSystem = std::move(fatFs);
        }

        // Mock
        {
            m_pMockFileSystem.reset(new MockFileSystem(m_pOriginalFileSystem.get()));
            ASSERT_NE(nullptr, m_pMockFileSystem.get());
        }

        // AesXtsFileSystem の生成
        {
            // 決め打ちの適当な鍵を使う
            static const char encryptionKeyGenerationKey[nn::fssystem::AesXtsFileSystem::EncryptionKeyGenerationKeySize] =
                { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15 };
            static const char macKey[nn::fssystem::AesXtsFileSystem::MacKeySize] =
                { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15 };

            auto pAesXtsFileSystem = new nn::fssystem::AesXtsFileSystem();
            pAesXtsFileSystem->Initialize(
                m_pMockFileSystem,
                encryptionKeyGenerationKey, sizeof(encryptionKeyGenerationKey),
                macKey, sizeof(macKey),
                GenerateRandomForAesXtsFileSystemForWin,
                xtsBlockSize);

            m_pFileSystem.reset(pAesXtsFileSystem);
        }
    }

    void TearDownImpl() NN_NOEXCEPT
    {
        reinterpret_cast<MockFileSystem*>(m_pMockFileSystem.get())->Reset();
    }

protected:
    nn::fs::fsa::IFileSystem* GetFileSystem() NN_NOEXCEPT
    {
        return m_pFileSystem.get();
    }

    MockFileSystem* GetMockFileSystem() NN_NOEXCEPT
    {
        return m_pMockFileSystem.get();
    }

    nn::fs::fsa::IFileSystem* GetOriginalFileSystem() NN_NOEXCEPT
    {
        return m_pOriginalFileSystem.get();
    }

private:
    std::unique_ptr<nn::fs::fsa::IFileSystem> m_pFileSystem;
    std::shared_ptr<MockFileSystem> m_pMockFileSystem;
    std::unique_ptr<nn::fs::fsa::IFileSystem> m_pOriginalFileSystem;

    std::unique_ptr<char[]> m_FatCacheBuffer;
    std::unique_ptr<char[]> m_MemoryStorageBuffer;
    std::unique_ptr<nn::fs::IStorage> m_MemoryStorage;
};

class AesXtsFileSystemTest : public AesXtsFileSystemTestBase, public ::testing::Test
{
protected:
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        SetUpImpl(nn::fssystem::AesXtsFileSystem::AesBlockSize);
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        TearDownImpl();
    }
};

TEST_F(AesXtsFileSystemTest, OpenReadDirectoryWithNotRequireFileSize)
{
    NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->CreateFile("/aesfile.txt", 1));

    std::unique_ptr<nn::fs::fsa::IDirectory> directory;
    NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->OpenDirectory(&directory, "/", static_cast<nn::fs::OpenDirectoryMode>(nn::fs::OpenDirectoryMode_All | nn::fs::OpenDirectoryModePrivate_NotRequireFileSize)));

    int64_t entryCount;
    std::unique_ptr<nn::fs::DirectoryEntry> entry(new nn::fs::DirectoryEntry);
    NNT_ASSERT_RESULT_SUCCESS(directory->Read(&entryCount, entry.get(), 1));
    EXPECT_EQ(0, entry->fileSize);
}

TEST_F(AesXtsFileSystemTest, OpenReadDirectoryWithNonEncryptedFile)
{
    NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->CreateDirectory("/test"));
    NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->CreateFile("/aesfile.txt", 1));
    NNT_ASSERT_RESULT_SUCCESS(GetOriginalFileSystem()->CreateFile("/test.txt", nn::fssystem::AesXtsFileHeader::Size));

    std::unique_ptr<nn::fs::fsa::IDirectory> directory;
    NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->OpenDirectory(&directory, "/", nn::fs::OpenDirectoryMode_All));

    int64_t entryCount;
    NNT_ASSERT_RESULT_SUCCESS(directory->GetEntryCount(&entryCount));
    EXPECT_EQ(3, entryCount);

    std::unique_ptr<nn::fs::DirectoryEntry[]> entries(new nn::fs::DirectoryEntry[static_cast<size_t>(entryCount)]);
    NNT_ASSERT_RESULT_SUCCESS(directory->Read(&entryCount, entries.get(), entryCount));
    EXPECT_EQ(3, entryCount);

    int count = 0;
    for (int i = 0; i < entryCount; ++i)
    {
        auto& entry = entries.get()[i];
        if (std::strcmp(entry.name, "aesfile.txt") == 0)
        {
            EXPECT_EQ(1, entry.fileSize);
            ++count;
        }
        else if (std::strcmp(entry.name, "test.txt") == 0)
        {
            EXPECT_EQ(0, entry.fileSize);
            ++count;
        }
        else if (std::strcmp(entry.name, "test") == 0)
        {
            ++count;
        }
    }
    EXPECT_EQ(3, count);
}

TEST_F(AesXtsFileSystemTest, OpenFileWithNonEncryptedFile)
{
    std::unique_ptr<nn::fs::fsa::IFile> pFile;
    NNT_ASSERT_RESULT_SUCCESS(GetOriginalFileSystem()->CreateFile("/test.txt", 0));
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultAesXtsFileSystemFileHeaderSizeCorruptedOnFileOpen, GetFileSystem()->OpenFile(&pFile, "/test.txt", nn::fs::OpenMode_Read));

    NNT_ASSERT_RESULT_SUCCESS(GetOriginalFileSystem()->CreateFile("/test2.txt", nn::fssystem::AesXtsFileHeader::Size));
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultAesXtsFileSystemFileNoHeaderOnFileOpen, GetFileSystem()->OpenFile(&pFile, "/test2.txt", nn::fs::OpenMode_Read));
}

TEST_F(AesXtsFileSystemTest, CreateFileWithNonEncryptedFile)
{
    NNT_ASSERT_RESULT_SUCCESS(GetOriginalFileSystem()->CreateFile("/test.txt", 0));
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultPathAlreadyExists, GetFileSystem()->CreateFile("/test.txt", 0));

    NNT_ASSERT_RESULT_SUCCESS(GetOriginalFileSystem()->CreateFile("/test2.txt", nn::fssystem::AesXtsFileHeader::Size));
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultPathAlreadyExists, GetFileSystem()->CreateFile("/test2.txt", 0));
}

TEST_F(AesXtsFileSystemTest, RenameFileWithNonEncryptedFile)
{
    NNT_ASSERT_RESULT_SUCCESS(GetOriginalFileSystem()->CreateFile("/test.txt", nn::fssystem::AesXtsFileHeader::Size));
    NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->RenameFile("/test.txt", "/test2.txt"));
}

TEST_F(AesXtsFileSystemTest, RenameDirectoryWithNonEncryptedFile)
{
    NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->CreateDirectory("/test"));
    NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->CreateFile("/test/aesfile.txt", 1));
    NNT_ASSERT_RESULT_SUCCESS(GetOriginalFileSystem()->CreateFile("/test/test.txt", nn::fssystem::AesXtsFileHeader::Size));
    NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->RenameDirectory("/test", "/test2"));
}

TEST_F(AesXtsFileSystemTest, GetEntryTypeWithNonEncryptedFile)
{
    nn::fs::DirectoryEntryType typeEntry;

    // サイズの足りないファイル
    NNT_ASSERT_RESULT_SUCCESS(GetOriginalFileSystem()->CreateFile("/test.txt", 0));
    NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->GetEntryType(&typeEntry, "/test.txt"));
    EXPECT_EQ(nn::fs::DirectoryEntryType_File, typeEntry);

    // サイズの足りているが暗号化されていないファイル
    NNT_ASSERT_RESULT_SUCCESS(GetOriginalFileSystem()->CreateFile("/test2.txt", nn::fssystem::AesXtsFileHeader::Size));
    NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->GetEntryType(&typeEntry, "/test2.txt"));
    EXPECT_EQ(nn::fs::DirectoryEntryType_File, typeEntry);

    // 検証エラーを返すファイル
    NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->CreateFile("/test3.txt", 0));
    NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->RenameFile("/test3.txt", "/test4.txt"));
    NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->GetEntryType(&typeEntry, "/test4.txt"));
    EXPECT_EQ(nn::fs::DirectoryEntryType_File, typeEntry);
}

TEST_F(AesXtsFileSystemTest, CreateFileWithWriteFailure)
{
    std::unique_ptr<nn::fs::fsa::IFile> pFile;

    GetMockFileSystem()->SetWriteFileFailureName("/test.txt");
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInternal, GetFileSystem()->CreateFile("/test.txt", nn::fssystem::AesXtsFileHeader::Size));
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultPathNotFound, GetFileSystem()->OpenFile(&pFile, "/test.txt", nn::fs::OpenMode_Read));
}

TEST_F(AesXtsFileSystemTest, RenameFileWithWriteFailure)
{
    std::unique_ptr<nn::fs::fsa::IFile> pFile;

    NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->CreateFile("/test.txt", nn::fssystem::AesXtsFileHeader::Size));
    GetMockFileSystem()->SetWriteFileFailureName("/test2.txt");
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInternal, GetFileSystem()->RenameFile("/test.txt", "/test2.txt"));
    NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->OpenFile(&pFile, "/test.txt", nn::fs::OpenMode_Read));
    pFile.reset();
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultPathNotFound, GetFileSystem()->OpenFile(&pFile, "/test2.txt", nn::fs::OpenMode_Read));
}

TEST_F(AesXtsFileSystemTest, RenameDirectoryWithWriteFailure)
{
    std::unique_ptr<nn::fs::fsa::IFile> pFile;

    NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->CreateDirectory("/test"));
    NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->CreateFile("/test/test1.txt", 0));
    NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->CreateFile("/test/test2.txt", 0));

    GetMockFileSystem()->SetWriteFileFailureName("/test2/test2.txt");
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInternal, GetFileSystem()->RenameDirectory("/test", "/test2"));

    NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->OpenFile(&pFile, "/test/test1.txt", nn::fs::OpenMode_Read));
    pFile.reset();
    NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->OpenFile(&pFile, "/test/test2.txt", nn::fs::OpenMode_Read));
    pFile.reset();

    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultPathNotFound, GetFileSystem()->OpenFile(&pFile, "/test2/test1..txt", nn::fs::OpenMode_Read));
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultPathNotFound, GetFileSystem()->OpenFile(&pFile, "/test2/test2.txt", nn::fs::OpenMode_Read));
}

TEST_F(AesXtsFileSystemTest, SetFileSizeWithWriteFailure)
{
    std::unique_ptr<nn::fs::fsa::IFile> pFile;

    NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->CreateFile("/test.txt", 5));
    NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->OpenFile(&pFile, "/test.txt", nn::fs::OpenMode_Read));
    GetMockFileSystem()->SetWriteFileFailureName("/test.txt");
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInternal, pFile->SetSize(0));
    int64_t fileSize;
    NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&fileSize));
    EXPECT_EQ(fileSize, 5);
}

TEST_F(AesXtsFileSystemTest, RenameFileOnBaseFileSystem)
{
    std::unique_ptr<nn::fs::fsa::IFile> pFile;

    NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->CreateFile("/test.txt", 0));
    NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->OpenFile(&pFile, "/test.txt", nn::fs::OpenMode_Read));
    pFile.reset();
    NNT_ASSERT_RESULT_SUCCESS(GetOriginalFileSystem()->RenameFile("/test.txt", "/test2.txt"));
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultAesXtsFileSystemFileHeaderCorruptedOnFileOpen, GetFileSystem()->OpenFile(&pFile, "/test2.txt", nn::fs::OpenMode_Read));
}

TEST_F(AesXtsFileSystemTest, RenameDirectoryOnBaseFileSystem)
{
    std::unique_ptr<nn::fs::fsa::IFile> pFile;

    NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->CreateDirectory("/test"));
    NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->CreateFile("/test/test.txt", 0));
    NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->OpenFile(&pFile, "/test/test.txt", nn::fs::OpenMode_Read));
    pFile.reset();
    NNT_ASSERT_RESULT_SUCCESS(GetOriginalFileSystem()->RenameDirectory("/test", "/test2"));
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultAesXtsFileSystemFileHeaderCorruptedOnFileOpen, GetFileSystem()->OpenFile(&pFile, "/test2/test.txt", nn::fs::OpenMode_Read));
}

TEST_F(AesXtsFileSystemTest, OpenFileWithGetSizeFailure)
{
    std::unique_ptr<nn::fs::fsa::IFile> pFile;

    NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->CreateFile("/test.txt", 0));
    GetMockFileSystem()->SetGetFileSizeFailure(true);
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInternal, GetFileSystem()->OpenFile(&pFile, "/test.txt", nn::fs::OpenMode_Write));
}

TEST_F(AesXtsFileSystemTest, SetFileSizeWithDataCorruption)
{
    std::unique_ptr<nn::fs::fsa::IFile> pFile;
    NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->CreateFile("/test.txt", 0));
    NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->OpenFile(&pFile, "/test.txt", nn::fs::OpenMode_Write));

    {
        auto pOriginalFile = nnt::fs::detail::GetBaseFile(static_cast<nn::fssystem::AesXtsFile*>(pFile.get()));
        char fillZeroData[64];
        std::memset(fillZeroData, 0, sizeof(fillZeroData));
        NNT_ASSERT_RESULT_SUCCESS(pOriginalFile->Write(0, fillZeroData, sizeof(fillZeroData), nn::fs::WriteOption::MakeValue(0)));
    }

    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultAesXtsFileSystemFileHeaderCorruptedOnFileSetSize, pFile->SetSize(0));
}

TEST_F(AesXtsFileSystemTest, SetFileSizeWithFileSizeCorruption)
{
    std::unique_ptr<nn::fs::fsa::IFile> pFile;
    NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->CreateFile("/test.txt", 0));

    NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->OpenFile(&pFile, "/test.txt", nn::fs::OpenMode_Write));

    {
        auto pOriginalFile = nnt::fs::detail::GetBaseFile(static_cast<nn::fssystem::AesXtsFile*>(pFile.get()));
        NNT_ASSERT_RESULT_SUCCESS(pOriginalFile->SetSize(0));
    }

    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultAesXtsFileSystemFileSizeCorruptedOnFileSetSize, pFile->SetSize(0));
}

TEST_F(AesXtsFileSystemTest, SetFileSizeOver)
{
    NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->CreateFile("/test.txt", 0));

    int64_t freeSpaceSize = 0;
    NNT_ASSERT_RESULT_SUCCESS(GetOriginalFileSystem()->GetFreeSpaceSize(&freeSpaceSize, "/"));

    {
        std::unique_ptr<nn::fs::fsa::IFile> pFile;
        NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->OpenFile(&pFile, "/test.txt", nn::fs::OpenMode_Write));

        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultUsableSpaceNotEnough, pFile->SetSize(freeSpaceSize + 1));
    }

    {
        std::unique_ptr<nn::fs::fsa::IFile> pFile;
        NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->OpenFile(&pFile, "/test.txt", nn::fs::OpenMode_Read));

        int64_t fileSize = 0;
        NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&fileSize));
        EXPECT_LE(fileSize, freeSpaceSize);
    }
}

TEST_F(AesXtsFileSystemTest, CheckBaseFileSizeOnCreteFileAndSetFileSize)
{
    // Create で 0、SetSize で 4
    {
        std::unique_ptr<nn::fs::fsa::IFile> pFile;
        NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->CreateFile("/test.txt", 0));
        NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->OpenFile(&pFile, "/test.txt", nn::fs::OpenMode_Write));

        {
            auto pOriginalFile = nnt::fs::detail::GetBaseFile(static_cast<nn::fssystem::AesXtsFile*>(pFile.get()));
            int64_t value = 0;
            NNT_ASSERT_RESULT_SUCCESS(pOriginalFile->GetSize(&value));
            EXPECT_EQ(value, nn::fssystem::AesXtsFileHeader::Size);
        }

        NNT_ASSERT_RESULT_SUCCESS(pFile->SetSize(4));

        {
            auto pOriginalFile = nnt::fs::detail::GetBaseFile(static_cast<nn::fssystem::AesXtsFile*>(pFile.get()));
            int64_t value = 0;
            NNT_ASSERT_RESULT_SUCCESS(pOriginalFile->GetSize(&value));
            EXPECT_EQ(value, nn::fssystem::AesXtsFileHeader::Size + nn::fssystem::AesXtsStorage::KeySize);
        }
    }

    // Create で 4、SetSize で 0
    {
        std::unique_ptr<nn::fs::fsa::IFile> pFile;
        NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->CreateFile("/test2.txt", 4));
        NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->OpenFile(&pFile, "/test2.txt", nn::fs::OpenMode_Write));

        {
            auto pOriginalFile = nnt::fs::detail::GetBaseFile(static_cast<nn::fssystem::AesXtsFile*>(pFile.get()));
            int64_t value = 0;
            NNT_ASSERT_RESULT_SUCCESS(pOriginalFile->GetSize(&value));
            EXPECT_EQ(value, nn::fssystem::AesXtsFileHeader::Size + nn::fssystem::AesXtsStorage::KeySize);
        }

        NNT_ASSERT_RESULT_SUCCESS(pFile->SetSize(0));

        {
            auto pOriginalFile = nnt::fs::detail::GetBaseFile(static_cast<nn::fssystem::AesXtsFile*>(pFile.get()));
            int64_t value = 0;
            NNT_ASSERT_RESULT_SUCCESS(pOriginalFile->GetSize(&value));
            EXPECT_EQ(value, nn::fssystem::AesXtsFileHeader::Size);
        }
    }
}

class VariousXtsBlockSizeTest : public AesXtsFileSystemTestBase, public testing::TestWithParam<int>
{
protected:
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        SetUpImpl(GetParam());
    }
};
INSTANTIATE_TEST_CASE_P(VariousXtsBlockSize, VariousXtsBlockSizeTest, testing::Values(16, 32, 64, 128, 256, 512, 1024));

TEST_P(VariousXtsBlockSizeTest, VariousXtsBlockSize)
{
    const char* path = "/aesxts.file";

    static const size_t SrcDataSize = 1024;
    uint8_t srcData[SrcDataSize] = {0};
    for( int i=0; i < SrcDataSize; ++i )
    {
        srcData[i] = i % 256;
    }

    // ファイル作成
    NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->CreateFile(path, /* size =*/0, /* option =*/0));

    // ファイルオープン
    std::unique_ptr<nn::fs::fsa::IFile> file;

    NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->OpenFile(
                                  &file, path,
                                  static_cast<nn::fs::OpenMode>(nn::fs::OpenMode::OpenMode_Read |
                                                                nn::fs::OpenMode::OpenMode_Write |
                                                                nn::fs::OpenMode::OpenMode_AllowAppend)
                                  ));

    // 読み書き確認
    nnt::fs::util::ConfirmWriteRead(file.get(), srcData, SrcDataSize);

    // ファイルクローズ
    file.reset();

    // ファイル削除
    NNT_ASSERT_RESULT_SUCCESS(GetFileSystem()->DeleteFile(path));

}

TEST(AesXtsFileSystemLargeTest, ReadWrite)
{
    nnt::fs::util::VirtualMemoryStorage memoryStorage(nnt::fs::util::LargeOffsetMax * 2);

    size_t cacheBufferSize = nn::fat::FatFileSystem::GetCacheBufferSize();
    auto fatCacheBuffer = nnt::fs::util::AllocateBuffer(cacheBufferSize);

    std::unique_ptr<nn::fat::FatFileSystem> pFatFs(new nn::fat::FatFileSystem());
    NNT_ASSERT_RESULT_SUCCESS(pFatFs->Initialize(&memoryStorage, fatCacheBuffer.get(), cacheBufferSize));
    NNT_ASSERT_RESULT_SUCCESS(pFatFs->Format());
    NNT_ASSERT_RESULT_SUCCESS(pFatFs->Mount());

    std::shared_ptr<nn::fs::fsa::IFileSystem> pBaseFileSystem(
        new nn::fssystem::ConcatenationFileSystem(std::move(pFatFs)));

    // 決め打ちの適当な鍵を使う
    static const char encryptionKeyGenerationKey[nn::fssystem::AesXtsFileSystem::EncryptionKeyGenerationKeySize] =
        { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15 };
    static const char macKey[nn::fssystem::AesXtsFileSystem::MacKeySize] =
        { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15 };

    // AesXtsFileSystem の生成
    nn::fssystem::AesXtsFileSystem aesXtsFileSystem;
    NNT_ASSERT_RESULT_SUCCESS(aesXtsFileSystem.Initialize(
        pBaseFileSystem,
        encryptionKeyGenerationKey, sizeof(encryptionKeyGenerationKey),
        macKey, sizeof(macKey),
        GenerateRandomForAesXtsFileSystemForWin,
        nn::fssystem::AesXtsFileSystem::AesBlockSize));

    nnt::fs::util::TestFileSystemAccessWithLargeOffset(&aesXtsFileSystem, "/file");
}

}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);

    // BufferPool 初期化
    static const size_t bufferPoolSize = nn::fssrv::detail::DeviceBufferSize;
    static NN_ALIGNAS(4096) char s_BufferPool[bufferPoolSize];
    nn::fssystem::InitializeBufferPool(s_BufferPool, bufferPoolSize);

    ::testing::InitGoogleTest(&argc, argv);
    int result = RUN_ALL_TESTS();

    nnt::Exit(result);
}
