﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nn/fssrv/fssrv_MemoryResourceFromStandardAllocator.h>
#include <nn/fssystem/fs_HostFileSystem.h>
#include <nn/fssystem/fs_DirectorySaveDataFileSystem.h>
#include <nn/fssystem/fs_ConcatenationFileSystem.h>
#include <nn/fat/fat_FatFileSystem.h>
#include <nn/fs/detail/fs_IFileDataCache.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/nnt_Argument.h>

#include "fsa/fs_FileSystemAccessor.h"

namespace nnt { namespace fs {

namespace {
    const auto BufferSize = 512 * 1024;
    char g_Buffer[BufferSize];
    nn::mem::StandardAllocator g_Allocator(g_Buffer, sizeof(g_Buffer));
    nn::fssrv::MemoryResourceFromStandardAllocator g_MemoryResource(&g_Allocator);
}

class FileSystemAccessorTest : public ::testing::Test
{
public:
    FileSystemAccessorTest() NN_NOEXCEPT
    {
        m_HostDirectory.Create();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountHostRoot());
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_FileSystem.Initialize(m_HostDirectory.GetPath().c_str()));
    }

    virtual ~FileSystemAccessorTest() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::fs::UnmountHostRoot();
        m_HostDirectory.Delete();
    }

protected:
    nn::fs::fsa::IFileSystem& GetFileSystem() NN_NOEXCEPT
    {
        return m_FileSystem;
    }

private:
    nn::fssystem::HostFileSystem m_FileSystem;
    nnt::fs::util::TemporaryHostDirectory m_HostDirectory;
};

typedef FileSystemAccessorTest FileSystemAccessorDeathTest;

TEST_F(FileSystemAccessorDeathTest, Commit)
{
    std::unique_ptr<nn::fssystem::DirectorySaveDataFileSystem> saveDataFileSystem(
        new nn::fssystem::DirectorySaveDataFileSystem(&GetFileSystem(), &g_MemoryResource));
    NNT_ASSERT_RESULT_SUCCESS(saveDataFileSystem->Initialize());

    nn::fs::detail::FileSystemAccessor fileSystem("save", std::move(saveDataFileSystem));
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile("/file", 1024, 0));
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateDirectory("/directory"));

    // OpenMode_Read でファイルを開いていてもコミットできる
    {
        std::unique_ptr<nn::fs::detail::FileAccessor> file;
        NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenFile(&file, "/file", nn::fs::OpenMode_Read));
        NNT_EXPECT_RESULT_SUCCESS(fileSystem.Commit());
    }

    // OpenMode_Write でファイルを開いているとコミットできない
    {
        std::unique_ptr<nn::fs::detail::FileAccessor> file;
        const auto data = 42;
        NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenFile(&file, "/file", nn::fs::OpenMode_Write));
        NNT_EXPECT_RESULT_SUCCESS(file->Write(
            0,
            &data,
            sizeof(data),
            nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
        EXPECT_DEATH_IF_SUPPORTED(fileSystem.Commit(), "");
    }

    // ディレクトリを開いていてもコミットできる
    {
        std::unique_ptr<nn::fs::detail::DirectoryAccessor> directory;
        NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenDirectory(
            &directory,
            "/directory",
            nn::fs::OpenDirectoryMode_All));
        NNT_EXPECT_RESULT_SUCCESS(fileSystem.Commit());
    }
}

// 4 GB を超えるオフセットでファイルアクセスするテスト
TEST(FileSystemAccessorLargeTest, ReadWrite)
{
    static const size_t AccessSize = 1024;
    static const int64_t FileSize = nnt::fs::util::LargeOffsetMax + AccessSize;
    static const char* FilePath = "/file";

    nnt::fs::util::VirtualMemoryStorage memoryStorage(nnt::fs::util::LargeOffsetMax * 2);

    size_t cacheBufferSize = nn::fat::FatFileSystem::GetCacheBufferSize();
    auto fatCacheBuffer = nnt::fs::util::AllocateBuffer(cacheBufferSize);

    std::unique_ptr<nn::fat::FatFileSystem> pFatFs(new nn::fat::FatFileSystem());
    NNT_ASSERT_RESULT_SUCCESS(pFatFs->Initialize(&memoryStorage, fatCacheBuffer.get(), cacheBufferSize));
    NNT_ASSERT_RESULT_SUCCESS(pFatFs->Format());
    NNT_ASSERT_RESULT_SUCCESS(pFatFs->Mount());

    std::unique_ptr<nn::fs::fsa::IFileSystem> pBaseFileSystem(new nn::fssystem::ConcatenationFileSystem(std::move(pFatFs)));

    nn::fs::detail::FileSystemAccessor fileSystem("test", std::move(pBaseFileSystem));

    NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile(FilePath, FileSize, nn::fs::CreateFileOptionFlag_BigFile));
    NN_UTIL_SCOPE_EXIT
    {
        NNT_ASSERT_RESULT_SUCCESS(fileSystem.DeleteFile(FilePath));
    };

    // バッファ初期化
    std::unique_ptr<char> readBuffer(new char[AccessSize]);
    std::unique_ptr<char> writeBufferList[nnt::fs::util::LargeOffsetListLength];
    for( auto& writeBuffer : writeBufferList )
    {
        writeBuffer.reset(new char[AccessSize]);
        nnt::fs::util::FillBufferWithRandomValue(writeBuffer.get(), AccessSize);
    }

    {
        // AllowAppend なしのテスト
        std::unique_ptr<nn::fs::detail::FileAccessor> pFile;
        NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenFile(
            &pFile,
            FilePath,
            static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write)));

        for( size_t i = 0; i < nnt::fs::util::LargeOffsetListLength; ++i )
        {
            NNT_ASSERT_RESULT_SUCCESS(pFile->Write(
                nnt::fs::util::LargeOffsetList[i],
                writeBufferList[i].get(),
                AccessSize,
                nn::fs::WriteOption()));
        }
        NNT_ASSERT_RESULT_SUCCESS(pFile->Flush());

        for( size_t i = 0; i < nnt::fs::util::LargeOffsetListLength; ++i )
        {
            size_t readSize = 0;
            NNT_ASSERT_RESULT_SUCCESS(pFile->Read(
                &readSize,
                nnt::fs::util::LargeOffsetList[i],
                readBuffer.get(),
                AccessSize,
                nn::fs::ReadOption()));
            ASSERT_EQ(AccessSize, readSize);
            NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBufferList[i].get(), readBuffer.get(), AccessSize);
        }

        // 範囲外への読み書きが失敗する
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultFileExtensionWithoutOpenModeAllowAppend,
            pFile->Write(FileSize, writeBufferList[0].get(), AccessSize, nn::fs::WriteOption()));

        size_t readSize = 0;
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultFileExtensionWithoutOpenModeAllowAppend,
            pFile->Read(&readSize, FileSize + 1, readBuffer.get(), AccessSize, nn::fs::ReadOption()));
        }
    {
        // AllowAppend ありのテスト
        std::unique_ptr<nn::fs::detail::FileAccessor> pFile;
        NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenFile(
            &pFile,
            FilePath,
            static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend)));

        // 範囲外への読み書き
        NNT_ASSERT_RESULT_SUCCESS(pFile->Write(FileSize, writeBufferList[0].get(), AccessSize, nn::fs::WriteOption()));
        NNT_ASSERT_RESULT_SUCCESS(pFile->Flush());

        size_t readSize = 0;
        NNT_ASSERT_RESULT_SUCCESS(pFile->Read(&readSize, FileSize, readBuffer.get(), AccessSize, nn::fs::ReadOption()));
        ASSERT_EQ(AccessSize, readSize);
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBufferList[0].get(), readBuffer.get(), AccessSize);

        // 拡張されたサイズをテスト
        int64_t size = 0;
        NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&size));
        EXPECT_GE(FileSize + static_cast<int64_t>(AccessSize), size);
    }
}

}}
