﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/fs.h>
#include <nn/fs/fs_GameCard.h>
#include <nn/fs/fs_IEventNotifier.h>
#include <nn/fs/fs_IStorage.h>

#include <nn/os.h>

using namespace nn;
using namespace nn::fs;

namespace {

// バッファのダンプ
void DumpBuffer(const void* buffer, size_t size)
{
    for(size_t i = 0; i < size; ++i)
    {
        if(i % 4   == 0)
        {
            NN_LOG(" ");
        }
        if(i % 32  == 0)
        {
            NN_LOG("\n");
        }

        uint8_t value8 = static_cast<const uint8_t*>(buffer)[i];
        NN_LOG("%02x", value8);
    }

    NN_LOG("\n\n");
}


// 挿入されたゲームカードの normal area 先頭 512 Bytes を読み込む
void ReadNormalArea()
{
    NN_LOG("Reading normal area.\n");

    const int NormalAreaOffset = 0x78 * 512; // card header, cert area, reserved area を読み飛ばす
    const size_t BufferSize = 512;

    GameCardHandle handle;
    Result result;

    // ゲームカードのハンドルを取得
    result = GetGameCardHandle(&handle);
    if( result.IsFailure() )
    {
        NN_ABORT("failed to open handle. (%d:%d)\n", result.GetModule(), result.GetDescription());
    }

    // normal area をオープン
    std::unique_ptr<nn::fs::IStorage> storage;
    result = OpenGameCardPartition(&storage, handle, GameCardPartitionRaw::NormalReadOnly);
    if( result.IsFailure() )
    {
        NN_ABORT("failed to open handle. (%d:%d)\n", result.GetModule(), result.GetDescription());
    }

    // normal area の先頭 512 バイトを読み込み
    // アクセスのオフセット・サイズは 512 バイト単位である必要がある
    char buffer[BufferSize];
    memset(buffer, 0xAB, BufferSize);
    result = (storage->Read(NormalAreaOffset, buffer, BufferSize));
    if( result.IsFailure() )
    {
        NN_ABORT("failed to read normal area. (%d:%d)\n", result.GetModule(), result.GetDescription());
    }

    DumpBuffer(buffer, BufferSize);

    // CUP バージョンを取得 & 表示

    nn::fs::GameCardUpdatePartitionInfo info;
    nn::fs::GetGameCardUpdatePartitionInfo(&info, handle);
    NN_LOG("CupVersion = %x\n", info.version);
    NN_LOG("CupId      = %x\n", info.id);
}

// 挿入されたゲームカードの secure area 先頭 512 bytes を読み込む
void ReadSecureArea()
{
    NN_LOG("Reading secure area.\n");

    const size_t BufferSize = 512;

    GameCardHandle handle;
    Result result;

    // ゲームカードのハンドルを取得
    result = GetGameCardHandle(&handle);
    if( result.IsFailure() )
    {
        NN_ABORT("failed to open handle. (%d:%d)\n", result.GetModule(), result.GetDescription());
    }

    // secure area をオープン
    std::unique_ptr<nn::fs::IStorage> storage;
    result = OpenGameCardPartition(&storage, handle, GameCardPartitionRaw::SecureReadOnly);
    if( result.IsFailure() )
    {
        NN_ABORT("failed to open handle. (%d:%d)\n", result.GetModule(), result.GetDescription());
    }

    // secure area の先頭 512 バイトを読み込み
    // アクセスのオフセット・サイズは 512 バイト単位である必要がある
    char buffer[BufferSize];
    memset(buffer, 0xAB, BufferSize);
    result = (storage->Read(0, buffer, BufferSize));
    if( result.IsFailure() )
    {
        NN_ABORT("failed to read secure area. (%d:%d)\n", result.GetModule(), result.GetDescription());
    }

    DumpBuffer(buffer, BufferSize);
}

// 挿入か抜去のイベント通知を 1 回待つ
void WaitAttachOrDetach()
{
    NN_LOG("Waiting attach/detach of game card.\n");

    Result result;

    // 挿抜イベント通知をオープン
    std::unique_ptr<nn::fs::IEventNotifier> deviceDetectionEventNotifier;
    result = OpenGameCardDetectionEventNotifier(&deviceDetectionEventNotifier);
    if( result.IsFailure() )
    {
        NN_ABORT("failed to open notifier. (%d:%d)\n", result.GetModule(), result.GetDescription());
    }

    // システムイベントと紐づけ
    nn::os::SystemEventType deviceDetectionEvent;
    result = deviceDetectionEventNotifier->BindEvent(&deviceDetectionEvent, nn::os::EventClearMode_ManualClear);
    if( result.IsFailure() )
    {
        NN_ABORT("failed to bind system event to notifier. (%d:%d)\n", result.GetModule(), result.GetDescription());
    }

    // イベント通知を待つ
    nn::os::WaitSystemEvent(&deviceDetectionEvent);
    nn::os::ClearSystemEvent(&deviceDetectionEvent);
    NN_LOG("%s detected.\n", IsGameCardInserted() ? "Attach" : "Detach");

    // イベントを破棄
    nn::os::DestroySystemEvent(&deviceDetectionEvent);
}

// ゲームカードが挿入されていなければ挿入を待つ
void WaitAttach()
{
    if( !IsGameCardInserted() )
    {
        Result result;

        NN_LOG("Waiting attach of game card.\n");

        // 挿抜イベント通知をオープン
        std::unique_ptr<nn::fs::IEventNotifier> deviceDetectionEventNotifier;
        result = OpenGameCardDetectionEventNotifier(&deviceDetectionEventNotifier);
        if( result.IsFailure() )
        {
            NN_ABORT("failed to open notifier. (%d:%d)\n", result.GetModule(), result.GetDescription());
        }

        // システムイベントと紐づけ
        nn::os::SystemEventType deviceDetectionEvent;
        result = deviceDetectionEventNotifier->BindEvent(&deviceDetectionEvent, nn::os::EventClearMode_ManualClear);
        if( result.IsFailure() )
        {
            NN_ABORT("failed to bind system event to notifier. (%d:%d)\n", result.GetModule(), result.GetDescription());
        }
        // 挿入を待つ
        while( !IsGameCardInserted() )
        {
            // イベント通知を待つ
            nn::os::WaitSystemEvent(&deviceDetectionEvent);
            nn::os::ClearSystemEvent(&deviceDetectionEvent);
        }

        // イベントを破棄
        nn::os::DestroySystemEvent(&deviceDetectionEvent);
    }
}

}


extern "C" void nnMain()
{
    WaitAttach();
    ReadNormalArea();
    ReadSecureArea();
    WaitAttachOrDetach();
}
