﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_CacheStorage.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SpeedEmulation.h>
#include <nn/fs/fs_TemporaryStorage.h>
#include <nn/result/result_HandlingUtility.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

namespace {

// nmeta 記述によって作成されていて利用可能であること
TEST(TemporaryStorage, Mount)
{
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountBis(nn::fs::BisPartitionId::User, nullptr));
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::ListDirectoryRecursive("@User:/"));
        nn::fs::Unmount("@User");
    }

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountTemporaryStorage("temp"));

    NN_LOG("%s %d: %d\n", __FUNCTION__, __LINE__, 0);
    nnt::fs::util::DumpDirectoryRecursive("temp:/");

    // アプリ終了でクリンナップされるので、必ず未存在である
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile("temp:/test.file", 8 * 1024 * 1024));
    NN_LOG("%s %d: %d\n", __FUNCTION__, __LINE__, 0);
    nnt::fs::util::DumpDirectoryRecursive("temp:/");

    NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::WriteFileWith32BitCount("temp:/test.file", 8 * 1024 * 1024, 0));

    nnt::fs::util::DumpDirectoryRecursive("temp:/");

    EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("temp:/test.file", 0));

    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CommitSaveData("temp")); // コミットしたとしても


    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountBis(nn::fs::BisPartitionId::User, nullptr));
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::ListDirectoryRecursive("@User:/"));
        nn::fs::Unmount("@User");
    }

    nn::fs::Unmount("temp");

    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountBis(nn::fs::BisPartitionId::User, nullptr));
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::DumpDirectoryRecursive("@User:/"));
        nn::fs::Unmount("@User");
    }

    // unmount しても内容は保持（アプリ終了まで）
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountTemporaryStorage("temp"));
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("temp:/test.file", 0));
        nn::fs::Unmount("temp");
    }

}

// nmeta 記述によって作成されていて利用可能であること
TEST(CacheStorage, MountAndCreate)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountCacheStorage("cache"));

    nnt::fs::util::DumpDirectoryRecursive("cache:/");
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile("cache:/test.file", 8 * 1024 * 1024));
    nnt::fs::util::DumpDirectoryRecursive("cache:/");
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CommitSaveData("cache"));

    NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::WriteFileWith32BitCount("cache:/test.file", 8 * 1024 * 1024, 0));

    nnt::fs::util::DumpDirectoryRecursive("cache:/");

    EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("cache:/test.file", 0));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CommitSaveData("cache"));

    nn::fs::Unmount("cache");
}

// 前回実行時の内容が残っていること(実装は通常のセーブデータ)
TEST(CacheStorage, MountAndVerify)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountCacheStorage("cache"));
    EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("cache:/test.file", 0));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteFile("cache:/test.file"));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CommitSaveData("cache"));

    nnt::fs::util::DumpDirectoryRecursive("cache:/");

    nn::fs::Unmount("cache");

    nnt::fs::util::DeleteAllTestSaveData();
}


}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();
    nn::fs::SetEnabledAutoAbort(false);

    auto result = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(result);
}

