﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/fs.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_ResultHandler.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

namespace {

    const char TestFileName[] = "file";

    const int64_t SystemSaveDataSize = 1 * 1024 * 1024;
    const int64_t SystemSaveDataJournalSize = 1 * 1024 * 1024;
    const int SystemSaveDataCountLimit = 48; // 51以上にするとデッドロックが発生

    nnt::fs::util::String GetSystemMountNameByIndex(int index) NN_NOEXCEPT
    {
        nnt::fs::util::String mountName = "syssave" + nnt::fs::util::ToString(index);
        return mountName;
    }

    nn::fs::SystemSaveDataId GetSystemSaveDataIdByIndex(int index) NN_NOEXCEPT
    {
        const nn::fs::SystemSaveDataId BaseId = 0x8000000000004100;
        return BaseId + index;
    }

    // セーブデータをダーティー化する
    void MakeDirtinessSaveData(const nnt::fs::util::String& mountName, int64_t sizeSaveData) NN_NOEXCEPT
    {
        nnt::fs::util::String TestPath = mountName + ":/" + TestFileName;

        int64_t fileSize = sizeSaveData - 16 * 1024 * 2;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile(TestPath.c_str(), fileSize));
        char buf[8] = {};

        nn::fs::FileHandle handle;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&handle, TestPath.c_str(), nn::fs::OpenMode_Write));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        int64_t offsetMin = 16 * 1024 - static_cast<int64_t>(sizeof(buf) / 2);
        for( int64_t offset = offsetMin; offset + static_cast<int64_t>(sizeof(buf)) < fileSize; offset += 32 * 1024 )
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteFile(
                handle, offset, buf, sizeof(buf),
                nn::fs::WriteOption()));
        }

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::FlushFile(handle));
    }

}

TEST(MountMultiSaveData, WriteWithoutCommit)
{
    // テストで使用するのと同 ID の既存セーブデータを削除
    for( int i = 1; i <= SystemSaveDataCountLimit; ++i )
    {
        const auto saveDataId = static_cast<nn::fs::SaveDataId>(GetSystemSaveDataIdByIndex(i));
        nn::fs::DeleteSaveData(saveDataId);
    }

    // システムセーブデータを作成
    for( int i = 1; i <= SystemSaveDataCountLimit; ++i )
    {
        NN_LOG("create system save data #%d\n", i);
        auto result = nn::fs::CreateSystemSaveData(
            GetSystemSaveDataIdByIndex(i), SystemSaveDataSize, SystemSaveDataJournalSize, 0);
        NNT_ASSERT_RESULT_SUCCESS(result);
    }
    NN_UTIL_SCOPE_EXIT
    {
        // 全システムセーブデータをアンマウント、削除
        for( int i = 1; i <= SystemSaveDataCountLimit; ++i )
        {
            nn::fs::Unmount(GetSystemMountNameByIndex(i).c_str());
            const auto saveDataId = static_cast<nn::fs::SaveDataId>(GetSystemSaveDataIdByIndex(i));
            nn::fs::DeleteSaveData(saveDataId);
        }
    };

    // システムセーブデータをマウント
    for( int i = 1; i <= SystemSaveDataCountLimit; ++i )
    {
        NN_LOG("mount system save data #%d\n", i);
        auto result = nn::fs::MountSystemSaveData(
            GetSystemMountNameByIndex(i).c_str(),
            GetSystemSaveDataIdByIndex(i));
        NNT_ASSERT_RESULT_SUCCESS(result);
    }

    // システムセーブを Dirty 化する
    for( int i = 1; i <= SystemSaveDataCountLimit; ++i )
    {
        NN_LOG("write system save data #%d\n", i);
        MakeDirtinessSaveData(GetSystemMountNameByIndex(i), SystemSaveDataSize);
    }
} // NOLINT(impl/function_size)

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);
    nn::fs::SetEnabledAutoAbort(false);

    auto testResult = RUN_ALL_TESTS();

    nnt::Exit(testResult);
}
