﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <functional>

#include <nn.h>
#include <nn/init.h>
#include <nn/ae.h>
#include <nn/time.h>
#include <nn/htc.h>
#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_CacheStoragePrivate.h>
#include <nn/fs/fs_CacheStorage.h>
#include <nn/fs/fs_Host.h>
#include <nn/fs/fs_PriorityPrivate.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SaveDataForDebug.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataManagementPrivate.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/fs/fs_TemporaryStorage.h>
#include <nn/fs/detail/fs_AccessLog.h>
#include <nn/fs/fs_SpeedEmulation.h>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>

#include "testFs_Stress_MountBase.h"
#include "testFs_Stress_Param.h"

class TestContext
{
public:
    NN_IMPLICIT TestContext(StressTestMountInfoBase* pTest) NN_NOEXCEPT
    : m_pTarget(pTest),
      m_IsSuccess(false),
      m_Priority(nn::fs::PriorityRaw_Normal),
      m_Random(nnt::fs::util::GetRandomSeed())
    {
    }

    //! マウント名を取得する
    const char* GetMountName() const NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(m_pTarget);
        return m_pTarget->GetMountName();
    }

    //! 読み込み専用かどうか
    bool IsReadOnly() const NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(m_pTarget);
        return m_pTarget->IsReadOnly();
    }

    //! セーブデータかどうか
    bool IsSaveData() const NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(m_pTarget);
        return m_pTarget->IsSaveData();
    }

    //! FAT へのアクセスが発生するかどうか
    bool IsAccessesFatDirectly() const NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(m_pTarget);
        return m_pTarget->IsAccessesFatDirectly();
    }

    //! アクセス優先度を取得
    nn::fs::PriorityRaw GetPriority() const NN_NOEXCEPT
    {
        return m_Priority;
    }

    //! アクセス優先度を設定
    void SetPriority(nn::fs::PriorityRaw priority) NN_NOEXCEPT
    {
        m_Priority = priority;
    }

    //! アクセス優先度をランダムに設定
    void SetPriorityRandom() NN_NOEXCEPT
    {
        m_Priority = static_cast<nn::fs::PriorityRaw>(std::uniform_int_distribution<int>(0, 4 - 1)(m_Random));
    }

    //! テスト成功状態に遷移する
    void Succeed() NN_NOEXCEPT
    {
        m_IsSuccess = true;
    }

    //! テスト失敗状態に遷移する
    void Fail() NN_NOEXCEPT
    {
        m_IsSuccess = false;
    }

    //! テストの成否を取得する
    bool IsSuccess() const NN_NOEXCEPT
    {
        return m_IsSuccess;
    }

    nn::os::Mutex& GetMutex() NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(m_pTarget);
        return m_pTarget->GetMutex();
    }

    //! テスト対象ストレージ情報を取得
    const StressTestMountInfoBase* GetTarget() const NN_NOEXCEPT
    {
        return m_pTarget;
    }

private:
    StressTestMountInfoBase* m_pTarget;     //!< テスト対象ストレージ情報
    bool m_IsSuccess;                       //!< テストの成否
    nn::fs::PriorityRaw m_Priority;         //!< アクセス優先度
    std::mt19937 m_Random;
};

class TestRunner
{
public:
    TestRunner() NN_NOEXCEPT
    : m_Random(nnt::fs::util::GetRandomSeed()),
      m_LoopCount(StressTestParam::LoopCount)
    {
    }

    ~TestRunner() NN_NOEXCEPT
    {
        for( size_t i = 0; i < m_Tests.size(); ++i )
        {
            delete m_Tests[i];
        }
    }

    virtual void RunTest(TestContext*, int) NN_NOEXCEPT
    {
    }

    void RunThreadTest() NN_NOEXCEPT;

    void AddContext(TestContext* p) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Tests.size() < StressTestParam::ThreadCountMax);
        m_Tests.push_back(p);
    }

    //! ループ回数を取得
    int GetLoopCount() const NN_NOEXCEPT
    {
        return m_LoopCount;
    }

    //! ループ回数を設定
    void SetLoopCount(int count) NN_NOEXCEPT
    {
        m_LoopCount = count;
    }

    //! テストが成功したかどうか取得
    bool IsSuccess() const NN_NOEXCEPT
    {
        for( size_t i = 0; i < m_Tests.size(); ++i )
        {
            if(! m_Tests[i]->IsSuccess() )
            {
                return false;
            }
        }
        return true;
    }

protected:
    std::mt19937 m_Random;

private:
    nnt::fs::util::Vector<TestContext *> m_Tests;
    int m_LoopCount;                            //!< ループ回数
};

