﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testFs_Stress_StressTest.h"

// 指定されたテスト情報でスレッドを起動します。
void TestRunner::RunThreadTest() NN_NOEXCEPT
{
    auto threadCount = m_Tests.size();

    // テストをスレッドで実行、テスト終了待ちを行います。
    {
        struct Argument
        {
            TestRunner* pRunner;
            TestContext* pTest;
            int threadIndex;
        };

        static NN_OS_ALIGNAS_THREAD_STACK char s_ThreadStack[StressTestParam::ThreadStackSize * StressTestParam::ThreadCountMax];
        static nn::os::ThreadType s_Threads[StressTestParam::ThreadCountMax];
        static Argument s_Arguments[StressTestParam::ThreadCountMax];

        for( size_t threadIndex = 0; threadIndex < threadCount; ++threadIndex )
        {
            s_Arguments[threadIndex].pRunner = this;
            s_Arguments[threadIndex].pTest = m_Tests[threadIndex];
            s_Arguments[threadIndex].threadIndex = static_cast<int>(threadIndex);

            NNT_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(
                s_Threads + threadIndex,
                [](void* pArgument) NN_NOEXCEPT
                {
                    const auto& argument = *reinterpret_cast<Argument*>(pArgument);

                    const auto loopCount = argument.pRunner->GetLoopCount();
                    ASSERT_GT(loopCount, 0);
                    for( auto count = 0; count < loopCount; ++count )
                    {
                        argument.pRunner->RunTest(argument.pTest, argument.threadIndex);
                    }
                },
                s_Arguments + threadIndex,
                s_ThreadStack + StressTestParam::ThreadStackSize * threadIndex,
                StressTestParam::ThreadStackSize,
                m_Tests[threadIndex]->GetPriority()
            ));
        }

        for( size_t threadIndex = 0; threadIndex < threadCount; ++threadIndex )
        {
            nn::os::StartThread(s_Threads + threadIndex);
        }

        for( size_t threadIndex = 0; threadIndex < threadCount; ++threadIndex )
        {
            nn::os::WaitThread(s_Threads + threadIndex);
            nn::os::DestroyThread(s_Threads + threadIndex);
        }
    }
} // NOLINT(impl/function_size)

void SetUpSaveData() NN_NOEXCEPT
{
    // セーブデータ一括削除
    nnt::fs::util::DeleteAllTestSaveData();
    {
        const nn::fs::SaveDataSpaceId SpaceIds[] = {
            nn::fs::SaveDataSpaceId::User,
            nn::fs::SaveDataSpaceId::System,
            nn::fs::SaveDataSpaceId::SdSystem,
            nn::fs::SaveDataSpaceId::Temporary
        };
        for( auto spaceId : SpaceIds )
        {
            while( NN_STATIC_CONDITION(true) )
            {
                nn::util::optional<nn::fs::SaveDataInfo> info;
                nnt::fs::util::FindSaveData(&info, spaceId, [&](const nn::fs::SaveDataInfo& saveDataInfo) NN_NOEXCEPT
                {
                    return ((saveDataInfo.saveDataUserId == StressTestParam::UserIds[0])
                         || (saveDataInfo.saveDataUserId == StressTestParam::UserIds[1]));
                });
                if( info == nn::util::nullopt )
                {
                    break;
                }

                auto saveId = info->saveDataId;
                NN_LOG("delete save data (%x, %llx).\n", spaceId, saveId);
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::DeleteSaveData(spaceId, saveId));
            }
        }
    }

    // セーブデータ作成
    nn::Result result;

    result = nn::fs::CreateSaveData(
                 StressTestParam::ApplicationIds[0],
                 StressTestParam::UserIds[0],
                 StressTestParam::OwnerIds[0],
                 StressTestParam::SaveDataSize,
                 StressTestParam::JournalSize,
                 0
             );
    if( !nn::fs::ResultPathAlreadyExists::Includes(result) )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    result = nn::fs::CreateSaveData(
                 StressTestParam::ApplicationIds[0],
                 StressTestParam::UserIds[1],
                 StressTestParam::OwnerIds[0],
                 StressTestParam::SaveDataSize,
                 StressTestParam::JournalSize,
                 0
             );
    if( !nn::fs::ResultPathAlreadyExists::Includes(result) )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    result = nn::fs::CreateSaveData(
                 StressTestParam::ApplicationIds[1],
                 StressTestParam::UserIds[0],
                 StressTestParam::OwnerIds[1],
                 StressTestParam::SaveDataSize,
                 StressTestParam::JournalSize,
                 0
             );
    if( !nn::fs::ResultPathAlreadyExists::Includes(result) )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    result = nn::fs::CreateCacheStorage(
                 nnt::fs::util::ApplicationId,
                 nn::fs::SaveDataSpaceId::SdUser,
                 StressTestParam::OwnerIds[0],
                 StressTestParam::SaveDataSize,
                 StressTestParam::JournalSize,
                 0
             );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

extern "C" void nninitStartup()
{
    const size_t MemoryHeapSize = 4 * 1024 * 1024;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::SetMemoryHeapSize(MemoryHeapSize));
    uintptr_t address;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::AllocateMemoryBlock(&address, MemoryHeapSize));
    nn::init::InitializeAllocator(reinterpret_cast<void*>(address), MemoryHeapSize);
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nn::time::Initialize();

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();
    nn::fs::SetEnabledAutoAbort(false);

    static const int SessionCountMax = 5;
    nn::fs::detail::DisableSessionForRealtimeOnlyAndChangeSessionCount(SessionCountMax);
    NN_UTIL_SCOPE_EXIT
    {
        nnt::fs::util::RevertSessionSettingToDefault();
    };

    SetUpSaveData();

    auto testResult = RUN_ALL_TESTS();

    nn::time::Finalize();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(testResult);
}
