﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testFs_Stress_StressTest.h"

// スレッドごとにファイルの作成、ランダムライトアクセス、削除を繰り返す
class TestHeavy : public TestRunner
{
public:
    virtual void RunTest(TestContext* pTest, int threadIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        // 優先度を初期化します
        auto priority = static_cast<nn::fs::PriorityRaw>(std::uniform_int_distribution<int>(0, 4 - 1)(m_Random));
        nn::fs::SetPriorityRawOnCurrentThread(priority);

        int operation = std::uniform_int_distribution<int>(0, 2 - 1)(m_Random);
        switch( operation )
        {
        case 0:
            RunTestFile(pTest, threadIndex);
            break;

        case 1:
            RunTestDirectory(pTest, threadIndex);
            break;

        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    void SetupDirectory(const char* mountName, int threadIndex, bool isCreateFile)
    {
        std::unique_ptr<char[]> path(new char[nn::fs::EntryNameLengthMax]);
        nn::util::SNPrintf(
            path.get(),
            nn::fs::EntryNameLengthMax,
            "%s:/dir%03d",
            mountName,
            threadIndex
        );
        nnt::fs::util::String dirPath(path.get());

        (void)nn::fs::DeleteDirectoryRecursively(path.get());

        // エントリを作成する
        for( auto depth = 0; depth < MaxDepth; ++depth )
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateDirectory(dirPath.c_str()));
            if( isCreateFile )
            {
                std::unique_ptr<char[]> pathFile(new char[nn::fs::EntryNameLengthMax]);
                nn::util::SNPrintf(pathFile.get(), nn::fs::EntryNameLengthMax, "%s/sub", dirPath.c_str());
                NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile(pathFile.get(), 0x400));
            }
            std::unique_ptr<char[]> pathCurrent(new char[nn::fs::EntryNameLengthMax]);
            nn::util::SNPrintf(pathCurrent.get(), nn::fs::EntryNameLengthMax, "/%d", depth + 1);
            dirPath += pathCurrent.get();
        }
    }

    void RunTestFile(TestContext* pTest, int threadIndex) NN_NOEXCEPT
    {
        pTest->Fail();

        for( auto loopFile = 0; loopFile < 20; ++loopFile )
        {
            std::unique_ptr<char[]> path(new char[nn::fs::EntryNameLengthMax]);
            nn::util::SNPrintf(
                path.get(),
                nn::fs::EntryNameLengthMax,
                "%s:/file%d_%d",
                pTest->GetMountName(),
                threadIndex,
                loopFile
            );
            if( !pTest->IsReadOnly() )
            {
                (void)nn::fs::DeleteFile(path.get());
                NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile(path.get(), 0x1000));
            }
            NN_UTIL_SCOPE_EXIT
            {
                if( !pTest->IsReadOnly() )
                {
                    NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteFile(path.get()));
                }
            };

            nn::fs::FileHandle fileHandle;
            auto openMode = pTest->IsReadOnly() ?
                  nn::fs::OpenMode_Read
                : nn::fs::OpenMode_Read | nn::fs::OpenMode_Write;
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fs::OpenFile(
                    &fileHandle,
                    path.get(),
                    openMode
                )
            );
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(fileHandle);
            };

            int64_t sizeFile;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&sizeFile, fileHandle));
            std::unique_ptr<char[]> buffer(new char[static_cast<size_t>(sizeFile)]);
            for( auto loopOperation = 0; loopOperation < 10; ++loopOperation )
            {
                // ファイル内のランダムな範囲に対してリードライトを行います。
                int operation = std::uniform_int_distribution<int>(0, 10 - 1)(m_Random);
                if( operation < 4 )
                {
                    int64_t sizeRead = std::uniform_int_distribution<int64_t>(0, sizeFile - 1)(m_Random);
                    int64_t offsetRead = std::uniform_int_distribution<int64_t>(0, sizeFile - sizeRead - 1)(m_Random);
                    size_t sizeResult;
                    NNT_ASSERT_RESULT_SUCCESS(
                        nn::fs::ReadFile(
                            &sizeResult,
                            fileHandle,
                            offsetRead,
                            &buffer[0],
                            static_cast<size_t>(sizeRead)
                        )
                    );
                }
                else if( operation < 8 )
                {
                    if( !pTest->IsReadOnly() )
                    {
                        int64_t sizeWrite = std::uniform_int_distribution<int64_t>(0, sizeFile - 1)(m_Random);
                        int64_t offsetWrite = std::uniform_int_distribution<int64_t>(0, sizeFile - sizeWrite - 1)(m_Random);
                        NNT_ASSERT_RESULT_SUCCESS(
                            nn::fs::WriteFile(
                                fileHandle,
                                offsetWrite,
                                &buffer[0],
                                static_cast<size_t>(sizeWrite),
                                nn::fs::WriteOption::MakeValue(0)
                            )
                        );
                        NNT_ASSERT_RESULT_SUCCESS(nn::fs::FlushFile(fileHandle));
                    }
                }
                else
                {
                    if( !pTest->IsReadOnly() && !pTest->IsAccessesFatDirectly() )
                    {
                        int64_t sizeAdd = std::uniform_int_distribution<int64_t>(0, 0x1000 - 1)(m_Random);
                        NNT_ASSERT_RESULT_SUCCESS(nn::fs::SetFileSize(fileHandle, sizeFile + sizeAdd));
                        NNT_ASSERT_RESULT_SUCCESS(nn::fs::FlushFile(fileHandle));
                        NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&sizeFile, fileHandle));
                        buffer.reset(new char[static_cast<size_t>(sizeFile)]);
                    }
                }
            }
        }

        if( pTest->IsSaveData() )
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::CommitSaveData(pTest->GetMountName()));
        }

        pTest->Succeed();
    } // NOLINT(impl/function_size)

    void RunTestDirectory(TestContext* pTest, int threadIndex) NN_NOEXCEPT
    {
        pTest->Fail();

        std::unique_ptr<char[]> path(new char[nn::fs::EntryNameLengthMax]);
        nn::util::SNPrintf(
            path.get(),
            nn::fs::EntryNameLengthMax,
            "%s:/dir%03d",
            pTest->GetMountName(),
            threadIndex
        );
        nnt::fs::util::String dirPath(path.get());

        int operation = std::uniform_int_distribution<int>(0, 4 - 1)(m_Random);
        switch( operation )
        {
        case 0:
            {
                // エントリを作成する
                if( !pTest->IsReadOnly() )
                {
                    SetupDirectory(pTest->GetMountName(), threadIndex, true);
                }

                // エントリを開く
                dirPath = path.get();
                for( auto depth = 0; depth < MaxDepth; ++depth )
                {
                    nn::fs::DirectoryEntryType type;
                    NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetEntryType(&type, dirPath.c_str()));
                    ASSERT_EQ(nn::fs::DirectoryEntryType_Directory, type);
                    std::unique_ptr<char[]> pathFile(new char[nn::fs::EntryNameLengthMax]);
                    nn::util::SNPrintf(pathFile.get(), nn::fs::EntryNameLengthMax, "%s/sub", dirPath.c_str());
                    nn::fs::FileHandle fileHandle;
                    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandle, pathFile.get(), nn::fs::OpenMode_Read));
                    nn::fs::CloseFile(fileHandle);
                    std::unique_ptr<char[]> pathCurrent(new char[nn::fs::EntryNameLengthMax]);
                    nn::util::SNPrintf(pathCurrent.get(), nn::fs::EntryNameLengthMax, "/%d", depth + 1);
                    dirPath += pathCurrent.get();
                }
            }
            break;

        case 1:
            if( !pTest->IsReadOnly() )
            {
                // エントリを作成する
                SetupDirectory(pTest->GetMountName(), threadIndex, false);
            }
            break;

        case 2:
            if( !pTest->IsReadOnly() )
            {
                // エントリを作成する
                SetupDirectory(pTest->GetMountName(), threadIndex, true);
            }
            // ディレクトリを大量に開く
            {
                dirPath = path.get();
                nn::fs::DirectoryHandle dirHandles[MaxDepth];
                for( auto depth = 0; depth < MaxDepth; ++depth )
                {
                    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenDirectory(&dirHandles[depth], dirPath.c_str(), nn::fs::OpenDirectoryMode_All));
                    std::unique_ptr<char[]> pathCurrent(new char[nn::fs::EntryNameLengthMax]);
                    nn::util::SNPrintf(pathCurrent.get(), nn::fs::EntryNameLengthMax, "/%d", depth + 1);
                    dirPath += pathCurrent.get();
                }
                NN_UTIL_SCOPE_EXIT
                {
                    for( auto depth = 0; depth < MaxDepth; ++depth )
                    {
                        nn::fs::CloseDirectory(dirHandles[depth]);
                    }
                };
                dirPath = path.get();
                for( auto depth = 0; depth < MaxDepth; ++depth )
                {
                    // ディレクトリを読み込む
                    int64_t countRead;
                    nn::fs::DirectoryEntry dirEntry;
                    NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadDirectory(&countRead, &dirEntry, dirHandles[depth], 1));
                    ASSERT_EQ(1, countRead);

                    // ディレクトリ以下のファイルを読み込む
                    std::unique_ptr<char[]> pathFile(new char[nn::fs::EntryNameLengthMax]);
                    nn::util::SNPrintf(pathFile.get(), nn::fs::EntryNameLengthMax, "%s/sub", dirPath.c_str());
                    nn::fs::FileHandle fileHandle;
                    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandle, pathFile.get(), nn::fs::OpenMode_Read));
                    nn::fs::CloseFile(fileHandle);

                    // 再度ディレクトリを読み込む
                    NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadDirectory(&countRead, &dirEntry, dirHandles[depth], 1));

                    std::unique_ptr<char[]> pathCurrent(new char[nn::fs::EntryNameLengthMax]);
                    nn::util::SNPrintf(pathCurrent.get(), nn::fs::EntryNameLengthMax, "/%d", depth + 1);
                    dirPath += pathCurrent.get();
                }
            }
            break;

        case 3:
            if( !pTest->IsReadOnly() )
            {
                // エントリを作成する
                SetupDirectory(pTest->GetMountName(), threadIndex, true);
            }
            // ファイルを大量に開く
            {
                dirPath = path.get();
                nn::fs::FileHandle fileHandles[MaxDepth];
                for( auto depth = 0; depth < MaxDepth; ++depth )
                {
                    std::unique_ptr<char[]> pathFile(new char[nn::fs::EntryNameLengthMax]);
                    nn::util::SNPrintf(pathFile.get(), nn::fs::EntryNameLengthMax, "%s/sub", dirPath.c_str());
                    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandles[depth], pathFile.get(), nn::fs::OpenMode_Read));
                    std::unique_ptr<char[]> pathCurrent(new char[nn::fs::EntryNameLengthMax]);
                    nn::util::SNPrintf(pathCurrent.get(), nn::fs::EntryNameLengthMax, "/%d", depth + 1);
                    dirPath += pathCurrent.get();
                }
                for( auto depth = 0; depth < MaxDepth; ++depth )
                {
                    nn::fs::CloseFile(fileHandles[depth]);
                }
            }
            break;

        default:
            NN_UNEXPECTED_DEFAULT;
        }

        if( !pTest->IsReadOnly() )
        {
            // エントリを削除する
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteDirectoryRecursively(path.get()));
        }
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CommitSaveData(pTest->GetMountName()));

        pTest->Succeed();
    } // NOLINT(impl/function_size)

    static const auto MaxDepth = 30;
};

TEST(MultiFsStressTest, Heavy)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;

    StressTestMountBis fsBis;
    {
        fsBis.Mount();
        list.push_back(&fsBis);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsBis.Unmount();
    };

#if !defined(NN_BUILD_CONFIG_OS_WIN)
    StressTestMountCacheStorage fsCache;
    {
        const nnt::fs::util::String mountName("cache");
        fsCache.SetMountName(mountName.c_str());
        fsCache.Mount();
        list.push_back(&fsCache);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsCache.Unmount();
    };
#endif

    StressTestMountHostFs fsHost;
    {
        const nnt::fs::util::String mountName("host");
        fsHost.SetMountName(mountName.c_str());
        fsHost.Mount();
        list.push_back(&fsHost);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsHost.Unmount();
    };

#if !defined(NN_BUILD_CONFIG_OS_WIN)
    StressTestMountOtherSaveData fsOtherSave;
    {
        const nnt::fs::util::String mountName("other");
        fsOtherSave.SetMountName(mountName.c_str());
        fsOtherSave.Mount();
        list.push_back(&fsOtherSave);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsOtherSave.Unmount();
    };
#endif

    StressTestMountRomFs fsRom;
    {
        const nnt::fs::util::String mountName("rom");
        fsRom.SetMountName(mountName.c_str());
        fsRom.Mount();
        list.push_back(&fsRom);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsRom.Unmount();
    };

    StressTestMountSaveData fsSave;
    {
        const nnt::fs::util::String mountName("save");
        fsSave.SetMountName(mountName.c_str());
        fsSave.Mount();
        list.push_back(&fsSave);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsSave.Unmount();
    };

    StressTestMountSdCard fsSdCard;
    {
        const nnt::fs::util::String mountName("sd");
        fsSdCard.SetMountName(mountName.c_str());
        fsSdCard.Mount();
        list.push_back(&fsSdCard);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsSdCard.Unmount();
    };

    TestHeavy test;
    test.SetLoopCount(500);

    nnt::fs::util::Vector<StressTestMountInfoBase *> listRandom;
    listRandom.reserve(list.size());
    std::copy(list.begin(), list.end(), std::back_inserter(listRandom));
    std::random_shuffle(listRandom.begin(), listRandom.end());
    auto threadCount = listRandom.size();

    for( size_t threadIndex = 0; threadIndex < threadCount; ++threadIndex )
    {
        auto pContext = new TestContext(listRandom[threadIndex]);
        test.AddContext(pContext);
    }
    test.RunThreadTest();
}

