﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>

namespace nnt { namespace fs {

class FsStressTest : public ::testing::Test
{
public:
    static const auto ThreadCountMax = 8;
    static const auto ThreadStackSize = 16 * 1024;

    static const auto EntryCountMax = 100;

public:
    class TestCase
    {
    public:
        TestCase() NN_NOEXCEPT;
        virtual ~TestCase() NN_NOEXCEPT;

        void Initialize(const char* mountName, const char* testDirectoryPath, int index) NN_NOEXCEPT
        {
            m_MountName = mountName;
            m_TestDirectoryPath = testDirectoryPath;
            m_Index = index;
        }

        const char* GetMountName() const NN_NOEXCEPT
        {
            return m_MountName;
        }

        const char* GetTestDirectoryPath() const NN_NOEXCEPT
        {
            return m_TestDirectoryPath;
        }

        int GetTestCaseIndex() const NN_NOEXCEPT
        {
            return m_Index;
        }

        bool Failed(int threadIndex) const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_RANGE(threadIndex, 0, FsStressTest::ThreadCountMax);
            return m_Failed[threadIndex];
        }

        bool FailedAny() const NN_NOEXCEPT
        {
            for( auto threadIndex = 0; threadIndex < FsStressTest::ThreadCountMax; ++threadIndex )
            {
                if( Failed(threadIndex) )
                {
                    return true;
                }
            }
            return false;
        }

        void Succeed(int threadIndex) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_RANGE(threadIndex, 0, FsStressTest::ThreadCountMax);
            m_Failed[threadIndex] = false;
        }

        void SucceedAll() NN_NOEXCEPT
        {
            for( auto threadIndex = 0; threadIndex < FsStressTest::ThreadCountMax; ++threadIndex )
            {
                Succeed(threadIndex);
            }
        }

        void Fail(int threadIndex) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_RANGE(threadIndex, 0, FsStressTest::ThreadCountMax);
            m_Failed[threadIndex] = true;
        }

        void FailAll() NN_NOEXCEPT
        {
            for( auto threadIndex = 0; threadIndex < FsStressTest::ThreadCountMax; ++threadIndex )
            {
                Fail(threadIndex);
            }
        }

        virtual int GetLoopCount() const NN_NOEXCEPT;
        virtual int GetThreadCount() const NN_NOEXCEPT;
        virtual int GetPriority(int threadIndex) const NN_NOEXCEPT;

        virtual void SetUp(FsStressTest* pTest) NN_NOEXCEPT;
        virtual void TearDown(FsStressTest* pTest) NN_NOEXCEPT;

        virtual void Test(FsStressTest* pTest, int threadIndex) NN_NOEXCEPT = 0;

    private:
        const char* m_MountName;
        const char* m_TestDirectoryPath;
        int m_Index;
        std::atomic_bool m_Failed[FsStressTest::ThreadCountMax];
    };

public:
    FsStressTest() NN_NOEXCEPT
        : m_Mutex(false),
          m_MountedFileSystemCount(0),
          m_IsProcessNameNeeded(true)
    {
    }

    virtual ~FsStressTest() NN_NOEXCEPT NN_OVERRIDE;

    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE;
    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE;
    void ResetTestStatus() NN_NOEXCEPT;

    virtual bool IsReadOnly(int fsIndex) const NN_NOEXCEPT;

    virtual bool IsSaveData(int fsIndex) const NN_NOEXCEPT;
    virtual uint64_t GetApplicationId(int fsIndex) const NN_NOEXCEPT;
    virtual nn::fs::UserId GetUserId(int fsIndex) const NN_NOEXCEPT;

    virtual bool AccessesFatDirectly(int fsIndex) const NN_NOEXCEPT;

    nn::os::Mutex& GetMutex() NN_NOEXCEPT
    {
        return m_Mutex;
    }

    const char* GetTestDirectoryPath(int fsIndex) const NN_NOEXCEPT
    {
        return IsReadOnly(fsIndex) ? "" : m_TestDirectoryPath;
    }

protected:
    template<typename TestCaseType>
    void Test(const char* mountName) NN_NOEXCEPT
    {
        TestCaseType testCase;
        TestCase* pTestCase = &testCase;
        testCase.Initialize(mountName, GetTestDirectoryPath(0), 0);
        TestCore(&pTestCase, 1);
    }

    template<typename TestCaseType1, typename TestCaseType2>
    void Test(const char* mountName1, const char* mountName2) NN_NOEXCEPT
    {
        TestCaseType1 testCase1;
        TestCaseType2 testCase2;
        TestCase* pTestCases[2] = { &testCase1, &testCase2 };
        testCase1.Initialize(mountName1, GetTestDirectoryPath(0), 0);
        testCase2.Initialize(mountName2, GetTestDirectoryPath(1), 1);
        TestCore(pTestCases, 2);
    }

    template<typename TestCaseType1, typename TestCaseType2, typename TestCaseType3>
    void Test(const char* mountName1, const char* mountName2, const char* mountName3) NN_NOEXCEPT
    {
        TestCaseType1 testCase1;
        TestCaseType2 testCase2;
        TestCaseType3 testCase3;
        TestCase* pTestCases[3] = { &testCase1, &testCase2, &testCase3 };
        testCase1.Initialize(mountName1, GetTestDirectoryPath(0), 0);
        testCase2.Initialize(mountName2, GetTestDirectoryPath(1), 1);
        testCase3.Initialize(mountName3, GetTestDirectoryPath(2), 2);
        TestCore(pTestCases, 3);
    }

    void NotifyMountFileSystem() NN_NOEXCEPT
    {
        ++m_MountedFileSystemCount;
    }

    void NotifyUnmountFileSystem() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER(m_MountedFileSystemCount, 0);
        --m_MountedFileSystemCount;
    }

    int GetMountedFileSystemCount() const NN_NOEXCEPT
    {
        return m_MountedFileSystemCount;
    }

private:
    void SetUpTestDirectory(TestCase** ppTestCases, int testCaseCount) NN_NOEXCEPT;
    void TearDownTestDirectory(TestCase** ppTestCases, int testCaseCount) NN_NOEXCEPT;

    void TestCore(TestCase** ppTestCases, int testCaseCount) NN_NOEXCEPT;

private:
    nn::os::Mutex m_Mutex;
    char m_TestDirectoryPath[16];
    int m_MountedFileSystemCount;
    bool m_IsProcessNameNeeded;
};

class SaveDataFsStressTest : public FsStressTest
{
public:
    virtual ~SaveDataFsStressTest() NN_NOEXCEPT NN_OVERRIDE;
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE;
    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE;
    virtual bool IsSaveData(int fsIndex) const NN_NOEXCEPT NN_OVERRIDE;
    virtual uint64_t GetApplicationId(int fsIndex) const NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::fs::UserId GetUserId(int fsIndex) const NN_NOEXCEPT NN_OVERRIDE;

protected:
    static const char* GetMountName() NN_NOEXCEPT
    {
        return "save";
    }
};

class MultipleSaveDataFsStressTest : public FsStressTest
{
public:
    virtual ~MultipleSaveDataFsStressTest() NN_NOEXCEPT NN_OVERRIDE;
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE;
    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE;
    virtual bool IsSaveData(int fsIndex) const NN_NOEXCEPT NN_OVERRIDE;
    virtual uint64_t GetApplicationId(int fsIndex) const NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::fs::UserId GetUserId(int fsIndex) const NN_NOEXCEPT NN_OVERRIDE;

protected:
    static const char* GetMountName(int fsIndex) NN_NOEXCEPT
    {
        switch( fsIndex )
        {
        case 0:
            return "save1";
        case 1:
            return "save2";
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
};

class OtherApplicationSaveDataFsStressTest : public FsStressTest
{
public:
    virtual ~OtherApplicationSaveDataFsStressTest() NN_NOEXCEPT NN_OVERRIDE;
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE;
    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE;
    virtual bool IsSaveData(int fsIndex) const NN_NOEXCEPT NN_OVERRIDE;
    virtual uint64_t GetApplicationId(int fsIndex) const NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::fs::UserId GetUserId(int fsIndex) const NN_NOEXCEPT NN_OVERRIDE;

protected:
    static const char* GetMountName() NN_NOEXCEPT
    {
        return "save";
    }
};

class SaveDataFsRomFsStressTest : public FsStressTest
{
public:
    virtual ~SaveDataFsRomFsStressTest() NN_NOEXCEPT NN_OVERRIDE;
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE;
    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE;
    virtual bool IsReadOnly(int fsIndex) const NN_NOEXCEPT NN_OVERRIDE;
    virtual bool IsSaveData(int fsIndex) const NN_NOEXCEPT NN_OVERRIDE;
    virtual uint64_t GetApplicationId(int fsIndex) const NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::fs::UserId GetUserId(int fsIndex) const NN_NOEXCEPT NN_OVERRIDE;

protected:
    static const char* GetMountName(int fsIndex) NN_NOEXCEPT
    {
        switch( fsIndex )
        {
        case 0:
            return "save";
        case 1:
            return "rom";
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

private:
    std::unique_ptr<char[]> m_CacheBuffer;
};

class RomFsStressTest : public FsStressTest
{
public:
    virtual ~RomFsStressTest() NN_NOEXCEPT NN_OVERRIDE;
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE;
    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE;
    virtual bool IsReadOnly(int fsIndex) const NN_NOEXCEPT NN_OVERRIDE;

protected:
    static const char* GetMountName() NN_NOEXCEPT
    {
        return "rom";
    }

private:
    std::unique_ptr<char[]> m_CacheBuffer;
};

#if !defined(NNT_FS_STRESS_TEST_A_EXCLUDING_ENSURESAVEDATA2)
class HostFsStressTest : public FsStressTest
{
public:
    virtual ~HostFsStressTest() NN_NOEXCEPT NN_OVERRIDE;
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE;
    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE;

protected:
    static const char* GetMountName() NN_NOEXCEPT
    {
        return "host";
    }
    nnt::fs::util::TemporaryHostDirectory m_HostDirectory;
};
#define NNT_FS_STRESS_TEST_SUPPORTS_HOST_FS 1 // NOLINT(preprocessor/const)
#endif // !defined(NNT_FS_STRESS_TEST_A_EXCLUDING_ENSURESAVEDATA2)

class SdCardFsStressTest : public FsStressTest
{
public:
    virtual ~SdCardFsStressTest() NN_NOEXCEPT NN_OVERRIDE;
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE;
    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE;
    virtual bool AccessesFatDirectly(int fsIndex) const NN_NOEXCEPT NN_OVERRIDE;

protected:
    static const char* GetMountName() NN_NOEXCEPT
    {
        return "sdcard";
    }
};

#if defined(NNT_FS_STRESS_TEST_A_ALL) && defined(NN_BUILD_CONFIG_HARDWARE_NX)
class TemporaryStorageFsStressTest : public FsStressTest
{
public:
    virtual ~TemporaryStorageFsStressTest() NN_NOEXCEPT NN_OVERRIDE;
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE;
    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE;
    virtual bool IsSaveData(int fsIndex) const NN_NOEXCEPT NN_OVERRIDE;
    virtual uint64_t GetApplicationId(int fsIndex) const NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::fs::UserId GetUserId(int fsIndex) const NN_NOEXCEPT NN_OVERRIDE;

protected:
    static const char* GetMountName() NN_NOEXCEPT
    {
        return "temporary";
    }
};
#define NNT_FS_STRESS_TEST_SUPPORTS_TEMPORARY_STORAGE 1 // NOLINT(preprocessor/const)
#endif // defined(NNT_FS_STRESS_TEST_A_ALL) && defined(NN_BUILD_CONFIG_HARDWARE_NX)

class SdCardFsRomFsStressTest : public FsStressTest
{
public:
    virtual ~SdCardFsRomFsStressTest() NN_NOEXCEPT NN_OVERRIDE;
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE;
    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE;
    virtual bool IsReadOnly(int fsIndex) const NN_NOEXCEPT NN_OVERRIDE;
    virtual bool AccessesFatDirectly(int fsIndex) const NN_NOEXCEPT NN_OVERRIDE;

protected:
    static const char* GetMountName(int fsIndex) NN_NOEXCEPT
    {
        switch( fsIndex )
        {
        case 0:
            return "sdcard";
        case 1:
            return "rom";
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

private:
    std::unique_ptr<char[]> m_CacheBuffer;
};

}}
