﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <numeric>
#include <random>
#include <nnt/fsUtil/testFs_util.h>
#include "testFs_Stress_AccessFilesTestCase.h"

namespace nnt { namespace fs {

class ReadFilesTestCase : public AccessFilesTestCase
{
public:
    static const auto FileSize = 1 * 1024;

public:
    ReadFilesTestCase(
        int threadCount,
        int fileCount,
        bool isConcurrent,
        bool isVerificationEnabled,
        bool isCache) NN_NOEXCEPT
        : AccessFilesTestCase(
            threadCount,
            fileCount,
            static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write),
            true,
            isVerificationEnabled,
            isCache),
          m_IsConcurrent(isConcurrent)
    {
    }

    virtual ~ReadFilesTestCase() NN_NOEXCEPT NN_OVERRIDE {}

    virtual void Test(FsStressTest*, int threadIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        Fail(threadIndex);
        ASSERT_LE(GetEntryCount() * GetThreadCount(), FsStressTest::EntryCountMax);
        std::unique_ptr<char[]> buffer(new char[FileSize]);
        const auto entryCount
            = m_IsConcurrent ? GetEntryCount() * GetThreadCount() : GetEntryCount();
        for( auto entryIndex = 0; entryIndex < entryCount; ++entryIndex )
        {
            auto succeeded = false;
            ReadFile(
                &succeeded,
                buffer.get(),
                FileSize,
                m_IsConcurrent ? entryIndex : GetEntryIndex(threadIndex, entryIndex));
            ASSERT_TRUE(succeeded);
        }
        Succeed(threadIndex);
    }

protected:
    virtual void SetUpFile(bool* outSucceeded, int entryIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(outSucceeded);
        *outSucceeded = false;

        const auto file = GetFile(entryIndex);

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::SetFileSize(file, FileSize));

        std::unique_ptr<char[]> buffer(new char[FileSize]);
        std::iota(buffer.get(), buffer.get() + FileSize, static_cast<char>(entryIndex));
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteFile(
            file,
            0,
            buffer.get(),
            FileSize,
            nn::fs::WriteOption()));
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::FlushFile(file));

        if(IsVerificationEnabled())
        {
            RegisterVerifyBuffer(entryIndex, buffer.get(), FileSize);
        }

        *outSucceeded = true;
    }

    virtual void ReadFile(
        bool* outSucceeded,
        char* buffer,
        size_t size,
        int entryIndex) NN_NOEXCEPT = 0;

private:
    bool m_IsConcurrent;
};

NN_DEFINE_STATIC_CONSTANT(const int ReadFilesTestCase::FileSize);

class InBulkReadFilesTestCase : public ReadFilesTestCase
{
public:
    InBulkReadFilesTestCase(
        int threadCount,
        int fileCount,
        bool isConcurrent,
        bool isVerificationEnabled,
        bool isCache) NN_NOEXCEPT
        : ReadFilesTestCase(threadCount, fileCount, isConcurrent, isVerificationEnabled, isCache)
    {
    }

    virtual int GetLoopCount() const NN_NOEXCEPT NN_OVERRIDE
    {
        return 30;
    }

    virtual ~InBulkReadFilesTestCase() NN_NOEXCEPT NN_OVERRIDE {}

protected:
    virtual void ReadFile(
        bool* outSucceeded,
        char* buffer,
        size_t size,
        int entryIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(outSucceeded);
        *outSucceeded = false;

        const auto file = GetFile(entryIndex);
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadFile(file, 0, buffer, size));
        if(IsVerificationEnabled())
        {
            ASSERT_TRUE(Verify(entryIndex, buffer, 0, size));
        }

        *outSucceeded = true;
    }
};

class SequentialReadFilesTestCase : public ReadFilesTestCase
{
public:
    SequentialReadFilesTestCase(
        int threadCount,
        int fileCount,
        bool isConcurrent,
        bool isVerificationEnabled,
        bool isCache) NN_NOEXCEPT
        : ReadFilesTestCase(threadCount, fileCount, isConcurrent, isVerificationEnabled, isCache)
    {
    }

    virtual int GetLoopCount() const NN_NOEXCEPT NN_OVERRIDE
    {
        return 5;
    }

    virtual ~SequentialReadFilesTestCase() NN_NOEXCEPT NN_OVERRIDE {}

protected:
    virtual void ReadFile(
        bool* outSucceeded,
        char* buffer,
        size_t bufferSize,
        int entryIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(outSucceeded);
        *outSucceeded = false;

        const auto file = GetFile(entryIndex);
        for( int64_t offset = 0; offset < static_cast<int64_t>(bufferSize); ++offset )
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadFile(file, offset, buffer + offset, 1));
            if(IsVerificationEnabled())
            {
                ASSERT_TRUE(Verify(entryIndex, buffer + offset, offset, 1));
            }
        }

        *outSucceeded = true;
    }
};

class RandomReadFilesTestCase : public ReadFilesTestCase
{
public:
    RandomReadFilesTestCase(
        int threadCount,
        int fileCount,
        bool isConcurrent,
        bool isVerificationEnabled,
        bool isCache) NN_NOEXCEPT
        : ReadFilesTestCase(threadCount, fileCount, isConcurrent, isVerificationEnabled, isCache)
    {
    }

    virtual int GetLoopCount() const NN_NOEXCEPT NN_OVERRIDE
    {
        return 5;
    }

    virtual ~RandomReadFilesTestCase() NN_NOEXCEPT NN_OVERRIDE {}

protected:
    virtual void ReadFile(
        bool* outSucceeded,
        char* buffer,
        size_t size,
        int entryIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(outSucceeded);
        *outSucceeded = false;

        const auto file = GetFile(entryIndex);
        std::unique_ptr<std::mt19937> rng(new std::mt19937(nnt::fs::util::GetRandomSeed()));
        for( size_t count = 0; count < size; ++count )
        {
            const auto offset = std::uniform_int_distribution<int64_t>(0, size - 1)(*rng);
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadFile(file, offset, buffer + offset, 1));
            if(IsVerificationEnabled())
            {
                ASSERT_TRUE(Verify(entryIndex, buffer + offset, offset, 1));
            }
        }

        *outSucceeded = true;
    }
};

class MultiReadSingleFilesTestCase : public ReadFilesTestCase
{
public:
    static const size_t ReadCount = 1 * 1024;

public:
    MultiReadSingleFilesTestCase(
        int threadCount,
        int fileCount,
        bool isConcurrent,
        bool isVerificationEnabled,
        bool isCache) NN_NOEXCEPT
        : ReadFilesTestCase(threadCount, fileCount, isConcurrent, isVerificationEnabled, isCache)
    {
    }

    virtual int GetLoopCount() const NN_NOEXCEPT NN_OVERRIDE
    {
        return 10;
    }

    virtual ~MultiReadSingleFilesTestCase() NN_NOEXCEPT NN_OVERRIDE {}

protected:
    virtual void ReadFile(
        bool* outSucceeded,
        char* buffer,
        size_t size,
        int entryIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(entryIndex);
        NN_SDK_REQUIRES_NOT_NULL(outSucceeded);
        *outSucceeded = false;

        const auto file = GetFile(0);
        for( size_t count = 0; count < ReadCount; ++count )
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadFile(file, 0, buffer, size));
            if(IsVerificationEnabled())
            {
                ASSERT_TRUE(Verify(0, buffer, 0, size));
            }
        }

        *outSucceeded = true;
    }
};

class InBulkReadSingleFile : public InBulkReadFilesTestCase
{
public:
    InBulkReadSingleFile() NN_NOEXCEPT
        : InBulkReadFilesTestCase(FsStressTest::ThreadCountMax, 1, false, true, false)
    {
    }
};

class InBulkReadSingleFileWithCache : public InBulkReadFilesTestCase
{
public:
    InBulkReadSingleFileWithCache() NN_NOEXCEPT
        : InBulkReadFilesTestCase(FsStressTest::ThreadCountMax, 1, false, true, true)
    {
    }
};

class SequentialReadSingleFile : public SequentialReadFilesTestCase
{
public:
    SequentialReadSingleFile() NN_NOEXCEPT
        : SequentialReadFilesTestCase(FsStressTest::ThreadCountMax, 1, false, true, false)
    {
    }
};

class SequentialReadSingleFileWithCache : public SequentialReadFilesTestCase
{
public:
    SequentialReadSingleFileWithCache() NN_NOEXCEPT
        : SequentialReadFilesTestCase(FsStressTest::ThreadCountMax, 1, false, true, true)
    {
    }
};

class RandomReadSingleFile : public RandomReadFilesTestCase
{
public:
    RandomReadSingleFile() NN_NOEXCEPT
        : RandomReadFilesTestCase(FsStressTest::ThreadCountMax, 1, false, true, false)
    {
    }
};

class RandomReadSingleFileWithCache : public RandomReadFilesTestCase
{
public:
    RandomReadSingleFileWithCache() NN_NOEXCEPT
        : RandomReadFilesTestCase(FsStressTest::ThreadCountMax, 1, false, true, true)
    {
    }
};

class InBulkReadNumerousFiles : public InBulkReadFilesTestCase
{
public:
    InBulkReadNumerousFiles() NN_NOEXCEPT
        : InBulkReadFilesTestCase(FsStressTest::ThreadCountMax, 10, false, true, false)
    {
    }
};

class InBulkReadNumerousFilesWithCache : public InBulkReadFilesTestCase
{
public:
    InBulkReadNumerousFilesWithCache() NN_NOEXCEPT
        : InBulkReadFilesTestCase(FsStressTest::ThreadCountMax, 10, false, true, true)
    {
    }
};

class SequentialReadNumerousFiles : public SequentialReadFilesTestCase
{
public:
    SequentialReadNumerousFiles() NN_NOEXCEPT
        : SequentialReadFilesTestCase(FsStressTest::ThreadCountMax, 10, false, true, false)
    {
    }
};

class SequentialReadNumerousFilesWithCache : public SequentialReadFilesTestCase
{
public:
    SequentialReadNumerousFilesWithCache() NN_NOEXCEPT
        : SequentialReadFilesTestCase(FsStressTest::ThreadCountMax, 10, false, true, true)
    {
    }
};

class RandomReadNumerousFiles : public RandomReadFilesTestCase
{
public:
    RandomReadNumerousFiles() NN_NOEXCEPT
        : RandomReadFilesTestCase(FsStressTest::ThreadCountMax, 10, false, true, false)
    {
    }
};

class RandomReadNumerousFilesWithCache : public RandomReadFilesTestCase
{
public:
    RandomReadNumerousFilesWithCache() NN_NOEXCEPT
        : RandomReadFilesTestCase(FsStressTest::ThreadCountMax, 10, false, true, true)
    {
    }
};

class InBulkConcurrentReadFiles : public InBulkReadFilesTestCase
{
public:
    InBulkConcurrentReadFiles() NN_NOEXCEPT
        : InBulkReadFilesTestCase(FsStressTest::ThreadCountMax, 5, true, true, false)
    {
    }
};

class InBulkConcurrentReadFilesWithCache : public InBulkReadFilesTestCase
{
public:
    InBulkConcurrentReadFilesWithCache() NN_NOEXCEPT
        : InBulkReadFilesTestCase(FsStressTest::ThreadCountMax, 5, true, true, true)
    {
    }
};

class SequentialConcurrentReadFiles : public SequentialReadFilesTestCase
{
public:
    SequentialConcurrentReadFiles() NN_NOEXCEPT
        : SequentialReadFilesTestCase(FsStressTest::ThreadCountMax, 5, true, true, false)
    {
    }
};

class SequentialConcurrentReadFilesWithCache : public SequentialReadFilesTestCase
{
public:
    SequentialConcurrentReadFilesWithCache() NN_NOEXCEPT
        : SequentialReadFilesTestCase(FsStressTest::ThreadCountMax, 5, true, true, true)
    {
    }
};

class RandomConcurrentReadFiles : public RandomReadFilesTestCase
{
public:
    RandomConcurrentReadFiles() NN_NOEXCEPT
        : RandomReadFilesTestCase(FsStressTest::ThreadCountMax, 5, true, true, false)
    {
    }
};

class RandomConcurrentReadFilesWithCache : public RandomReadFilesTestCase
{
public:
    RandomConcurrentReadFilesWithCache() NN_NOEXCEPT
        : RandomReadFilesTestCase(FsStressTest::ThreadCountMax, 5, true, true, true)
    {
    }
};

class InBulkReadFewerFiles : public InBulkReadFilesTestCase
{
public:
    InBulkReadFewerFiles() NN_NOEXCEPT
        : InBulkReadFilesTestCase(FsStressTest::ThreadCountMax / 2, 10, false, true, false)
    {
    }
};

class InBulkReadFewerFilesWithCache : public InBulkReadFilesTestCase
{
public:
    InBulkReadFewerFilesWithCache() NN_NOEXCEPT
        : InBulkReadFilesTestCase(FsStressTest::ThreadCountMax / 2, 10, false, true, true)
    {
    }
};

class SequentialReadFewerFiles : public SequentialReadFilesTestCase
{
public:
    SequentialReadFewerFiles() NN_NOEXCEPT
        : SequentialReadFilesTestCase(FsStressTest::ThreadCountMax / 2, 10, false, true, false)
    {
    }
};

class SequentialReadFewerFilesWithCache : public SequentialReadFilesTestCase
{
public:
    SequentialReadFewerFilesWithCache() NN_NOEXCEPT
        : SequentialReadFilesTestCase(FsStressTest::ThreadCountMax / 2, 10, false, true, true)
    {
    }
};

class RandomReadFewerFiles : public RandomReadFilesTestCase
{
public:
    RandomReadFewerFiles() NN_NOEXCEPT
        : RandomReadFilesTestCase(FsStressTest::ThreadCountMax / 2, 10, false, true, false)
    {
    }
};

class RandomReadFewerFilesWithCache : public RandomReadFilesTestCase
{
public:
    RandomReadFewerFilesWithCache() NN_NOEXCEPT
        : RandomReadFilesTestCase(FsStressTest::ThreadCountMax / 2, 10, false, true, true)
    {
    }
};

class MultiReadSingleFiles : public MultiReadSingleFilesTestCase
{
public:
    MultiReadSingleFiles() NN_NOEXCEPT
        : MultiReadSingleFilesTestCase(FsStressTest::ThreadCountMax / 2, 10, true, true, false)
    {
    }
};

class MultiReadSingleFilesWithCache : public MultiReadSingleFilesTestCase
{
public:
    MultiReadSingleFilesWithCache() NN_NOEXCEPT
        : MultiReadSingleFilesTestCase(FsStressTest::ThreadCountMax / 2, 10, true, true, true)
    {
    }
};

TEST_F(SaveDataFsStressTest, InBulkReadSingleFile)
{
    Test<InBulkReadSingleFile>(GetMountName());
}

TEST_F(SaveDataFsStressTest, SequentialReadSingleFile)
{
    Test<SequentialReadSingleFile>(GetMountName());
}

TEST_F(SaveDataFsStressTest, RandomReadSingleFile)
{
    Test<RandomReadSingleFile>(GetMountName());
}

TEST_F(SaveDataFsStressTest, InBulkReadNumerousFiles)
{
    Test<InBulkReadNumerousFiles>(GetMountName());
}

TEST_F(SaveDataFsStressTest, SequentialReadNumerousFiles)
{
    Test<SequentialReadNumerousFiles>(GetMountName());
}

TEST_F(SaveDataFsStressTest, RandomReadNumerousFiles)
{
    Test<RandomReadNumerousFiles>(GetMountName());
}

TEST_F(SaveDataFsStressTest, InBulkConcurrentReadFiles)
{
    Test<InBulkConcurrentReadFiles>(GetMountName());
}

TEST_F(SaveDataFsStressTest, SequentialConcurrentReadFiles)
{
    Test<SequentialConcurrentReadFiles>(GetMountName());
}

TEST_F(SaveDataFsStressTest, RandomConcurrentReadFiles)
{
    Test<RandomConcurrentReadFiles>(GetMountName());
}

TEST_F(SaveDataFsStressTest, MultiReadSingleFiles)
{
    Test<MultiReadSingleFiles>(GetMountName());
}

TEST_F(MultipleSaveDataFsStressTest, InBulkReadNumerousFiles)
{
    Test<InBulkReadFewerFiles, InBulkReadFewerFiles>(GetMountName(0), GetMountName(1));
}

TEST_F(MultipleSaveDataFsStressTest, SequentialReadNumerousFiles)
{
    Test<SequentialReadFewerFiles, SequentialReadFewerFiles>(GetMountName(0), GetMountName(1));
}

TEST_F(MultipleSaveDataFsStressTest, RandomReadNumerousFiles)
{
    Test<RandomReadFewerFiles, RandomReadFewerFiles>(GetMountName(0), GetMountName(1));
}

TEST_F(MultipleSaveDataFsStressTest, InBulkSequentialReadNumerousFiles)
{
    Test<InBulkReadFewerFiles, SequentialReadFewerFiles>(GetMountName(0), GetMountName(1));
}

TEST_F(MultipleSaveDataFsStressTest, SequentialRandomReadNumerousFiles)
{
    Test<SequentialReadFewerFiles, RandomReadFewerFiles>(GetMountName(0), GetMountName(1));
}

TEST_F(MultipleSaveDataFsStressTest, RandomInBulkReadNumerousFiles)
{
    Test<RandomReadFewerFiles, InBulkReadFewerFiles>(GetMountName(0), GetMountName(1));
}

TEST_F(MultipleSaveDataFsStressTest, MultiReadSingleFiles)
{
    Test<MultiReadSingleFiles, MultiReadSingleFiles>(GetMountName(0), GetMountName(1));
}

TEST_F(OtherApplicationSaveDataFsStressTest, InBulkReadSingleFile)
{
    Test<InBulkReadSingleFile>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, SequentialReadSingleFile)
{
    Test<SequentialReadSingleFile>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, RandomReadSingleFile)
{
    Test<RandomReadSingleFile>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, InBulkReadNumerousFiles)
{
    Test<InBulkReadNumerousFiles>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, SequentialReadNumerousFiles)
{
    Test<SequentialReadNumerousFiles>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, RandomReadNumerousFiles)
{
    Test<RandomReadNumerousFiles>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, InBulkConcurrentReadFiles)
{
    Test<InBulkConcurrentReadFiles>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, SequentialConcurrentReadFiles)
{
    Test<SequentialConcurrentReadFiles>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, RandomConcurrentReadFiles)
{
    Test<RandomConcurrentReadFiles>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, MultiReadSingleFiles)
{
    Test<MultiReadSingleFiles>(GetMountName());
}

TEST_F(SaveDataFsRomFsStressTest, InBulkReadNumerousFiles)
{
    Test<InBulkReadFewerFiles, InBulkReadFewerFiles>(GetMountName(0), GetMountName(1));
}

TEST_F(SaveDataFsRomFsStressTest, InBulkReadNumerousFilesWithCache)
{
    Test<InBulkReadFewerFilesWithCache, InBulkReadFewerFilesWithCache>(GetMountName(0), GetMountName(1));
}

TEST_F(SaveDataFsRomFsStressTest, SequentialReadNumerousFiles)
{
    Test<SequentialReadFewerFiles, SequentialReadFewerFiles>(GetMountName(0), GetMountName(1));
}

TEST_F(SaveDataFsRomFsStressTest, SequentialReadNumerousFilesWithCache)
{
    Test<SequentialReadFewerFilesWithCache, SequentialReadFewerFilesWithCache>(GetMountName(0), GetMountName(1));
}

TEST_F(SaveDataFsRomFsStressTest, RandomReadNumerousFiles)
{
    Test<RandomReadFewerFiles, RandomReadFewerFiles>(GetMountName(0), GetMountName(1));
}

TEST_F(SaveDataFsRomFsStressTest, RandomReadNumerousFilesWithCache)
{
    Test<RandomReadFewerFilesWithCache, RandomReadFewerFilesWithCache>(GetMountName(0), GetMountName(1));
}

TEST_F(SaveDataFsRomFsStressTest, MultiReadSingleFiles)
{
    Test<MultiReadSingleFiles, MultiReadSingleFiles>(GetMountName(0), GetMountName(1));
}

TEST_F(SaveDataFsRomFsStressTest, MultiReadSingleFilesWithCache)
{
    Test<MultiReadSingleFilesWithCache, MultiReadSingleFilesWithCache>(GetMountName(0), GetMountName(1));
}

TEST_F(RomFsStressTest, InBulkReadSingleFile)
{
    Test<InBulkReadSingleFile>(GetMountName());
}

TEST_F(RomFsStressTest, InBulkReadSingleFileWithCache)
{
    Test<InBulkReadSingleFileWithCache>(GetMountName());
}

TEST_F(RomFsStressTest, SequentialReadSingleFile)
{
    Test<SequentialReadSingleFile>(GetMountName());
}

TEST_F(RomFsStressTest, SequentialReadSingleFileWithCache)
{
    Test<SequentialReadSingleFileWithCache>(GetMountName());
}

TEST_F(RomFsStressTest, RandomReadSingleFile)
{
    Test<RandomReadSingleFile>(GetMountName());
}

TEST_F(RomFsStressTest, RandomReadSingleFileWithCache)
{
    Test<RandomReadSingleFileWithCache>(GetMountName());
}

TEST_F(RomFsStressTest, InBulkReadNumerousFiles)
{
    Test<InBulkReadNumerousFiles>(GetMountName());
}

TEST_F(RomFsStressTest, InBulkReadNumerousFilesWithCache)
{
    Test<InBulkReadNumerousFilesWithCache>(GetMountName());
}

TEST_F(RomFsStressTest, SequentialReadNumerousFiles)
{
    Test<SequentialReadNumerousFiles>(GetMountName());
}

TEST_F(RomFsStressTest, SequentialReadNumerousFilesWithCache)
{
    Test<SequentialReadNumerousFilesWithCache>(GetMountName());
}

TEST_F(RomFsStressTest, RandomReadNumerousFiles)
{
    Test<RandomReadNumerousFiles>(GetMountName());
}

TEST_F(RomFsStressTest, RandomReadNumerousFilesWithCache)
{
    Test<RandomReadNumerousFilesWithCache>(GetMountName());
}

TEST_F(RomFsStressTest, InBulkConcurrentReadFiles)
{
    Test<InBulkConcurrentReadFiles>(GetMountName());
}

TEST_F(RomFsStressTest, InBulkConcurrentReadFilesWithCache)
{
    Test<InBulkConcurrentReadFilesWithCache>(GetMountName());
}

TEST_F(RomFsStressTest, SequentialConcurrentReadFiles)
{
    Test<SequentialConcurrentReadFiles>(GetMountName());
}

TEST_F(RomFsStressTest, SequentialConcurrentReadFilesWithCache)
{
    Test<SequentialConcurrentReadFilesWithCache>(GetMountName());
}

TEST_F(RomFsStressTest, RandomConcurrentReadFiles)
{
    Test<RandomConcurrentReadFiles>(GetMountName());
}

TEST_F(RomFsStressTest, RandomConcurrentReadFilesWithCache)
{
    Test<RandomConcurrentReadFilesWithCache>(GetMountName());
}

TEST_F(RomFsStressTest, MultiReadSingleFiles)
{
    Test<MultiReadSingleFiles>(GetMountName());
}

TEST_F(RomFsStressTest, MultiReadSingleFilesWithCache)
{
    Test<MultiReadSingleFilesWithCache>(GetMountName());
}

#if defined(NNT_FS_STRESS_TEST_SUPPORTS_HOST_FS)
TEST_F(HostFsStressTest, InBulkReadSingleFile)
{
    Test<InBulkReadSingleFile>(GetMountName());
}

TEST_F(HostFsStressTest, SequentialReadSingleFile)
{
    Test<SequentialReadSingleFile>(GetMountName());
}

TEST_F(HostFsStressTest, RandomReadSingleFile)
{
    Test<RandomReadSingleFile>(GetMountName());
}

TEST_F(HostFsStressTest, InBulkReadNumerousFiles)
{
    Test<InBulkReadNumerousFiles>(GetMountName());
}

TEST_F(HostFsStressTest, SequentialReadNumerousFiles)
{
    Test<SequentialReadNumerousFiles>(GetMountName());
}

TEST_F(HostFsStressTest, RandomReadNumerousFiles)
{
    Test<RandomReadNumerousFiles>(GetMountName());
}

TEST_F(HostFsStressTest, InBulkConcurrentReadFiles)
{
    Test<InBulkConcurrentReadFiles>(GetMountName());
}

TEST_F(HostFsStressTest, SequentialConcurrentReadFiles)
{
    Test<SequentialConcurrentReadFiles>(GetMountName());
}

TEST_F(HostFsStressTest, RandomConcurrentReadFiles)
{
    Test<RandomConcurrentReadFiles>(GetMountName());
}

TEST_F(HostFsStressTest, MultiReadSingleFiles)
{
    Test<MultiReadSingleFiles>(GetMountName());
}
#endif // defined(NNT_FS_STRESS_TEST_SUPPORTS_HOST_FS)

TEST_F(SdCardFsStressTest, InBulkReadSingleFile)
{
    Test<InBulkReadSingleFile>(GetMountName());
}

TEST_F(SdCardFsStressTest, SequentialReadSingleFile)
{
    Test<SequentialReadSingleFile>(GetMountName());
}

TEST_F(SdCardFsStressTest, RandomReadSingleFile)
{
    Test<RandomReadSingleFile>(GetMountName());
}

TEST_F(SdCardFsStressTest, InBulkReadNumerousFiles)
{
    Test<InBulkReadNumerousFiles>(GetMountName());
}

TEST_F(SdCardFsStressTest, SequentialReadNumerousFiles)
{
    Test<SequentialReadNumerousFiles>(GetMountName());
}

TEST_F(SdCardFsStressTest, RandomReadNumerousFiles)
{
    Test<RandomReadNumerousFiles>(GetMountName());
}

TEST_F(SdCardFsStressTest, InBulkConcurrentReadFiles)
{
    Test<InBulkConcurrentReadFiles>(GetMountName());
}

TEST_F(SdCardFsStressTest, SequentialConcurrentReadFiles)
{
    Test<SequentialConcurrentReadFiles>(GetMountName());
}

TEST_F(SdCardFsStressTest, RandomConcurrentReadFiles)
{
    Test<RandomConcurrentReadFiles>(GetMountName());
}

TEST_F(SdCardFsStressTest, MultiReadSingleFiles)
{
    Test<MultiReadSingleFiles>(GetMountName());
}

#if defined(NNT_FS_STRESS_TEST_SUPPORTS_TEMPORARY_STORAGE)
TEST_F(TemporaryStorageFsStressTest, InBulkReadSingleFile)
{
    Test<InBulkReadSingleFile>(GetMountName());
}

TEST_F(TemporaryStorageFsStressTest, SequentialReadSingleFile)
{
    Test<SequentialReadSingleFile>(GetMountName());
}

TEST_F(TemporaryStorageFsStressTest, RandomReadSingleFile)
{
    Test<RandomReadSingleFile>(GetMountName());
}

TEST_F(TemporaryStorageFsStressTest, InBulkReadNumerousFiles)
{
    Test<InBulkReadNumerousFiles>(GetMountName());
}

TEST_F(TemporaryStorageFsStressTest, SequentialReadNumerousFiles)
{
    Test<SequentialReadNumerousFiles>(GetMountName());
}

TEST_F(TemporaryStorageFsStressTest, RandomReadNumerousFiles)
{
    Test<RandomReadNumerousFiles>(GetMountName());
}

TEST_F(TemporaryStorageFsStressTest, InBulkConcurrentReadFiles)
{
    Test<InBulkConcurrentReadFiles>(GetMountName());
}

TEST_F(TemporaryStorageFsStressTest, SequentialConcurrentReadFiles)
{
    Test<SequentialConcurrentReadFiles>(GetMountName());
}

TEST_F(TemporaryStorageFsStressTest, RandomConcurrentReadFiles)
{
    Test<RandomConcurrentReadFiles>(GetMountName());
}

TEST_F(TemporaryStorageFsStressTest, MultiReadSingleFiles)
{
    Test<MultiReadSingleFiles>(GetMountName());
}
#endif // defined(NNT_FS_STRESS_TEST_SUPPORTS_TEMPORARY_STORAGE)

}}
