﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>
#include <nnt/fsUtil/testFs_util.h>
#include "testFs_Stress_AccessDirectoriesTestCase.h"
#include "testFs_Stress_AccessDeepDirectoriesTestCase.h"

namespace nnt { namespace fs {

class ReadDirectoriesTestCase : public AccessDirectoriesTestCase
{
public:
    ReadDirectoriesTestCase(int threadCount, int directoryCount, int subentryCount) NN_NOEXCEPT
        : AccessDirectoriesTestCase(
            threadCount,
            directoryCount,
            subentryCount)
    {
    }

    virtual ~ReadDirectoriesTestCase() NN_NOEXCEPT NN_OVERRIDE {}

    virtual int GetLoopCount() const NN_NOEXCEPT NN_OVERRIDE
    {
        return 3;
    }

    virtual void Test(FsStressTest*, int threadIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        Fail(threadIndex);

        ASSERT_LE(GetEntryCount() * GetThreadCount(), FsStressTest::EntryCountMax);

        std::unique_ptr<nn::fs::DirectoryEntry[]> entries(
            new nn::fs::DirectoryEntry[GetSubentryCount()]);

        for( auto entryIndex = 0; entryIndex < GetEntryCount(); ++entryIndex )
        {
            nn::fs::DirectoryHandle directory;
            char path[PathLength];
            MakePath(path, GetEntryIndex(threadIndex, entryIndex));
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenDirectory(
                &directory,
                path,
                nn::fs::OpenDirectoryMode_All));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseDirectory(directory);
            };

            auto succeeded = false;
            ReadDirectory(
                &succeeded,
                entries.get(),
                GetSubentryCount(),
                directory,
                path);
            ASSERT_TRUE(succeeded);
        }

        Succeed(threadIndex);
    }

protected:
    virtual void ReadDirectory(
        bool* outSucceeded,
        nn::fs::DirectoryEntry* pEntries,
        int64_t count,
        nn::fs::DirectoryHandle directory,
        const char* path) NN_NOEXCEPT = 0;
};

class InBulkReadDirectoriesTestCase : public ReadDirectoriesTestCase
{
public:
    InBulkReadDirectoriesTestCase(int threadCount, int directoryCount) NN_NOEXCEPT
        : ReadDirectoriesTestCase(threadCount, directoryCount, 10)
    {
    }

    virtual ~InBulkReadDirectoriesTestCase() NN_NOEXCEPT NN_OVERRIDE {}

protected:
    virtual void ReadDirectory(
        bool* outSucceeded,
        nn::fs::DirectoryEntry* pEntries,
        int64_t count,
        nn::fs::DirectoryHandle directory,
        const char* path) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(outSucceeded);
        *outSucceeded = false;

        int64_t entryCount = 0;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetDirectoryEntryCount(&entryCount, directory));
        ASSERT_EQ(count, entryCount);

        int64_t readCount = 0;
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadDirectory(&readCount, pEntries, directory, count));
        EXPECT_EQ(count, readCount);

        for( auto entryIndex = 0; entryIndex < count; ++entryIndex )
        {
            char subentryPath[PathLength];
            nn::util::SNPrintf(subentryPath, PathLength, "%s/%s", path, pEntries[entryIndex].name);

            nn::fs::DirectoryEntryType type;
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetEntryType(&type, subentryPath));
            EXPECT_EQ(pEntries[entryIndex].directoryEntryType, type);
        }

        *outSucceeded = true;
    }
};

class SequentialReadDirectoriesTestCase : public ReadDirectoriesTestCase
{
public:
    SequentialReadDirectoriesTestCase(int threadCount, int directoryCount) NN_NOEXCEPT
        : ReadDirectoriesTestCase(threadCount, directoryCount, 10)
    {
    }

    virtual ~SequentialReadDirectoriesTestCase() NN_NOEXCEPT NN_OVERRIDE {}

protected:
    virtual void ReadDirectory(
        bool* outSucceeded,
        nn::fs::DirectoryEntry* pEntries,
        int64_t count,
        nn::fs::DirectoryHandle directory,
        const char* path) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(outSucceeded);
        *outSucceeded = false;

        int64_t entryCount = 0;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetDirectoryEntryCount(&entryCount, directory));
        ASSERT_EQ(count, entryCount);

        for( auto entryIndex = 0; entryIndex < count; ++entryIndex )
        {
            int64_t readCount = 0;
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadDirectory(
                &readCount,
                pEntries + entryIndex,
                directory,
                1));
            EXPECT_EQ(1, readCount);

            char subentryPath[PathLength];
            nn::util::SNPrintf(subentryPath, PathLength, "%s/%s", path, pEntries[entryIndex].name);

            nn::fs::DirectoryEntryType type;
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetEntryType(&type, subentryPath));
            EXPECT_EQ(pEntries[entryIndex].directoryEntryType, type);
        }

        *outSucceeded = true;
    }
};

class InBulkReadDirectories : public InBulkReadDirectoriesTestCase
{
public:
    InBulkReadDirectories() NN_NOEXCEPT
        : InBulkReadDirectoriesTestCase(FsStressTest::ThreadCountMax, 10)
    {
    }
};

class InBulkReadFewerDirectories : public InBulkReadDirectoriesTestCase
{
public:
    InBulkReadFewerDirectories() NN_NOEXCEPT
        : InBulkReadDirectoriesTestCase(FsStressTest::ThreadCountMax / 2, 10)
    {
    }
};

class SequentialReadDirectories : public SequentialReadDirectoriesTestCase
{
public:
    SequentialReadDirectories() NN_NOEXCEPT
        : SequentialReadDirectoriesTestCase(FsStressTest::ThreadCountMax, 10)
    {
    }
};

class SequentialReadFewerDirectories : public SequentialReadDirectoriesTestCase
{
public:
    SequentialReadFewerDirectories() NN_NOEXCEPT
        : SequentialReadDirectoriesTestCase(FsStressTest::ThreadCountMax / 2, 10)
    {
    }
};

class ReadDeepDirectoriesTestCase : public AccessDeepDirectoriesTestCase
{
public:
    static const auto DirectoryDepthMax = 30;

public:
    explicit ReadDeepDirectoriesTestCase(int threadCount) NN_NOEXCEPT
        : AccessDeepDirectoriesTestCase(threadCount, DirectoryDepthMax)
    {
    }

    virtual ~ReadDeepDirectoriesTestCase() NN_NOEXCEPT NN_OVERRIDE {}

    virtual int GetLoopCount() const NN_NOEXCEPT NN_OVERRIDE
    {
        return 3;
    }

    virtual void Test(FsStressTest*, int threadIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        Fail(threadIndex);

        std::unique_ptr<char[]> path(new char[PathLength]);
        std::unique_ptr<nn::fs::DirectoryEntry> entry(new nn::fs::DirectoryEntry);
        MakePath(path.get(), PathLength, threadIndex, 0);

        for( auto depth = 0; ; ++depth )
        {
            nn::fs::DirectoryHandle directory;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenDirectory(
                &directory,
                path.get(),
                nn::fs::OpenDirectoryMode_All));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseDirectory(directory);
            };

            auto currentLength = strnlen(path.get(), PathLength - 1);
            auto foundSubdirectory = false;
            for( ; ; )
            {
                int64_t readCount = 0;
                NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadDirectory(
                    &readCount,
                    entry.get(),
                    directory,
                    1));
                if( readCount == 0 )
                {
                    break;
                }
                else if( entry->directoryEntryType == nn::fs::DirectoryEntryType_Directory )
                {
                    EXPECT_FALSE(foundSubdirectory) << entry->name << ' ' << path.get();
                    currentLength += nn::util::SNPrintf(
                        path.get() + currentLength,
                        PathLength - currentLength,
                        "/%s", entry->name);
                    foundSubdirectory = true;
                }
            }

            if( !foundSubdirectory )
            {
                EXPECT_EQ(DirectoryDepthMax, depth + 1);
                break;
            }
        }

        Succeed(threadIndex);
    }
};

NN_DEFINE_STATIC_CONSTANT(const int ReadDeepDirectoriesTestCase::DirectoryDepthMax);

class ReadDeepDirectories : public ReadDeepDirectoriesTestCase
{
public:
    ReadDeepDirectories() NN_NOEXCEPT
        : ReadDeepDirectoriesTestCase(FsStressTest::ThreadCountMax)
    {
    }
};

class ReadFewerDeepDirectories : public ReadDeepDirectoriesTestCase
{
public:
    ReadFewerDeepDirectories() NN_NOEXCEPT
        : ReadDeepDirectoriesTestCase(FsStressTest::ThreadCountMax / 2)
    {
    }
};

TEST_F(SaveDataFsStressTest, InBulkReadDirectories)
{
    Test<InBulkReadDirectories>(GetMountName());
}

TEST_F(SaveDataFsStressTest, SequentialReadDirectories)
{
    Test<SequentialReadDirectories>(GetMountName());
}

TEST_F(SaveDataFsStressTest, ReadDeepDirectories)
{
    Test<ReadDeepDirectories>(GetMountName());
}

TEST_F(MultipleSaveDataFsStressTest, InBulkReadDirectories)
{
    Test<InBulkReadFewerDirectories, InBulkReadFewerDirectories>(GetMountName(0), GetMountName(1));
}

TEST_F(MultipleSaveDataFsStressTest, SequentialReadDirectories)
{
    Test<SequentialReadFewerDirectories, SequentialReadFewerDirectories>(
        GetMountName(0),
        GetMountName(1));
}

TEST_F(MultipleSaveDataFsStressTest, ReadDeepDirectories)
{
    Test<ReadFewerDeepDirectories, ReadFewerDeepDirectories>(GetMountName(0), GetMountName(1));
}

TEST_F(OtherApplicationSaveDataFsStressTest, InBulkReadDirectories)
{
    Test<InBulkReadDirectories>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, SequentialReadDirectories)
{
    Test<SequentialReadDirectories>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, ReadDeepDirectories)
{
    Test<ReadDeepDirectories>(GetMountName());
}

TEST_F(SaveDataFsRomFsStressTest, InBulkReadDirectories)
{
    Test<InBulkReadFewerDirectories, InBulkReadFewerDirectories>(GetMountName(0), GetMountName(1));
}

TEST_F(SaveDataFsRomFsStressTest, SequentialReadDirectories)
{
    Test<SequentialReadFewerDirectories, SequentialReadFewerDirectories>(
        GetMountName(0),
        GetMountName(1));
}

TEST_F(SaveDataFsRomFsStressTest, ReadDeepDirectories)
{
    Test<ReadFewerDeepDirectories, ReadFewerDeepDirectories>(GetMountName(0), GetMountName(1));
}

TEST_F(RomFsStressTest, InBulkReadDirectories)
{
    Test<InBulkReadDirectories>(GetMountName());
}

TEST_F(RomFsStressTest, SequentialReadDirectories)
{
    Test<SequentialReadDirectories>(GetMountName());
}

TEST_F(RomFsStressTest, ReadDeepDirectories)
{
    Test<ReadDeepDirectories>(GetMountName());
}

#if defined(NNT_FS_STRESS_TEST_SUPPORTS_HOST_FS)
TEST_F(HostFsStressTest, InBulkReadDirectories)
{
    Test<InBulkReadDirectories>(GetMountName());
}

TEST_F(HostFsStressTest, SequentialReadDirectories)
{
    Test<SequentialReadDirectories>(GetMountName());
}

TEST_F(HostFsStressTest, ReadDeepDirectories)
{
    Test<ReadDeepDirectories>(GetMountName());
}
#endif // defined(NNT_FS_STRESS_TEST_SUPPORTS_HOST_FS)

TEST_F(SdCardFsStressTest, InBulkReadDirectories)
{
    Test<InBulkReadDirectories>(GetMountName());
}

TEST_F(SdCardFsStressTest, SequentialReadDirectories)
{
    Test<SequentialReadDirectories>(GetMountName());
}

TEST_F(SdCardFsStressTest, ReadDeepDirectories)
{
    Test<ReadDeepDirectories>(GetMountName());
}

#if defined(NNT_FS_STRESS_TEST_SUPPORTS_TEMPORARY_STORAGE)
TEST_F(TemporaryStorageFsStressTest, InBulkReadDirectories)
{
    Test<InBulkReadDirectories>(GetMountName());
}

TEST_F(TemporaryStorageFsStressTest, SequentialReadDirectories)
{
    Test<SequentialReadDirectories>(GetMountName());
}

TEST_F(TemporaryStorageFsStressTest, ReadDeepDirectories)
{
    Test<ReadDeepDirectories>(GetMountName());
}
#endif // defined(NNT_FS_STRESS_TEST_SUPPORTS_TEMPORARY_STORAGE)

}}
