﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testFs_Stress_StressTest.h"

namespace nnt { namespace fs {

namespace {
    const auto FileSizeMin = 1;
    const auto FileSizeMax = 16 * 1024;
    const auto EntryCountMin = 1;
    const auto EntryCountMax = 5;

    const auto PathLength = 64;

    const auto FilePrefix = "file";
    const auto DirectoryPrefix = "dir";

    std::mt19937 g_Rngs[FsStressTest::ThreadCountMax];
    char g_Buffer[FileSizeMax * FsStressTest::ThreadCountMax];
    nn::fs::DirectoryEntry g_Entries[EntryCountMax];
}

class OperateVariouslyTestCase : public FsStressTest::TestCase
{
public:
    explicit OperateVariouslyTestCase(int threadCount) NN_NOEXCEPT
        : m_ThreadCount(threadCount),
          m_FileCount(0),
          m_DirectoryCount(0)
    {
    }

    virtual ~OperateVariouslyTestCase() NN_NOEXCEPT NN_OVERRIDE {}

    virtual int GetLoopCount() const NN_NOEXCEPT NN_OVERRIDE
    {
        return 1000;
    }

    virtual int GetThreadCount() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_ThreadCount;
    }

    virtual void SetUp(FsStressTest* pTest) NN_NOEXCEPT NN_OVERRIDE
    {
        FailAll();

        for( auto& rng : g_Rngs )
        {
            rng.seed(nnt::fs::util::GetRandomSeed());
        }

        if( pTest->IsReadOnly(GetTestCaseIndex()) )
        {
            char path[PathLength];
            nn::util::SNPrintf(
                path,
                sizeof(path),
                "%s:%s/",
                GetMountName(),
                GetTestDirectoryPath());

            nn::fs::DirectoryHandle directory;
            int64_t count = 0;
            {
                NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenDirectory(
                    &directory,
                    path,
                    nn::fs::OpenDirectoryMode_File));
                NN_UTIL_SCOPE_EXIT
                {
                    nn::fs::CloseDirectory(directory);
                };
                NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetDirectoryEntryCount(&count, directory));
                m_FileCount = static_cast<int>(count);
            }
            {
                NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenDirectory(
                    &directory,
                    path,
                    nn::fs::OpenDirectoryMode_Directory));
                NN_UTIL_SCOPE_EXIT
                {
                    nn::fs::CloseDirectory(directory);
                };
                NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetDirectoryEntryCount(&count, directory));
                m_DirectoryCount = static_cast<int>(count);
            }
        }

        SucceedAll();
    }

    virtual void Test(FsStressTest* pTest, int threadIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        Fail(threadIndex);

        enum class Operation
        {
            ReadDirectory,
            ReadFile,
            ReadOnlyCount,

            CreateDirectory = ReadOnlyCount,
            DeleteDirectory,
            CreateFile,
            DeleteFile,
            ResizeFile,
            WriteFile,
            WritableCount,
        };

        for( ; ; )
        {
            const auto operationMax = pTest->IsReadOnly(GetTestCaseIndex())
                ? static_cast<int>(Operation::ReadOnlyCount) - 1
                : static_cast<int>(Operation::WritableCount) - 1;
            std::uniform_int_distribution<> distribution(0, operationMax);

            switch( static_cast<Operation>(distribution(g_Rngs[threadIndex])) )
            {
            case Operation::ReadDirectory:
                {
                    const auto result = ReadDirectory(pTest, threadIndex);
                    if( !nn::fs::ResultPathNotFound::Includes(result) )
                    {
                        NNT_ASSERT_RESULT_SUCCESS(result);
                        Succeed(threadIndex);
                        return;
                    }
                }
                break;
            case Operation::ReadFile:
                {
                    const auto result = ReadFile(pTest, threadIndex);
                    if( !nn::fs::ResultPathNotFound::Includes(result)
                        && !nn::fs::ResultTargetLocked::Includes(result) )
                    {
                        NNT_ASSERT_RESULT_SUCCESS(result);
                        Succeed(threadIndex);
                        return;
                    }
                }
                break;
            case Operation::CreateDirectory:
                NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(pTest, threadIndex));
                Succeed(threadIndex);
                return;
            case Operation::DeleteDirectory:
                {
                    const auto result = DeleteDirectory(pTest, threadIndex);
                    if( !nn::fs::ResultTargetLocked::Includes(result)
                        && !nn::fs::ResultTargetNotFound::Includes(result) )
                    {
                        NNT_ASSERT_RESULT_SUCCESS(result);
                        Succeed(threadIndex);
                        return;
                    }
                }
                break;
            case Operation::CreateFile:
                NNT_ASSERT_RESULT_SUCCESS(CreateFile(pTest, threadIndex));
                Succeed(threadIndex);
                return;
            case Operation::DeleteFile:
                {
                    const auto result = DeleteFile(pTest, threadIndex);
                    if( !nn::fs::ResultTargetLocked::Includes(result)
                        && !nn::fs::ResultTargetNotFound::Includes(result) )
                    {
                        NNT_ASSERT_RESULT_SUCCESS(result);
                        Succeed(threadIndex);
                        return;
                    }
                }
                break;
            case Operation::ResizeFile:
                {
                    const auto result = ResizeFile(pTest, threadIndex);
                    if( !nn::fs::ResultPathNotFound::Includes(result)
                        && !nn::fs::ResultTargetLocked::Includes(result) )
                    {
                        NNT_ASSERT_RESULT_SUCCESS(result);
                        Succeed(threadIndex);
                        return;
                    }
                }
                break;
            case Operation::WriteFile:
                {
                    const auto result = WriteFile(pTest, threadIndex);
                    if( !nn::fs::ResultPathNotFound::Includes(result)
                        && !nn::fs::ResultTargetLocked::Includes(result) )
                    {
                        NNT_ASSERT_RESULT_SUCCESS(result);
                        Succeed(threadIndex);
                        return;
                    }
                }
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
    } // NOLINT(impl/function_size)

private:
    nn::Result ReadDirectory(FsStressTest* pTest, int threadIndex) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(0 < m_DirectoryCount, nn::fs::ResultPathNotFound());

        auto& rng = g_Rngs[threadIndex];
        const auto index = std::uniform_int_distribution<>(0, m_DirectoryCount - 1)(rng);

        char path[PathLength];
        MakePath(path, DirectoryPrefix, index);

        nn::fs::DirectoryHandle directory;
        {
            std::lock_guard<nn::os::Mutex> lock(pTest->GetMutex());
            NN_RESULT_DO(nn::fs::OpenDirectory(&directory, path, nn::fs::OpenDirectoryMode_All));
        }
        NN_UTIL_SCOPE_EXIT
        {
            std::lock_guard<nn::os::Mutex> lock(pTest->GetMutex());
            nn::fs::CloseDirectory(directory);
        };

        {
            std::lock_guard<nn::os::Mutex> lock(pTest->GetMutex());
            int64_t count = 0;
            NN_RESULT_DO(nn::fs::ReadDirectory(&count, g_Entries, directory, EntryCountMax));
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result ReadFile(FsStressTest* pTest, int threadIndex) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(0 < m_FileCount, nn::fs::ResultPathNotFound());

        auto& rng = g_Rngs[threadIndex];
        const auto index = std::uniform_int_distribution<>(0, m_FileCount - 1)(rng);

        char path[PathLength];
        MakePath(path, FilePrefix, index);

        nn::fs::FileHandle file;
        {
            std::lock_guard<nn::os::Mutex> lock(pTest->GetMutex());
            NN_RESULT_DO(nn::fs::OpenFile(&file, path, nn::fs::OpenMode_Read));
        }
        NN_UTIL_SCOPE_EXIT
        {
            std::lock_guard<nn::os::Mutex> lock(pTest->GetMutex());
            nn::fs::CloseFile(file);
        };

        int64_t fileSize = 0;
        {
            std::lock_guard<nn::os::Mutex> lock(pTest->GetMutex());
            NN_RESULT_DO(nn::fs::GetFileSize(&fileSize, file));
        }
        {
            std::lock_guard<nn::os::Mutex> lock(pTest->GetMutex());
            const auto offset = std::uniform_int_distribution<int64_t>(0, fileSize - 1)(rng);
            const auto sizeMax = static_cast<size_t>(fileSize - offset);
            const auto size = std::uniform_int_distribution<size_t>(1, sizeMax)(rng);
            NN_RESULT_DO(nn::fs::ReadFile(file, offset, g_Buffer + FileSizeMax * threadIndex, size));
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result CreateDirectory(FsStressTest* pTest, int threadIndex) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(pTest->GetMutex());

        char path[PathLength];
        NN_RESULT_DO(nn::fs::CreateDirectory(MakePath(path, DirectoryPrefix, m_DirectoryCount)));

        ++m_DirectoryCount;

        auto& rng = g_Rngs[threadIndex];
        const auto count = std::uniform_int_distribution<>(EntryCountMin, EntryCountMax)(rng);
        const auto pathLength = strnlen(path, sizeof(path));
        for( auto index = 0; index < count; ++index )
        {
            nn::util::SNPrintf(path + pathLength, sizeof(path) - pathLength, "/sub%d", index);
            NN_RESULT_DO(nn::fs::CreateFile(path, 0));
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result DeleteDirectory(FsStressTest* pTest, int) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(pTest->GetMutex());

        NN_RESULT_THROW_UNLESS(0 < m_DirectoryCount, nn::fs::ResultTargetNotFound());

        char path[PathLength];
        MakePath(path, DirectoryPrefix, m_DirectoryCount - 1);
        NN_RESULT_DO(nn::fs::DeleteDirectoryRecursively(path));

        --m_DirectoryCount;
        NN_RESULT_SUCCESS;
    }

    nn::Result CreateFile(FsStressTest* pTest, int threadIndex) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(pTest->GetMutex());

        auto& rng = g_Rngs[threadIndex];
        const auto size = std::uniform_int_distribution<int64_t>(FileSizeMin, FileSizeMax)(rng);

        char path[PathLength];
        NN_RESULT_DO(nn::fs::CreateFile(MakePath(path, FilePrefix, m_FileCount), size));

        ++m_FileCount;
        NN_RESULT_SUCCESS;
    }

    nn::Result DeleteFile(FsStressTest* pTest, int) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(pTest->GetMutex());

        NN_RESULT_THROW_UNLESS(0 < m_FileCount, nn::fs::ResultTargetNotFound());

        char path[PathLength];
        NN_RESULT_DO(nn::fs::DeleteFile(MakePath(path, FilePrefix, m_FileCount - 1)));

        --m_FileCount;
        NN_RESULT_SUCCESS;
    }

    nn::Result ResizeFile(FsStressTest* pTest, int threadIndex) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(0 < m_FileCount, nn::fs::ResultPathNotFound());

        auto& rng = g_Rngs[threadIndex];
        const auto index = std::uniform_int_distribution<>(0, m_FileCount - 1)(rng);

        char path[PathLength];
        MakePath(path, FilePrefix, index);

        nn::fs::FileHandle file;
        {
            std::lock_guard<nn::os::Mutex> lock(pTest->GetMutex());
            NN_RESULT_DO(nn::fs::OpenFile(&file, path, nn::fs::OpenMode_Write));
        }
        NN_UTIL_SCOPE_EXIT
        {
            std::lock_guard<nn::os::Mutex> lock(pTest->GetMutex());
            nn::fs::CloseFile(file);
        };

        const auto size = std::uniform_int_distribution<int64_t>(FileSizeMin, FileSizeMax)(rng);
        {
            std::lock_guard<nn::os::Mutex> lock(pTest->GetMutex());
            NN_RESULT_DO(nn::fs::SetFileSize(file, size));
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result WriteFile(FsStressTest* pTest, int threadIndex) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(0 < m_FileCount, nn::fs::ResultPathNotFound());

        auto& rng = g_Rngs[threadIndex];
        const auto index = std::uniform_int_distribution<>(0, m_FileCount - 1)(rng);

        char path[PathLength];
        MakePath(path, FilePrefix, index);

        nn::fs::FileHandle file;
        {
            std::lock_guard<nn::os::Mutex> lock(pTest->GetMutex());
            NN_RESULT_DO(nn::fs::OpenFile(&file, path, nn::fs::OpenMode_Write));
        }
        NN_UTIL_SCOPE_EXIT
        {
            std::lock_guard<nn::os::Mutex> lock(pTest->GetMutex());
            nn::fs::CloseFile(file);
        };

        int64_t fileSize = 0;
        {
            std::lock_guard<nn::os::Mutex> lock(pTest->GetMutex());
            NN_RESULT_DO(nn::fs::GetFileSize(&fileSize, file));
        }
        {
            std::lock_guard<nn::os::Mutex> lock(pTest->GetMutex());
            const auto offset = std::uniform_int_distribution<int64_t>(0, fileSize - 1)(rng);
            const auto sizeMax = static_cast<size_t>(fileSize - offset);
            const auto size = std::uniform_int_distribution<size_t>(1, sizeMax)(rng);
            NN_RESULT_DO(nn::fs::WriteFile(
                file,
                offset,
                g_Buffer + FileSizeMax * threadIndex,
                size,
                nn::fs::WriteOption()));
        }
        {
            std::lock_guard<nn::os::Mutex> lock(pTest->GetMutex());
            NN_RESULT_DO(nn::fs::FlushFile(file));
        }

        NN_RESULT_SUCCESS;
    }

    template<size_t Length>
    const char* MakePath(char (&path)[Length], const char* pPrefix, int index) const NN_NOEXCEPT
    {
        return MakePath(path, Length, pPrefix, index);
    }

    const char* MakePath(
        char* path,
        size_t length,
        const char* pPrefix,
        int index) const NN_NOEXCEPT
    {
        nn::util::SNPrintf(
            path,
            length,
            "%s:%s/%s%d",
            GetMountName(),
            GetTestDirectoryPath(),
            pPrefix,
            index);
        return path;
    }

private:
    int m_ThreadCount;
    std::atomic<int> m_FileCount;
    std::atomic<int> m_DirectoryCount;
};

class OperateVariously : public OperateVariouslyTestCase
{
public:
    OperateVariously() NN_NOEXCEPT
        : OperateVariouslyTestCase(FsStressTest::ThreadCountMax)
    {
    }
};

class OperateVariouslyFewerThreads : public OperateVariouslyTestCase
{
public:
    OperateVariouslyFewerThreads() NN_NOEXCEPT
        : OperateVariouslyTestCase(FsStressTest::ThreadCountMax / 2)
    {
    }
};

TEST_F(SaveDataFsStressTest, OperateVariously)
{
    Test<OperateVariously>(GetMountName());
}

TEST_F(MultipleSaveDataFsStressTest, OperateVariously)
{
    Test<OperateVariouslyFewerThreads, OperateVariouslyFewerThreads>(
        GetMountName(0),
        GetMountName(1));
}

TEST_F(OtherApplicationSaveDataFsStressTest, OperateVariously)
{
    Test<OperateVariously>(GetMountName());
}

TEST_F(SaveDataFsRomFsStressTest, OperateVariously)
{
    Test<OperateVariouslyFewerThreads, OperateVariouslyFewerThreads>(
        GetMountName(0),
        GetMountName(1));
}

TEST_F(RomFsStressTest, OperateVariously)
{
    Test<OperateVariously>(GetMountName());
}

#if defined(NNT_FS_STRESS_TEST_SUPPORTS_HOST_FS)
TEST_F(HostFsStressTest, OperateVariously)
{
    Test<OperateVariously>(GetMountName());
}
#endif // defined(NNT_FS_STRESS_TEST_SUPPORTS_HOST_FS)

TEST_F(SdCardFsStressTest, OperateVariously)
{
    Test<OperateVariously>(GetMountName());
}

#if defined(NNT_FS_STRESS_TEST_SUPPORTS_TEMPORARY_STORAGE)
TEST_F(TemporaryStorageFsStressTest, OperateVariously)
{
    Test<OperateVariously>(GetMountName());
}
#endif // defined(NNT_FS_STRESS_TEST_SUPPORTS_TEMPORARY_STORAGE)

}}
