﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testFs_Stress_OpenEntriesTestCase.h"

namespace nnt { namespace fs {

class OpenFilesTestCase : public OpenEntriesTestCase
{
public:
    OpenFilesTestCase(int threadCount, int fileCount) NN_NOEXCEPT
        : OpenEntriesTestCase(threadCount, fileCount, nn::fs::DirectoryEntryType_File)
    {
    }

    virtual ~OpenFilesTestCase() NN_NOEXCEPT NN_OVERRIDE {}

    virtual void Test(FsStressTest*, int threadIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        Fail(threadIndex);
        ASSERT_LE(GetEntryCount() * GetThreadCount(), FsStressTest::EntryCountMax);
        auto openedEntryCount = 0;
        NN_UTIL_SCOPE_EXIT
        {
            for( auto entryIndex = 0; entryIndex < openedEntryCount; ++entryIndex )
            {
                nn::fs::CloseFile(m_Files[threadIndex * GetEntryCount() + entryIndex]);
            }
        };
        for( auto entryIndex = 0; entryIndex < GetEntryCount(); ++entryIndex )
        {
            char path[PathLength];
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(
                m_Files + GetEntryIndex(threadIndex, entryIndex),
                MakePath(path, GetEntryIndex(threadIndex, entryIndex)),
                nn::fs::OpenMode_Read));
            ++openedEntryCount;
        }
        Succeed(threadIndex);
    }

private:
    nn::fs::FileHandle m_Files[FsStressTest::EntryCountMax];
};

class OpenSingleFile : public OpenFilesTestCase
{
public:
    OpenSingleFile() NN_NOEXCEPT
        : OpenFilesTestCase(FsStressTest::ThreadCountMax, 1)
    {
    }

    virtual int GetLoopCount() const NN_NOEXCEPT NN_OVERRIDE
    {
        return 1000;
    }
};

class OpenNumerousFiles : public OpenFilesTestCase
{
public:
    OpenNumerousFiles() NN_NOEXCEPT
        : OpenFilesTestCase(FsStressTest::ThreadCountMax, 10)
    {
    }

    virtual int GetLoopCount() const NN_NOEXCEPT NN_OVERRIDE
    {
        return 200;
    }
};

class OpenFewerFiles : public OpenFilesTestCase
{
public:
    OpenFewerFiles() NN_NOEXCEPT
        : OpenFilesTestCase(FsStressTest::ThreadCountMax / 2, 10)
    {
    }

    virtual int GetLoopCount() const NN_NOEXCEPT NN_OVERRIDE
    {
        return 200;
    }
};

TEST_F(SaveDataFsStressTest, OpenSingleFile)
{
    Test<OpenSingleFile>(GetMountName());
}

TEST_F(SaveDataFsStressTest, OpenNumerousFiles)
{
    Test<OpenNumerousFiles>(GetMountName());
}

TEST_F(MultipleSaveDataFsStressTest, OpenNumerousFiles)
{
    Test<OpenFewerFiles, OpenFewerFiles>(GetMountName(0), GetMountName(1));
}

TEST_F(OtherApplicationSaveDataFsStressTest, OpenSingleFile)
{
    Test<OpenSingleFile>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, OpenNumerousFiles)
{
    Test<OpenNumerousFiles>(GetMountName());
}

TEST_F(SaveDataFsRomFsStressTest, OpenNumerousFiles)
{
    Test<OpenFewerFiles, OpenFewerFiles>(GetMountName(0), GetMountName(1));
}

TEST_F(RomFsStressTest, OpenSingleFile)
{
    Test<OpenSingleFile>(GetMountName());
}

TEST_F(RomFsStressTest, OpenNumerousFiles)
{
    Test<OpenNumerousFiles>(GetMountName());
}

#if defined(NNT_FS_STRESS_TEST_SUPPORTS_HOST_FS)
TEST_F(HostFsStressTest, OpenSingleFile)
{
    Test<OpenSingleFile>(GetMountName());
}

TEST_F(HostFsStressTest, OpenNumerousFiles)
{
    Test<OpenNumerousFiles>(GetMountName());
}
#endif // defined(NNT_FS_STRESS_TEST_SUPPORTS_HOST_FS)

TEST_F(SdCardFsStressTest, OpenSingleFile)
{
    Test<OpenSingleFile>(GetMountName());
}

TEST_F(SdCardFsStressTest, OpenNumerousFiles)
{
    Test<OpenNumerousFiles>(GetMountName());
}

#if defined(NNT_FS_STRESS_TEST_SUPPORTS_TEMPORARY_STORAGE)
TEST_F(TemporaryStorageFsStressTest, OpenSingleFile)
{
    Test<OpenSingleFile>(GetMountName());
}

TEST_F(TemporaryStorageFsStressTest, OpenNumerousFiles)
{
    Test<OpenNumerousFiles>(GetMountName());
}
#endif // defined(NNT_FS_STRESS_TEST_SUPPORTS_TEMPORARY_STORAGE)

}}
