﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs/fs_SaveDataForDebug.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include "testFs_Stress_StressTest.h"

namespace nnt { namespace fs {

class MountFileSystemsTestBase : public FsStressTest::TestCase
{
public:
    virtual ~MountFileSystemsTestBase() NN_NOEXCEPT NN_OVERRIDE {}

    virtual int GetLoopCount() const NN_NOEXCEPT NN_OVERRIDE
    {
        return 500;
    }

    virtual void SetUp(FsStressTest* pTest) NN_NOEXCEPT NN_OVERRIDE
    {
        FsStressTest::TestCase::SetUp(pTest);

        FailAll();
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&m_CacheSize));
        m_CacheBuffer.reset(new char[m_CacheSize]);
        SucceedAll();
    }

protected:
    char* GetCacheBuffer() NN_NOEXCEPT
    {
        return m_CacheBuffer.get();
    }

    size_t GetCacheSize() const NN_NOEXCEPT
    {
        return m_CacheSize;
    }

private:
    std::unique_ptr<char[]> m_CacheBuffer;
    size_t m_CacheSize;
};

class MountFileSystems : public MountFileSystemsTestBase
{
public:
    virtual void Test(FsStressTest*, int threadIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        Fail(threadIndex);
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataForDebug("save"));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount("save");
        };
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountRom("rom", GetCacheBuffer(), GetCacheSize()));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount("rom");
        };
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHost("host", nnt::fs::util::GetHostTemporaryPath()));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount("host");
        };
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHostRoot());
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::UnmountHostRoot();
        };
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSdCardForDebug("sdcard"));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount("sdcard");
        };
        Succeed(threadIndex);
    }
};

class MountFileSystemsAndCreateDeleteFile : public MountFileSystemsTestBase
{
public:
    virtual ~MountFileSystemsAndCreateDeleteFile() NN_NOEXCEPT NN_OVERRIDE {}

    virtual void Test(FsStressTest*, int threadIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        Fail(threadIndex);

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataForDebug("test"));
        CreateDeleteFileAndUnmount();

#if defined(NNT_FS_STRESS_TEST_SUPPORTS_HOST_FS)
        {
            nnt::fs::util::TemporaryHostDirectory directory;
            directory.Create();
            NN_UTIL_SCOPE_EXIT
            {
                directory.Delete();
            };

            NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHost("test", directory.GetPath().c_str()));
            CreateDeleteFileAndUnmount();
        }
#endif // defined(NNT_FS_STRESS_TEST_SUPPORTS_HOST_FS)

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSdCardForDebug("test"));
        CreateDeleteFileAndUnmount();

        Succeed(threadIndex);
    }

    virtual void SetUp(FsStressTest* pTest) NN_NOEXCEPT NN_OVERRIDE
    {
        MountFileSystemsTestBase::SetUp(pTest);

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSdCardForDebug("test"));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount("test");
        };

        nnt::fs::util::DeleteFileOrDirectoryIfExists(GetFilePath());
    }

private:
    static const char* GetFilePath() NN_NOEXCEPT
    {
#if defined(NNT_FS_STRESS_TEST_A_EXCLUDING_ENSURESAVEDATA)
        const char* TestFilePath = "test:/testFile1";
#elif defined(NNT_FS_STRESS_TEST_A_EXCLUDING_ENSURESAVEDATA2)
        const char* TestFilePath = "test:/testFile2";
#else
        const char* TestFilePath = "test:/testFile";
#endif
        return TestFilePath;
    }

    void CreateDeleteFileAndUnmount() NN_NOEXCEPT
    {
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount("test");
        };
        {
            const size_t TestFileSize = 16 * 1024;
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fs::CreateFile(GetFilePath(), TestFileSize)
            );
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fs::DeleteFile(GetFilePath())
            );
        }
    }
};

TEST_F(FsStressTest, MountFileSystems)
{
    Test<MountFileSystems>(nullptr);
}

TEST_F(FsStressTest, MountFileSystemsAndCreateDeleteFile)
{
    Test<MountFileSystemsAndCreateDeleteFile>(nullptr);
}

}}
