﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <random>

#include <nn/nn_TimeSpan.h>
#include <nn/fs/fs_AddOnContent.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileDataCache.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_MemoryManagement.h>
#include <nn/fs/fs_Rom.h>
#include <nn/os/os_ThreadApi.h>
#include <nn/os/os_ThreadCommon.h>
#include <nn/os/os_ThreadTypes.h>
#include <nn/util/util_ScopeExit.h>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util_allocator.h>
#include <nnt/fsUtil/testFs_util_function.h>
#include <nnt/fsUtil/testFs_util_GoogleTestUtility.h>
#include <nnt/result/testResult_Assert.h>

namespace {

const size_t StackSize = 32 * 1024;
const int ThreadCount = 8;
NN_OS_ALIGNAS_THREAD_STACK char s_ThreadStack[StackSize * ThreadCount] = {};

class MountRomTest : public ::testing::Test
{
public:
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        SetUp(2 * 1024 * 1024 + 1);
    }

    void SetUp(int64_t fileSize) NN_NOEXCEPT
    {
        size_t mountRomCacheBufferSize;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&mountRomCacheBufferSize));
        m_MountRomCacheBuffer.reset(new char[mountRomCacheBufferSize]);
        ASSERT_TRUE(m_MountRomCacheBuffer);
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountRom("rom", m_MountRomCacheBuffer.get(), mountRomCacheBufferSize));

        char filePath[256];
        nn::util::SNPrintf(filePath, sizeof(filePath), "rom:/%lld", fileSize);
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&m_FileHandle, filePath, nn::fs::OpenMode_Read));
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&m_FileSize, m_FileHandle));
        ASSERT_EQ(m_FileSize, fileSize);
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::fs::CloseFile(m_FileHandle);
        nn::fs::Unmount("rom");
    }

    nn::fs::FileHandle GetFileHandle() NN_NOEXCEPT
    {
        return m_FileHandle;
    }

    int64_t GetFileSize() const NN_NOEXCEPT
    {
        return m_FileSize;
    }

    void RunTest(nn::os::ThreadFunction func) NN_NOEXCEPT
    {
        // シングルスレッドで実行
        func(this);

        // マルチスレッドで実行
        nn::os::ThreadType threads[ThreadCount];

        int createdThreadCount = 0;
        NN_UTIL_SCOPE_EXIT
        {
            for( int i = 0; i < createdThreadCount; ++i )
            {
                nn::os::DestroyThread(&threads[i]);
            }
        };

        for( int i = 0; i < ThreadCount; ++i )
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(
                &threads[i],
                func,
                this,
                &s_ThreadStack[StackSize * i],
                StackSize,
                nn::os::DefaultThreadPriority));
            ++createdThreadCount;
        }

        for( int i = 0; i < createdThreadCount; ++i )
        {
            nn::os::StartThread(&threads[i]);
        }
        for( int i = 0; i < createdThreadCount; ++i )
        {
            nn::os::WaitThread(&threads[i]);
        }
    }

private:
    std::unique_ptr<char[]> m_MountRomCacheBuffer;
    nn::fs::FileHandle m_FileHandle;
    int64_t m_FileSize;
};

struct FileDataCacheTestParam
{
    int64_t fileSize;
    size_t fileDataCacheBufferSize;
};

class FileDataCacheTest : public MountRomTest, public ::testing::WithParamInterface<FileDataCacheTestParam>
{
public:
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        MountRomTest::SetUp(GetParam().fileSize);

        m_FileDataCacheBufferSize = GetParam().fileDataCacheBufferSize;
        m_FileDataCacheBuffer.reset(new char[m_FileDataCacheBufferSize]);
        ASSERT_TRUE(m_FileDataCacheBuffer);
        nn::fs::EnableGlobalFileDataCache(m_FileDataCacheBuffer.get(), m_FileDataCacheBufferSize);
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::fs::DisableGlobalFileDataCache();

        MountRomTest::TearDown();
    }

    size_t GetFileDataCacheBufferSize() const NN_NOEXCEPT
    {
        return m_FileDataCacheBufferSize;
    }

private:
    std::unique_ptr<char[]> m_FileDataCacheBuffer;
    size_t m_FileDataCacheBufferSize;
};

class SafeFileReader
{
public:
    SafeFileReader(nn::fs::FileHandle fileHandle, size_t bufferSize) NN_NOEXCEPT
        : m_FileHandle(fileHandle)
        , m_Buffer(new uint8_t[bufferSize + VerifySize * 2])
        , m_BufferSize(bufferSize)
    {
        NN_ABORT_UNLESS(m_Buffer);

        std::memset(m_Buffer.get(), VerifyPattern, VerifySize);
        std::memset(m_Buffer.get() + VerifySize + m_BufferSize, VerifyPattern, VerifySize);
    }

    bool VerifyBuffer() NN_NOEXCEPT
    {
        for( int i = 0; i < VerifySize; ++i )
        {
            if( m_Buffer[i] != VerifyPattern )
            {
                return false;
            }
            if( m_Buffer[i + VerifySize + m_BufferSize] != VerifyPattern )
            {
                return false;
            }
        }
        return true;
    }

    void Read(int64_t offset, size_t size) NN_NOEXCEPT
    {
        NNT_FS_SCOPED_TRACE("offset: %lld size: %zd\n", offset, size);

        void* pBuffer = m_Buffer.get() + VerifySize;
        size_t readSize;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(&readSize, m_FileHandle, offset, pBuffer, size));
        EXPECT_TRUE(nnt::fs::util::IsFilledWith8BitCount(pBuffer, readSize, offset));
        ASSERT_TRUE(VerifyBuffer());
    }

private:
    static const int VerifySize = 1024;
    static const uint8_t VerifyPattern = 0xfe;

private:
    nn::fs::FileHandle m_FileHandle;
    std::unique_ptr<uint8_t[]> m_Buffer;
    size_t m_BufferSize;
};

// オフセット -32 ～ +32, サイズ 16 KB の読み込み
TEST_P(FileDataCacheTest, ReadOffset32)
{
    RunTest(
        [](void* pArg) NN_NOEXCEPT
        {
            const auto pTest = reinterpret_cast<FileDataCacheTest*>(pArg);

            const size_t BufferSize = 16 * 1024;
            SafeFileReader reader(pTest->GetFileHandle(), BufferSize);

            for( int offsetBegin = -32; offsetBegin <= 32; ++offsetBegin )
            {
                for( int64_t offset = offsetBegin; offset < pTest->GetFileSize(); offset += BufferSize )
                {
                    if( offset >= 0 )
                    {
                        reader.Read(offset, BufferSize);
                    }
                }
            }
        });
}

// ファイルの後ろから先頭にオフセット -32 ～ +32, サイズ 16 KB の読み込み
TEST_P(FileDataCacheTest, ReadOffset32Reverse)
{
    RunTest(
        [](void* pArg) NN_NOEXCEPT
        {
            const auto pTest = reinterpret_cast<FileDataCacheTest*>(pArg);

            const size_t BufferSize = 16 * 1024;
            SafeFileReader reader(pTest->GetFileHandle(), BufferSize);

            for( int offsetBegin = -32; offsetBegin <= 32; ++offsetBegin )
            {
                for( int64_t offset = pTest->GetFileSize() - BufferSize + offsetBegin; offset >= 0; offset -= BufferSize )
                {
                    reader.Read(offset, BufferSize);
                }
            }
        });
}
// オフセット -32 ～ +32, サイズ 16 KB の読み込みを二重に行う
TEST_P(FileDataCacheTest, ReadOffset32Twice)
{
    RunTest(
        [](void* pArg) NN_NOEXCEPT
        {
            const auto pTest = reinterpret_cast<FileDataCacheTest*>(pArg);

            const size_t BufferSize = 16 * 1024;
            SafeFileReader reader(pTest->GetFileHandle(), BufferSize);

            for( int offsetBegin = -32; offsetBegin <= 32; ++offsetBegin )
            {
                for( int64_t offset = offsetBegin; offset < pTest->GetFileSize(); offset += BufferSize )
                {
                    if( offset >= 0 )
                    {
                        reader.Read(offset, BufferSize);
                        reader.Read(offset, BufferSize);
                    }
                }
            }
        });
}

// サイズ 16 KB -32 ～ +32 バイトの読み込み
TEST_P(FileDataCacheTest, ReadSize16K32B)
{
    RunTest(
        [](void* pArg) NN_NOEXCEPT
        {
            const auto pTest = reinterpret_cast<FileDataCacheTest*>(pArg);

            const size_t BufferSize = 16 * 1024 + 32;
            SafeFileReader reader(pTest->GetFileHandle(), BufferSize);

            for( size_t size = 16 * 1024 - 32; size <= 16 * 1024 + 32; ++size )
            {
                for( int64_t offset = 0; offset < pTest->GetFileSize(); offset += size )
                {
                    reader.Read(offset, size);
                }
            }
        });
}

// サイズ 1 ～ 32 バイトと 16 KB を交互に読み込み
TEST_P(FileDataCacheTest, ReadSize32ByteAnd16KB)
{
    RunTest(
        [](void* pArg) NN_NOEXCEPT
        {
            const auto pTest = reinterpret_cast<FileDataCacheTest*>(pArg);

            const size_t BufferSize = 16 * 1024;
            SafeFileReader reader(pTest->GetFileHandle(), BufferSize);

            for( size_t subSize = 1; subSize <= 32; ++subSize )
            {
                for( int64_t offset = 0; offset + static_cast<int64_t>(BufferSize + subSize) <= pTest->GetFileSize(); offset += BufferSize + subSize )
                {
                    reader.Read(offset, subSize);
                    reader.Read(offset, BufferSize);
                }
            }
        });
}

// 16 KB おきに 1 バイトの読み込み
TEST_P(FileDataCacheTest, Read1BytePer16KB)
{
    RunTest(
        [](void* pArg) NN_NOEXCEPT
        {
            const auto pTest = reinterpret_cast<FileDataCacheTest*>(pArg);

            const size_t BufferSize = 1;
            SafeFileReader reader(pTest->GetFileHandle(), BufferSize);

            for( int offsetBegin = -32; offsetBegin <= 32; ++offsetBegin )
            {
                for( int64_t offset = offsetBegin; offset < pTest->GetFileSize(); offset += 16 * 1024 )
                {
                    if( offset >= 0 )
                    {
                        reader.Read(offset, BufferSize);
                    }
                }
            }
        });
}

// サイズ 16 KB で 8 KB ずつずらす読み込み
TEST_P(FileDataCacheTest, ReadOverlap8KB)
{
    RunTest(
        [](void* pArg) NN_NOEXCEPT
        {
            const auto pTest = reinterpret_cast<FileDataCacheTest*>(pArg);

            const size_t BufferSize = 16 * 1024;
            SafeFileReader reader(pTest->GetFileHandle(), BufferSize);

            for( int offsetBegin = -32; offsetBegin <= 32; ++offsetBegin )
            {
                for( int64_t offset = offsetBegin; offset < pTest->GetFileSize(); offset += BufferSize / 2 )
                {
                    if( offset >= 0 )
                    {
                        reader.Read(offset, BufferSize);
                    }
                }
            }
        });
}

// ファイルの後ろから先頭へサイズ 16 KB で 8 KB ずつずらす読み込み
TEST_P(FileDataCacheTest, ReadOverlap8KBReverse)
{
    RunTest(
        [](void* pArg) NN_NOEXCEPT
        {
            const auto pTest = reinterpret_cast<FileDataCacheTest*>(pArg);

            const size_t BufferSize = 16 * 1024;
            SafeFileReader reader(pTest->GetFileHandle(), BufferSize);

            for( int offsetBegin = -32; offsetBegin <= 32; ++offsetBegin )
            {
                for( int64_t offset = pTest->GetFileSize() - BufferSize + offsetBegin; offset >= 0; offset -= BufferSize / 2 )
                {
                    reader.Read(offset, BufferSize);
                }
            }
        });
}

// 8 KB 読み込んでからそれを覆うように 16 KB 読み込み
TEST_P(FileDataCacheTest, Read8KBAnd16KB)
{
    RunTest(
        [](void* pArg) NN_NOEXCEPT
        {
            const auto pTest = reinterpret_cast<FileDataCacheTest*>(pArg);

            const size_t BufferSize = 16 * 1024;
            SafeFileReader reader(pTest->GetFileHandle(), BufferSize);

            for( int offset8KB = -32; offset8KB <= 32; ++offset8KB )
            {
                for( int64_t offset = 0; offset + static_cast<int64_t>(BufferSize) <= pTest->GetFileSize(); offset += BufferSize )
                {
                    if( offset8KB >= 0 )
                    {
                        reader.Read(offset + offset8KB, BufferSize / 2);
                    }
                    else
                    {
                        reader.Read(offset + BufferSize / 2 + offset8KB, BufferSize / 2);
                    }
                    reader.Read(offset, BufferSize);
                }
            }
        });
}

// 16 KB 読み込んでからその中の 8 KB 読み込み
TEST_P(FileDataCacheTest, Read16KBAnd8KB)
{
    RunTest(
        [](void* pArg) NN_NOEXCEPT
        {
            const auto pTest = reinterpret_cast<FileDataCacheTest*>(pArg);

            const size_t BufferSize = 16 * 1024;
            SafeFileReader reader(pTest->GetFileHandle(), BufferSize);

            for( int offset8KB = -32; offset8KB <= 32; ++offset8KB )
            {
                for( int64_t offset = 0; offset + static_cast<int64_t>(BufferSize) <= pTest->GetFileSize(); offset += BufferSize )
                {
                    reader.Read(offset, BufferSize);
                    if( offset8KB >= 0 )
                    {
                        reader.Read(offset + offset8KB, BufferSize / 2);
                    }
                    else
                    {
                        reader.Read(offset + BufferSize / 2 + offset8KB, BufferSize / 2);
                    }
                }
            }
        });
}

// 8 KB おきに 1 バイト読み込んだ後で同じ領域を 64 KB で読み込み
TEST_P(FileDataCacheTest, Read1BytePer8KBAnd64KB)
{
    RunTest(
        [](void* pArg) NN_NOEXCEPT
        {
            const auto pTest = reinterpret_cast<FileDataCacheTest*>(pArg);

            const size_t BufferSize = 64 * 1024;
            SafeFileReader reader(pTest->GetFileHandle(), BufferSize);

            for( int64_t offset = 0; offset + static_cast<int64_t>(BufferSize) <= pTest->GetFileSize(); offset += BufferSize )
            {
                for( int64_t offset1Byte = offset; offset1Byte < offset + static_cast<int64_t>(BufferSize); offset1Byte += 8 * 1024 )
                {
                    reader.Read(offset1Byte, 1);
                }
                reader.Read(offset, BufferSize);
            }
        });
}

// 64 KB 読み込んだ後で同じ領域を 8 KB おきに 1 バイト読み込み
TEST_P(FileDataCacheTest, Read64KBAnd1BytePer8KB)
{
    RunTest(
        [](void* pArg) NN_NOEXCEPT
        {
            const auto pTest = reinterpret_cast<FileDataCacheTest*>(pArg);

            const size_t BufferSize = 64 * 1024;
            SafeFileReader reader(pTest->GetFileHandle(), BufferSize);

            for( int64_t offset = 0; offset + static_cast<int64_t>(BufferSize) <= pTest->GetFileSize(); offset += BufferSize )
            {
                reader.Read(offset, BufferSize);
                for( int64_t offset1Byte = offset; offset1Byte < offset + static_cast<int64_t>(BufferSize); offset1Byte += 8 * 1024 )
                {
                    reader.Read(offset1Byte, 1);
                }
            }
        });
}

// 16 KB 間隔を開けてファイルデータキャッシュのバッファサイズ分読み込んだ後で 32 KB ずつ読み込み
TEST_P(FileDataCacheTest, ReadCachedAndUncached)
{
    {
        const auto fileDataCacheBufferSize = GetFileDataCacheBufferSize();
        NNT_FS_UTIL_SKIP_TEST_UNLESS(static_cast<int64_t>(fileDataCacheBufferSize) * 2 < GetFileSize());
    }

    RunTest(
        [](void* pArg) NN_NOEXCEPT
        {
            const auto pTest = reinterpret_cast<FileDataCacheTest*>(pArg);

            const size_t BufferSize = 32 * 1024;
            SafeFileReader reader(pTest->GetFileHandle(), BufferSize);

            const auto fileDataCacheBufferSize = pTest->GetFileDataCacheBufferSize();

            for( int64_t offset = 0; offset < static_cast<int64_t>(fileDataCacheBufferSize) * 2; offset += BufferSize )
            {
                reader.Read(offset, BufferSize / 2);
            }

            for( int64_t offset = 0; offset < static_cast<int64_t>(fileDataCacheBufferSize) * 2; offset += BufferSize )
            {
                reader.Read(offset, BufferSize);
            }
        });
}

// ファイルデータキャッシュのバッファサイズより大きい読み込み
TEST_P(FileDataCacheTest, ReadOverFileDataCacheBufferSize)
{
    {
        const auto fileDataCacheBufferSize = GetFileDataCacheBufferSize();
        const size_t bufferSize = fileDataCacheBufferSize * 2;
        NNT_FS_UTIL_SKIP_TEST_UNLESS(static_cast<int64_t>(bufferSize) < GetFileSize());
    }

    RunTest(
        [](void* pArg) NN_NOEXCEPT
        {
            const auto pTest = reinterpret_cast<FileDataCacheTest*>(pArg);

            const auto fileDataCacheBufferSize = pTest->GetFileDataCacheBufferSize();
            const size_t bufferSize = fileDataCacheBufferSize * 2;

            SafeFileReader reader(pTest->GetFileHandle(), bufferSize);

            for( int offsetBegin = -32; offsetBegin <= 32; ++offsetBegin )
            {
                for( int64_t offset = offsetBegin; offset < pTest->GetFileSize(); offset += bufferSize )
                {
                    if( offset >= 0 )
                    {
                        reader.Read(offset, bufferSize);
                    }
                }
            }
        });
}

// スレッドごとに異なる領域を読み込む
TEST_P(FileDataCacheTest, ReadMultiThread)
{
    {
        const size_t BufferSize = 16 * 1024;
        NNT_FS_UTIL_SKIP_TEST_UNLESS(static_cast<int64_t>(ThreadCount * BufferSize) < GetFileSize());
    }

    struct ThreadArg
    {
        int64_t fileSize;
        nn::fs::FileHandle file;
        int threadIndex;
    };

    const auto ThreadFunc = [](void* pArg) NN_NOEXCEPT
    {
        const auto pThreadArg = reinterpret_cast<ThreadArg*>(pArg);

        const size_t BufferSize = 16 * 1024;
        SafeFileReader reader(pThreadArg->file, BufferSize);

        const int64_t readAreaBegin = pThreadArg->fileSize / ThreadCount * pThreadArg->threadIndex;

        for( int offsetBegin = -32; offsetBegin <= 32; ++offsetBegin )
        {
            for( int64_t offset = readAreaBegin + offsetBegin; offset < readAreaBegin + static_cast<int64_t>(BufferSize); offset += BufferSize )
            {
                if( offset >= 0 )
                {
                    reader.Read(offset, BufferSize);
                }
            }
        }
    };

    nn::os::ThreadType threads[ThreadCount];
    ThreadArg threadArgs[ThreadCount];
    int createdThreadCount = 0;
    NN_UTIL_SCOPE_EXIT
    {
        for( int i = 0; i < createdThreadCount; ++i )
        {
            nn::os::DestroyThread(&threads[i]);
        }
    };

    for( int i = 0; i < ThreadCount; ++i )
    {
        threadArgs[i].file = GetFileHandle();
        threadArgs[i].fileSize = GetFileSize();
        threadArgs[i].threadIndex = i;
        NNT_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(
            &threads[i],
            ThreadFunc,
            &threadArgs[i],
            &s_ThreadStack[StackSize * i],
            StackSize,
            nn::os::DefaultThreadPriority));
        ++createdThreadCount;
    }

    for( int i = 0; i < createdThreadCount; ++i )
    {
        nn::os::StartThread(&threads[i]);
    }
    for( int i = 0; i < createdThreadCount; ++i )
    {
        nn::os::WaitThread(&threads[i]);
    }
}

// スレッドごとに異なるマウント名でマウントしてファイルを読み込む
TEST_P(FileDataCacheTest, ReadMultiMount)
{
    struct ThreadArg
    {
        int64_t fileSize;
        int threadIndex;
    };

    const auto ThreadFunc = [](void* pArg) NN_NOEXCEPT
    {
        const auto pThreadArg = reinterpret_cast<ThreadArg*>(pArg);

        size_t mountRomCacheBufferSize;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&mountRomCacheBufferSize));
        auto mountRomBuffer = nnt::fs::util::AllocateBuffer(mountRomCacheBufferSize);

        char mountName[256];
        nn::util::SNPrintf(mountName, sizeof(mountName), "rom%d", pThreadArg->threadIndex);
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountRom(mountName, mountRomBuffer.get(), mountRomCacheBufferSize));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(mountName);
        };

        char filePath[256];
        nn::util::SNPrintf(filePath, sizeof(filePath), "%s:/%d", mountName, pThreadArg->fileSize);
        nn::fs::FileHandle file;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&file, filePath, nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(file);
        };

        int64_t fileSize;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, file));
        ASSERT_EQ(pThreadArg->fileSize, fileSize);

        const size_t BufferSize = 16 * 1024;
        SafeFileReader reader(file, BufferSize);

        for( int offsetBegin = -32; offsetBegin <= 32; ++offsetBegin )
        {
            for( int64_t offset = offsetBegin; offset < fileSize; offset += BufferSize )
            {
                if( offset >= 0 )
                {
                    reader.Read(offset, BufferSize);
                }
            }
        }
    };

    nn::os::ThreadType threads[ThreadCount];
    ThreadArg threadArgs[ThreadCount];
    int createdThreadCount = 0;
    NN_UTIL_SCOPE_EXIT
    {
        for( int i = 0; i < createdThreadCount; ++i )
        {
            nn::os::DestroyThread(&threads[i]);
        }
    };

    for( int i = 0; i < ThreadCount; ++i )
    {
        threadArgs[i].fileSize = GetFileSize();
        threadArgs[i].threadIndex = i;
        NNT_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(
            &threads[i],
            ThreadFunc,
            &threadArgs[i],
            &s_ThreadStack[StackSize * i],
            StackSize,
            nn::os::DefaultThreadPriority));
        ++createdThreadCount;
    }

    for( int i = 0; i < createdThreadCount; ++i )
    {
        nn::os::StartThread(&threads[i]);
    }
    for( int i = 0; i < createdThreadCount; ++i )
    {
        nn::os::WaitThread(&threads[i]);
    }
}

// ROM と AOC を同時にマウントする
TEST_P(FileDataCacheTest, ReadRomAndAoc)
{
    struct ThreadArg
    {
        int64_t fileSize;
        int threadIndex;
    };

    const auto ThreadFunc = [](void* pArg) NN_NOEXCEPT
    {
        const auto pThreadArg = reinterpret_cast<ThreadArg*>(pArg);

        std::unique_ptr<char[]> mountCacheBuffer;

        char mountName[256];
        if( pThreadArg->threadIndex % 2 == 0 )
        {
            size_t mountCacheBufferSize;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&mountCacheBufferSize));
            mountCacheBuffer.reset(new char[mountCacheBufferSize]);
            ASSERT_TRUE(mountCacheBuffer);

            nn::util::SNPrintf(mountName, sizeof(mountName), "rom%d", pThreadArg->threadIndex);
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountRom(mountName, mountCacheBuffer.get(), mountCacheBufferSize));
        }
        else
        {
            const nn::aoc::AddOnContentIndex index = 1;
            size_t mountCacheBufferSize;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::QueryMountAddOnContentCacheSize(&mountCacheBufferSize, index));
            mountCacheBuffer.reset(new char[mountCacheBufferSize]);
            ASSERT_TRUE(mountCacheBuffer);

            nn::util::SNPrintf(mountName, sizeof(mountName), "aoc%d", pThreadArg->threadIndex);
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountAddOnContent(mountName, index, mountCacheBuffer.get(), mountCacheBufferSize));
        }

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(mountName);
        };

        char filePath[256];
        nn::util::SNPrintf(filePath, sizeof(filePath), "%s:/%d", mountName, pThreadArg->fileSize);
        nn::fs::FileHandle file;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&file, filePath, nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(file);
        };

        int64_t fileSize;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, file));
        ASSERT_EQ(pThreadArg->fileSize, fileSize);

        const size_t BufferSize = 16 * 1024;
        SafeFileReader reader(file, BufferSize);

        for( int offsetBegin = -32; offsetBegin <= 32; ++offsetBegin )
        {
            for( int64_t offset = offsetBegin; offset < fileSize; offset += BufferSize )
            {
                if( offset >= 0 )
                {
                    reader.Read(offset, BufferSize);
                }
            }
        }
    };

    nn::os::ThreadType threads[ThreadCount];
    ThreadArg threadArgs[ThreadCount];
    int createdThreadCount = 0;
    NN_UTIL_SCOPE_EXIT
    {
        for( int i = 0; i < createdThreadCount; ++i )
        {
            nn::os::DestroyThread(&threads[i]);
        }
    };

    for( int i = 0; i < ThreadCount; ++i )
    {
        threadArgs[i].fileSize = GetFileSize();
        threadArgs[i].threadIndex = i;
        NNT_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(
            &threads[i],
            ThreadFunc,
            &threadArgs[i],
            &s_ThreadStack[StackSize * i],
            StackSize,
            nn::os::DefaultThreadPriority));
        ++createdThreadCount;
    }

    for( int i = 0; i < createdThreadCount; ++i )
    {
        nn::os::StartThread(&threads[i]);
    }
    for( int i = 0; i < createdThreadCount; ++i )
    {
        nn::os::WaitThread(&threads[i]);
    }
}

// ランダムな読み込み
TEST_P(FileDataCacheTest, ReadRandom)
{
    const size_t BufferSize = 4 * 1024 * 1024;
    SafeFileReader reader(GetFileHandle(), BufferSize);

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    const int LoopCount = 1000;
    for( int i = 0; i < LoopCount; ++i )
    {
        const int64_t offset = std::uniform_int_distribution<int64_t>(0, GetFileSize() - 1)(mt);
        const size_t size = std::uniform_int_distribution<size_t>(0, BufferSize)(mt);
        reader.Read(offset, size);
    }
}

const FileDataCacheTestParam FileDataCacheTestParams[] =
{
    {1,                     1 * 1024 * 1024},
    {256,                   1 * 1024 * 1024},
    {16  * 1024 - 1,        1 * 1024 * 1024},
    {16  * 1024,            1 * 1024 * 1024},
    {16  * 1024 + 1,        1 * 1024 * 1024},
    {0x12345,               1 * 1024 * 1024},
    {123456,                1 * 1024 * 1024},
    {1   * 1024 * 1024 - 1, 1 * 1024 * 1024},
    {1   * 1024 * 1024,     1 * 1024 * 1024},
    {1   * 1024 * 1024 + 1, 1 * 1024 * 1024},
    {0x123456,              1 * 1024 * 1024},
    {1234567,               1 * 1024 * 1024},
    {2   * 1024 * 1024 - 1, 1 * 1024 * 1024},
    {2   * 1024 * 1024,     1 * 1024 * 1024},
    {2   * 1024 * 1024 + 1, 1 * 1024 * 1024},
    {2   * 1024 * 1024 - 1, 2 * 1024 * 1024},
    {2   * 1024 * 1024,     2 * 1024 * 1024},
    {2   * 1024 * 1024 + 1, 2 * 1024 * 1024},
};

INSTANTIATE_TEST_CASE_P(FileDataCacheTest,
                        FileDataCacheTest,
                        ::testing::ValuesIn(FileDataCacheTestParams));

// ファイルデータキャッシュを無効化してから再度読み込み
TEST_F(MountRomTest, DisableFileDataCache)
{
    const size_t BufferSize = 16 * 1024;
    SafeFileReader reader(GetFileHandle(), BufferSize);

    {
        const size_t fileDataCacheBufferSize = 1 * 1024 * 1024;
        auto fileDataCacheBuffer = nnt::fs::util::AllocateBuffer(fileDataCacheBufferSize);
        nn::fs::EnableGlobalFileDataCache(fileDataCacheBuffer.get(), fileDataCacheBufferSize);
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::DisableGlobalFileDataCache();
        };

        for( int offsetBegin = -32; offsetBegin <= 32; ++offsetBegin )
        {
            for( int64_t offset = offsetBegin; offset < GetFileSize(); offset += BufferSize )
            {
                if( offset >= 0 )
                {
                    reader.Read(offset, BufferSize);
                }
            }
        }
    }

    for( int offsetBegin = -32; offsetBegin <= 32; ++offsetBegin )
    {
        for( int64_t offset = offsetBegin; offset < GetFileSize(); offset += BufferSize )
        {
            if( offset >= 0 )
            {
                reader.Read(offset, BufferSize);
            }
        }
    }
}

// ファイルデータキャッシュを有効化してから再度読み込み
TEST_F(MountRomTest, EnableFileDataCache)
{
    const size_t BufferSize = 16 * 1024;
    SafeFileReader reader(GetFileHandle(), BufferSize);

    for( int offsetBegin = -32; offsetBegin <= 32; ++offsetBegin )
    {
        for( int64_t offset = offsetBegin; offset < GetFileSize(); offset += BufferSize )
        {
            if( offset >= 0 )
            {
                reader.Read(offset, BufferSize);
            }
        }
    }

    const size_t fileDataCacheBufferSize = 1 * 1024 * 1024;
    auto fileDataCacheBuffer = nnt::fs::util::AllocateBuffer(fileDataCacheBufferSize);
    nn::fs::EnableGlobalFileDataCache(fileDataCacheBuffer.get(), fileDataCacheBufferSize);
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::DisableGlobalFileDataCache();
    };

    for( int offsetBegin = -32; offsetBegin <= 32; ++offsetBegin )
    {
        for( int64_t offset = offsetBegin; offset < GetFileSize(); offset += BufferSize )
        {
            if( offset >= 0 )
            {
                reader.Read(offset, BufferSize);
            }
        }
    }
}

// ファイルデータキャッシュ有効・無効を切り替えながら読み込む
TEST_F(MountRomTest, EnableAndDisableFileDataCache)
{
    const auto FileDataCacheEnableAndDisable = [](void* pArg) NN_NOEXCEPT
    {
        const size_t fileDataCacheBufferSize = 1 * 1024 * 1024;
        auto fileDataCacheBuffer = nnt::fs::util::AllocateBuffer(fileDataCacheBufferSize);

        const bool* const pIsEnd = reinterpret_cast<bool*>(pArg);

        while( !*pIsEnd )
        {
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            nn::fs::EnableGlobalFileDataCache(fileDataCacheBuffer.get(), fileDataCacheBufferSize);
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            nn::fs::DisableGlobalFileDataCache();
        }
    };

    bool isFileDataCacheEnableAndDisableEnd = false;
    nn::os::ThreadType thread;
    NNT_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(
        &thread,
        FileDataCacheEnableAndDisable,
        &isFileDataCacheEnableAndDisableEnd,
        s_ThreadStack,
        StackSize,
        nn::os::DefaultThreadPriority));
    nn::os::StartThread(&thread);

    const size_t BufferSize = 16 * 1024;
    SafeFileReader reader(GetFileHandle(), BufferSize);

    for( int offsetBegin = -32; offsetBegin <= 32; ++offsetBegin )
    {
        for( int64_t offset = offsetBegin; offset < GetFileSize(); offset += BufferSize )
        {
            if( offset >= 0 )
            {
                reader.Read(offset, BufferSize);
            }
        }
    }

    isFileDataCacheEnableAndDisableEnd = true;
    nn::os::WaitThread(&thread);
    nn::os::DestroyThread(&thread);
}

// サイズ 0 のファイル読み込み
TEST_F(MountRomTest, ReadZero)
{
    const size_t fileDataCacheBufferSize = 1 * 1024 * 1024;
    auto fileDataCacheBuffer = nnt::fs::util::AllocateBuffer(fileDataCacheBufferSize);
    nn::fs::EnableGlobalFileDataCache(fileDataCacheBuffer.get(), fileDataCacheBufferSize);
    NN_UTIL_SCOPE_EXIT
    {
        if( fileDataCacheBuffer )
        {
            nn::fs::DisableGlobalFileDataCache();
        }
    };

    nn::fs::FileHandle file;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&file, "rom:/0", nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(file);
    };

    int64_t fileSize;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, file));
    ASSERT_EQ(0, fileSize);

    char buf;
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadFile(file, 0, &buf, 0));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadFile(file, 0, nullptr, 0));
}

// 4 GB を超えるオフセットで読み込み
TEST_F(MountRomTest, ReadLargeOffset)
{
    nn::fs::FileHandle file;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&file, "rom:/4294967360", nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(file);
    };

    const int64_t Size4GB = static_cast<int64_t>(4) * 1024 * 1024 * 1024;

    int64_t fileSize;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, file));
    EXPECT_GE(fileSize, Size4GB + 64);

    // ファイルの内容は Fill だとオフセット 4 G + offset と offset の内容が一致して
    // オフセット 4 G 以降が切り捨てられても分からないので Random にしている
    // ファイルデータキャッシュなしで読み込み、これを正解の内容とする
    const size_t BufferSize = 32;
    const size_t BufferSizeForNoCache = BufferSize + 64;
    auto bufferForNoCache = nnt::fs::util::AllocateBuffer(BufferSizeForNoCache);
    {
        size_t readSize;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(&readSize, file, Size4GB - 32, bufferForNoCache.get(), BufferSizeForNoCache));
        ASSERT_EQ(BufferSizeForNoCache, readSize);
    }

    // ファイルデータキャッシュを有効化
    const size_t fileDataCacheBufferSize = 1 * 1024 * 1024;
    auto fileDataCacheBuffer = nnt::fs::util::AllocateBuffer(fileDataCacheBufferSize);
    nn::fs::EnableGlobalFileDataCache(fileDataCacheBuffer.get(), fileDataCacheBufferSize);
    NN_UTIL_SCOPE_EXIT
    {
        if( fileDataCacheBuffer )
        {
            nn::fs::DisableGlobalFileDataCache();
        }
    };

    // ファイルデータキャッシュ有効で読み込み、比較
    auto buffer = nnt::fs::util::AllocateBuffer(BufferSize);

    for( int64_t offset = -32 + Size4GB; offset <= 32 + Size4GB; ++offset )
    {
        NNT_FS_SCOPED_TRACE("offset: %lld size: %lld\n", offset, static_cast<uint64_t>(BufferSize));

        size_t readSize;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(&readSize, file, offset, buffer.get(), BufferSize));
        ASSERT_EQ(BufferSize, readSize);

        NNT_FS_UTIL_EXPECT_MEMCMPEQ(bufferForNoCache.get() + offset - (Size4GB - 32), buffer.get(), readSize);
    }
}

// ファイルデータキャッシュ有効な状態でマウントし直す
TEST(FileDataCacheRemountTest, Remount)
{
    const size_t fileDataCacheBufferSize = 1 * 1024 * 1024;
    auto fileDataCacheBuffer = nnt::fs::util::AllocateBuffer(fileDataCacheBufferSize);

    size_t mountRomCacheBufferSize;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&mountRomCacheBufferSize));
    auto mountRomCacheBuffer = nnt::fs::util::AllocateBuffer(mountRomCacheBufferSize);

    const auto MountAndReadTest = [&mountRomCacheBuffer, mountRomCacheBufferSize](const char* mountName) NN_NOEXCEPT
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountRom(mountName, mountRomCacheBuffer.get(), mountRomCacheBufferSize));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(mountName);
        };

        nn::fs::FileHandle file;
        nnt::fs::util::String filePath = nnt::fs::util::String(mountName) + ":/2097153";
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&file, filePath.c_str(), nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(file);
        };

        int64_t fileSize;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, file));

        const size_t BufferSize = 16 * 1024;
        SafeFileReader reader(file, BufferSize);

        for( int offsetBegin = -32; offsetBegin <= 32; ++offsetBegin )
        {
            for( int64_t offset = offsetBegin; offset < fileSize; offset += BufferSize )
            {
                if( offset >= 0 )
                {
                    reader.Read(offset, BufferSize);
                }
            }
        }
    };

    {
        nn::fs::EnableGlobalFileDataCache(fileDataCacheBuffer.get(), fileDataCacheBufferSize);
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::DisableGlobalFileDataCache();
        };

        // 同じマウント名で再マウント
        MountAndReadTest("mount");
        MountAndReadTest("mount");
    }
    {
        nn::fs::EnableGlobalFileDataCache(fileDataCacheBuffer.get(), fileDataCacheBufferSize);
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::DisableGlobalFileDataCache();
        };

        // 異なるマウント名で再マウント
        MountAndReadTest("mount1");
        MountAndReadTest("mount2");
    }
}

}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();
    nn::fs::SetEnabledAutoAbort(false);

    auto testResult = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(testResult);
}
