﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForApplications.h>
#include <nn/fs.h>
#include <nn/fs/fs_BcatSaveData.h>
#include <nn/fs/fs_DeviceSaveData.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SaveDataManagementPrivate.h>
#include <nn/fs/fs_ResultHandler.h>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/fsUtil/testFs_util_function.h>

using namespace nn;
using namespace nn::fs;
using namespace nn::fs::detail;
using namespace nnt::fs::util;

namespace nn {
    namespace fs {
        namespace detail {
            // fs_SaveDataManagementPrivate.h に宣言がないのでここでプロトタイプ宣言
            Result ReadSaveDataFileSystemExtraData(SaveDataExtraData *outValue, SaveDataId saveDataId) NN_NOEXCEPT;
        }
    }
}

namespace {

    const int64_t ownerId               = 0x0005000C10000001;
    const size_t saveDataSize           = 1024 * 1024;
    const size_t saveDataJournalSize    = 1024 * 1024;
    const int cacheStorageIndex         = 0;


    // テスト用のアカウント
    nn::account::Uid GetUidForTest()
    {
        static nn::account::Uid user = nn::account::InvalidUid;

        // user が有効ならそのまま返す
        if(user)
        {
            return user;
        }

        // リストアップした最初のユーザを返す
        int userCount = 0;
        nn::oe::Initialize();
        nn::account::Initialize();
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::account::ListAllUsers(&userCount, &user, 1)
        );
        EXPECT_TRUE(userCount > 0);

        return user;
    }


    // テスト用のセーブデータの作成・マウント・削除

    // デバイスセーブデータ（本体内蔵メモリの User 領域）
    void CreateDeviceSaveDataImpl()
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::CreateDeviceSaveData(
                nnt::fs::util::ApplicationId,
                ownerId,
                saveDataSize,
                saveDataJournalSize,
                0
            )
        );
    }
    void DeleteDeviceSaveDataImpl()
    {
        Vector<nn::fs::SaveDataInfo> infos;

        // 削除対象のセーブデータをリストアップ
        FindSaveData(&infos, nn::fs::SaveDataSpaceId::User, [&](const SaveDataInfo& i) NN_NOEXCEPT
        {
            if(i.applicationId == nnt::fs::util::ApplicationId && i.saveDataType == SaveDataType::Device)
            {
                return true;
            }
            else
            {
                return false;
            }
        });

        // リストアップされた全てのセーブデータを削除
        for(nn::fs::SaveDataInfo i : infos)
        {
            nn::Result result = nn::fs::DeleteSaveData(i.saveDataSpaceId, i.saveDataId);
            if(!nn::fs::ResultTargetNotFound::Includes(result))
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }
        }
    }
    void MountDeviceSaveDataImpl(const char* mountName)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::MountDeviceSaveData(mountName)
        );
    }

    // ユーザアカウントセーブデータ（本体内蔵メモリの User 領域）
    void CreateUserAccountSaveDataImpl()
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::EnsureSaveData(GetUidForTest())
        );
    }
    void DeleteUserAccountSaveDataImpl()
    {
        Vector<SaveDataInfo> infos;

        // 削除対象のセーブデータをリストアップ
        FindSaveData(&infos, SaveDataSpaceId::User, [&](const SaveDataInfo& i) NN_NOEXCEPT
        {
            if(i.applicationId == nnt::fs::util::ApplicationId && i.saveDataType == SaveDataType::Account)
            {
                return true;
            }
            else
            {
                return false;
            }
        });

        // リストアップされた全てのセーブデータを削除
        for(nn::fs::SaveDataInfo i : infos)
        {
            nn::Result result = nn::fs::DeleteSaveData(i.saveDataSpaceId, i.saveDataId);
            if(!nn::fs::ResultTargetNotFound::Includes(result))
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }
        }
    }
    void MountUserAccountSaveDataImpl(const char* mountName)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::MountSaveData(mountName, GetUidForTest())
        );
    }
    void ExtendUserAccountSaveDataImpl()
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::ExtendSaveData(
                GetUidForTest(),
                saveDataSize * 2,
                saveDataJournalSize * 2
            )
        );
    }

    // BCAT セーブデータ（本体内蔵メモリの User 領域）
    void CreateBcatSaveDataImpl()
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::CreateBcatSaveData(
                nnt::fs::util::ApplicationId,
                saveDataSize
            )
        );
    }
    void DeleteBcatSaveDataImpl()
    {
        Vector<SaveDataInfo> infos;

        // 削除対象のセーブデータをリストアップ
        FindSaveData(&infos, SaveDataSpaceId::User, [&](const SaveDataInfo& i) NN_NOEXCEPT
        {
            if(i.applicationId == nnt::fs::util::ApplicationId && i.saveDataType == SaveDataType::Bcat)
            {
                return true;
            }
            else
            {
                return false;
            }
        });

        // リストアップされた全てのセーブデータを削除
        for(nn::fs::SaveDataInfo i : infos)
        {
            nn::Result result = nn::fs::DeleteSaveData(i.saveDataSpaceId, i.saveDataId);
            if(!nn::fs::ResultTargetNotFound::Includes(result))
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }
        }
    }
    void MountBcatSaveDataImpl(const char* mountName)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::MountBcatSaveData(
                mountName,
                nnt::fs::util::ApplicationId
            )
        );
    }

    // キャッシュストレージ（SD カードの User 領域）
    void CreateCacheStorageImpl()
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::CreateCacheStorage(
                cacheStorageIndex,
                saveDataSize,
                saveDataJournalSize
            )
        );
    }
    void DeleteCacheStorageImpl()
    {
        nn::Result result;
        result = nn::fs::DeleteCacheStorage(cacheStorageIndex);
        if(!nn::fs::ResultTargetNotFound::Includes(result))
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
    }
    void MountCacheStorageImpl(const char* mountName)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::MountCacheStorage(mountName, cacheStorageIndex)
        );
    }

    // 指定スペースの nnt::fs::util::ApplicationId のセーブデータをすべて削除します。
    void DeleteSaveData(nn::fs::SaveDataSpaceId spaceId)
    {
        Vector<nn::fs::SaveDataInfo> infos;

        // 削除対象のセーブデータをリストアップ
        FindSaveData(&infos, spaceId, [&](const SaveDataInfo& i) NN_NOEXCEPT
        {
            return true;
        });

        // リストアップされた全てのセーブデータを削除
        for(nn::fs::SaveDataInfo i : infos)
        {
            nn::fs::DeleteSaveData(i.saveDataSpaceId, i.saveDataId);
        }
    }

    // 本体内蔵メモリの User 領域の次の自動採番の SaveDataId を取得します。
    // 内部でデバイスセーブデータを作成して削除するため、
    // 自動採番の SaveDataId がインクリメントされます。
    nn::fs::SaveDataId GetNextUserSaveDataId()
    {
        Vector<nn::fs::SaveDataInfo> infos;

        // デバイスセーブデータを作成
        DeleteDeviceSaveDataImpl();
        CreateDeviceSaveDataImpl();

        // SD カードの User 領域の全てのセーブデータをリストアップ
        FindSaveData(&infos, nn::fs::SaveDataSpaceId::User, [&](const SaveDataInfo& i) NN_NOEXCEPT
        {
            return true;
        });
        NN_ABORT_UNLESS(infos.size() > 0);

        // 最後の ID の次が次の自動採番の ID になるはず
        nn::fs::SaveDataId nextSaveDataId = infos.back().saveDataId + 1;

        // デバイスセーブデータを削除
        DeleteDeviceSaveDataImpl();

        return nextSaveDataId;
    }


    // SD カードの User 領域の次の自動採番の SaveDataId を取得します。
    // 内部でキャッシュストレージを作成して削除するため、
    // 自動採番の SaveDataId がインクリメントされます。
    nn::fs::SaveDataId GetNextSdUserSaveDataId()
    {
        Vector<nn::fs::SaveDataInfo> infos;

        // CacheStorage を作成
        DeleteCacheStorageImpl();
        CreateCacheStorageImpl();

        // SD カードの User 領域の全てのセーブデータをリストアップ
        FindSaveData(&infos, nn::fs::SaveDataSpaceId::SdUser, [&](const SaveDataInfo& i) NN_NOEXCEPT
        {
            return true;
        });
        NN_ABORT_UNLESS(infos.size() > 0);

        // 最後の ID の次が次の自動採番の ID になるはず
        nn::fs::SaveDataId nextSaveDataId = infos.back().saveDataId + 1;

        // CacheStorage を削除
        DeleteCacheStorageImpl();

        return nextSaveDataId;
    }

    // 本体内蔵メモリの User 領域の自動採番の SaveDataId をインクリメントします。
    void IncrementUserSaveDataId(int step)
    {
        for(int i=0; i<step; ++i)
        {
            DeleteDeviceSaveDataImpl();
            CreateDeviceSaveDataImpl();
        }
        DeleteDeviceSaveDataImpl();
    }

    // SD カードの User 領域の自動採番の SaveDataId をインクリメントします。
    void IncrementSdUserSaveDataId(int step)
    {
        for(int i=0; i<step; ++i)
        {
            DeleteCacheStorageImpl();
            CreateCacheStorageImpl();
        }
        DeleteCacheStorageImpl();
    }

    // 以下、テストデータ作成・検証用

    void GetRandomOffsetAndSize(int64_t* pOutOffset, size_t* pOutSize, int64_t fileSize, int64_t maxSize)
    {
        std::mt19937 random(nnt::fs::util::GetRandomSeed());

        int64_t offset;
        int64_t size;

        offset = random() % fileSize;
        size   = std::min(static_cast<int64_t>(random()), maxSize);

        if (offset + size > fileSize)
        {
            size = fileSize - offset;
        }

        *pOutOffset = offset;
        *pOutSize   = static_cast<size_t>(size);
    }

    nn::Result RandomReadWithCheck(const char* path)
    {
        FileHandle handle;
        NN_RESULT_DO(OpenFile(&handle, path, OpenMode_Read));
        NN_UTIL_SCOPE_EXIT{
            CloseFile(handle);
        };

        int64_t fileSize = 0;
        NN_RESULT_DO(GetFileSize(&fileSize, handle));
        if (fileSize == 0)
        {
            NN_RESULT_SUCCESS;
        }

        // とりあえず 16MB に設定する
        const size_t BufferSize = 16 * 1024 * 1024;
        auto buffer = AllocateBuffer(BufferSize);
        if (buffer.get() == nullptr)
        {
            NN_LOG("BufferAllocate Error %s , l.%d\n", __FUNCTION__, __LINE__);
            NN_RESULT_DO(nn::fs::ResultUnknown());
        }
        memset(buffer.get(), 0, BufferSize);

        int64_t offset = 0;
        size_t readSize = 0;

        GetRandomOffsetAndSize(&offset, &readSize, fileSize, BufferSize);

        auto result = (ReadFile(handle, offset, buffer.get(), readSize));
        if (result.IsFailure())
        {
            NN_LOG("offset = 0x%llx, size = 0x%llx, fileSize = %llx\n", offset, readSize, fileSize);
            NN_RESULT_DO(result);
        }

        // Verify
        offset += GetOffsetByFileName(path);
        if (!IsFilledWith32BitCount(buffer.get(), readSize, offset))
        {
            NN_LOG("read data mismatch \n");
            NN_LOG("path = %s fileSize = %lld\n", path, fileSize);
            NN_LOG("offset = %lld readSize= %d nameOffset = %lld\n", offset, readSize, GetOffsetByFileName(path));
            return nn::fs::ResultUnknown();
        }

        NN_RESULT_SUCCESS;
    }

    void RandomVerify(const char* mountName)
    {
        String testDirectory = String(mountName) + ":/";
        Vector<String> fileList;
        Vector<String> dirList;

        NN_ABORT_UNLESS_RESULT_SUCCESS(ListDirectoryRecursive(&fileList, &dirList, testDirectory.c_str()));

        // ファイル数が 0 なら以下の処理は行わない
        if (fileList.size() == 0)
        {
            return;
        }

        for(int i=0; i<10; ++i)
        {
            // fs 内のファイルをランダムに選びランダムなレンジで read して verify する
            std::mt19937 random(nnt::fs::util::GetRandomSeed());
            int fileIndex = random() % fileList.size();
            NN_ABORT_UNLESS_RESULT_SUCCESS(RandomReadWithCheck(fileList[fileIndex].c_str()));
        }
    }

    void GetDirectoryTreeHash(Hash* outValue, const char* mountName)
    {
        String testDirectory = String(mountName) + ":/";
        const size_t bufferSize = 1024 * 1024;
        auto buffer = AllocateBuffer(bufferSize);

        NN_ABORT_UNLESS_RESULT_SUCCESS(
            CalculateDirectoryTreeHash(outValue, testDirectory.c_str(), buffer.get(), bufferSize)
        );
    }


    class SaveDataOperator
    {
    public:
        explicit SaveDataOperator(const char* mountName, nn::fs::SaveDataId saveDataId)
            : m_MountName(mountName)
            , m_SaveDataId(saveDataId)
        {}

        // セーブデータを作成
        virtual void Create() = 0;

        // セーブデータを削除
        virtual void Delete() = 0;

        // マウント
        virtual void Mount() = 0;

        // SaveDataSpaceId を取得
        virtual nn::fs::SaveDataSpaceId GetSaveDataSpaceId() = 0;

        // SaveDataId を取得
        virtual nn::fs::SaveDataId GetSaveDataId()
        {
            return m_SaveDataId;
        }

        // マウント名を取得
        virtual String GetMountName()
        {
            return m_MountName;
        }

        // アンマウント
        virtual void Unmount()
        {
            nn::fs::Unmount(m_MountName.c_str());
        }

        // コミット
        virtual void Commit()
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                nn::fs::Commit(m_MountName.c_str())
            );
        }

        // テストデータを作成
        virtual void CreateTestData()
        {
            const int64_t totalSize = 1024;
            const int64_t totalEntryCount = 32;

            Mount();

            // ファイルの作成、書き込み
            nnt::fs::util::CreateTestDirectoryTreeRandomlyByFileNameOffset((m_MountName + ":/").c_str(), totalSize, totalEntryCount);

            // コミット
            Commit();

            // ハッシュを計算
            GetDirectoryTreeHash(&m_DirectoryTreeHash, m_MountName.c_str());

            Unmount();
        }

        // テストデータを検証
        virtual void VerifyTestData()
        {
            Mount();

            // CreateTestData で生成したファイルを検証
            RandomVerify(m_MountName.c_str());

            // ハッシュを検証
            Hash after;
            GetDirectoryTreeHash(&after, m_MountName.c_str());
            NNT_FS_UTIL_EXPECT_MEMCMPEQ(&m_DirectoryTreeHash.value, after.value, sizeof(m_DirectoryTreeHash.value));

            Unmount();
        }

        // SaveDataExtraData をランダムに設定
        virtual void SetSaveDataExtraData()
        {
            SaveDataExtraData saveDataExtraData;
            FillBufferWithRandomValue(&saveDataExtraData, sizeof(saveDataExtraData));

            // マスクなしで設定（flags を設定）
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                nn::fs::detail::WriteSaveDataFileSystemExtraData(GetSaveDataSpaceId(), GetSaveDataId(), saveDataExtraData)
            );
            // マスクありで設定（padding1 を設定）
            SaveDataExtraData mask;
            memset(&mask, 0, sizeof(mask));
            memset(mask.padding1, 0xFF, sizeof(mask.padding1));
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                nn::fs::detail::WriteSaveDataFileSystemExtraData(GetSaveDataSpaceId(), GetSaveDataId(), saveDataExtraData, mask)
            );

            // 値を読み出して確認
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                nn::fs::detail::ReadSaveDataFileSystemExtraData(&m_SaveDataExtraData, GetSaveDataSpaceId(), GetSaveDataId())
            );
            EXPECT_TRUE(m_SaveDataExtraData.flags == saveDataExtraData.flags);
            NNT_FS_UTIL_EXPECT_MEMCMPEQ(m_SaveDataExtraData.padding1, saveDataExtraData.padding1, sizeof(saveDataExtraData.padding1));
        }

        // SaveDataExtraData を検証
        virtual void VerifySaveDataExtraData()
        {
            SaveDataExtraData saveDataExtraData;
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                nn::fs::detail::ReadSaveDataFileSystemExtraData(&saveDataExtraData, GetSaveDataSpaceId(), GetSaveDataId())
            );
            NNT_FS_UTIL_EXPECT_MEMCMPEQ(&m_SaveDataExtraData, &saveDataExtraData, sizeof(SaveDataExtraData));

            // 本体内蔵メモリ上のセーブデータの場合は space id 指定なし版でも検証
            if(GetSaveDataSpaceId() == SaveDataSpaceId::User || GetSaveDataSpaceId() == SaveDataSpaceId::System)
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(
                    nn::fs::detail::ReadSaveDataFileSystemExtraData(&saveDataExtraData, GetSaveDataId())
                );
                NNT_FS_UTIL_EXPECT_MEMCMPEQ(&m_SaveDataExtraData, &saveDataExtraData, sizeof(SaveDataExtraData));
            }
        }

        // 影響がある可能性のある操作を行う
        virtual void OperateSaveData()
        {
            Create();
            CreateTestData();
            VerifyTestData();
            SetSaveDataExtraData();
            VerifySaveDataExtraData();
            Delete();
        }

    private:
        String m_MountName;
        nn::fs::SaveDataId m_SaveDataId;
        Hash m_DirectoryTreeHash;
        SaveDataExtraData m_SaveDataExtraData;
    };

    // デバイスセーブデータ（本体内蔵メモリの User 領域）
    class DeviceSaveDataOperator : public SaveDataOperator
    {
    public:
        explicit DeviceSaveDataOperator(const char* mountName, nn::fs::SaveDataId saveDataId)
            : SaveDataOperator(mountName, saveDataId)
        {}

        virtual void Create()
        {
            CreateDeviceSaveDataImpl();
        }

        virtual void Delete()
        {
            DeleteDeviceSaveDataImpl();
        }

        virtual void Mount()
        {
            MountDeviceSaveDataImpl(GetMountName().c_str());
        }

        virtual nn::fs::SaveDataSpaceId GetSaveDataSpaceId()
        {
            return SaveDataSpaceId::User;
        }
    };

    // ユーザアカウントセーブデータ（本体内蔵メモリの User 領域）
    class UserAccountSaveDataOperator : public SaveDataOperator
    {
    public:
        explicit UserAccountSaveDataOperator(const char* mountName, nn::fs::SaveDataId saveDataId)
            : SaveDataOperator(mountName, saveDataId)
        {}

        virtual void Create()
        {
            CreateUserAccountSaveDataImpl();
        }

        virtual void Delete()
        {
            DeleteUserAccountSaveDataImpl();
        }

        virtual void Mount()
        {
            MountUserAccountSaveDataImpl(GetMountName().c_str());
        }

        virtual nn::fs::SaveDataSpaceId GetSaveDataSpaceId()
        {
            return SaveDataSpaceId::User;
        }

        virtual void OperateSaveData()
        {
            Create();
            CreateTestData();
            VerifyTestData();
            SetSaveDataExtraData();
            VerifySaveDataExtraData();

            // セーブデータ拡張
            ExtendUserAccountSaveDataImpl();

            Delete();
        }
    };

    // BCAT セーブデータ（本体内蔵メモリの User 領域）
    class BcatSaveDataOperator : public SaveDataOperator
    {
    public:
        explicit BcatSaveDataOperator(const char* mountName, nn::fs::SaveDataId saveDataId)
            : SaveDataOperator(mountName, saveDataId)
        {}

        virtual void Create()
        {
            CreateBcatSaveDataImpl();
        }

        virtual void Delete()
        {
            DeleteBcatSaveDataImpl();
        }

        virtual nn::fs::SaveDataSpaceId GetSaveDataSpaceId()
        {
            return SaveDataSpaceId::User;
        }

        virtual void Mount()
        {
            MountBcatSaveDataImpl(GetMountName().c_str());
        }
    };

    // キャッシュストレージ（SD カードの User 領域）
    class CacheStorageOperator : public SaveDataOperator
    {
    public:
        explicit CacheStorageOperator(const char* mountName, nn::fs::SaveDataId saveDataId)
            : SaveDataOperator(mountName, saveDataId)
        {}

        virtual void Create()
        {
            CreateCacheStorageImpl();
        }

        virtual void Delete()
        {
            DeleteCacheStorageImpl();
        }

        virtual void Mount()
        {
            MountCacheStorageImpl(GetMountName().c_str());
        }

        virtual nn::fs::SaveDataSpaceId GetSaveDataSpaceId()
        {
            return SaveDataSpaceId::SdUser;
        }
    };

}

class TestSaveDataIdDuplication : public ::testing::Test
{
public:
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::Result result;
        Vector<SaveDataInfo> infos;

        // 全てのユーザセーブデータを削除
        DeleteSaveData(nn::fs::SaveDataSpaceId::User);
        DeleteSaveData(nn::fs::SaveDataSpaceId::SdUser);

        // 次の自動採番のセーブデータ ID を取得
        nn::fs::SaveDataId nextSdUserSaveDataId = GetNextSdUserSaveDataId();
        nn::fs::SaveDataId nextUserSaveDataId = GetNextUserSaveDataId();

        // 自動採番のセーブデータ ID が揃うように調整
        int64_t saveDataIdDiff = nextSdUserSaveDataId - nextUserSaveDataId;
        if(saveDataIdDiff > 0)
        {
            IncrementUserSaveDataId(saveDataIdDiff);
        }
        else if(saveDataIdDiff < 0)
        {
            IncrementSdUserSaveDataId(-saveDataIdDiff);
        }

        // セーブデータ ID が揃っていることを確認
        nextSdUserSaveDataId = GetNextSdUserSaveDataId();
        nextUserSaveDataId = GetNextUserSaveDataId();
        NN_ABORT_UNLESS(nextSdUserSaveDataId == nextUserSaveDataId, "  nextSdUserSaveDataId: 0x%016llX\n  nextUserSaveDataId: 0x%016llX", nextSdUserSaveDataId, nextUserSaveDataId);

        m_SaveDataId = nextUserSaveDataId;
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        // 全てのユーザセーブデータを削除
        DeleteSaveData(nn::fs::SaveDataSpaceId::User);
        DeleteSaveData(nn::fs::SaveDataSpaceId::SdUser);
    }


    // 2 つのセーブデータを指定してテストを実行する
    void DoDuprecatedSaveDataIdTest(SaveDataOperator& target, SaveDataOperator& duplicated)
    {
        // 対象のセーブデータを作成してテストデータを作成
        target.Create();
        target.CreateTestData();
        target.SetSaveDataExtraData();

        // ID 被りのセーブデータを作成して色々と操作を行う
        duplicated.OperateSaveData();

        // 対象のセーブデータを検証
        target.VerifyTestData();
        target.VerifySaveDataExtraData();
        target.Delete();
    }

    // テスト対象セーブデータ ID を取得する
    nn::fs::SaveDataId GetSaveDataId()
    {
        return m_SaveDataId;
    }

private:
    nn::fs::SaveDataId m_SaveDataId;
};


// キャッシュストレージにデバイスセーブデータを被せる
TEST_F(TestSaveDataIdDuplication, CacheStorage_DeviceSaveData)
{
    auto targetSave = CacheStorageOperator("target", GetSaveDataId());
    auto duplicatedSave = DeviceSaveDataOperator("duplicated", GetSaveDataId());
    DoDuprecatedSaveDataIdTest(targetSave, duplicatedSave);
}

// キャッシュストレージにユーザアカウントセーブデータを被せる
TEST_F(TestSaveDataIdDuplication, CacheStorage_UserAccountSaveData)
{
    auto targetSave = CacheStorageOperator("target", GetSaveDataId());
    auto duplicatedSave = UserAccountSaveDataOperator("duplicated", GetSaveDataId());
    DoDuprecatedSaveDataIdTest(targetSave, duplicatedSave);
}

// キャッシュストレージに BCAT セーブデータを被せる
TEST_F(TestSaveDataIdDuplication, CacheStorage_BcatSaveData)
{
    auto targetSave = CacheStorageOperator("target", GetSaveDataId());
    auto duplicatedSave = BcatSaveDataOperator("duplicated", GetSaveDataId());
    DoDuprecatedSaveDataIdTest(targetSave, duplicatedSave);
}

// デバイスセーブデータにキャッシュストレージを被せる
TEST_F(TestSaveDataIdDuplication, DeviceSaveData_CacheStorage)
{
    auto targetSave = DeviceSaveDataOperator("target", GetSaveDataId());
    auto duplicatedSave = CacheStorageOperator("duplicated", GetSaveDataId());
    DoDuprecatedSaveDataIdTest(targetSave, duplicatedSave);
}

// ユーザアカウントセーブデータにキャッシュストレージを被せる
TEST_F(TestSaveDataIdDuplication, UserAccountSaveData_CacheStorage)
{
    auto targetSave = UserAccountSaveDataOperator("target", GetSaveDataId());
    auto duplicatedSave = CacheStorageOperator("duplicated", GetSaveDataId());
    DoDuprecatedSaveDataIdTest(targetSave, duplicatedSave);
}

// BCAT セーブデータにキャッシュストレージを被せる
TEST_F(TestSaveDataIdDuplication, BcatSaveData_CacheStorage)
{
    auto targetSave = BcatSaveDataOperator("target", GetSaveDataId());
    auto duplicatedSave = CacheStorageOperator("duplicated", GetSaveDataId());
    DoDuprecatedSaveDataIdTest(targetSave, duplicatedSave);
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nn::fs::SetEnabledAutoAbort(false);
    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();

    auto result = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(result);
}
