﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <cstring>

#include <nn/util/util_FormatString.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/base/testBase_Exit.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_ContentStorage.h>
#include <nn/fs/fs_CacheStoragePrivate.h>
#include <nn/fs/fs_CacheStorage.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataManagementPrivate.h>
#include <nn/fs/fs_TemporaryStorage.h>
#include <nnt/nnt_Argument.h>
#include <nn/fat/fat_FatFileSystem.h>
#include <nn/fs/fsa/fs_Registrar.h>
#include <nn/fs/fs_SpeedEmulation.h>
#include <nn/fs/fs_MmcPrivate.h>

using namespace nn::fs;
using namespace nnt::fs::util;

void* operator new(size_t size)
{
    return malloc(size);
}

void operator delete(void* p) NN_NOEXCEPT
{
    free(p);
}

namespace{

const char TestDirectoryName[]  = "perftestdir";
const char TestFileName[]       = "0";
const char DummyTestFileName[]  = "1";
const int  FileCount            = 4096;

const int DivisionCountMax      = 1   * 1024; // 小さいバッファサイズの場合、この数以上の read/write はしない
const int64_t AlignmentPadding  = 32;         // アライメントずれテスト用パラメータ
const size_t TestReadWriteSize  = 128 * 1024 * 1024;
const int BufferSizeMin         = 1   * 1024;
const int BufferSizeMax         = 4   * 1024 * 1024;

const char MountName[]          = "perftest";

const char SpeedEmulationModeNameNone[] = "Off";
const char SpeedEmulationModeNameSlower[] = "Slower";
const char SpeedEmulationModeNameFaster[] = "Faster";
const char SpeedEmulationModeNameRandom[] = "Random";
const char* g_SpeedEmulationModeName = SpeedEmulationModeNameNone;

const char* g_AccessPriorityName = "Default";
String g_BackgroundThreadTag;

NN_ALIGNAS(4096) char g_ReadWriteBuffer[256 * 1024 * 1024];


class PerformanceTestBase : public ::testing::Test, public nnt::fs::util::PrepareWorkDirFixture
{
public:
    virtual ~PerformanceTestBase() NN_NOEXCEPT NN_OVERRIDE
    {
    }
};

typedef PerformanceTestBase PerformanceTestFlushHostFs;
typedef PerformanceTestBase PerformanceTestReadWriteHostFs;
typedef PerformanceTestBase PerformanceTestFlushSdCard;
typedef PerformanceTestBase PerformanceTestFlushBisFs;
typedef PerformanceTestBase PerformanceTestOpenModeHostFs;
typedef PerformanceTestBase PerformanceTestOpenModeSdCard;
typedef PerformanceTestBase PerformanceTestOpenModeBisFs;
typedef PerformanceTestBase PerformanceTestContentStorageSd;
typedef PerformanceTestBase PerformanceTestContentStorageNand;

Vector<PerformanceResult> g_PerformanceResultsRead;
Vector<PerformanceResult> g_PerformanceResultsWrite;

void DumpPerformanceResultsFlush(Vector<PerformanceResult>& results) NN_NOEXCEPT
{
    NN_LOG("BufferSize[bytes]   Elapsed[usec]\n");
    for(auto result : results)
    {
        NN_LOG("        %9d       %9lld\n",
            result.bufferSize,
            result.elapsedTimeUsec);
    }
}

enum class AccessType
{
    Read,
    Write
};

enum class Alignment
{
    Aligned,
    Unaligned,
};

enum class RelationWithSpeedEmulation
{
    NoEffect,
    HasEffect
};

void GetTeamCityTag(
    nnt::fs::util::String* outValue,
    const char* prefix,
    AccessType accessType,
    Alignment alignment,
    const char* speedEmulationModeName
) NN_NOEXCEPT
{
    *outValue = prefix;

    switch( accessType )
    {
    case AccessType::Read:
        *outValue += "Read";
        break;

    case AccessType::Write:
        *outValue += "Write";
        break;

    default:
        NN_LOG("Unknown access type\n");
        nnt::Exit(1);
        break;
    }

    if( alignment == Alignment::Unaligned )
    {
        *outValue += "Unaligned";
    }

    *outValue += speedEmulationModeName;

    if( std::strcmp(g_AccessPriorityName, "Default") != 0 )
    {
        *outValue += g_AccessPriorityName;
    }

    *outValue += g_BackgroundThreadTag;
}

void GetTeamCityTag(
    nnt::fs::util::String* outValue,
    const char* prefix,
    AccessType accessType,
    Alignment alignment,
    RelationWithSpeedEmulation relationWithSpeedEmulation
) NN_NOEXCEPT
{
    GetTeamCityTag(
        outValue,
        prefix,
        accessType,
        alignment,
        relationWithSpeedEmulation == RelationWithSpeedEmulation::HasEffect
            ? g_SpeedEmulationModeName : "");
}

void CheckPerformanceResult(
    const char* tagPrefix,
    AccessType accessType,
    Alignment alignment,
    RelationWithSpeedEmulation relationWithSpeedEmulation,
    const Vector<PerformanceResult>& results
) NN_NOEXCEPT
{
    {
        nnt::fs::util::String tag;
        GetTeamCityTag(&tag, tagPrefix, accessType, alignment, relationWithSpeedEmulation);
        EXPECT_TRUE(nnt::fs::util::CheckPerformanceResult(tag.c_str(), results, false));
    }

    if( relationWithSpeedEmulation == RelationWithSpeedEmulation::HasEffect
        && g_SpeedEmulationModeName == SpeedEmulationModeNameSlower )
    {
        nnt::fs::util::String tag;
        GetTeamCityTag(&tag, tagPrefix, accessType, alignment, SpeedEmulationModeNameNone);
        EXPECT_TRUE(nnt::fs::util::CheckPerformanceResult(tag.c_str(), results, true));
    }
}

/**
* @brief ファイルのシーケンシャル書き込みを計測
*/
void WriteSequential(const String mountName, int64_t offsetDifference, bool isSaveData) NN_NOEXCEPT
{
    Vector<PerformanceResult> perfResults;
    const String TestPath = mountName + ":/" + TestDirectoryName + "/" + TestFileName;

    for(int bufferSize = BufferSizeMin ; bufferSize <= BufferSizeMax ; bufferSize *= 2)
    {
        NNT_FS_SCOPED_TRACE_SAFE("buffer size: %d\n", bufferSize);

        int64_t totalSize = std::min(static_cast<int64_t>(TestReadWriteSize),
                                    static_cast<int64_t>(bufferSize) * DivisionCountMax);
        ASSERT_EQ(0, totalSize % bufferSize);

        FileHandle handle;
        NNT_ASSERT_RESULT_SUCCESS(OpenFile(&handle, TestPath.c_str(), OpenMode_Write));
        int64_t fileSize;
        NNT_ASSERT_RESULT_SUCCESS(GetFileSize(&fileSize, handle));
        ASSERT_GE(fileSize, totalSize + offsetDifference);
        FillBufferWithRandomValue(g_ReadWriteBuffer, bufferSize);

        TimeCount timeCount;
        timeCount.StartTime();
        for(int64_t offset = 0; offset < totalSize; offset += bufferSize)
        {
            NNT_ASSERT_RESULT_SUCCESS(WriteFile(
                handle, offset + offsetDifference, g_ReadWriteBuffer, bufferSize,
                WriteOption()));
        }
        NNT_ASSERT_RESULT_SUCCESS(FlushFile(handle));
        timeCount.StopTime();

        CloseFile(handle);
        if( isSaveData )
        {
            timeCount.StartTime();
            NNT_ASSERT_RESULT_SUCCESS(CommitSaveData(mountName.c_str()));
            timeCount.StopTime();
        }

        // パフォーマンス計測結果 push_back 処理（シングルスレッド用）
        AddPerformanceResults(
            g_PerformanceResultsWrite, bufferSize, totalSize, timeCount.GetTotalTime() / 1000);
    }
}

/**
* @brief ファイルのランダムな箇所からの書き込みを計測
*/
void WriteShuffle(const String mountName, int64_t offsetDifference, bool isSaveData) NN_NOEXCEPT
{
    Vector<PerformanceResult> perfResults;
    const String TestPath = mountName + ":/" + TestDirectoryName + "/" + TestFileName;

    for(int bufferSize = BufferSizeMin ; bufferSize <= BufferSizeMax ; bufferSize *= 2)
    {
        NNT_FS_SCOPED_TRACE_SAFE("buffer size: %d\n", bufferSize);

        int64_t totalSize = std::min(static_cast<int64_t>(TestReadWriteSize),
                                    static_cast<int64_t>(bufferSize) * DivisionCountMax);
        ASSERT_EQ(0, totalSize % bufferSize);

        FileHandle handle;
        NNT_ASSERT_RESULT_SUCCESS(OpenFile(&handle, TestPath.c_str(), OpenMode_Write));
        int64_t fileSize;
        NNT_ASSERT_RESULT_SUCCESS(GetFileSize(&fileSize, handle));
        ASSERT_GE(fileSize, totalSize + offsetDifference);
        FillBufferWithRandomValue(g_ReadWriteBuffer, bufferSize);

        // オフセット値格納用の配列を作成
        int arrayNum = static_cast<int>(totalSize / bufferSize);
        std::unique_ptr<int[]> offsetArray(new int[arrayNum]);
        CreateRandomArray(offsetArray.get(), arrayNum, bufferSize);

        TimeCount timeCount;
        timeCount.StartTime();
        for(int i = 0; i < arrayNum; i++)
        {
            NNT_ASSERT_RESULT_SUCCESS(WriteFile(
                handle, offsetArray[i] + offsetDifference, g_ReadWriteBuffer, bufferSize,
                WriteOption()));
            NNT_ASSERT_RESULT_SUCCESS(FlushFile(handle));
        }
        timeCount.StopTime();
        CloseFile(handle);

        if( isSaveData )
        {
            timeCount.StartTime();
            NNT_ASSERT_RESULT_SUCCESS(CommitSaveData(mountName.c_str()));
            timeCount.StopTime();
        }

        // パフォーマンス計測結果 push_back 処理（シングルスレッド用）
        AddPerformanceResults(
            g_PerformanceResultsWrite, bufferSize, totalSize, timeCount.GetTotalTime() / 1000);
    }
}

void DummyWriteAndRead(const String mountName, bool isCommitFunctionValid) NN_NOEXCEPT
{
    const String DummyPath = mountName + ":/" + TestDirectoryName + "/" + DummyTestFileName;

    {
        FileHandle handle;
        NNT_ASSERT_RESULT_SUCCESS(OpenFile(&handle, DummyPath.c_str(), OpenMode_Read | OpenMode_Write));
        NN_UTIL_SCOPE_EXIT
        {
            CloseFile(handle);
        };
        FillBufferWithRandomValue(g_ReadWriteBuffer, BufferSizeMax);

        for(int offset = 0 ; offset < TestReadWriteSize ; offset += BufferSizeMax)
        {
            NNT_ASSERT_RESULT_SUCCESS(WriteFile(handle, offset, g_ReadWriteBuffer, BufferSizeMax, WriteOption()));
        }
        NNT_ASSERT_RESULT_SUCCESS(FlushFile(handle));
    }

    if( isCommitFunctionValid )
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CommitSaveData(mountName.c_str()));
    }

    {
        FileHandle handle;
        NNT_ASSERT_RESULT_SUCCESS(OpenFile(&handle, DummyPath.c_str(), OpenMode_Read | OpenMode_Write));
        NN_UTIL_SCOPE_EXIT
        {
            CloseFile(handle);
        };

        for(int offset = 0 ; offset < TestReadWriteSize ; offset += BufferSizeMax)
        {
            NNT_ASSERT_RESULT_SUCCESS(ReadFile(handle, offset, g_ReadWriteBuffer, BufferSizeMax));
        }
    }
}

/**
* @brief ファイルのシーケンシャル読み込みを計測
*/
void ReadSequential(const String mountName, int64_t offsetDifference, bool isSaveData) NN_NOEXCEPT
{
    Vector<PerformanceResult> perfResults;
    const String TestPath = mountName + ":/" + TestDirectoryName + "/" + TestFileName;

    for(int bufferSize = BufferSizeMin ; bufferSize <= BufferSizeMax ; bufferSize *= 2)
    {
        FileHandle handle;
        NNT_ASSERT_RESULT_SUCCESS(OpenFile(&handle, TestPath.c_str(), OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            CloseFile(handle);
        };

        int64_t totalSize = std::min(static_cast<int64_t>(TestReadWriteSize),
                                    static_cast<int64_t>(bufferSize) * DivisionCountMax);
        ASSERT_EQ(0, totalSize % bufferSize);
        int64_t fileSize;
        NNT_ASSERT_RESULT_SUCCESS(GetFileSize(&fileSize, handle));
        ASSERT_GE(fileSize, totalSize + offsetDifference);

        DummyWriteAndRead(mountName, isSaveData);

        TimeCount timeCount;
        timeCount.StartTime();
        for(int64_t offset = 0; offset < totalSize; offset += bufferSize)
        {
            NNT_ASSERT_RESULT_SUCCESS(ReadFile(handle, offset + offsetDifference
                                    , g_ReadWriteBuffer, bufferSize));
        }
        timeCount.StopTime();

        // パフォーマンス計測結果 push_back 処理（シングルスレッド用）
        AddPerformanceResults(g_PerformanceResultsRead, bufferSize, totalSize, timeCount.GetEndTime() / 1000);
    }
}


/**
* @brief ファイルのランダムな箇所からの読み込みを計測
*/
void ReadShuffle(const String mountName, int64_t offsetDifference, bool isSaveData) NN_NOEXCEPT
{
    Vector<PerformanceResult> perfResults;
    const String TestPath = mountName + ":/" + TestDirectoryName + "/" + TestFileName;

    for(int bufferSize = BufferSizeMin ; bufferSize <= BufferSizeMax ; bufferSize *= 2)
    {
        FileHandle handle;
        NNT_ASSERT_RESULT_SUCCESS(OpenFile(&handle, TestPath.c_str(), OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            CloseFile(handle);
        };

        int64_t totalSize = std::min(static_cast<int64_t>(TestReadWriteSize),
                                    static_cast<int64_t>(bufferSize) * DivisionCountMax);
        ASSERT_EQ(0, totalSize % bufferSize);
        int64_t fileSize;
        NNT_ASSERT_RESULT_SUCCESS(GetFileSize(&fileSize, handle));
        ASSERT_GE(fileSize, totalSize + offsetDifference);

        // オフセット値格納用の配列を作成
        int arrayNum = static_cast<int>(totalSize / bufferSize);
        std::unique_ptr<int[]> offsetArray(new int[arrayNum]);
        CreateRandomArray(offsetArray.get(), arrayNum, bufferSize);

        DummyWriteAndRead(mountName, isSaveData);

        TimeCount timeCount;
        timeCount.StartTime();
        for(int i = 0; i < arrayNum; i++)
        {
            NNT_ASSERT_RESULT_SUCCESS(ReadFile(handle, offsetArray[i] + offsetDifference
                                    , g_ReadWriteBuffer, bufferSize));
        }
        timeCount.StopTime();

        // パフォーマンス計測結果 push_back 処理（シングルスレッド用）
        AddPerformanceResults(g_PerformanceResultsRead, bufferSize, totalSize, timeCount.GetEndTime() / 1000);
    }
}

/**
* @brief FlushFileの実行時間を計測
* @pre mountName に対象ファイルシステムがマウント済み
* @param[in] bufferSize 対象となるファイルサイズ
*/
void FlushFileTest(const String mountName, const int bufferSize) NN_NOEXCEPT
{
    Vector<PerformanceResult> perfResults;
    const String TestPath = mountName + ":/" + TestDirectoryName + "/" + TestFileName;

    FileHandle handle;
    NNT_ASSERT_RESULT_SUCCESS(OpenFile(&handle, TestPath.c_str(), OpenMode_Write | OpenMode_AllowAppend));

    FillBufferWithRandomValue(g_ReadWriteBuffer, bufferSize);

    NNT_ASSERT_RESULT_SUCCESS(WriteFile(handle, 0, g_ReadWriteBuffer, bufferSize, WriteOption()));

    TimeCount timeCount;

    timeCount.StartTime();
    NNT_ASSERT_RESULT_SUCCESS(FlushFile(handle));
    timeCount.StopTime();

    // パフォーマンス計測結果 push_back 処理（シングルスレッド用）
    AddPerformanceResults(g_PerformanceResultsWrite, bufferSize, bufferSize, timeCount.GetEndTime() / 1000);

    CloseFile(handle);
}

/**
* @brief GetFileOpenModeの実行時間を計測
* @pre mountName に対象ファイルシステムがマウント済み
* @param[in] openMode 対象となるファイルのオープンモード
*/
void GetFileOpenModeTest(const String mountName, const int openMode) NN_NOEXCEPT
{
    auto comment = AllocateBuffer(32);
    nn::util::SNPrintf(comment.get(), 32, "%s%s%s", (openMode == OpenMode_Read ? " Read" :openMode & OpenMode_Read ? " Read |" :"")
                                            , (openMode & OpenMode_Write ? openMode & OpenMode_AllowAppend ? " Write |" :" Write" :"")
                                            , (openMode & OpenMode_AllowAppend ? " AllowAppend" :""));
    NN_LOG("%32s", comment.get());

    const String TestPath = mountName + ":/" + TestDirectoryName + "/" + TestFileName;

    TimeCount timeCount;
    FileHandle handle;
    for(int i = 0; i < FileCount; i++)
    {
        NNT_FS_SCOPED_TRACE_SAFE("loop count: %d, file path: %s\n", i, TestPath.c_str());
        NNT_ASSERT_RESULT_SUCCESS(OpenFile(&handle, TestPath.c_str(), openMode));

        timeCount.StartTime();
        GetFileOpenMode(handle);
        timeCount.StopTime();

        CloseFile(handle);
    }

    NN_LOG("        %9lld\n", timeCount.GetTotalTime() / 1000);

}


} // namespace

//!< HostFs で 1 ファイルを作成し、指定されたバッファサイズで 256 MB 分読み書きする時間を計測する
TEST_F(PerformanceTestReadWriteHostFs, ReadFile_WriteFile)
{
    static const char* TestTag = "HostFs";
    const String mountName = "perftest";
    const String directoryPath = mountName + ":/" + TestDirectoryName;
    const String TestPath = mountName + ":/" + TestDirectoryName + "/" + TestFileName;
    const String DummyPath = mountName + ":/" + TestDirectoryName + "/" + DummyTestFileName;
    CreateWorkRootPath();
    NNT_EXPECT_RESULT_SUCCESS(MountHost(mountName.c_str(), GetWorkRootPath().c_str()));
    DeleteDirectoryRecursively(directoryPath.c_str());
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(directoryPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile(TestPath.c_str(), static_cast<int64_t>(TestReadWriteSize + AlignmentPadding)));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile(DummyPath.c_str(), static_cast<int64_t>(TestReadWriteSize + AlignmentPadding)));

    {
        g_PerformanceResultsWrite.clear();
        WriteSequential(mountName, 0, false);
        WriteShuffle(mountName, 0, false);

        NN_LOG("Write Performance Test\n");
        NN_LOG("                                                          Sequential                           Random\n");
        DumpPerformanceResults(g_PerformanceResultsWrite);
        NN_LOG("\n\n");

        nnt::fs::util::String tag;
        GetTeamCityTag(&tag, TestTag, AccessType::Write, Alignment::Aligned, RelationWithSpeedEmulation::NoEffect);
        WriteTeamCityPerformanceResults(tag.c_str(), g_PerformanceResultsWrite);
    }

    {
        g_PerformanceResultsRead.clear();
        ReadSequential(mountName, 0, false);
        ReadShuffle(mountName, 0, false);

        NN_LOG("Read Performance Test\n");
        NN_LOG("                                                          Sequential                           Random\n");
        DumpPerformanceResults(g_PerformanceResultsRead);
        NN_LOG("\n\n");

        nnt::fs::util::String tag;
        GetTeamCityTag(&tag, TestTag, AccessType::Read, Alignment::Aligned, RelationWithSpeedEmulation::NoEffect);
        WriteTeamCityPerformanceResults(tag.c_str(), g_PerformanceResultsRead);
    }

    NNT_ASSERT_RESULT_SUCCESS(DeleteFile(TestPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(DeleteFile(DummyPath.c_str()));
    Unmount(mountName.c_str());
    DeleteWorkRootPath();
}

//!< HostFs で 1 ファイルを作成し、指定されたバッファサイズで 256 MB 分読み書きする時間を計測する (非アライメント)
TEST_F(PerformanceTestReadWriteHostFs, ReadFile_WriteFileUnaligned)
{
    static const char* TestTag = "HostFs";
    String mountName = "perftest";
    String directoryPath = mountName + ":/" + TestDirectoryName;
    const String TestPath = mountName + ":/" + TestDirectoryName + "/" + TestFileName;
    const String DummyPath = mountName + ":/" + TestDirectoryName + "/" + DummyTestFileName;
    CreateWorkRootPath();
    NNT_EXPECT_RESULT_SUCCESS(MountHost(mountName.c_str(), GetWorkRootPath().c_str()));
    DeleteDirectoryRecursively(directoryPath.c_str());
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(directoryPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile(TestPath.c_str(), static_cast<int64_t>(TestReadWriteSize + AlignmentPadding)));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile(DummyPath.c_str(), static_cast<int64_t>(TestReadWriteSize + AlignmentPadding)));

    {
        g_PerformanceResultsWrite.clear();
        WriteSequential(mountName, AlignmentPadding, false);
        WriteShuffle(mountName, AlignmentPadding, false);

        NN_LOG("Unaligned Write Performance Test\n");
        NN_LOG("                                                          Sequential                           Random\n");
        DumpPerformanceResults(g_PerformanceResultsWrite);
        NN_LOG("\n\n");

        nnt::fs::util::String tag;
        GetTeamCityTag(&tag, TestTag, AccessType::Write, Alignment::Unaligned, RelationWithSpeedEmulation::NoEffect);
        WriteTeamCityPerformanceResults(tag.c_str(), g_PerformanceResultsWrite);
    }

    {
        g_PerformanceResultsRead.clear();
        ReadSequential(mountName, AlignmentPadding, false);
        ReadShuffle(mountName, AlignmentPadding, false);

        NN_LOG("Unaligned Read Performance Test\n");
        NN_LOG("                                                          Sequential                           Random\n");
        DumpPerformanceResults(g_PerformanceResultsRead);
        NN_LOG("\n\n");

        nnt::fs::util::String tag;
        GetTeamCityTag(&tag, TestTag, AccessType::Read, Alignment::Unaligned, RelationWithSpeedEmulation::NoEffect);
        WriteTeamCityPerformanceResults(tag.c_str(), g_PerformanceResultsRead);
    }

    NNT_ASSERT_RESULT_SUCCESS(DeleteFile(TestPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(DeleteFile(DummyPath.c_str()));
    Unmount(mountName.c_str());
    DeleteWorkRootPath();
}

//!< SdCard で 1 ファイルを作成し、指定されたバッファサイズで 256 MB 分読み書きする時間を計測する
TEST(PerformanceTestReadWriteSdCard, ReadFile_WriteFile)
{
    static const char* TestTag = "SdCard";
    const String mountName = "perftest";
    const String directoryPath = mountName + ":/" + TestDirectoryName;
    const String TestPath = mountName + ":/" + TestDirectoryName + "/" + TestFileName;
    const String DummyPath = mountName + ":/" + TestDirectoryName + "/" + DummyTestFileName;
    NNT_ASSERT_RESULT_SUCCESS(MountSdCardForDebug(mountName.c_str()));
    DeleteDirectoryRecursively(directoryPath.c_str());
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(directoryPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile(TestPath.c_str(), static_cast<int64_t>(TestReadWriteSize + AlignmentPadding)));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile(DummyPath.c_str(), static_cast<int64_t>(TestReadWriteSize + AlignmentPadding)));

    {
        g_PerformanceResultsWrite.clear();
        WriteSequential(mountName, 0, false);
        WriteShuffle(mountName, 0, false);

        NN_LOG("Write Performance Test\n");
        NN_LOG("                                                          Sequential                           Random\n");
        DumpPerformanceResults(g_PerformanceResultsWrite);
        NN_LOG("\n\n");

        nnt::fs::util::String tag;
        GetTeamCityTag(&tag, TestTag, AccessType::Write, Alignment::Aligned, RelationWithSpeedEmulation::NoEffect);
        WriteTeamCityPerformanceResults(tag.c_str(), g_PerformanceResultsWrite);
    }

    {
        g_PerformanceResultsRead.clear();
        ReadSequential(mountName, 0, false);
        ReadShuffle(mountName, 0, false);

        NN_LOG("Read Performance Test\n");
        NN_LOG("                                                          Sequential                           Random\n");
        DumpPerformanceResults(g_PerformanceResultsRead);
        NN_LOG("\n\n");

        nnt::fs::util::String tag;
        GetTeamCityTag(&tag, TestTag, AccessType::Read, Alignment::Aligned, RelationWithSpeedEmulation::NoEffect);
        WriteTeamCityPerformanceResults(tag.c_str(), g_PerformanceResultsRead);

        EXPECT_TRUE(nnt::fs::util::CheckPerformanceResult(tag.c_str(), g_PerformanceResultsRead, false));
    }

    NNT_ASSERT_RESULT_SUCCESS(DeleteFile(TestPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(DeleteFile(DummyPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(DeleteDirectoryRecursively(directoryPath.c_str()));
    Unmount(MountName);
}

//!< SdCard で 1 ファイルを作成し、指定されたバッファサイズで 256 MB 分読み書きする時間を計測する (非アライメント)
TEST(PerformanceTestReadWriteSdCard, ReadFile_WriteFileUnaligned)
{
    static const char* TestTag = "SdCard";
    String mountName = "perftest";
    String directoryPath = mountName + ":/" + TestDirectoryName;
    const String TestPath = mountName + ":/" + TestDirectoryName + "/" + TestFileName;
    const String DummyPath = mountName + ":/" + TestDirectoryName + "/" + DummyTestFileName;
    NNT_ASSERT_RESULT_SUCCESS(MountSdCardForDebug(mountName.c_str()));
    DeleteDirectoryRecursively(directoryPath.c_str());
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(directoryPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile(TestPath.c_str(), static_cast<int64_t>(TestReadWriteSize + AlignmentPadding)));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile(DummyPath.c_str(), static_cast<int64_t>(TestReadWriteSize + AlignmentPadding)));

    {
        g_PerformanceResultsWrite.clear();
        WriteSequential(mountName, AlignmentPadding, false);
        WriteShuffle(mountName, AlignmentPadding, false);

        NN_LOG("Unaligned Write Performance Test\n");
        NN_LOG("                                                          Sequential                           Random\n");
        DumpPerformanceResults(g_PerformanceResultsWrite);
        NN_LOG("\n\n");

        nnt::fs::util::String tag;
        GetTeamCityTag(&tag, TestTag, AccessType::Write, Alignment::Unaligned, RelationWithSpeedEmulation::NoEffect);
        WriteTeamCityPerformanceResults(tag.c_str(), g_PerformanceResultsWrite);
    }

    {
        g_PerformanceResultsRead.clear();
        ReadSequential(mountName, AlignmentPadding, false);
        ReadShuffle(mountName, AlignmentPadding, false);

        NN_LOG("Unaligned Read Performance Test\n");
        NN_LOG("                                                          Sequential                           Random\n");
        DumpPerformanceResults(g_PerformanceResultsRead);
        NN_LOG("\n\n");

        nnt::fs::util::String tag;
        GetTeamCityTag(&tag, TestTag, AccessType::Read, Alignment::Unaligned, RelationWithSpeedEmulation::NoEffect);
        WriteTeamCityPerformanceResults(tag.c_str(), g_PerformanceResultsRead);
    }

    NNT_ASSERT_RESULT_SUCCESS(DeleteFile(TestPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(DeleteFile(DummyPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(DeleteDirectoryRecursively(directoryPath.c_str()));
    Unmount(MountName);
}

//!< BisFs で 1 ファイルを作成し、指定されたバッファサイズで 256 MB 分読み書きする時間を計測する
TEST(PerformanceTestReadWriteBisFs, ReadFile_WriteFile)
{
    const char* TestTag = "BisFs";
    const String mountName(nn::fs::GetBisMountName(nn::fs::BisPartitionId::User));
    const String directoryPath = mountName + ":/" + TestDirectoryName;
    const String TestPath = mountName + ":/" + TestDirectoryName + "/" + TestFileName;
    const String DummyPath = mountName + ":/" + TestDirectoryName + "/" + DummyTestFileName;
    NNT_ASSERT_RESULT_SUCCESS(MountBis(nn::fs::BisPartitionId::User, nullptr));
    DeleteDirectoryRecursively(directoryPath.c_str());
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(directoryPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile(TestPath.c_str(), static_cast<int64_t>(TestReadWriteSize + AlignmentPadding)));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile(DummyPath.c_str(), static_cast<int64_t>(TestReadWriteSize + AlignmentPadding)));

    {
        g_PerformanceResultsWrite.clear();
        WriteSequential(mountName, 0, false);
        WriteShuffle(mountName, 0, false);

        NN_LOG("Write Performance Test\n");
        NN_LOG("                                                          Sequential                           Random\n");
        DumpPerformanceResults(g_PerformanceResultsWrite);
        NN_LOG("\n\n");

        nnt::fs::util::String tag;
        GetTeamCityTag(&tag, TestTag, AccessType::Write, Alignment::Aligned, RelationWithSpeedEmulation::NoEffect);
        WriteTeamCityPerformanceResults(tag.c_str(), g_PerformanceResultsWrite);
    }

    {
        g_PerformanceResultsRead.clear();
        ReadSequential(mountName, 0, false);
        ReadShuffle(mountName, 0, false);

        NN_LOG("Read Performance Test\n");
        NN_LOG("                                                          Sequential                           Random\n");
        DumpPerformanceResults(g_PerformanceResultsRead);
        NN_LOG("\n\n");

        nnt::fs::util::String tag;
        GetTeamCityTag(&tag, TestTag, AccessType::Read, Alignment::Aligned, RelationWithSpeedEmulation::NoEffect);
        WriteTeamCityPerformanceResults(tag.c_str(), g_PerformanceResultsRead);

        EXPECT_TRUE(nnt::fs::util::CheckPerformanceResult(tag.c_str(), g_PerformanceResultsRead, false));
    }

    NNT_ASSERT_RESULT_SUCCESS(DeleteFile(TestPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(DeleteFile(DummyPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(DeleteDirectoryRecursively(directoryPath.c_str()));
    Unmount(mountName.c_str());
}

//!< BisFs で 1 ファイルを作成し、指定されたバッファサイズで 256 MB 分読み書きする時間を計測する (非アライメント)
TEST(PerformanceTestReadWriteBisFs, ReadFile_WriteFileUnaligned)
{
    const char* TestTag = "BisFs";
    String mountName(nn::fs::GetBisMountName(nn::fs::BisPartitionId::User));
    String directoryPath = mountName + ":/" + TestDirectoryName;
    const String TestPath = mountName + ":/" + TestDirectoryName + "/" + TestFileName;
    const String DummyPath = mountName + ":/" + TestDirectoryName + "/" + DummyTestFileName;
    NNT_ASSERT_RESULT_SUCCESS(MountBis(nn::fs::BisPartitionId::User, nullptr));
    DeleteDirectoryRecursively(directoryPath.c_str());
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(directoryPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile(TestPath.c_str(), static_cast<int64_t>(TestReadWriteSize + AlignmentPadding)));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile(DummyPath.c_str(), static_cast<int64_t>(TestReadWriteSize + AlignmentPadding)));

    {
        g_PerformanceResultsWrite.clear();
        WriteSequential(mountName, AlignmentPadding, false);
        WriteShuffle(mountName, AlignmentPadding, false);

        NN_LOG("Unaligned Write Performance Test\n");
        NN_LOG("                                                          Sequential                           Random\n");
        DumpPerformanceResults(g_PerformanceResultsWrite);
        NN_LOG("\n\n");

        nnt::fs::util::String tag;
        GetTeamCityTag(&tag, TestTag, AccessType::Write, Alignment::Unaligned, RelationWithSpeedEmulation::NoEffect);
        WriteTeamCityPerformanceResults(tag.c_str(), g_PerformanceResultsWrite);
    }

    {
        g_PerformanceResultsRead.clear();
        ReadSequential(mountName, AlignmentPadding, false);
        ReadShuffle(mountName, AlignmentPadding, false);

        NN_LOG("Unaligned Read Performance Test\n");
        NN_LOG("                                                          Sequential                           Random\n");
        DumpPerformanceResults(g_PerformanceResultsRead);
        NN_LOG("\n\n");

        nnt::fs::util::String tag;
        GetTeamCityTag(&tag, TestTag, AccessType::Read, Alignment::Unaligned, RelationWithSpeedEmulation::NoEffect);
        WriteTeamCityPerformanceResults(tag.c_str(), g_PerformanceResultsRead);
    }

    NNT_ASSERT_RESULT_SUCCESS(DeleteFile(TestPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(DeleteFile(DummyPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(DeleteDirectoryRecursively(directoryPath.c_str()));
    Unmount(mountName.c_str());
}

/**
* @brief ContentStorageSd で 1 ファイルを作成し、
* @brief 指定されたバッファサイズで 256 MB 分読み書きする時間を計測する
*/
TEST_F(PerformanceTestContentStorageSd, ReadFile_WriteFile)
{
    static const char* TestTag = "ContentStorageSd";
    const String mountName(nn::fs::GetContentStorageMountName(nn::fs::ContentStorageId::SdCard));
    const String directoryPath = mountName + ":/" + TestDirectoryName;
    const String TestPath = mountName + ":/" + TestDirectoryName + "/" + TestFileName;
    const String DummyPath = mountName + ":/" + TestDirectoryName + "/" + DummyTestFileName;
    CreateWorkRootPath();
    NNT_ASSERT_RESULT_SUCCESS(MountContentStorage(nn::fs::ContentStorageId::SdCard));
    DeleteDirectoryRecursively(directoryPath.c_str());
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(directoryPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile(TestPath.c_str(), static_cast<int64_t>(TestReadWriteSize + AlignmentPadding)));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile(DummyPath.c_str(), static_cast<int64_t>(TestReadWriteSize + AlignmentPadding)));

    {
        g_PerformanceResultsWrite.clear();
        WriteSequential(mountName, 0, false);
        WriteShuffle(mountName, 0, false);

        NN_LOG("Write Performance Test\n");
        NN_LOG("                                                          Sequential                           Random\n");
        DumpPerformanceResults(g_PerformanceResultsWrite);
        NN_LOG("\n\n");

        nnt::fs::util::String tag;
        GetTeamCityTag(&tag, TestTag, AccessType::Write, Alignment::Aligned, RelationWithSpeedEmulation::NoEffect);
        WriteTeamCityPerformanceResults(tag.c_str(), g_PerformanceResultsWrite);
    }

    {
        g_PerformanceResultsRead.clear();
        ReadSequential(mountName, 0, false);
        ReadShuffle(mountName, 0, false);

        NN_LOG("Read Performance Test\n");
        NN_LOG("                                                          Sequential                           Random\n");
        DumpPerformanceResults(g_PerformanceResultsRead);
        NN_LOG("\n\n");

        nnt::fs::util::String tag;
        GetTeamCityTag(&tag, TestTag, AccessType::Read, Alignment::Aligned, RelationWithSpeedEmulation::NoEffect);
        WriteTeamCityPerformanceResults(tag.c_str(), g_PerformanceResultsRead);
    }

    NNT_ASSERT_RESULT_SUCCESS(DeleteFile(TestPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(DeleteFile(DummyPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(DeleteDirectoryRecursively(directoryPath.c_str()));
    Unmount(mountName.c_str());
}

/**
* @brief ContentStorageSd で 1 ファイルを作成し、
* @brief 指定されたバッファサイズで 256 MB 分読み書きする時間を計測する
*/
TEST_F(PerformanceTestContentStorageSd, ReadFile_WriteFileUnaligned)
{
    static const char* TestTag = "ContentStorageSd";
    String mountName(nn::fs::GetContentStorageMountName(nn::fs::ContentStorageId::SdCard));
    String directoryPath = mountName + ":/" + TestDirectoryName;
    const String TestPath = mountName + ":/" + TestDirectoryName + "/" + TestFileName;
    const String DummyPath = mountName + ":/" + TestDirectoryName + "/" + DummyTestFileName;
    CreateWorkRootPath();
    NNT_ASSERT_RESULT_SUCCESS(MountContentStorage(nn::fs::ContentStorageId::SdCard));
    DeleteDirectoryRecursively(directoryPath.c_str());
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(directoryPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile(TestPath.c_str(), static_cast<int64_t>(TestReadWriteSize + AlignmentPadding)));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile(DummyPath.c_str(), static_cast<int64_t>(TestReadWriteSize + AlignmentPadding)));

    {
        g_PerformanceResultsWrite.clear();
        WriteSequential(mountName, AlignmentPadding, false);
        WriteShuffle(mountName, AlignmentPadding, false);

        NN_LOG("Unalignd Write Performance Test\n");
        NN_LOG("                                                          Sequential                           Random\n");
        DumpPerformanceResults(g_PerformanceResultsWrite);
        NN_LOG("\n\n");

        nnt::fs::util::String tag;
        GetTeamCityTag(&tag, TestTag, AccessType::Write, Alignment::Unaligned, RelationWithSpeedEmulation::NoEffect);
        WriteTeamCityPerformanceResults(tag.c_str(), g_PerformanceResultsWrite);
    }

    {
        g_PerformanceResultsRead.clear();
        ReadSequential(mountName, AlignmentPadding, false);
        ReadShuffle(mountName, AlignmentPadding, false);

        NN_LOG("Read Performance Test\n");
        NN_LOG("                                                          Sequential                           Random\n");
        DumpPerformanceResults(g_PerformanceResultsRead);
        NN_LOG("\n\n");

        nnt::fs::util::String tag;
        GetTeamCityTag(&tag, TestTag, AccessType::Read, Alignment::Unaligned, RelationWithSpeedEmulation::NoEffect);
        WriteTeamCityPerformanceResults(tag.c_str(), g_PerformanceResultsRead);
    }

    NNT_ASSERT_RESULT_SUCCESS(DeleteFile(TestPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(DeleteFile(DummyPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(DeleteDirectoryRecursively(directoryPath.c_str()));
    Unmount(mountName.c_str());
}

/**
* @brief ContentStorageNand で 1 ファイルを作成し、
* @brief 指定されたバッファサイズで 256 MB 分読み書きする時間を計測する
*/
TEST_F(PerformanceTestContentStorageNand, ReadFile_WriteFile)
{
    static const char* TestTag = "ContentStorageNand";
    const String mountName(nn::fs::GetContentStorageMountName(nn::fs::ContentStorageId::System));
    const String directoryPath = mountName + ":/" + TestDirectoryName;
    const String TestPath = mountName + ":/" + TestDirectoryName + "/" + TestFileName;
    const String DummyPath = mountName + ":/" + TestDirectoryName + "/" + DummyTestFileName;
    CreateWorkRootPath();
    NNT_ASSERT_RESULT_SUCCESS(MountContentStorage(nn::fs::ContentStorageId::System));
    DeleteDirectoryRecursively(directoryPath.c_str());
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(directoryPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile(TestPath.c_str(), static_cast<int64_t>(TestReadWriteSize + AlignmentPadding)));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile(DummyPath.c_str(), static_cast<int64_t>(TestReadWriteSize + AlignmentPadding)));

    {
        g_PerformanceResultsWrite.clear();
        WriteSequential(mountName, 0, false);
        WriteShuffle(mountName, 0, false);

        NN_LOG("Write Performance Test\n");
        NN_LOG("                                                          Sequential                           Random\n");
        DumpPerformanceResults(g_PerformanceResultsWrite);
        NN_LOG("\n\n");

        nnt::fs::util::String tag;
        GetTeamCityTag(&tag, TestTag, AccessType::Write, Alignment::Aligned, RelationWithSpeedEmulation::NoEffect);
        WriteTeamCityPerformanceResults(tag.c_str(), g_PerformanceResultsWrite);
    }

    {
        g_PerformanceResultsRead.clear();
        ReadSequential(mountName, 0, false);
        ReadShuffle(mountName, 0, false);

        NN_LOG("Read Performance Test\n");
        NN_LOG("                                                          Sequential                           Random\n");
        DumpPerformanceResults(g_PerformanceResultsRead);
        NN_LOG("\n\n");

        nnt::fs::util::String tag;
        GetTeamCityTag(&tag, TestTag, AccessType::Read, Alignment::Aligned, RelationWithSpeedEmulation::NoEffect);
        WriteTeamCityPerformanceResults(tag.c_str(), g_PerformanceResultsRead);
    }

    NNT_ASSERT_RESULT_SUCCESS(DeleteFile(TestPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(DeleteFile(DummyPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(DeleteDirectoryRecursively(directoryPath.c_str()));
    Unmount(mountName.c_str());
}

/**
* @brief ContentStorageNand で 1 ファイルを作成し、
* @brief 指定されたバッファサイズで 256 MB 分読み書きする時間を計測する
* @brief (非アライメント)
*/
TEST_F(PerformanceTestContentStorageNand, ReadFile_WriteFileUnaligned)
{
    static const char* TestTag = "ContentStorageNand";
    String mountName(nn::fs::GetContentStorageMountName(nn::fs::ContentStorageId::System));
    String directoryPath = mountName + ":/" + TestDirectoryName;
    const String TestPath = mountName + ":/" + TestDirectoryName + "/" + TestFileName;
    const String DummyPath = mountName + ":/" + TestDirectoryName + "/" + DummyTestFileName;
    CreateWorkRootPath();
    NNT_ASSERT_RESULT_SUCCESS(MountContentStorage(nn::fs::ContentStorageId::System));
    DeleteDirectoryRecursively(directoryPath.c_str());
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(directoryPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile(TestPath.c_str(), static_cast<int64_t>(TestReadWriteSize + AlignmentPadding)));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile(DummyPath.c_str(), static_cast<int64_t>(TestReadWriteSize + AlignmentPadding)));

    {
        g_PerformanceResultsWrite.clear();
        WriteSequential(mountName, AlignmentPadding, false);
        WriteShuffle(mountName, AlignmentPadding, false);

        NN_LOG("Unaligned Write Performance Test\n");
        NN_LOG("                                                          Sequential                           Random\n");
        DumpPerformanceResults(g_PerformanceResultsWrite);
        NN_LOG("\n\n");

        nnt::fs::util::String tag;
        GetTeamCityTag(&tag, TestTag, AccessType::Write, Alignment::Unaligned, RelationWithSpeedEmulation::NoEffect);
        WriteTeamCityPerformanceResults(tag.c_str(), g_PerformanceResultsWrite);
    }

    {
        g_PerformanceResultsRead.clear();
        ReadSequential(mountName, AlignmentPadding, false);
        ReadShuffle(mountName, AlignmentPadding, false);

        NN_LOG("Unaligned Read Performance Test\n");
        NN_LOG("                                                          Sequential                           Random\n");
        DumpPerformanceResults(g_PerformanceResultsRead);
        NN_LOG("\n\n");

        nnt::fs::util::String tag;
        GetTeamCityTag(&tag, TestTag, AccessType::Read, Alignment::Unaligned, RelationWithSpeedEmulation::NoEffect);
        WriteTeamCityPerformanceResults(tag.c_str(), g_PerformanceResultsRead);
    }

    NNT_ASSERT_RESULT_SUCCESS(DeleteFile(TestPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(DeleteFile(DummyPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(DeleteDirectoryRecursively(directoryPath.c_str()));
    Unmount(mountName.c_str());
}

void PerformanceTestReadWriteSystemSaveDataImpl(bool isAligned)
{
    const SystemSaveDataId id = 0x8000000000000003;
    const size_t saveSize = TestReadWriteSize * 2 + 1024 * 1024;
    const String mountName = "save";
    const String directoryPath = mountName + ":/" + TestDirectoryName;
    const String TestPath = mountName + ":/" + TestDirectoryName + "/" + TestFileName;
    const String DummyPath = mountName + ":/" + TestDirectoryName + "/" + DummyTestFileName;
    nnt::fs::util::DeleteAllTestSaveData();
    NNT_ASSERT_RESULT_SUCCESS(CreateSystemSaveData(id, saveSize, saveSize, 0));
    NNT_ASSERT_RESULT_SUCCESS(MountSystemSaveData(mountName.c_str(), id));
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(directoryPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile(TestPath.c_str(), static_cast<int64_t>(TestReadWriteSize + AlignmentPadding)));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile(DummyPath.c_str(), static_cast<int64_t>(TestReadWriteSize + AlignmentPadding)));
    NNT_ASSERT_RESULT_SUCCESS(CommitSaveData(mountName.c_str()));

    std::function<nn::Result()> postTest = [=]() -> nn::Result {
        NN_RESULT_SUCCESS;
    };

    const char TestTag[] = "SaveData";

    {
        g_PerformanceResultsWrite.clear();
        WriteSequential(mountName, isAligned ? 0 : AlignmentPadding, true);
        WriteShuffle(mountName, isAligned ? 0 : AlignmentPadding, true);

        NN_LOG("%sWrite Performance Test\n", isAligned ? "" : "Unaligned ");
        NN_LOG("                                                          Sequential                           Random\n");
        DumpPerformanceResults(g_PerformanceResultsWrite);
        NN_LOG("\n\n");

        nnt::fs::util::String tag;
        GetTeamCityTag(
            &tag,
            TestTag,
            AccessType::Write,
            isAligned ? Alignment::Aligned : Alignment::Unaligned,
            RelationWithSpeedEmulation::HasEffect
        );
        WriteTeamCityPerformanceResults(tag.c_str(), g_PerformanceResultsWrite);
    }

    {
        g_PerformanceResultsRead.clear();
        ReadSequential(mountName, isAligned ? 0 : AlignmentPadding, true);
        ReadShuffle(mountName, isAligned ? 0 : AlignmentPadding, true);

        NN_LOG("%sRead Performance Test\n", isAligned ? "" : "Unaligned ");
        NN_LOG("                                                          Sequential                           Random\n");
        DumpPerformanceResults(g_PerformanceResultsRead);
        NN_LOG("\n\n");

        nnt::fs::util::String tag;
        GetTeamCityTag(
            &tag,
            TestTag,
            AccessType::Read,
            isAligned ? Alignment::Aligned : Alignment::Unaligned,
            RelationWithSpeedEmulation::HasEffect
        );
        WriteTeamCityPerformanceResults(tag.c_str(), g_PerformanceResultsRead);

        EXPECT_TRUE(nnt::fs::util::CheckPerformanceResult(tag.c_str(), g_PerformanceResultsRead, false));
        if( g_SpeedEmulationModeName == SpeedEmulationModeNameNone && isAligned )
        {
            nnt::fs::util::String tagSlower;
            GetTeamCityTag(&tagSlower, TestTag, AccessType::Read, Alignment::Aligned, SpeedEmulationModeNameSlower);
            EXPECT_TRUE(nnt::fs::util::CheckPerformanceResult(tagSlower.c_str(), g_PerformanceResultsRead, true));
        }
    }

    NNT_ASSERT_RESULT_SUCCESS(DeleteFile(TestPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(DeleteFile(DummyPath.c_str()));
    Unmount(mountName.c_str());
    nnt::fs::util::DeleteAllTestSaveData();
}


//!< SystemSaveData で 1 ファイルを作成し、指定されたバッファサイズで 256 MB 分読み書きする時間を計測する
TEST(PerformanceTestReadWriteSystemSaveData, ReadFile_WriteFile)
{
    PerformanceTestReadWriteSystemSaveDataImpl(true);
}

TEST(PerformanceTestReadWriteSystemSaveData, ReadFile_WriteFileUnaligned)
{
    PerformanceTestReadWriteSystemSaveDataImpl(false);
}

// CacheStorage, TemporaryStorage は Windows ではテストしない
#if !defined(NN_BUILD_CONFIG_OS_WIN)

void PerformanceTestReadWriteCacheStorageImpl(bool isAligned)
{
    const size_t saveSize = TestReadWriteSize * 2 + 1024 * 1024;
    const auto spaceId = nn::fs::SaveDataSpaceId::SdUser;
    const String mountName = "cache";
    const String directoryPath = mountName + ":/" + TestDirectoryName;
    const String TestPath = mountName + ":/" + TestDirectoryName + "/" + TestFileName;
    const String DummyPath = mountName + ":/" + TestDirectoryName + "/" + DummyTestFileName;
    nnt::fs::util::DeleteAllTestSaveData();
    NNT_ASSERT_RESULT_SUCCESS(CreateCacheStorage(nnt::fs::util::ApplicationId, spaceId, 0, saveSize, saveSize, 0));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountCacheStorage("cache", nnt::fs::util::ApplicationId));
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(directoryPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile(TestPath.c_str(), static_cast<int64_t>(TestReadWriteSize + AlignmentPadding)));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile(DummyPath.c_str(), static_cast<int64_t>(TestReadWriteSize + AlignmentPadding)));
    NNT_ASSERT_RESULT_SUCCESS(CommitSaveData(mountName.c_str()));

    const char TestTag[] = "CacheStorage";

    {
        g_PerformanceResultsWrite.clear();
        WriteSequential(mountName, isAligned ? 0 : AlignmentPadding, true);
        WriteShuffle(mountName, isAligned ? 0 : AlignmentPadding, true);

        NN_LOG("%sWrite Performance Test\n", isAligned ? "" : "Unaligned ");
        NN_LOG("                                                          Sequential                           Random\n");
        DumpPerformanceResults(g_PerformanceResultsWrite);
        NN_LOG("\n\n");

        nnt::fs::util::String tag;
        GetTeamCityTag(
            &tag,
            TestTag,
            AccessType::Write,
            isAligned ? Alignment::Aligned : Alignment::Unaligned,
            RelationWithSpeedEmulation::HasEffect
        );
        WriteTeamCityPerformanceResults(tag.c_str() ,g_PerformanceResultsWrite);
    }

    {
        g_PerformanceResultsRead.clear();
        ReadSequential(mountName, isAligned ? 0 : AlignmentPadding, true);
        ReadShuffle(mountName, isAligned ? 0 : AlignmentPadding, true);

        NN_LOG("%sRead Performance Test\n", isAligned ? "" : "Unaligned ");
        NN_LOG("                                                          Sequential                           Random\n");
        DumpPerformanceResults(g_PerformanceResultsRead);
        NN_LOG("\n\n");

        nnt::fs::util::String tag;
        GetTeamCityTag(
            &tag,
            TestTag,
            AccessType::Read,
            isAligned ? Alignment::Aligned : Alignment::Unaligned,
            RelationWithSpeedEmulation::HasEffect
        );
        WriteTeamCityPerformanceResults(tag.c_str(), g_PerformanceResultsRead);

        EXPECT_TRUE(nnt::fs::util::CheckPerformanceResult(tag.c_str(), g_PerformanceResultsRead, false));
        if( g_SpeedEmulationModeName == SpeedEmulationModeNameNone && isAligned )
        {
            nnt::fs::util::String tagSlower;
            GetTeamCityTag(&tagSlower, TestTag, AccessType::Read, Alignment::Aligned, SpeedEmulationModeNameSlower);
            EXPECT_TRUE(nnt::fs::util::CheckPerformanceResult(tagSlower.c_str(), g_PerformanceResultsRead, true));
        }
    }

    NNT_ASSERT_RESULT_SUCCESS(DeleteFile(TestPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(DeleteFile(DummyPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(DeleteDirectoryRecursively(directoryPath.c_str()));
    Unmount(mountName.c_str());
    nnt::fs::util::DeleteAllTestSaveData();
}

/**
* @brief CacheStorage で 1 ファイルを作成し、
* @brief 指定されたバッファサイズで 256 MB 分読み書きする時間を計測する
*/
TEST(PerformanceTestReadWriteCacheStorage, ReadFile_WriteFile)
{
    PerformanceTestReadWriteCacheStorageImpl(true);
}

/**
* @brief CacheStorage で 1 ファイルを作成し、
* @brief 指定されたバッファサイズで 256 MB 分読み書きする時間を計測する
*/
TEST(PerformanceTestReadWriteCacheStorage, ReadFile_WriteFileUnaligned)
{
    PerformanceTestReadWriteCacheStorageImpl(false);
}

void PerformanceTestReadWriteTemporaryStorageImpl(bool isAligned)
{
    const String mountName = "temp";
    const String directoryPath = mountName + ":/" + TestDirectoryName;
    const String TestPath = mountName + ":/" + TestDirectoryName + "/" + TestFileName;
    const String DummyPath = mountName + ":/" + TestDirectoryName + "/" + DummyTestFileName;
    nnt::fs::util::DeleteAllTestSaveData();
    NNT_ASSERT_RESULT_SUCCESS(MountTemporaryStorage(mountName.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(directoryPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile(TestPath.c_str(), static_cast<int64_t>(TestReadWriteSize + AlignmentPadding)));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile(DummyPath.c_str(), static_cast<int64_t>(TestReadWriteSize + AlignmentPadding)));
    NNT_ASSERT_RESULT_SUCCESS(CommitSaveData(mountName.c_str()));

    const char TestTag[] = "TemporaryStorage";

    {
        g_PerformanceResultsWrite.clear();
        WriteSequential(mountName, isAligned ? 0 : AlignmentPadding, true);
        WriteShuffle(mountName, isAligned ? 0 : AlignmentPadding, true);

        NN_LOG("%sWrite Performance Test\n", isAligned ? "" : "Unaligned ");
        NN_LOG("                                                          Sequential                           Random\n");
        DumpPerformanceResults(g_PerformanceResultsWrite);
        NN_LOG("\n\n");

        nnt::fs::util::String tag;
        GetTeamCityTag(
            &tag,
            TestTag,
            AccessType::Write,
            isAligned ? Alignment::Aligned : Alignment::Unaligned,
            RelationWithSpeedEmulation::HasEffect
        );
        WriteTeamCityPerformanceResults(tag.c_str(), g_PerformanceResultsWrite);
    }

    {
        g_PerformanceResultsRead.clear();
        ReadSequential(mountName, isAligned ? 0 : AlignmentPadding, true);
        ReadShuffle(mountName, isAligned ? 0 : AlignmentPadding, true);

        NN_LOG("%sRead Performance Test\n", isAligned ? "" : "Unaligned ");
        NN_LOG("                                                          Sequential                           Random\n");
        DumpPerformanceResults(g_PerformanceResultsRead);
        NN_LOG("\n\n");

        nnt::fs::util::String tag;
        GetTeamCityTag(
            &tag,
            TestTag,
            AccessType::Read,
            isAligned ? Alignment::Aligned : Alignment::Unaligned,
            RelationWithSpeedEmulation::HasEffect
        );
        WriteTeamCityPerformanceResults(tag.c_str(), g_PerformanceResultsRead);

        EXPECT_TRUE(nnt::fs::util::CheckPerformanceResult(tag.c_str(), g_PerformanceResultsRead, false));
        if( g_SpeedEmulationModeName == SpeedEmulationModeNameNone && isAligned )
        {
            nnt::fs::util::String tagSlower;
            GetTeamCityTag(&tagSlower, TestTag, AccessType::Read, Alignment::Aligned, SpeedEmulationModeNameSlower);
            EXPECT_TRUE(nnt::fs::util::CheckPerformanceResult(tagSlower.c_str(), g_PerformanceResultsRead, true));
        }
    }

    NNT_ASSERT_RESULT_SUCCESS(DeleteFile(TestPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(DeleteFile(DummyPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(DeleteDirectoryRecursively(directoryPath.c_str()));
    Unmount(mountName.c_str());
    nnt::fs::util::DeleteAllTestSaveData();
}

/**
* @brief TemporaryStorage で 1 ファイルを作成し、
* @brief 指定されたバッファサイズで 256 MB 分読み書きする時間を計測する
*/
TEST(PerformanceTestReadWriteTemporaryStorage, ReadFile_WriteFile)
{
    PerformanceTestReadWriteTemporaryStorageImpl(true);
}

/**
* @brief TemporaryStorage で 1 ファイルを作成し、
* @brief 指定されたバッファサイズで 256 MB 分読み書きする時間を計測する
*/
TEST(PerformanceTestReadWriteTemporaryStorage, ReadFile_WriteFileUnaligned)
{
    PerformanceTestReadWriteTemporaryStorageImpl(false);
}

#endif

//!< HostFs で指定されたバッファサイズで書き込まれたファイルを Flush する時間を計測する
TEST_F(PerformanceTestFlushHostFs, FlushFile)
{
    NN_LOG("\nFlushFile HostFs Performance Test\n");

    const String mountName = "perftest";
    const String directoryPath = mountName + ":/" + TestDirectoryName;
    const String TestPath = mountName + ":/" + TestDirectoryName + "/" + TestFileName;
    g_PerformanceResultsWrite.clear();

    int bufferSizes[] = {1 * 1024, 16 * 1024, 32 * 1024, 256 * 1024, 1 * 1024 * 1024, 16 * 1024 * 1024, 32 * 1024 * 1024, 256 * 1024 * 1024 };

    CreateWorkRootPath();
    NNT_EXPECT_RESULT_SUCCESS(MountHost(mountName.c_str(), GetWorkRootPath().c_str()));
    DeleteDirectoryRecursively(directoryPath.c_str());
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(directoryPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile(TestPath.c_str(), 0));
    {
        for (auto bufferSize : bufferSizes)
        {
            FlushFileTest(mountName, bufferSize);
        }
    }
    NNT_ASSERT_RESULT_SUCCESS(DeleteDirectoryRecursively(directoryPath.c_str()));
    Unmount(mountName.c_str());
    DeleteWorkRootPath();

    DumpPerformanceResultsFlush(g_PerformanceResultsWrite);
}

//!< SdCard で指定されたバッファサイズで書き込まれたファイルを Flush する時間を計測する
TEST_F(PerformanceTestFlushSdCard, FlushFile)
{
    NN_LOG("\nFlushFile SdCard Performance Test\n");

    const String mountName = "perftest";
    const String directoryPath = mountName + ":/" + TestDirectoryName;
    const String TestPath = mountName + ":/" + TestDirectoryName + "/" + TestFileName;
    g_PerformanceResultsWrite.clear();

    int bufferSizes[] = {1 * 1024, 16 * 1024, 32 * 1024, 256 * 1024, 1 * 1024 * 1024, 16 * 1024 * 1024, 32 * 1024 * 1024, 256 * 1024 * 1024 };

    NNT_ASSERT_RESULT_SUCCESS(MountSdCardForDebug(mountName.c_str()));
    DeleteDirectoryRecursively(directoryPath.c_str());
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(directoryPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile(TestPath.c_str(), 0));
    {
        for (auto bufferSize : bufferSizes)
        {
            NNT_FS_SCOPED_TRACE("buffer size: %d\n", bufferSize);
            FlushFileTest(mountName, bufferSize);
        }
    }
    NNT_ASSERT_RESULT_SUCCESS(DeleteDirectoryRecursively(directoryPath.c_str()));
    Unmount(mountName.c_str());

    DumpPerformanceResultsFlush(g_PerformanceResultsWrite);
}

//!< BisFs で指定されたバッファサイズで書き込まれたファイルを Flush する時間を計測する
TEST_F(PerformanceTestFlushBisFs, FlushFile)
{
    NN_LOG("\nFlushFile BisFs Performance Test\n");

    const String mountName(nn::fs::GetBisMountName(nn::fs::BisPartitionId::User));
    const String directoryPath = mountName + ":/" + TestDirectoryName;
    const String TestPath = mountName + ":/" + TestDirectoryName + "/" + TestFileName;
    g_PerformanceResultsWrite.clear();

    int bufferSizes[] = {1 * 1024, 16 * 1024, 32 * 1024, 256 * 1024, 1 * 1024 * 1024, 16 * 1024 * 1024, 32 * 1024 * 1024, 256 * 1024 * 1024 };

    NNT_ASSERT_RESULT_SUCCESS(MountBis(nn::fs::BisPartitionId::User, nullptr));
    DeleteDirectoryRecursively(directoryPath.c_str());
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(directoryPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile(TestPath.c_str(), 0));
    {
        for (auto bufferSize : bufferSizes)
        {
            NNT_FS_SCOPED_TRACE("buffer size: %d\n", bufferSize);
            FlushFileTest(mountName, bufferSize);
        }
    }
    NNT_ASSERT_RESULT_SUCCESS(DeleteDirectoryRecursively(directoryPath.c_str()));
    Unmount(mountName.c_str());

    DumpPerformanceResultsFlush(g_PerformanceResultsWrite);
}


//!< HostFs で指定された OpenMode で開かれた 4096 個のファイルの OpenMode を取得する時間を計測する
TEST_F(PerformanceTestOpenModeHostFs, GetFileOpenMode)
{
    NN_LOG("\nGetFileOpenMode HostFs Performance Test\n");

    int openModes[] = { OpenMode_Read,
                        OpenMode_Write,
                        OpenMode_Read  | OpenMode_Write,
                        OpenMode_Read  | OpenMode_AllowAppend,
                        OpenMode_Write | OpenMode_AllowAppend,
                        OpenMode_Read  | OpenMode_Write | OpenMode_AllowAppend};

    const String mountName = "perftest";
    const String directoryPath = mountName + ":/" + TestDirectoryName;
    const String TestPath = mountName + ":/" + TestDirectoryName + "/" + TestFileName;

    NN_LOG("                        OpenMode    Elapsed[usec]\n");
    CreateWorkRootPath();
    NNT_EXPECT_RESULT_SUCCESS(MountHost(mountName.c_str(), GetWorkRootPath().c_str()));
    DeleteDirectoryRecursively(directoryPath.c_str());
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(directoryPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile(TestPath.c_str(), static_cast<int64_t>(TestReadWriteSize)));
    {
        for (auto openMode : openModes)
        {
            GetFileOpenModeTest(mountName, openMode);
        }
    }
    NNT_ASSERT_RESULT_SUCCESS(DeleteDirectoryRecursively(directoryPath.c_str()));
    Unmount(mountName.c_str());
    DeleteWorkRootPath();
    NN_LOG("\n");
}

//!< SdCard で指定された OpenMode で開かれた 4096 個のファイルの OpenMode を取得する時間を計測する
TEST_F(PerformanceTestOpenModeSdCard, GetFileOpenMode)
{
    NN_LOG("\nGetFileOpenMode SdCard Performance Test\n");

    int openModes[] = { OpenMode_Read,
                        OpenMode_Write,
                        OpenMode_Read  | OpenMode_Write,
                        OpenMode_Read  | OpenMode_AllowAppend,
                        OpenMode_Write | OpenMode_AllowAppend,
                        OpenMode_Read  | OpenMode_Write | OpenMode_AllowAppend};

    const String mountName = "perftest";
    const String directoryPath = mountName + ":/" + TestDirectoryName;
    const String TestPath = mountName + ":/" + TestDirectoryName + "/" + TestFileName;

    NN_LOG("                        OpenMode    Elapsed[usec]\n");
    NNT_ASSERT_RESULT_SUCCESS(MountSdCardForDebug(mountName.c_str()));
    DeleteDirectoryRecursively(directoryPath.c_str());
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(directoryPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile(TestPath.c_str(), static_cast<int64_t>(TestReadWriteSize)));
    {
        for (auto openMode : openModes)
        {
            GetFileOpenModeTest(mountName, openMode);
        }
    }
    NNT_ASSERT_RESULT_SUCCESS(DeleteDirectoryRecursively(directoryPath.c_str()));
    Unmount(mountName.c_str());
    NN_LOG("\n");
}

//!< BisFs で指定された OpenMode で開かれた 4096 個のファイルの OpenMode を取得する時間を計測する
TEST_F(PerformanceTestOpenModeBisFs, GetFileOpenMode)
{
    NN_LOG("\nGetFileOpenMode BisFs Performance Test\n");

    int openModes[] = { OpenMode_Read,
                        OpenMode_Write,
                        OpenMode_Read  | OpenMode_Write,
                        OpenMode_Read  | OpenMode_AllowAppend,
                        OpenMode_Write | OpenMode_AllowAppend,
                        OpenMode_Read  | OpenMode_Write | OpenMode_AllowAppend};

    const String mountName(nn::fs::GetBisMountName(nn::fs::BisPartitionId::User));
    const String directoryPath = mountName + ":/" + TestDirectoryName;
    const String TestPath = mountName + ":/" + TestDirectoryName + "/" + TestFileName;

    NN_LOG("                        OpenMode    Elapsed[usec]\n");
    NNT_ASSERT_RESULT_SUCCESS(MountBis(nn::fs::BisPartitionId::User, nullptr));
    DeleteDirectoryRecursively(directoryPath.c_str());
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(directoryPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile(TestPath.c_str(), static_cast<int64_t>(TestReadWriteSize)));
    {
        for (auto openMode : openModes)
        {
            GetFileOpenModeTest(mountName, openMode);
        }
    }
    NNT_ASSERT_RESULT_SUCCESS(DeleteDirectoryRecursively(directoryPath.c_str()));
    Unmount(mountName.c_str());
    NN_LOG("\n");
}


extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    nnt::fs::util::SetFsTestPerformanceConfiguration();

    ::testing::InitGoogleTest(&argc, argv);

    SetAllocator(Allocate, Deallocate);
    ResetAllocateCount();
    nn::fs::SetEnabledAutoAbort(false);
    nn::fat::FatFileSystem::SetAllocatorForFat(nnt::fs::util::GetTestLibraryAllocator());

    bool isBackgroundAccessThreadRun = false;
    {
        BackgroundAccessThreadParameter parameter;
        ParseBackgroundAccessThreadParameter(&parameter, &argc, argv);
        if( ::testing::Test::HasFatalFailure() )
        {
            nnt::Exit(1);
        }
        if( parameter.isEnabled )
        {
            parameter.Dump();

            RunBackgroundAccessThread(parameter);
            if( ::testing::Test::HasFatalFailure() )
            {
                nnt::Exit(1);
            }

            isBackgroundAccessThreadRun = true;
            g_BackgroundThreadTag = parameter.GetTag();
        }
    }

    int argumentIndex = 1;

    nn::fs::SetSpeedEmulationMode(nn::fs::SpeedEmulationMode::None);
    g_SpeedEmulationModeName = SpeedEmulationModeNameNone;
    if( argumentIndex < argc )
    {
        auto pSpeedEmulationMode = argv[argumentIndex];

        if( strncmp(pSpeedEmulationMode, SpeedEmulationModeNameSlower, strlen(SpeedEmulationModeNameSlower)) == 0 )
        {
            nn::fs::SetSpeedEmulationMode(nn::fs::SpeedEmulationMode::Slower);
            g_SpeedEmulationModeName = SpeedEmulationModeNameSlower;
        }
        else if( strncmp(pSpeedEmulationMode, SpeedEmulationModeNameFaster, strlen(SpeedEmulationModeNameFaster)) == 0 )
        {
            nn::fs::SetSpeedEmulationMode(nn::fs::SpeedEmulationMode::Faster);
            g_SpeedEmulationModeName = SpeedEmulationModeNameFaster;
        }
        else if( strncmp(pSpeedEmulationMode, SpeedEmulationModeNameRandom, strlen(SpeedEmulationModeNameRandom)) == 0 )
        {
            nn::fs::SetSpeedEmulationMode(nn::fs::SpeedEmulationMode::Random);
            g_SpeedEmulationModeName = SpeedEmulationModeNameRandom;
        }
        else
        {
            --argumentIndex;
        }

        ++argumentIndex;
    }

    for( ; argumentIndex < argc; ++argumentIndex )
    {
        auto pCurrentArgument = argv[argumentIndex];
        if( strcmp(pCurrentArgument, "--access-priority") == 0 )
        {
            ++argumentIndex;
            if( argumentIndex >= argc )
            {
                NN_LOG("access priority not specified\n");
                nnt::Exit(1);
            }

            pCurrentArgument = argv[argumentIndex];
            nn::fs::PriorityRaw priority = ConvertStringToPriorityRaw(pCurrentArgument);
            nn::fs::SetPriorityRawOnCurrentThread(priority);

            NN_LOG("access priority is set to %s\n", pCurrentArgument);
            g_AccessPriorityName = pCurrentArgument;
        }
    }

    // MmcPatrol 休止
    nn::fs::SuspendMmcPatrol();

    auto testResult = RUN_ALL_TESTS();

    // MmcPatrol 再開
    nn::fs::ResumeMmcPatrol();

    nn::fs::SetSpeedEmulationMode(nn::fs::SpeedEmulationMode::None);
    if( isBackgroundAccessThreadRun )
    {
        StopBackgroundAccessThread();
    }

    if (CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(testResult);
}
