﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <cstring>

#include <nn/util/util_FormatString.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/base/testBase_Exit.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_MmcPrivate.h>
#include <nnt/nnt_Argument.h>
#include <nn/fat/fat_FatFileSystem.h>
#include <nn/fs/fsa/fs_Registrar.h>
#include <nn/fs/fs_PriorityPrivate.h>
#include <nn/fs/fs_SpeedEmulation.h>

using namespace nn::fs;
using namespace nnt::fs::util;

void* operator new(size_t size)
{
    return malloc(size);
}

void operator delete(void* p) NN_NOEXCEPT
{
    free(p);
}

namespace{

const char TestDirectoryName[]  = "perftestdir";
const char TestFileName[]       = "1";

const int DivisionCountMax = 1   * 1024; // 小さいバッファサイズの場合、この数以上の read/write はしない
const int64_t AlignmentPadding  = 32;         // アライメントずれテスト用パラメータ
const int64_t FileSize     = 128 * 1024 * 1024;
const int BufferSizeMin    = 1   * 1024;
const int BufferSizeMax    = 4   * 1024 * 1024;

nnt::fs::util::String g_StorageName;
nnt::fs::util::String g_PriorityName;
nnt::fs::util::String g_BackgroundThreadTag;
const char TestRomRead[]              = "RomRead_";
const char TestRomReadUnaligned[]     = "RomReadUnaligned_";
const char TestNandSlow[]             = "NANDSlower";
const char TestNandFast[]             = "NANDFaster";
const char TestNandRand[]             = "NANDRand";
const char TestNandOff[]              = "NANDOff";
const char TestGC50[]                 = "GC50";
const char TestGC25[]                 = "GC25";
const char TestSdCard[]               = "SdCard";

Vector<PerformanceResult> g_perfResultsRead;

void DumpPerformanceResultsFlush(Vector<PerformanceResult>& results) NN_NOEXCEPT
{
    NN_LOG("BufferSize[bytes]   Elapsed[usec]\n");
    for(auto result : results)
    {
        NN_LOG("        %9d       %9lld\n",
            result.bufferSize,
            result.elapsedTimeUsec);
    }
}
/**
* @brief ROM 内のファイルのシーケンシャル読み込みを計測
* @pre MountRom が成功している
*/
void ReadSequentialForRom(const String mountName, bool isAligned) NN_NOEXCEPT
{
    Vector<PerformanceResult> perfResults;
    const String TestPath = mountName + ":/" + TestDirectoryName + "/" + TestFileName;

    for(int bufferSize = BufferSizeMin ; bufferSize <= BufferSizeMax ; bufferSize *= 2)
    {
        int64_t fileSize = std::min(
            FileSize + (isAligned ? 0 : AlignmentPadding),
            static_cast<int64_t>(bufferSize) * DivisionCountMax + (isAligned ? 0 : AlignmentPadding));
        ReadSequentialCore(g_perfResultsRead, TestPath, bufferSize, fileSize, isAligned ? 0 : AlignmentPadding);
    }
}

/**
* @brief ROM 内のファイルのランダムな箇所からの読み込みを計測
* @pre MountRom が成功している
*/
void ReadShuffleForRom(const String mountName, bool isAligned) NN_NOEXCEPT
{
    Vector<PerformanceResult> perfResults;
    const String TestPath = mountName + ":/" + TestDirectoryName + "/" + TestFileName;

    for(int bufferSize = BufferSizeMin ; bufferSize <= BufferSizeMax ; bufferSize *= 2)
    {
        int64_t fileSize = std::min(
            FileSize + (isAligned ? 0 : AlignmentPadding),
            static_cast<int64_t>(bufferSize) * DivisionCountMax + (isAligned ? 0 : AlignmentPadding));
        ReadShuffleCore(g_perfResultsRead, TestPath, bufferSize, fileSize, isAligned ? 0 : AlignmentPadding);
    }
}

void GetTestName(nnt::fs::util::String* outValue, bool isAligned, const nnt::fs::util::String& storageName) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(outValue);

    *outValue = isAligned ? TestRomRead : TestRomReadUnaligned;
    *outValue += storageName + g_PriorityName + g_BackgroundThreadTag;
}

void GetTestName(nnt::fs::util::String* outValue, bool isAligned) NN_NOEXCEPT
{
    GetTestName(outValue, isAligned, g_StorageName);
}

void CheckPerformanceResult(const Vector<PerformanceResult>& result, bool isAligned) NN_NOEXCEPT
{
    {
        nnt::fs::util::String testName;
        GetTestName(&testName, isAligned);
        EXPECT_TRUE(nnt::fs::util::CheckPerformanceResult(testName.c_str(), result, false));
    }

    if( g_BackgroundThreadTag.length() == 0 )
    {
        if(    g_StorageName.compare(TestNandFast) == 0
            || g_StorageName.compare(TestNandOff) == 0
            || g_StorageName.compare(TestGC50) == 0
            || g_StorageName.compare(TestGC25) == 0
            || g_StorageName.compare(TestSdCard) == 0 )
        {
            nnt::fs::util::String testName;
            GetTestName(&testName, isAligned, TestNandSlow);
            EXPECT_TRUE(nnt::fs::util::CheckPerformanceResult(testName.c_str(), result, true));
        }
    }
}

void PerformanceTestImpl(bool isAligned) NN_NOEXCEPT
{
    size_t mountRomCacheBufferSize = 0;
    NNT_ASSERT_RESULT_SUCCESS(QueryMountRomCacheSize(&mountRomCacheBufferSize));
    auto mountRomCacheBuffer = AllocateBuffer(mountRomCacheBufferSize);
    g_perfResultsRead.clear();

    String mountName = "perftest";
    NNT_ASSERT_RESULT_SUCCESS(MountRom(mountName.c_str(), mountRomCacheBuffer.get(), mountRomCacheBufferSize));
    {
       ReadSequentialForRom(mountName, isAligned);
       ReadShuffleForRom(mountName, isAligned);
    }
    Unmount(mountName.c_str());
    NN_LOG("Read Rom Performance Test\n");
    NN_LOG("                                                          Sequential                           Random\n");
    DumpPerformanceResults(g_perfResultsRead);
    NN_LOG("\n\n");

    nnt::fs::util::String testName;
    GetTestName(&testName, isAligned);
    WriteTeamCityPerformanceResults(testName.c_str(), g_perfResultsRead);
    CheckPerformanceResult(g_perfResultsRead, isAligned);
}

} // namespace

//!< ROM をマウント後 ROM 内の特定ファイルを各バッファサイズで読み込む時間を計測する
TEST(PerformanceTestReadRomFs, ReadFile)
{
    PerformanceTestImpl(true);
}

//!< ROM をマウント後 ROM 内の特定ファイルを各バッファサイズで読み込む時間を計測する (非アライメント)
TEST(PerformanceTestReadRomFs, ReadFileUnaligned)
{
    PerformanceTestImpl(false);
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    nnt::fs::util::SetFsTestPerformanceConfiguration();

    ::testing::InitGoogleTest(&argc, argv);

    SetAllocator(Allocate, Deallocate);
    ResetAllocateCount();
    nn::fs::SetEnabledAutoAbort(false);

    bool isBackgroundAccessThreadRun = false;
    {
        BackgroundAccessThreadParameter parameter;
        ParseBackgroundAccessThreadParameter(&parameter, &argc, argv);
        if( ::testing::Test::HasFatalFailure() )
        {
            nnt::Exit(1);
        }
        if( parameter.isEnabled )
        {
            parameter.Dump();

            RunBackgroundAccessThread(parameter);
            if( ::testing::Test::HasFatalFailure() )
            {
                nnt::Exit(1);
            }

            isBackgroundAccessThreadRun = true;
            g_BackgroundThreadTag = parameter.GetTag();
        }
    }

    int argumentIndex = 1;

    // 引数にテスト(ストレージ)名をもらう
    if( argumentIndex < argc )
    {
        auto pStorageName = argv[argumentIndex];

        g_StorageName = pStorageName;

        // テストの名前によってはエミュレーションを行う
        if(strncmp(pStorageName, TestNandSlow, strlen(TestNandSlow)) == 0)
        {
            nn::fs::SetSpeedEmulationMode(nn::fs::SpeedEmulationMode::Slower);
        }
        else if(strncmp(pStorageName, TestNandFast, strlen(TestNandFast)) == 0)
        {
            nn::fs::SetSpeedEmulationMode(nn::fs::SpeedEmulationMode::Faster);
        }
        else if(strncmp(pStorageName, TestNandRand, strlen(TestNandRand)) == 0)
        {
            nn::fs::SetSpeedEmulationMode(nn::fs::SpeedEmulationMode::Random);
        }
        else
        {
            nn::fs::SetSpeedEmulationMode(nn::fs::SpeedEmulationMode::None);
        }

        ++argumentIndex;
    }

    for( ; argumentIndex < argc; ++argumentIndex )
    {
        auto pCurrentArgument = argv[argumentIndex];
        if( strcmp(pCurrentArgument, "--access-priority") == 0 )
        {
            ++argumentIndex;
            if( argumentIndex >= argc )
            {
                NN_LOG("access priority not specified\n");
                nnt::Exit(1);
            }

            pCurrentArgument = argv[argumentIndex];
            nn::fs::PriorityRaw priority = ConvertStringToPriorityRaw(pCurrentArgument);
            nn::fs::SetPriorityRawOnCurrentThread(priority);

            NN_LOG("access priority is set to %s\n", pCurrentArgument);

            if( strcmp(pCurrentArgument, "Default") != 0 )
            {
                g_PriorityName = pCurrentArgument;
            }
        }
    }

    // MmcPatrol 休止
    nn::fs::SuspendMmcPatrol();

    auto testResult = RUN_ALL_TESTS();

    // MmcPatrol 再開
    nn::fs::ResumeMmcPatrol();

    // 元に戻す
    nn::fs::SetSpeedEmulationMode(nn::fs::SpeedEmulationMode::None);
    if( isBackgroundAccessThreadRun )
    {
        StopBackgroundAccessThread();
    }

    // 検証
    if (CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(testResult);
}
