﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SpeedEmulation.h>
#include <nn/fs/fs_MmcPrivate.h>
#include <nn/util/util_IntUtil.h>
#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>

void* operator new(size_t size) NN_NOEXCEPT
{
    return malloc(size);
}

void operator delete(void* p) NN_NOEXCEPT
{
    free(p);
}

namespace {

const int DivisionCountMax = 128;
const int BufferSizeMin    = 1 * 1024;
const int BufferSizeMax    = 4 * 1024 * 1024;

const char TestNandSlow[] = "NANDSlow";
const char TestNandFast[] = "NANDFast";
const char TestNandRand[] = "NANDRand";

nnt::fs::util::Vector<nn::fs::DirectoryEntry> g_Files;
nnt::fs::util::Vector<nnt::fs::util::PerformanceResult> g_PerformanceResults;

// ファイル一覧を取得します。
nn::Result GetFiles(const char* path) NN_NOEXCEPT
{
    g_Files.clear();

    nn::fs::DirectoryHandle directory;
    NN_RESULT_DO(nn::fs::OpenDirectory(&directory, path, nn::fs::OpenDirectoryMode_File));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseDirectory(directory);
    };

    int64_t getCount = 0;
    NN_RESULT_DO(nn::fs::GetDirectoryEntryCount(&getCount, directory));
    NN_ABORT_UNLESS(nn::util::IsIntValueRepresentable<size_t>(getCount));

    g_Files.resize(static_cast<size_t>(getCount));

    int64_t readCount = 0;
    NN_RESULT_DO(nn::fs::ReadDirectory(&readCount, g_Files.data(), directory, getCount));
    NN_ABORT_UNLESS(getCount == readCount);

    NN_RESULT_SUCCESS;
}

// シーケンシャルリードの計測をします。
nn::Result ReadSequentialCore(
    int64_t* pElapsedTime, const char* filePath, int bufferSize, int64_t fileSize) NN_NOEXCEPT
{
    nn::fs::FileHandle handle;
    NN_RESULT_DO(nn::fs::OpenFile(&handle, filePath, nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    std::unique_ptr<char[]> readBuffer(new char[bufferSize]);
    NN_ABORT_UNLESS_NOT_NULL(readBuffer);

    nnt::fs::util::TimeCount timeCount;
    for( int64_t offset = 0; offset < fileSize; offset += bufferSize )
    {
        NNT_FS_SCOPED_TRACE_SAFE("buffer size: %d, offset count: %d\n", bufferSize, offset);
        timeCount.StartTime();
        NN_RESULT_DO(nn::fs::ReadFile(handle, offset, readBuffer.get(), bufferSize));
        timeCount.StopTime();
    }

    *pElapsedTime = timeCount.GetTotalTime();

    NN_RESULT_SUCCESS;
}

// ROM 内のファイルのシーケンシャル読み込みを計測
nn::Result ReadSequential(const char* path) NN_NOEXCEPT
{
    for( int bufferSize = BufferSizeMin; bufferSize <= BufferSizeMax; bufferSize <<= 1 )
    {
        NN_LOG("Sequencial read: %d\n", bufferSize);

        int64_t totalReadSize = 0;
        int64_t totalElapsedTime = 0; // 単位はナノ秒

        // ファイルもシーケンシャル
        for( const auto& file : g_Files )
        {
            if( bufferSize <= file.fileSize )
            {
                nnt::fs::util::String filePath(path);
                filePath += file.name;

                const auto readSize = std::min<int64_t>(
                    nn::util::align_down(file.fileSize, bufferSize),
                    bufferSize * DivisionCountMax
                );

                // シーケンシャルリード
                int64_t elapsedTime = 0;
                NN_RESULT_DO(
                    ReadSequentialCore(&elapsedTime, filePath.c_str(), bufferSize, readSize));

                totalReadSize += readSize;
                totalElapsedTime += elapsedTime;
            }
        }

        nnt::fs::util::AddPerformanceResults(
            g_PerformanceResults, bufferSize, totalReadSize, totalElapsedTime / 1000);
    }

    NN_RESULT_SUCCESS;
}

// ランダムリードの計測をします。
nn::Result ReadShuffleCore(
    int64_t* pElapsedTime, const char* filePath, int bufferSize, int64_t fileSize) NN_NOEXCEPT
{
    nn::fs::FileHandle handle;
    NN_RESULT_DO(nn::fs::OpenFile(&handle, filePath, nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    // オフセット値格納用の配列を作成
    int arrayCount = static_cast<int>(fileSize / bufferSize);
    std::unique_ptr<int[]> offsetArray(new int[arrayCount]);
    NN_ABORT_UNLESS_NOT_NULL(offsetArray);
    nnt::fs::util::CreateRandomArray(offsetArray.get(), arrayCount, bufferSize);

    std::unique_ptr<char[]> readBuffer(new char[bufferSize]);
    NN_ABORT_UNLESS_NOT_NULL(readBuffer);

    nnt::fs::util::TimeCount timeCount;
    for( int i = 0; i < arrayCount; ++i )
    {
        const auto offset = offsetArray[i];

        NNT_FS_SCOPED_TRACE_SAFE("buffer size: %d, offset count: %d\n", bufferSize, offset);
        timeCount.StartTime();
        NN_RESULT_DO(nn::fs::ReadFile(handle, offset, readBuffer.get(), bufferSize));
        timeCount.StopTime();
    }

    *pElapsedTime = timeCount.GetTotalTime();

    NN_RESULT_SUCCESS;
}

// ROM 内のファイルのランダムな箇所からの読み込みを計測
nn::Result ReadShuffle(const char* path) NN_NOEXCEPT
{
    nnt::fs::util::Vector<const nn::fs::DirectoryEntry*> pFiles;
    pFiles.reserve(g_Files.size());

    for( const auto& file : g_Files )
    {
        pFiles.push_back(&file);
    }

    std::mt19937 random(nnt::fs::util::GetRandomSeed());
    for( int bufferSize = BufferSizeMin; bufferSize <= BufferSizeMax; bufferSize <<= 1 )
    {
        NN_LOG("Random read: %d\n", bufferSize);

        std::shuffle(pFiles.begin(), pFiles.end(), random);

        int64_t totalReadSize = 0;
        int64_t totalElapsedTime = 0; // 単位はナノ秒

        // ファイルもランダム
        for( auto pFile : pFiles )
        {
            if( bufferSize <= pFile->fileSize )
            {
                nnt::fs::util::String filePath(path);
                filePath += pFile->name;

                const auto readSize = std::min<int64_t>(
                    nn::util::align_down(pFile->fileSize, bufferSize),
                    bufferSize * DivisionCountMax
                );

                // ランダムリード
                int64_t elapsedTime = 0;
                NN_RESULT_DO(
                    ReadShuffleCore(&elapsedTime, filePath.c_str(), bufferSize, readSize));

                totalReadSize += readSize;
                totalElapsedTime += elapsedTime;
            }
        }

        nnt::fs::util::AddPerformanceResults(
            g_PerformanceResults, bufferSize, totalReadSize, totalElapsedTime / 1000);
    }

    NN_RESULT_SUCCESS;
}

} // namespace

// ROM をマウント後 ROM 内の全ファイルを各バッファサイズで読み込む時間を計測する
TEST(PerformanceTestReadPatchReal, ReadFile)
{
    size_t mountRomCacheBufferSize = 0;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&mountRomCacheBufferSize));
    auto mountRomCacheBuffer = nnt::fs::util::AllocateBuffer(mountRomCacheBufferSize);

    NNT_ASSERT_RESULT_SUCCESS(
        nn::fs::MountRom("rom", mountRomCacheBuffer.get(), mountRomCacheBufferSize));
    {
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount("rom");
        };

        NNT_ASSERT_RESULT_SUCCESS(GetFiles("rom:/"));

        g_PerformanceResults.clear();

        NNT_ASSERT_RESULT_SUCCESS(ReadSequential("rom:/"));
        NNT_ASSERT_RESULT_SUCCESS(ReadShuffle("rom:/"));
    }

    NN_LOG("Read Rom Performance Test\n");
    NN_LOG("                                                          Sequential                           Random\n");
    nnt::fs::util::DumpPerformanceResults(g_PerformanceResults);
    NN_LOG("\n\n");
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    nnt::fs::util::SetFsTestPerformanceConfiguration();

    ::testing::InitGoogleTest(&argc, argv);

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();
    nn::fs::SetEnabledAutoAbort(false);

    // 引数にテスト(ストレージ)名をもらう
    if( 1 < argc )
    {
        // テストの名前によってはエミュレーションを行う
        if( std::strncmp(argv[1], TestNandSlow, strlen(TestNandSlow)) == 0 )
        {
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fs::SetSpeedEmulationMode(nn::fs::SpeedEmulationMode::Slower));
        }
        else if ( std::strncmp(argv[1], TestNandFast, std::strlen(TestNandFast)) == 0 )
        {
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fs::SetSpeedEmulationMode(nn::fs::SpeedEmulationMode::Faster));
        }
        else if( std::strncmp(argv[1], TestNandRand, std::strlen(TestNandRand)) == 0 )
        {
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fs::SetSpeedEmulationMode(nn::fs::SpeedEmulationMode::Random));
        }
        else
        {
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fs::SetSpeedEmulationMode(nn::fs::SpeedEmulationMode::None));
        }
    }

    // MmcPatrol 休止
    nn::fs::SuspendMmcPatrol();

    auto testResult = RUN_ALL_TESTS();

    // MmcPatrol 再開
    nn::fs::ResumeMmcPatrol();

    // 元に戻す
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::SetSpeedEmulationMode(nn::fs::SpeedEmulationMode::None));
    if ( nnt::fs::util::CheckMemoryLeak() )
    {
        nnt::Exit(1);
    }

    nnt::Exit(testResult);
}
