﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdlib>

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>

#include <nn/os/os_Thread.h>

#include <nn/fs.h>
#include <nn/fs/fs_ResultHandler.h>

#include <nnt/nntest.h>


#define FILE_DATA_CACHE_TEST_INFO_LOG(...)
#define FILE_DATA_CACHE_TEST_ERROR_LOG(...)  NN_LOG(__VA_ARGS__)

extern const size_t DefaultFileDataCacheSize;

// ----------------------------------------------------------------------------

struct RandomReadTestArg
{
    const char* mountName;
    int64_t seed;
    int trialCount;
    nn::TimeSpan trialTime;
    int64_t trialSize;

    bool success;
};

void RandomReadTest(void* theArg);
bool RandomReadTest(const char* mountName, int64_t seed, int trialCount, nn::TimeSpan trialTime);
bool RandomReadTest(const char* mountName, int64_t seed, int trialCount, nn::TimeSpan trialTime, int64_t trialSize);

// ----------------------------------------------------------------------------

struct ThreadInfo
{
    static const size_t ThreadStackSize = 32768;
    NN_OS_ALIGNAS_THREAD_STACK nn::Bit8 threadStack[ThreadStackSize];
    nn::os::ThreadType threadType;
};

// ----------------------------------------------------------------------------

void ExpectNoMemoryLeak();

// ----------------------------------------------------------------------------

class ScopedRomFsMounter
{
private:
    void* m_pCache;
    char m_MountName[nn::fs::MountNameLengthMax + 1];
public:
    explicit ScopedRomFsMounter(const char* name) NN_NOEXCEPT
    {
        size_t fileSystemCacheSize;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&fileSystemCacheSize));

        m_pCache = std::malloc(fileSystemCacheSize);
        NN_ABORT_UNLESS_NOT_NULL(m_pCache);

        std::strncpy(m_MountName, name, nn::fs::MountNameLengthMax);
        m_MountName[nn::fs::MountNameLengthMax] = '\0';

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountRom(m_MountName, m_pCache, fileSystemCacheSize));

        FILE_DATA_CACHE_TEST_INFO_LOG("RomFs mounted\n");
    }
    void Unmount() NN_NOEXCEPT
    {
        if (m_pCache)
        {
            nn::fs::Unmount(m_MountName);
            std::free(m_pCache);
            m_pCache = nullptr;
            FILE_DATA_CACHE_TEST_INFO_LOG("RomFs unmounted\n");
        }
    }
    ~ScopedRomFsMounter() NN_NOEXCEPT
    {
        Unmount();
    }
};

class ScopedGlobalFileDataCacheEnabler
{
private:
    void* m_pCache;
public:
    explicit ScopedGlobalFileDataCacheEnabler(size_t cacheSize) NN_NOEXCEPT
    {
        m_pCache = std::malloc(cacheSize);
        NN_ABORT_UNLESS_NOT_NULL(m_pCache);

        nn::fs::EnableGlobalFileDataCache(m_pCache, cacheSize);

        FILE_DATA_CACHE_TEST_INFO_LOG("GlobalFileDataCache enabled\n");
    }
    void Disable() NN_NOEXCEPT
    {
        if (m_pCache)
        {
            nn::fs::DisableGlobalFileDataCache();
            std::free(m_pCache);
            m_pCache = nullptr;
            FILE_DATA_CACHE_TEST_INFO_LOG("GlobalFileDataCache disabled\n");
        }
    }
    ~ScopedGlobalFileDataCacheEnabler() NN_NOEXCEPT
    {
        Disable();
    }
};

// ----------------------------------------------------------------------------

extern const size_t MinAcceptableCacheMemorySize;

size_t GetMaxAllocatableCacheMemorySize() NN_NOEXCEPT;

// ----------------------------------------------------------------------------

class AutoAbortDisabledTestFixture : public ::testing::Test
{
protected:
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::fs::SetEnabledAutoAbort(false);
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::fs::SetEnabledAutoAbort(true);
    }
};

class ReadFileTestFixture : public AutoAbortDisabledTestFixture
{
private:
    const char* RomMountName = "rom";
    const char* TestTargetFilePath = "rom:/1m";

private:
    void* m_pCache;
    nn::fs::FileHandle m_FileHandle;

    void* m_pFileSizeBuffer;
    size_t m_FileSizeBufferSize;

public:
    nn::fs::FileHandle GetFileHandle() const NN_NOEXCEPT
    {
        return m_FileHandle;
    }

    void* GetFileSizeBuffer() const NN_NOEXCEPT
    {
        return m_pFileSizeBuffer;
    }

    size_t GetFileSizeBufferSize() const NN_NOEXCEPT
    {
        return m_FileSizeBufferSize;
    }

protected:
    void SetUpFileSystem() NN_NOEXCEPT;
    void SetUpFileHandle() NN_NOEXCEPT;
    void TearDownFileHandle() NN_NOEXCEPT;
    void TearDownFileSystem() NN_NOEXCEPT;

    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE;
    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE;
};
typedef ReadFileTestFixture ReadFileBehaviorDifference;

void TestReadFile(nn::fs::FileHandle fileHandle, int64_t offset, void* buffer, size_t size);
void TestReadFile(nn::fs::FileHandle fileHandle, int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option);
void TestReadFile(size_t* readSize, nn::fs::FileHandle fileHandle, int64_t offset, void* buffer, size_t size);
void TestReadFile(size_t* readSize, nn::fs::FileHandle fileHandle, int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option);

void DeathTestReadFile(nn::fs::FileHandle fileHandle, int64_t offset, void* buffer, size_t size);
void DeathTestReadFile(nn::fs::FileHandle fileHandle, int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option);
void DeathTestReadFile(size_t* readSize, nn::fs::FileHandle fileHandle, int64_t offset, void* buffer, size_t size);
void DeathTestReadFile(size_t* readSize, nn::fs::FileHandle fileHandle, int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option);

// ----------------------------------------------------------------------------

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#include "testFs_Integration_FileDataCache_Util-os.horizon.h"
#endif
