﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdlib>
#include <cstring>

#include <nn/nn_Common.h>

#include <nn/aoc.h>
#include <nn/fs.h>
#include <nn/fs/fs_SystemData.h>

#include "testFs_Integration_FileDataCache_Util.h"


class ScopedAddOnContentMounter
{
private:
    void* m_pCache;
    char m_MountName[nn::fs::MountNameLengthMax + 1];
    nn::aoc::AddOnContentIndex m_Index;
public:
    ScopedAddOnContentMounter(const char* name, nn::aoc::AddOnContentIndex index) NN_NOEXCEPT
        : m_Index(index)
    {
        size_t fileSystemCacheSize;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountAddOnContentCacheSize(&fileSystemCacheSize, index));

        m_pCache = std::malloc(fileSystemCacheSize);
        NN_ABORT_UNLESS_NOT_NULL(m_pCache);

        std::strncpy(m_MountName, name, nn::fs::MountNameLengthMax);
        m_MountName[nn::fs::MountNameLengthMax] = '\0';

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountAddOnContent(m_MountName, index, m_pCache, fileSystemCacheSize));

        FILE_DATA_CACHE_TEST_INFO_LOG("AddOnContent %d mounted\n", index);
    }
    void Unmount() NN_NOEXCEPT
    {
        if (m_pCache)
        {
            nn::fs::Unmount(m_MountName);
            std::free(m_pCache);
            m_pCache = nullptr;

            NN_UNUSED(m_Index);
            FILE_DATA_CACHE_TEST_INFO_LOG("AddOnContent %d unmounted\n", m_Index);
        }
    }
    ~ScopedAddOnContentMounter() NN_NOEXCEPT
    {
        Unmount();
    }
};

class ScopedSystemDataMounter
{
private:
    void* m_pCache;
    char m_MountName[nn::fs::MountNameLengthMax + 1];
public:
    ScopedSystemDataMounter(const char* name, nn::ncm::SystemDataId id) NN_NOEXCEPT
    {
        size_t fileSystemCacheSize;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountSystemDataCacheSize(&fileSystemCacheSize, id));

        m_pCache = std::malloc(fileSystemCacheSize);
        NN_ABORT_UNLESS_NOT_NULL(m_pCache);

        std::strncpy(m_MountName, name, nn::fs::MountNameLengthMax);
        m_MountName[nn::fs::MountNameLengthMax] = '\0';

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountSystemData(m_MountName, id, m_pCache, fileSystemCacheSize));

        FILE_DATA_CACHE_TEST_INFO_LOG("SystemData %s mounted\n", m_MountName);
    }
    void Unmount() NN_NOEXCEPT
    {
        if (m_pCache)
        {
            nn::fs::Unmount(m_MountName);
            std::free(m_pCache);
            m_pCache = nullptr;

            FILE_DATA_CACHE_TEST_INFO_LOG("SystemData %s unmounted\n", m_MountName);
        }
    }
    ~ScopedSystemDataMounter() NN_NOEXCEPT
    {
        Unmount();
    }
};
