﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>

#include <nn/fs.h>

#include <nnt/nntest.h>
#include <nnt/fsUtil/testFs_util.h>

#include "testFs_Integration_FileDataCache_Util.h"


namespace
{

class MountRomTestFixture : public AutoAbortDisabledTestFixture
{
private:
    void* m_pCache;
    size_t m_CacheSize;

public:
    void* GetFileSystemCacheBuffer() const NN_NOEXCEPT
    {
        return m_pCache;
    }

    size_t GetFileSystemCacheBufferSize() const NN_NOEXCEPT
    {
        return m_CacheSize;
    }

protected:
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&m_CacheSize));

        m_pCache = std::malloc(m_CacheSize);
        NN_ABORT_UNLESS_NOT_NULL(m_pCache);

        AutoAbortDisabledTestFixture::SetUp();
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        AutoAbortDisabledTestFixture::TearDown();

        std::free(m_pCache);
    }
};
typedef MountRomTestFixture MountRomBehaviorDifference;

void TestMountRom(const char* name, void* buffer, size_t bufferSize)
{
    nn::Result cacheEnabledResult;
    nn::Result cacheDisabledResult;
    {
        ScopedGlobalFileDataCacheEnabler fileDataCache(DefaultFileDataCacheSize);
        cacheEnabledResult = nn::fs::MountRom(name, buffer, bufferSize);
        if (cacheEnabledResult.IsSuccess())
        {
            nn::fs::Unmount(name);
        }
    }
    {
        cacheDisabledResult = nn::fs::MountRom(name, buffer, bufferSize);
        if (cacheDisabledResult.IsSuccess())
        {
            nn::fs::Unmount(name);
        }
    }
    EXPECT_EQ(cacheEnabledResult.GetInnerValueForDebug(), cacheDisabledResult.GetInnerValueForDebug());
}

void DeathTestMountRom(const char* name, void* buffer, size_t bufferSize)
{
    {
        ScopedGlobalFileDataCacheEnabler fileDataCache(DefaultFileDataCacheSize);
        EXPECT_DEATH_IF_SUPPORTED(
            nn::fs::MountRom(name, buffer, bufferSize), "");
    }
    {
        EXPECT_DEATH_IF_SUPPORTED(
            nn::fs::MountRom(name, buffer, bufferSize), "");
    }
}

}  // namespace unnamed


TEST_F(MountRomBehaviorDifference, InvalidMountName)
{
    TestMountRom(nullptr, GetFileSystemCacheBuffer(), GetFileSystemCacheBufferSize());
    TestMountRom("C", GetFileSystemCacheBuffer(), GetFileSystemCacheBufferSize());
    TestMountRom("@rom", GetFileSystemCacheBuffer(), GetFileSystemCacheBufferSize());
    TestMountRom("rom:", GetFileSystemCacheBuffer(), GetFileSystemCacheBufferSize());
    TestMountRom("rom/", GetFileSystemCacheBuffer(), GetFileSystemCacheBufferSize());
    {
        ScopedRomFsMounter romFs("rom");
        TestMountRom("rom", GetFileSystemCacheBuffer(), GetFileSystemCacheBufferSize());
    }
    TestMountRom("", GetFileSystemCacheBuffer(), GetFileSystemCacheBufferSize());
    TestMountRom("12345678901234567890", GetFileSystemCacheBuffer(), GetFileSystemCacheBufferSize());
}

TEST_F(MountRomBehaviorDifference, NullPointerBuffer)
{
    TestMountRom("rom", nullptr, 0);
    TestMountRom("rom", nullptr, GetFileSystemCacheBufferSize());
}

TEST_F(MountRomBehaviorDifference, TooSmallBufferSize)
{
#if defined(NN_BUILD_TARGET_PLATFORM_OS_NN)

    // AutoAbort の設定に関わらず内部アボートする
    DeathTestMountRom("rom", GetFileSystemCacheBuffer(), 0);
    DeathTestMountRom("rom", GetFileSystemCacheBuffer(), GetFileSystemCacheBufferSize() - 1);

#elif defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)

    TestMountRom("rom", GetFileSystemCacheBuffer(), 0);
    TestMountRom("rom", GetFileSystemCacheBuffer(), GetFileSystemCacheBufferSize() - 1);

#else
#error "unsupported os"
#endif
}

TEST_F(MountRomBehaviorDifference, Mount10FileSystem)
{
    void* caches[9];
    size_t cacheSize;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&cacheSize));

    for (int i = 0; i < 10; i++)
    {
        if (i == 10 - 1)
        {
            TestMountRom("rom10", GetFileSystemCacheBuffer(), GetFileSystemCacheBufferSize());
        }
        else
        {
            char mountName[8];
            ::snprintf(mountName, sizeof(mountName), "rom%d", i);

            caches[i] = std::malloc(cacheSize);
            NN_ABORT_UNLESS_NOT_NULL(caches[i]);

            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountRom(mountName, caches[i], cacheSize));
        }
    }
    for (int i = 0; i < 9; i++)
    {
        char mountName[8];
        ::snprintf(mountName, sizeof(mountName), "rom%d", i);

        nn::fs::Unmount(mountName);

        std::free(caches[i]);
    }
}

TEST_F(ReadFileBehaviorDifference, InvalidFileHandle)
{
    size_t readSize;
    nn::fs::FileHandle fileHandle = { nullptr };

    // ファイルデータキャッシュの有無に関わらずクラッシュする
    DeathTestReadFile(fileHandle, 0, GetFileSizeBuffer(), GetFileSizeBufferSize());
    DeathTestReadFile(fileHandle, 0, GetFileSizeBuffer(), GetFileSizeBufferSize(), nn::fs::ReadOption::MakeValue(0));
    DeathTestReadFile(&readSize, fileHandle, 0, GetFileSizeBuffer(), GetFileSizeBufferSize());
    DeathTestReadFile(&readSize, fileHandle, 0, GetFileSizeBuffer(), GetFileSizeBufferSize(), nn::fs::ReadOption::MakeValue(0));
}

TEST_F(ReadFileBehaviorDifference, NullPointerReadSize)
{
    TestReadFile(nullptr, GetFileHandle(), 0, GetFileSizeBuffer(), GetFileSizeBufferSize());
    TestReadFile(nullptr, GetFileHandle(), 0, GetFileSizeBuffer(), GetFileSizeBufferSize(), nn::fs::ReadOption::MakeValue(0));
}

TEST_F(ReadFileBehaviorDifference, NegativeOffset)
{
    size_t readSize;

    TestReadFile(GetFileHandle(), -1, GetFileSizeBuffer(), 0);
    TestReadFile(GetFileHandle(), -1, GetFileSizeBuffer(), 0, nn::fs::ReadOption::MakeValue(0));
    TestReadFile(&readSize, GetFileHandle(), -1, GetFileSizeBuffer(), 0);
    TestReadFile(&readSize, GetFileHandle(), -1, GetFileSizeBuffer(), 0, nn::fs::ReadOption::MakeValue(0));

    TestReadFile(GetFileHandle(), -1, GetFileSizeBuffer(), GetFileSizeBufferSize());
    TestReadFile(GetFileHandle(), -1, GetFileSizeBuffer(), GetFileSizeBufferSize(), nn::fs::ReadOption::MakeValue(0));
    TestReadFile(&readSize, GetFileHandle(), -1, GetFileSizeBuffer(), GetFileSizeBufferSize());
    TestReadFile(&readSize, GetFileHandle(), -1, GetFileSizeBuffer(), GetFileSizeBufferSize(), nn::fs::ReadOption::MakeValue(0));

    TestReadFile(GetFileHandle(), INT64_MIN, GetFileSizeBuffer(), GetFileSizeBufferSize());
    TestReadFile(GetFileHandle(), INT64_MIN, GetFileSizeBuffer(), GetFileSizeBufferSize(), nn::fs::ReadOption::MakeValue(0));
    TestReadFile(&readSize, GetFileHandle(), INT64_MIN, GetFileSizeBuffer(), GetFileSizeBufferSize());
    TestReadFile(&readSize, GetFileHandle(), INT64_MIN, GetFileSizeBuffer(), GetFileSizeBufferSize(), nn::fs::ReadOption::MakeValue(0));
}

TEST_F(ReadFileBehaviorDifference, TooLargeOffset)
{
    size_t readSize;

    TestReadFile(GetFileHandle(), static_cast<int64_t>(GetFileSizeBufferSize() + 1), GetFileSizeBuffer(), 0);
    TestReadFile(GetFileHandle(), static_cast<int64_t>(GetFileSizeBufferSize() + 1), GetFileSizeBuffer(), 0, nn::fs::ReadOption::MakeValue(0));
    TestReadFile(&readSize, GetFileHandle(), static_cast<int64_t>(GetFileSizeBufferSize() + 1), GetFileSizeBuffer(), 0);
    TestReadFile(&readSize, GetFileHandle(), static_cast<int64_t>(GetFileSizeBufferSize() + 1), GetFileSizeBuffer(), 0, nn::fs::ReadOption::MakeValue(0));

    TestReadFile(GetFileHandle(), static_cast<int64_t>(GetFileSizeBufferSize() + 1), GetFileSizeBuffer(), GetFileSizeBufferSize());
    TestReadFile(GetFileHandle(), static_cast<int64_t>(GetFileSizeBufferSize() + 1), GetFileSizeBuffer(), GetFileSizeBufferSize(), nn::fs::ReadOption::MakeValue(0));
    TestReadFile(&readSize, GetFileHandle(), static_cast<int64_t>(GetFileSizeBufferSize() + 1), GetFileSizeBuffer(), GetFileSizeBufferSize());
    TestReadFile(&readSize, GetFileHandle(), static_cast<int64_t>(GetFileSizeBufferSize() + 1), GetFileSizeBuffer(), GetFileSizeBufferSize(), nn::fs::ReadOption::MakeValue(0));

    TestReadFile(GetFileHandle(), INT64_MAX, GetFileSizeBuffer(), GetFileSizeBufferSize());
    TestReadFile(GetFileHandle(), INT64_MAX, GetFileSizeBuffer(), GetFileSizeBufferSize(), nn::fs::ReadOption::MakeValue(0));
    TestReadFile(&readSize, GetFileHandle(), INT64_MAX, GetFileSizeBuffer(), GetFileSizeBufferSize());
    TestReadFile(&readSize, GetFileHandle(), INT64_MAX, GetFileSizeBuffer(), GetFileSizeBufferSize(), nn::fs::ReadOption::MakeValue(0));
}

TEST_F(ReadFileBehaviorDifference, NullPointerBuffer)
{
    size_t readSize;

    TestReadFile(GetFileHandle(), 0, nullptr, 0);
    TestReadFile(GetFileHandle(), 0, nullptr, 0, nn::fs::ReadOption::MakeValue(0));
    TestReadFile(&readSize, GetFileHandle(), 0, nullptr, 0);
    TestReadFile(&readSize, GetFileHandle(), 0, nullptr, 0, nn::fs::ReadOption::MakeValue(0));

    TestReadFile(GetFileHandle(), 0, nullptr, GetFileSizeBufferSize());
    TestReadFile(GetFileHandle(), 0, nullptr, GetFileSizeBufferSize(), nn::fs::ReadOption::MakeValue(0));
    TestReadFile(&readSize, GetFileHandle(), 0, nullptr, GetFileSizeBufferSize());
    TestReadFile(&readSize, GetFileHandle(), 0, nullptr, GetFileSizeBufferSize(), nn::fs::ReadOption::MakeValue(0));
}

TEST_F(ReadFileBehaviorDifference, ZeroSize)
{
    size_t readSize;

    TestReadFile(GetFileHandle(), 0, GetFileSizeBuffer(), 0);
    TestReadFile(GetFileHandle(), 0, GetFileSizeBuffer(), 0, nn::fs::ReadOption::MakeValue(0));
    TestReadFile(&readSize, GetFileHandle(), 0, GetFileSizeBuffer(), 0);
    TestReadFile(&readSize, GetFileHandle(), 0, GetFileSizeBuffer(), 0, nn::fs::ReadOption::MakeValue(0));

    TestReadFile(GetFileHandle(), static_cast<int64_t>(GetFileSizeBufferSize()), GetFileSizeBuffer(), 0);
    TestReadFile(GetFileHandle(), static_cast<int64_t>(GetFileSizeBufferSize()), GetFileSizeBuffer(), 0, nn::fs::ReadOption::MakeValue(0));
    TestReadFile(&readSize, GetFileHandle(), static_cast<int64_t>(GetFileSizeBufferSize()), GetFileSizeBuffer(), 0);
    TestReadFile(&readSize, GetFileHandle(), static_cast<int64_t>(GetFileSizeBufferSize()), GetFileSizeBuffer(), 0, nn::fs::ReadOption::MakeValue(0));
}

TEST_F(ReadFileBehaviorDifference, TooLargeSize)
{
    size_t readSize;

    TestReadFile(GetFileHandle(), 0, GetFileSizeBuffer(), SIZE_MAX);
    TestReadFile(GetFileHandle(), 0, GetFileSizeBuffer(), SIZE_MAX, nn::fs::ReadOption::MakeValue(0));
    TestReadFile(&readSize, GetFileHandle(), 0, GetFileSizeBuffer(), SIZE_MAX);
    TestReadFile(&readSize, GetFileHandle(), 0, GetFileSizeBuffer(), SIZE_MAX, nn::fs::ReadOption::MakeValue(0));
}

TEST_F(ReadFileBehaviorDifference, ReadBeyondItsSizeLimit)
{
    size_t readSize;

    TestReadFile(GetFileHandle(), 0, GetFileSizeBuffer(), GetFileSizeBufferSize() * 2);
    TestReadFile(GetFileHandle(), 0, GetFileSizeBuffer(), GetFileSizeBufferSize() * 2, nn::fs::ReadOption::MakeValue(0));
    TestReadFile(&readSize, GetFileHandle(), 0, GetFileSizeBuffer(), GetFileSizeBufferSize() * 2);
    TestReadFile(&readSize, GetFileHandle(), 0, GetFileSizeBuffer(), GetFileSizeBufferSize() * 2, nn::fs::ReadOption::MakeValue(0));

    TestReadFile(GetFileHandle(), static_cast<int64_t>(GetFileSizeBufferSize()), GetFileSizeBuffer(), GetFileSizeBufferSize() * 2);
    TestReadFile(GetFileHandle(), static_cast<int64_t>(GetFileSizeBufferSize()), GetFileSizeBuffer(), GetFileSizeBufferSize() * 2, nn::fs::ReadOption::MakeValue(0));
    TestReadFile(&readSize, GetFileHandle(), static_cast<int64_t>(GetFileSizeBufferSize()), GetFileSizeBuffer(), GetFileSizeBufferSize() * 2);
    TestReadFile(&readSize, GetFileHandle(), static_cast<int64_t>(GetFileSizeBufferSize()), GetFileSizeBuffer(), GetFileSizeBufferSize() * 2, nn::fs::ReadOption::MakeValue(0));
}
