﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <memory>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <nn/aoc.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/util/util_Optional.h>

#include <nn/fs.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_RomFsFileSystem.h>
#include <nn/fs/fsa/fs_Registrar.h>
#include <nn/fssrv/sf/fssrv_IFileSystemProxy.h>
#include "shim/fs_FileSystemProxyServiceObject.h"
#include "shim/fs_StorageServiceObjectAdapter.h"

#include "testFs_Integration_FileDataCache_Util.h"


namespace
{

class ErrorEmulationStorage : public nn::fs::IStorage
{
private:
    std::unique_ptr<nn::fs::IStorage> m_pStorage;
    nn::util::optional<nn::Result> m_EmulationResult;

public:
    explicit ErrorEmulationStorage(std::unique_ptr<nn::fs::IStorage>&& pStorage) NN_NOEXCEPT
        : m_pStorage(std::move(pStorage))
        , m_EmulationResult(nn::util::nullopt)
    {
    }

    void SetErrorResult(nn::Result result) NN_NOEXCEPT
    {
        m_EmulationResult = result;
    }
    void RemoveErrorResult() NN_NOEXCEPT
    {
        m_EmulationResult = nn::util::nullopt;
    }

    virtual nn::Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        if (m_EmulationResult)
        {
            return m_EmulationResult.value();
        }
        else
        {
            return m_pStorage->Read(offset, buffer, size);
        }
    }
    virtual nn::Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        if (m_EmulationResult)
        {
            return m_EmulationResult.value();
        }
        else
        {
            return m_pStorage->Flush();
        }
    }
    virtual nn::Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        if (m_EmulationResult)
        {
            return m_EmulationResult.value();
        }
        else
        {
            return m_pStorage->GetSize(outValue);
        }
    }
};

// ----------------------------------------------------------------------------

class MountAddOnContentTestFixture : public AutoAbortDisabledTestFixture
{
private:
    struct AddOnContentInfo
    {
        nn::aoc::AddOnContentIndex index;
        std::unique_ptr<nn::Bit8> cache;
        size_t cacheSize;
    };

private:
    int m_AocCount;
    std::unique_ptr<AddOnContentInfo[]> m_AocInfos;

public:
    int GetAddOnContentCount() const NN_NOEXCEPT
    {
        return m_AocCount;
    }

    void* GetFileSystemCacheBuffer(int index) const NN_NOEXCEPT
    {
        for (int i = 0; i < m_AocCount; i++)
        {
            if (m_AocInfos[i].index == index)
            {
                return m_AocInfos[i].cache.get();
            }
        }
        NN_ABORT("aoc index %d not detected", index);
    }

    size_t GetFileSystemCacheBufferSize(int index) const NN_NOEXCEPT
    {
        for (int i = 0; i < m_AocCount; i++)
        {
            if (m_AocInfos[i].index == index)
            {
                return m_AocInfos[i].cacheSize;
            }
        }
        NN_ABORT("aoc index %d not detected", index);
    }

protected:
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        m_AocCount = nn::aoc::CountAddOnContent();
        NN_ABORT_UNLESS(m_AocCount > 0);

        std::unique_ptr<nn::aoc::AddOnContentIndex[]> indexes(new nn::aoc::AddOnContentIndex[m_AocCount]);
        m_AocInfos.reset(new AddOnContentInfo[m_AocCount]);

        int listedCount = nn::aoc::ListAddOnContent(indexes.get(), 0, m_AocCount);
        NN_ABORT_UNLESS_EQUAL(listedCount, m_AocCount);

        for (int i = 0; i < m_AocCount; i++)
        {
            m_AocInfos[i].index = indexes[i];
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountAddOnContentCacheSize(&m_AocInfos[i].cacheSize, m_AocInfos[i].index));
            m_AocInfos[i].cache.reset(new nn::Bit8[m_AocInfos[i].cacheSize]);
        }

        AutoAbortDisabledTestFixture::SetUp();
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        AutoAbortDisabledTestFixture::TearDown();
    }
};
typedef MountAddOnContentTestFixture MountAddOnContentBahaviorDifference;

class ReadFileErrorEmulationTestFixture : public ReadFileTestFixture
{
private:
    const char* RomMountName = "rom";

private:
    void* m_FileSystemCache;
    ErrorEmulationStorage* m_pErrorEmulationStorage;

private:
    void SetUpErrorEmulationFileSystem() NN_NOEXCEPT
    {
        size_t fileSystemCacheSize;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&fileSystemCacheSize));

        m_FileSystemCache = std::malloc(fileSystemCacheSize);
        NN_ABORT_UNLESS_NOT_NULL(m_FileSystemCache);

        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = nn::fs::detail::GetFileSystemProxyServiceObject();
        nn::sf::SharedPointer<nn::fssrv::sf::IStorage> storage;
        NN_ABORT_UNLESS_RESULT_SUCCESS(fileSystemProxy->OpenDataStorageByCurrentProcess(&storage));

        std::unique_ptr<nn::fs::IStorage> storageAbstract(new nn::fs::detail::StorageServiceObjectAdapter(std::move(storage)));
        NN_ABORT_UNLESS_NOT_NULL(storageAbstract);

        std::unique_ptr<ErrorEmulationStorage> errorEmulationStorage(new ErrorEmulationStorage(std::move(storageAbstract)));
        m_pErrorEmulationStorage = errorEmulationStorage.get();  // ホントは shared_ptr とか weak_ptr とかにしたいが、ライブラリ側を変えるほどではないと思う
        NN_ABORT_UNLESS_NOT_NULL(errorEmulationStorage);

        std::unique_ptr<nn::fs::RomFsFileSystem> fileSystem(new nn::fs::RomFsFileSystem());
        NN_ABORT_UNLESS_NOT_NULL(fileSystem);
        NN_ABORT_UNLESS_RESULT_SUCCESS(fileSystem->Initialize(std::move(errorEmulationStorage), m_FileSystemCache, fileSystemCacheSize, true));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::fsa::Register(RomMountName, std::move(fileSystem), nullptr, true, true, false));
    }

    void TearDownErrorEmulationFileSystem() NN_NOEXCEPT
    {
        nn::fs::Unmount(RomMountName);

        // Unmount の中で delete されるはず
        m_pErrorEmulationStorage = nullptr;
    }

protected:
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        SetUpErrorEmulationFileSystem();
        SetUpFileHandle();
        AutoAbortDisabledTestFixture::SetUp();
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        m_pErrorEmulationStorage->RemoveErrorResult();

        AutoAbortDisabledTestFixture::TearDown();
        TearDownFileHandle();
        TearDownErrorEmulationFileSystem();
    }

public:
    void SetErrorResult(nn::Result result) NN_NOEXCEPT
    {
        m_pErrorEmulationStorage->SetErrorResult(result);
    }
};
typedef ReadFileErrorEmulationTestFixture ReadFileErrorBehaviorDifference;

// ----------------------------------------------------------------------------

void TestMountAddOnContent(const char* name, nn::aoc::AddOnContentIndex index, void* buffer, size_t bufferSize)
{
    nn::Result cacheEnabledResult;
    nn::Result cacheDisabledResult;
    {
        ScopedGlobalFileDataCacheEnabler fileDataCache(DefaultFileDataCacheSize);
        cacheEnabledResult = nn::fs::MountAddOnContent(name, index, buffer, bufferSize);
        if (cacheEnabledResult.IsSuccess())
        {
            nn::fs::Unmount(name);
        }
    }
    {
        cacheDisabledResult = nn::fs::MountAddOnContent(name, index, buffer, bufferSize);
        if (cacheDisabledResult.IsSuccess())
        {
            nn::fs::Unmount(name);
        }
    }
    EXPECT_EQ(cacheEnabledResult.GetInnerValueForDebug(), cacheDisabledResult.GetInnerValueForDebug());
}

void DeathTestMountAddOnContent(const char* name, nn::aoc::AddOnContentIndex index, void* buffer, size_t bufferSize)
{
    {
        ScopedGlobalFileDataCacheEnabler fileDataCache(DefaultFileDataCacheSize);
        EXPECT_DEATH_IF_SUPPORTED(
            nn::fs::MountAddOnContent(name, index, buffer, bufferSize), "");
    }
    {
        EXPECT_DEATH_IF_SUPPORTED(
            nn::fs::MountAddOnContent(name, index, buffer, bufferSize), "");
    }
}

}  // namespace unnamed


TEST_F(MountAddOnContentBahaviorDifference, InvalidMountName)
{
    TestMountAddOnContent(nullptr, 1, GetFileSystemCacheBuffer(1), GetFileSystemCacheBufferSize(1));
    TestMountAddOnContent("C", 1, GetFileSystemCacheBuffer(1), GetFileSystemCacheBufferSize(1));
    TestMountAddOnContent("@aoc", 1, GetFileSystemCacheBuffer(1), GetFileSystemCacheBufferSize(1));
    TestMountAddOnContent("aoc:", 1, GetFileSystemCacheBuffer(1), GetFileSystemCacheBufferSize(1));
    TestMountAddOnContent("aoc/", 1, GetFileSystemCacheBuffer(1), GetFileSystemCacheBufferSize(1));
    {
        ScopedAddOnContentMounter aoc("aoc", 1);
        TestMountAddOnContent("aoc", 1, GetFileSystemCacheBuffer(1), GetFileSystemCacheBufferSize(1));
    }

    // AutoAbort の設定に関わらず内部アボートする
    DeathTestMountAddOnContent("", 1, GetFileSystemCacheBuffer(1), GetFileSystemCacheBufferSize(1));
    DeathTestMountAddOnContent("12345678901234567890", 1, GetFileSystemCacheBuffer(1), GetFileSystemCacheBufferSize(1));
}

TEST_F(MountAddOnContentBahaviorDifference, InvalidAddOnContentIndex)
{
    TestMountAddOnContent("aoc", -1, GetFileSystemCacheBuffer(1), GetFileSystemCacheBufferSize(1));
    TestMountAddOnContent("aoc", 0, GetFileSystemCacheBuffer(1), GetFileSystemCacheBufferSize(1));
    TestMountAddOnContent("aoc", GetAddOnContentCount() + 1, GetFileSystemCacheBuffer(1), GetFileSystemCacheBufferSize(1));
}

TEST_F(MountAddOnContentBahaviorDifference, NullPointerBuffer)
{
    TestMountAddOnContent("aoc", 1, nullptr, 0);
    TestMountAddOnContent("aoc", 1, nullptr, GetFileSystemCacheBufferSize(1));
}

TEST_F(MountAddOnContentBahaviorDifference, TooSmallBufferSize)
{
#if defined(NN_BUILD_TARGET_PLATFORM_OS_NN)

    // AutoAbort の設定に関わらず内部アボートする
    DeathTestMountAddOnContent("aoc", 1, GetFileSystemCacheBuffer(1), 0);
    DeathTestMountAddOnContent("aoc", 1, GetFileSystemCacheBuffer(1), GetFileSystemCacheBufferSize(1) - 1);

#else
#error "unsupported os"
#endif
}

TEST_F(MountAddOnContentBahaviorDifference, Mount128FileSystem)
{
    const nn::aoc::AddOnContentIndex targetIndex = 1;

    void* caches[127];
    size_t cacheSize;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountAddOnContentCacheSize(&cacheSize, targetIndex));

    for (int i = 0; i < 128; i++)
    {
        if (i == 128 - 1)
        {
            TestMountAddOnContent("aoc127", targetIndex, GetFileSystemCacheBuffer(targetIndex), GetFileSystemCacheBufferSize(targetIndex));
        }
        else
        {
            char mountName[8];
            ::snprintf(mountName, sizeof(mountName), "aoc%d", i);

            caches[i] = std::malloc(cacheSize);
            NN_ABORT_UNLESS_NOT_NULL(caches[i]);

            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountAddOnContent(mountName, targetIndex, caches[i], cacheSize));
        }
    }
    for (int i = 0; i < 127; i++)
    {
        char mountName[8];
        ::snprintf(mountName, sizeof(mountName), "aoc%d", i);

        nn::fs::Unmount(mountName);

        std::free(caches[i]);
    }
}
TEST_F(ReadFileErrorBehaviorDifference, ResultRomNonRealDataVerificationFailed)
{
    SetErrorResult(nn::fs::ResultRomNonRealDataVerificationFailed());

    size_t readSize;

    TestReadFile(GetFileHandle(), 0, GetFileSizeBuffer(), GetFileSizeBufferSize());
    TestReadFile(GetFileHandle(), 0, GetFileSizeBuffer(), GetFileSizeBufferSize(), nn::fs::ReadOption::MakeValue(0));
    TestReadFile(&readSize, GetFileHandle(), 0, GetFileSizeBuffer(), GetFileSizeBufferSize());
    TestReadFile(&readSize, GetFileHandle(), 0, GetFileSizeBuffer(), GetFileSizeBufferSize(), nn::fs::ReadOption::MakeValue(0));
}

TEST_F(ReadFileErrorBehaviorDifference, ResultUnclearedRomRealDataVerificationFailed)
{
    SetErrorResult(nn::fs::ResultUnclearedRomRealDataVerificationFailed());

    size_t readSize;

    TestReadFile(GetFileHandle(), 0, GetFileSizeBuffer(), GetFileSizeBufferSize());
    TestReadFile(GetFileHandle(), 0, GetFileSizeBuffer(), GetFileSizeBufferSize(), nn::fs::ReadOption::MakeValue(0));
    TestReadFile(&readSize, GetFileHandle(), 0, GetFileSizeBuffer(), GetFileSizeBufferSize());
    TestReadFile(&readSize, GetFileHandle(), 0, GetFileSizeBuffer(), GetFileSizeBufferSize(), nn::fs::ReadOption::MakeValue(0));
}

TEST_F(ReadFileErrorBehaviorDifference, ResultInvalidOffset)
{
    SetErrorResult(nn::fs::ResultInvalidOffset());

    size_t readSize;

    TestReadFile(GetFileHandle(), 0, GetFileSizeBuffer(), GetFileSizeBufferSize());
    TestReadFile(GetFileHandle(), 0, GetFileSizeBuffer(), GetFileSizeBufferSize(), nn::fs::ReadOption::MakeValue(0));
    TestReadFile(&readSize, GetFileHandle(), 0, GetFileSizeBuffer(), GetFileSizeBufferSize());
    TestReadFile(&readSize, GetFileHandle(), 0, GetFileSizeBuffer(), GetFileSizeBufferSize(), nn::fs::ReadOption::MakeValue(0));
}
