﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/fs/fs_RightsId.h>
#include <nn/fs/fs_Content.h>
#include <nn/fs/fs_ResultHandler.h>

#include <nn/es.h>
#include <nn/es/es_MaxTicketSize.h>
#include <nn/spl/spl_Api.h>

using namespace nn;
using namespace nn::fs;
using namespace nn::fs::detail;
using namespace nnt::fs::util;

namespace {

    //! @pre この RightsId, KeyGeneration で作成済み
    const RightsId g_RightsId = { { 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0xB1, 0xC5,  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, } };
    const int g_KeyGeneration = 5;

#if !defined(NN_BUILD_CONFIG_OS_WIN)

    //! @pre 0x0005000c10000000 のアプリがインストール済み
    TEST(GetRightsId, Application)
    {
        RightsId rightsId;
        nn::ncm::ProgramId programId = { 0x010000000000B1C5ULL };
        nn::ncm::StorageId storageId = nn::ncm::StorageId::BuildInUser;

        NNT_EXPECT_RESULT_SUCCESS(GetRightsId(&rightsId, programId, storageId));

        RightsId rightsIdExpected = { { 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0xB1, 0xC5,  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, } };
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(&rightsIdExpected, &rightsId, sizeof(RightsId));
    }

    TEST(ExternalKey, Register)
    {
        RightsId rightsId = { { 0 } };
        nn::spl::AccessKey accessKey = { { 0 } };

        NNT_EXPECT_RESULT_SUCCESS(RegisterExternalKey(rightsId, accessKey));
        NNT_EXPECT_RESULT_SUCCESS(UnregisterAllExternalKey());
    }

    TEST(ExternalKey, DuplicatedRegister)
    {
        RightsId rightsId = { { 0 } };
        nn::spl::AccessKey accessKey = { { 0 } };

        for (int i = 0; i < 128 * 1024; i++)
        {
            NNT_EXPECT_RESULT_SUCCESS(RegisterExternalKey(rightsId, accessKey));
        }

        NNT_EXPECT_RESULT_SUCCESS(UnregisterAllExternalKey());
    }

    void GetExternalKey(spl::AccessKey* pOutValue)
    {
        es::RightsIdIncludingKeyId esRightsId;
        std::memcpy(&esRightsId._data, &g_RightsId.data, sizeof(esRightsId._data));

        size_t ticketSize;
        uint8_t ticketBuffer[es::MaxTicketSize] = {};

        // コモンチケットのデータを取得する
        NNT_ASSERT_RESULT_SUCCESS(es::GetCommonTicketData(&ticketSize, ticketBuffer, sizeof(ticketBuffer), esRightsId));

        // チケット内のタイトル鍵のオフセットとサイズ
        const int TicketTitleKeyOffset = 0x180;
        const int TicketAesTitleKeySize = 16;

        spl::InitializeForEs();
        NN_UTIL_SCOPE_EXIT
        {
            spl::Finalize();
        };

        // タイトル鍵のアクセスキーを取得する
        NNT_ASSERT_RESULT_SUCCESS(spl::PrepareCommonEsTitleKey(pOutValue, &ticketBuffer[TicketTitleKeyOffset], TicketAesTitleKeySize, g_KeyGeneration));
    }


    //! @pre 本テストが作成済み
    TEST(ExternalKey, MountContent)
    {
        nn::spl::AccessKey accessKey;

        size_t BufferSize = 8 * 1024;
        auto buffer = AllocateBuffer(BufferSize);

        // 登録前はマウントできないこと
        {
            NNT_EXPECT_RESULT_SUCCESS(UnregisterAllExternalKey());
            NNT_EXPECT_RESULT_FAILURE(ResultNcaExternalKeyUnregistered, MountRom("rom", buffer.get(), BufferSize));
        }

        // 登録後マウントできること
        GetExternalKey(&accessKey);
        NNT_EXPECT_RESULT_SUCCESS(RegisterExternalKey(g_RightsId, accessKey));

        {
            NNT_EXPECT_RESULT_SUCCESS(MountRom("rom", buffer.get(), BufferSize));
            Unmount("rom");
        }

        // 外部鍵登録削除後はマウントできないこと
        {
            NNT_EXPECT_RESULT_SUCCESS(UnregisterAllExternalKey());
            NNT_EXPECT_RESULT_FAILURE(ResultNcaExternalKeyUnregistered, MountRom("rom", buffer.get(), BufferSize));
        }
    }


    //! @pre --ticket でこのテストが作成済み＆インストール済み
    TEST(ExternalKey, Performance)
    {
        NNT_EXPECT_RESULT_SUCCESS(UnregisterAllExternalKey());

        // 8192 個登録できること、時間がかからないこと
        {
            TimeCount registerTime;
            registerTime.StartTime();

            const int KeyCount = 8192;
            {
                for (int i = 0; i < KeyCount; i++)
                {
                    RightsId rightsId;
                    nn::spl::AccessKey accessKey;
                    nnt::fs::util::FillBufferWith8BitCount(&rightsId, sizeof(rightsId), i);
                    nnt::fs::util::FillBufferWith8BitCount(&accessKey, sizeof(accessKey), i);

                    NNT_EXPECT_RESULT_SUCCESS(RegisterExternalKey(rightsId, accessKey));
                }
            }

            registerTime.StopTime();
            registerTime.ViewTime(registerTime.GetTotalTime());
            EXPECT_LE(registerTime.GetTotalTime(), 1000 * 1000 * 1000); // <= 1000ms
        }


        // 8192 個登録後のルックアップに時間がかからないこと
        {
            // 自分の鍵を登録しなおし
            nn::spl::AccessKey accessKey;
            GetExternalKey(&accessKey);
            NNT_ASSERT_RESULT_SUCCESS(fs::RegisterExternalKey(g_RightsId, accessKey));

            {
                size_t BufferSize = 8 * 1024;
                auto buffer = AllocateBuffer(BufferSize);

                TimeCount mountTime;
                mountTime.StartTime();
                {
                    NNT_EXPECT_RESULT_SUCCESS(MountRom("rom", buffer.get(), BufferSize));
                }
                mountTime.StopTime();
                mountTime.ViewTime(mountTime.GetTotalTime());
                EXPECT_LE(mountTime.GetTotalTime(), 100 * 1000 * 1000); // <= 100ms

                Unmount("rom");
            }

        }

        // 8192 個登録後の全クリアに時間がかからないこと
        {
            TimeCount unregisterTime;
            unregisterTime.StartTime();

            NNT_EXPECT_RESULT_SUCCESS(UnregisterAllExternalKey());

            unregisterTime.StopTime();
            unregisterTime.ViewTime(unregisterTime.GetTotalTime());
            EXPECT_LE(unregisterTime.GetTotalTime(), 100 * 1000 * 1000); // <= 100ms

        }

    }

#endif

}

extern "C" void nnMain()
{
    int     argc = nn::os::GetHostArgc();
    char**  argv = nn::os::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);
    nn::fs::SetEnabledAutoAbort(false);
    nn::es::Initialize();

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
