﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/mem.h>
#include <nn/audio.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>

using namespace nn;
using namespace nn::fs;
using namespace nn::fs::detail;
using namespace nnt::fs::util;

namespace {

const int FileSize = 1 * 1024 * 1024;

// AudioRenderer サンプルから抜粋

NN_ALIGNAS(4096) char g_WorkBuffer[8 * 1024 * 1024];
NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_WaveBufferPoolMemory[14 * 1024 * 1024];

nn::mem::StandardAllocator g_Allocator;
nn::audio::AudioRendererHandle g_Handle;

void SetUpBuffer()
{
    g_Allocator.Initialize(g_WorkBuffer, sizeof(g_WorkBuffer));

    // レンダラのパラメータを指定します。
    nn::audio::AudioRendererParameter parameter;
    nn::audio::InitializeAudioRendererParameter(&parameter);

    // レンダラのワークバッファを準備します。
    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
    void* workBuffer = g_Allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(workBuffer);

    nn::os::SystemEvent systemEvent;

    // レンダラを初期化します。
    NN_ABORT_UNLESS(
        nn::audio::OpenAudioRenderer(&g_Handle, &systemEvent, parameter, workBuffer, workBufferSize).IsSuccess(),
        "Failed to open AudioRenderer"
    );

    // AudioRendererConfig を初期化します。
    size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(parameter);
    void* configBuffer = g_Allocator.Allocate(configBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(configBuffer);
    nn::audio::AudioRendererConfig config;
    nn::audio::InitializeAudioRendererConfig(&config, parameter, configBuffer, configBufferSize);

    // WaveBuffer に追加するサンプルデータを保持するためのメモリプールを準備します。
    nn::audio::MemoryPoolType waveBufferMemoryPool;
    auto rt = AcquireMemoryPool(&config, &waveBufferMemoryPool, g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory));
    NN_ABORT_UNLESS(rt);
    rt = RequestAttachMemoryPool(&waveBufferMemoryPool);
    NN_ABORT_UNLESS(rt);

    auto result = nn::audio::RequestUpdateAudioRenderer(g_Handle, &config);
    NN_ABORT_UNLESS(result.IsSuccess());
}


void TearDownBuffer()
{
    // レンダリングを終了します。
    nn::audio::CloseAudioRenderer(g_Handle);
}


void WriteReadFileWithDeviceSharedMemory(const char* path)
{
    DeleteFile(path);
    NNT_ASSERT_RESULT_SUCCESS(CreateFile(path, FileSize));

    FileHandle handle;
    NNT_ASSERT_RESULT_SUCCESS(OpenFile(&handle, path, OpenMode_Read | OpenMode_Write));

    void* buffer = g_WaveBufferPoolMemory;

    FillBufferWith32BitCount(buffer, FileSize, 0);

    NNT_EXPECT_RESULT_SUCCESS(WriteFile(handle, 0, buffer, FileSize, WriteOption()));
    NNT_EXPECT_RESULT_SUCCESS(FlushFile(handle));

    InvalidateVariable<char>(static_cast<char*>(buffer), FileSize);
    NNT_EXPECT_RESULT_SUCCESS(ReadFile(handle, 0, buffer, FileSize, ReadOption()));

    EXPECT_TRUE(IsFilledWith32BitCount(buffer, FileSize, 0));

    CloseFile(handle);
    NNT_EXPECT_RESULT_SUCCESS(DeleteFile(path));
}

TEST(DeviceSharedMemory, HostFs)
{
    nnt::fs::util::TemporaryHostDirectory hostDirectory;
    hostDirectory.Create();
    NNT_ASSERT_RESULT_SUCCESS(MountHostRoot());

    WriteReadFileWithDeviceSharedMemory((hostDirectory.GetPath() + "/file").c_str());

    UnmountHostRoot();
}

TEST(DeviceSharedMemory, BisFs)
{
    NNT_ASSERT_RESULT_SUCCESS(MountBis("bis", BisPartitionId::User));

    WriteReadFileWithDeviceSharedMemory("bis:/file");

    Unmount("bis");
}


}

extern "C" void nnMain()
{
    int     argc = nn::os::GetHostArgc();
    char**  argv = nn::os::GetHostArgv();

    nn::fs::SetEnabledAutoAbort(false);

    SetUpBuffer();

    ::testing::InitGoogleTest(&argc, argv);

    auto ret = RUN_ALL_TESTS();

    TearDownBuffer();

    nnt::Exit(ret);
}
