﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>
#include <memory>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/fs.h>
#include <nn/fs/fs_GameCard.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/fs/fs_IEventNotifier.h>
#include <nn/fs/fs_IStorage.h>

#include <nn/os.h>

#define ABORT_IF_FAILURE(result, ...) \
    if (result.IsFailure()) { NN_LOG(__VA_ARGS__); NN_LOG("###test###  Result: Module %d, Description %d\n", result.GetModule(), result.GetDescription()); NN_ABORT_UNLESS(result.IsSuccess()); }

#define NN_TEST_CSWS_LOG(...)  NN_LOG("###test### " __VA_ARGS__)

using namespace nn;
using namespace nn::fs;

namespace {

class Time
{
public:
    Time() : m_StartTime(0), m_EndTime(0) {}
    ~Time() {}

    void Start()
    {
        m_StartTime = nn::os::GetSystemTick();
    }

    int64_t GetElapsedTimeMsec()
    {
        m_EndTime = nn::os::GetSystemTick();
        return (m_EndTime - m_StartTime).ToTimeSpan().GetMilliSeconds();
    }

    int64_t GetElapsedTimeSec()
    {
        return GetElapsedTimeMsec() / 1000;
    }

private:
    nn::os::Tick m_StartTime;
    nn::os::Tick m_EndTime;
};


const size_t g_BufferSize = 4 * 1024 * 1024;
char g_Buffer[g_BufferSize];


// 挿入されたゲームカードの secure area 先頭 512 bytes を読み込む
void ReadGcSecureArea()
{
    NN_TEST_CSWS_LOG("[gc] Reading secure area.\n");

    GameCardHandle handle;

    // ゲームカードのハンドルを取得
    nn::Result result = GetGameCardHandle(&handle);
    ABORT_IF_FAILURE(result, "[gc] failed to open handle\n");

    // secure area をオープン
    std::unique_ptr<nn::fs::IStorage> storage;
    result = OpenGameCardPartition(&storage, handle, GameCardPartitionRaw::SecureReadOnly);
    ABORT_IF_FAILURE(result, "[gc] failed to open partition\n");

    // secure area の先頭 512 バイトを読み込み
    // アクセスのオフセット・サイズは 512 バイト単位である必要がある
    memset(g_Buffer, 0xAB, g_BufferSize);
    result = (storage->Read(0, g_Buffer, g_BufferSize));
    ABORT_IF_FAILURE(result, "[gc] failed to read secure area\n");
}

void SetData(void* pData, size_t dataSize) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pData);
    if (dataSize == 0)
    {
        return;
    }
    uint32_t* pCurrentData = reinterpret_cast<uint32_t*>(pData);
    for (size_t i = 0; i < (dataSize / sizeof(uint32_t)); i++)
    {
        pCurrentData[i] = i;
    }
}

bool CheckData(void* pData, size_t dataSize) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pData);
    if (dataSize == 0)
    {
        return true;
    }
    uint32_t* pCurrentData = reinterpret_cast<uint32_t*>(pData);
    for (size_t i = 0; i < (dataSize / sizeof(uint32_t)); i++)
    {
        if(pCurrentData[i] != i)
        {
            return false;
        }
    }
    return true;
}

void ReadSdCard()
{
    nn::Result result = nn::fs::MountSdCardForDebug("sdcard");
    ABORT_IF_FAILURE(result, "nn::fs::MountSdCardForDebug\n");

    result = nn::fs::DeleteFile("sdcard:/data.bin");
    if (result.IsFailure())
    {
        NN_TEST_CSWS_LOG("[sd] nn::fs::DeleteFile is failure. Module:%d, Description:%d\n", result.GetModule(), result.GetDescription());
    }

    result = nn::fs::CreateFile("sdcard:/data.bin", 0);
    ABORT_IF_FAILURE(result, "nn::fs::CreateFile\n");

    NN_TEST_CSWS_LOG("[sd] Write / Read %.1f(KB)\n", sizeof(g_Buffer) / 1024.0 );
    nn::fs::FileHandle fileHandle;
    result = nn::fs::OpenFile(&fileHandle, "sdcard:/data.bin", nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend);
    ABORT_IF_FAILURE(result, "[sd] nn::fs::OpenFile\n");
    SetData(g_Buffer, g_BufferSize);
    result = nn::fs::WriteFile(fileHandle, 0, g_Buffer, g_BufferSize, nn::fs::WriteOption());
    ABORT_IF_FAILURE(result, "[sd] nn::fs::WriteFile\n");
    result = nn::fs::FlushFile(fileHandle);
    ABORT_IF_FAILURE(result, "[sd] nn::fs::FlushFile\n");
    nn::fs::CloseFile(fileHandle);

    result = nn::fs::OpenFile(&fileHandle, "sdcard:/data.bin", nn::fs::OpenMode_Read);
    ABORT_IF_FAILURE(result, "[sd] nn::fs::OpenFile\n");
    std::memset(g_Buffer, 0xA5, sizeof(g_Buffer));
    result = nn::fs::ReadFile(fileHandle, 0, g_Buffer, g_BufferSize);
    ABORT_IF_FAILURE(result, "[sd] nn::fs::ReadFile\n");
    nn::fs::CloseFile(fileHandle);

    bool isDataValid = CheckData(g_Buffer, g_BufferSize);
    NN_ABORT_UNLESS(isDataValid, "[sd] invalid data detected in verification\n");

    nn::fs::Unmount("sdcard");
}

}

extern "C" void nndiagStartup()
{
}

extern "C" void nnMain()
{
    // 挿抜イベント通知をオープン
    std::unique_ptr<nn::fs::IEventNotifier> gcDetectionEventNotifier;
    std::unique_ptr<nn::fs::IEventNotifier> sdDetectionEventNotifier;
    nn::os::SystemEventType gcDetectionEvent;
    nn::os::SystemEventType sdDetectionEvent;
    Result result;

    // GC の登録
    {
        result = OpenGameCardDetectionEventNotifier(&gcDetectionEventNotifier);
        ABORT_IF_FAILURE(result, "failed to open gc event notifier\n");

        // システムイベントと紐づけ
        result = gcDetectionEventNotifier->BindEvent(&gcDetectionEvent, nn::os::EventClearMode_ManualClear);
        ABORT_IF_FAILURE(result, "failed to bind gc detection event\n");
    }

    // SD の登録
    {
        result = OpenSdCardDetectionEventNotifier(&sdDetectionEventNotifier);
        ABORT_IF_FAILURE(result, "failed to open sd event notifier\n");

        // システムイベントと紐づけ
        result = sdDetectionEventNotifier->BindEvent(&sdDetectionEvent, nn::os::EventClearMode_ManualClear);
        ABORT_IF_FAILURE(result, "failed to bind sd detection event\n");
    }

    // 挿入状態のチェック
    NN_ABORT_UNLESS(IsSdCardInserted(), "SD card is not inserted\n");
    NN_ABORT_UNLESS(IsGameCardInserted(), "Game card is not inserted\n");

    // イベントをクリア
    nn::os::ClearSystemEvent(&gcDetectionEvent);
    nn::os::ClearSystemEvent(&sdDetectionEvent);

    // 引数取得
    int waitSecCount = 30 * 60;
    int readMsecCount = 1000;
    int sleepMsecCount = 1000;
    {
        int argc = nn::os::GetHostArgc();
        char** argv = nn::os::GetHostArgv();
        if(argc == 1)
        {
            NN_TEST_CSWS_LOG("usage: CardStateWhenSleep <testTimeSec> <readTimeMsec> <sleepTimeMsec>\n");
        }
        if(argc > 1)
        {
            waitSecCount = atoi(argv[1]);
        }
        if(argc > 2)
        {
            readMsecCount = atoi(argv[2]);
        }
        if(argc > 3)
        {
            sleepMsecCount = atoi(argv[3]);
        }
    }
    NN_TEST_CSWS_LOG("Test - card state while sleeping: %d sec - (read %d msec, sleep %d msec)\n", waitSecCount, readMsecCount, sleepMsecCount);

    Time sleepTimer;
    sleepTimer.Start();

    // たまに読みながらスリープ：リードに失敗したり挿抜イベントが上がると ABORT させる
    do
    {
        Time actionTimer;
        actionTimer.Start();
        NN_TEST_CSWS_LOG("read %d msec\n", readMsecCount);
        do
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));

            // 読む（読めなかったら ABORT）
            ReadGcSecureArea();
            ReadSdCard();

            // イベントがないことを確認する
            nn::os::TryWaitSystemEvent(&gcDetectionEvent);
        } while (actionTimer.GetElapsedTimeSec() < (readMsecCount / 1000));

        // アクセスしていない状態も一応作る
        NN_TEST_CSWS_LOG("sleep %d msec\n", sleepMsecCount);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(sleepMsecCount));

        // イベントが起きていたら ABORT
        bool isGcEventOccurred = nn::os::TryWaitSystemEvent(&gcDetectionEvent);
        bool isSdEventOccurred = nn::os::TryWaitSystemEvent(&sdDetectionEvent);
        if (isGcEventOccurred || isSdEventOccurred)
        {
            NN_TEST_CSWS_LOG("Error: Event detected - gc: %d, sd: %d\n", isGcEventOccurred, isSdEventOccurred);
            NN_ABORT("TEST FAILURE\n");
        }

        NN_TEST_CSWS_LOG("time: %.2f / %d\n", sleepTimer.GetElapsedTimeMsec() / 1000.0, waitSecCount);
    } while (sleepTimer.GetElapsedTimeSec() < waitSecCount);

    // イベントを破棄
    nn::os::DestroySystemEvent(&gcDetectionEvent);
    nn::os::DestroySystemEvent(&sdDetectionEvent);

    NN_TEST_CSWS_LOG("[FS SLEEP TEST SUCCESS]\n");
}
