﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsApi/testFs_Api.h>
#include <nnt/fsApi/testFs_Integration_Api_SaveData.h>
#include <nnt/fsApiSaveDataExtraData/testFs_ApiSaveDataExtraData.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

#include <nn/fs/fs_TemporaryStorage.h>
#include <nn/fs/fs_SaveDataManagementPrivate.h>

#include <nnt/fsUtil/testFs_util_GlobalNewDeleteChecker.Impl.h>

namespace {
    const char MountName[] = "TmpStorage";
    const int HeapSize = 128 * 1024;
    const int64_t StorageSize = 40ULL * 1024 * 1024;
    char g_HeapStack[HeapSize];
    char g_RootDirPath[256];
    nn::fs::SaveDataId g_SaveDataId;

    void SetupTemporaryStorage() NN_NOEXCEPT
    {
        nn::util::optional<nn::fs::SaveDataInfo> info;
        nnt::fs::util::FindSaveData(&info, nn::fs::SaveDataSpaceId::Temporary, [&](const nn::fs::SaveDataInfo& i) NN_NOEXCEPT
        {
            return i.applicationId.value == 0x0100000000010000ull;
        });
        ASSERT_TRUE(static_cast<bool>(info));
        g_SaveDataId = info->saveDataId;

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountTemporaryStorage(MountName));
        nn::util::SNPrintf(g_RootDirPath, sizeof(g_RootDirPath), "%s:/API_TEST", MountName);
        nn::fs::DeleteDirectoryRecursively(g_RootDirPath);
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateDirectory(g_RootDirPath));
        nn::fs::Unmount(MountName);
    }

    void TeardownTemporaryStorage() NN_NOEXCEPT
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountTemporaryStorage(MountName));
        nn::fs::DeleteDirectory(g_RootDirPath);
        nn::fs::Unmount(MountName);
    }
}

namespace nnt { namespace fs { namespace api {
    class FsApiTemporaryStorage : public FsApiIntegrationTestFileSystem, public ITestSaveDataExtraDataAccessor
    {
    public:
        FsApiTemporaryStorage() NN_NOEXCEPT : FsApiIntegrationTestFileSystem(MountName)
        {
            NN_ABORT_UNLESS(nn::fs::MountTemporaryStorage(MountName).IsSuccess());
        }
        ~FsApiTemporaryStorage() NN_NOEXCEPT
        {
            nn::fs::Unmount(MountName);
        }

    public:
        virtual Result WriteExtraData(const nn::fs::SaveDataExtraData& extraData) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(nn::fs::detail::WriteSaveDataFileSystemExtraData(nn::fs::SaveDataSpaceId::Temporary, g_SaveDataId, extraData));
            NN_RESULT_SUCCESS;
        }

        virtual Result CommitExtraData(bool updateTimeStamp) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(updateTimeStamp);
            NN_RESULT_SUCCESS;
        }

        virtual Result ReadExtraData(nn::fs::SaveDataExtraData* outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(nn::fs::detail::ReadSaveDataFileSystemExtraData(outValue, nn::fs::SaveDataSpaceId::Temporary, g_SaveDataId));
            NN_RESULT_SUCCESS;
        }
    };

    void GetTestFileSystemInfo(std::unique_ptr<TestFileSystemInfo>* outValue, int index)
    {
        NN_UNUSED(index);
        std::unique_ptr<FsApiTemporaryStorage> tmpStorageFs(new FsApiTemporaryStorage());

        std::unique_ptr<TestFileSystemInfo> info(new TestFileSystemInfo(FsApiTestType::Integration));
        info->fileSystem = std::move(tmpStorageFs);
        info->type = FileSystemType_SaveDataFileSystem;
        info->rootDirPath = g_RootDirPath;

        auto& attribute = info->attribute;

        attribute.directoryNameLengthMax = nn::fssystem::dbm::MaxDirectoryNameLength;
        attribute.fileNameLengthMax      = nn::fssystem::dbm::MaxFileNameLength;
        // EntryNameLengthMax 以上のパスが shim で弾かれるので設定できるのは EntryNameLengthMax まで
        attribute.directoryPathLengthMax = nn::fs::EntryNameLengthMax;
        attribute.filePathLengthMax      = nn::fs::EntryNameLengthMax;

        attribute.fileSizeMax = 0xFFFFFFFF;
        attribute.storageSize = StorageSize;

        attribute.fileOpenMax      = 30;
        attribute.directoryOpenMax = 20;

        attribute.speedEmulationTypeFlag = static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::MmcStorageContextEnabled);

        attribute.isSupportedMultiBytePath = false;
        attribute.isSupportedGetFreeSpaceSize = true;
        attribute.isConcatenationFileSystem = false;
        attribute.isSupportedQueryRange = true;
        attribute.isSupportedFileOpenCheckOnCommit = true;

        attribute.fileSizeAlignment = 1;

        *outValue = std::move(info);
    }

    void GetTestSaveDataExtraDataInfo(std::unique_ptr<TestSaveDataExtraDataInfo>* outValue, ITestFileSystem* pFileSystem) NN_NOEXCEPT
    {
        std::unique_ptr<TestSaveDataExtraDataInfo> info(new TestSaveDataExtraDataInfo);
        info->saveDataExtraDataAccessor = reinterpret_cast<FsApiTemporaryStorage*>(pFileSystem);
        info->attribute.isCommitSaveDataExtraDataWhenWrite = true;
        *outValue = std::move(info);
    }
}}}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nnt::fs::api::LoadAllTests();
    nnt::fs::api::LoadAllSaveDataExtraDataTests();

    nnt::fs::util::InitializeTestLibraryHeap(g_HeapStack, HeapSize);
    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();
    nn::fs::SetEnabledAutoAbort(false);

    SetupTemporaryStorage();
    if (::testing::Test::HasFatalFailure())
    {
        NN_LOG("Setup TemoiraryStorag Failed\n");
        nnt::Exit(1);
    }

    auto ret = RUN_ALL_TESTS();

    TeardownTemporaryStorage();
    if (::testing::Test::HasFatalFailure())
    {
        NN_LOG("Teardown TemoiraryStorag Failed\n");
        nnt::Exit(1);
    }

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(ret);
}
