﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>

#include <nn/fs.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/os.h>

#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fssystem/fs_AesXtsFile.h>
#include <nn/util/util_FormatString.h>

#include <nn/fs/fs_SaveDataManagementPrivate.h>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>

#include <nnt/base/testBase_Exit.h>
#include <nnt/fsApi/testFs_Api.h>
#include <nnt/fsApi/testFs_Integration_Api_SaveData.h>
#include <nnt/fsApiSaveDataExtraData/testFs_ApiSaveDataExtraData.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>

#include <nnt/fsUtil/testFs_util_GlobalNewDeleteChecker.Impl.h>

namespace {
    const int HeapSize = 128 * 1024;
    char g_HeapStack[HeapSize];
    nnt::fs::api::SaveDataTestParameters g_Parameters;
    nn::fs::SaveDataSpaceId g_SaveDataSpaceId = nn::fs::SaveDataSpaceId::SdSystem;
}

namespace nnt { namespace fs { namespace api {
    class FsApiSdSaveData : public FsApiIntegrationTestFileSystem, public ITestSaveDataExtraDataAccessor
    {
    public:
        FsApiSdSaveData() NN_NOEXCEPT : FsApiIntegrationTestFileSystem(g_Parameters.mountName)
        {
            NN_ABORT_UNLESS(nn::fs::MountSystemSaveData(g_Parameters.mountName, g_SaveDataSpaceId, g_Parameters.saveDataId).IsSuccess());
        }
        virtual ~FsApiSdSaveData() NN_NOEXCEPT NN_OVERRIDE
        {
            nn::fs::Unmount(g_Parameters.mountName);
        }

    public:
        virtual Result WriteExtraData(const nn::fs::SaveDataExtraData& extraData) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(nn::fs::detail::WriteSaveDataFileSystemExtraData(g_SaveDataSpaceId, g_Parameters.saveDataId, extraData));
            NN_RESULT_SUCCESS;
        }

        virtual Result CommitExtraData(bool updateTimeStamp) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(updateTimeStamp);
            NN_RESULT_SUCCESS;
        }

        virtual Result ReadExtraData(nn::fs::SaveDataExtraData* outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(nn::fs::detail::ReadSaveDataFileSystemExtraData(outValue, g_SaveDataSpaceId, g_Parameters.saveDataId));
            NN_RESULT_SUCCESS;
        }
    };

    void GetTestFileSystemInfo(std::unique_ptr<TestFileSystemInfo>* outValue, int index)
    {
        NN_UNUSED(index);
        std::unique_ptr<FsApiSdSaveData> sdFs(new FsApiSdSaveData());

        std::unique_ptr<TestFileSystemInfo> info(new TestFileSystemInfo(FsApiTestType::Integration));
        info->fileSystem = std::move(sdFs);
        info->type = FileSystemType_SaveDataFileSystem;
        info->rootDirPath = g_Parameters.rootPath;
        info->attribute = GetSaveDataFileSystemAttribute(g_Parameters);
        info->attribute.isReadOverloadsSupported = false; // ストレージが遅いので無効化
        info->attribute.speedEmulationTypeFlag = static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::SdCardStorageContextEnabled)
                                               | static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::UsbStorageContextEnabled);

        *outValue = std::move(info);
    }

    void GetTestSaveDataExtraDataInfo(std::unique_ptr<TestSaveDataExtraDataInfo>* outValue, ITestFileSystem* pFileSystem) NN_NOEXCEPT
    {
        std::unique_ptr<TestSaveDataExtraDataInfo> info(new TestSaveDataExtraDataInfo);
        info->saveDataExtraDataAccessor = reinterpret_cast<FsApiSdSaveData*>(pFileSystem);
        info->attribute.isCommitSaveDataExtraDataWhenWrite = true;
        *outValue = std::move(info);
    }
}}}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    g_SaveDataSpaceId = nn::fs::SaveDataSpaceId::SdSystem;
    if (argc >= 2)
    {
        auto saveDataSpaceName = argv[1];
        if (std::strcmp(saveDataSpaceName, "User") == 0)
        {
            g_SaveDataSpaceId = nn::fs::SaveDataSpaceId::SdUser;
        }
        else if (std::strcmp(saveDataSpaceName, "System") == 0)
        {
            g_SaveDataSpaceId = nn::fs::SaveDataSpaceId::SdSystem;
        }
        else
        {
            NN_LOG("Unknown SaveDataSpace Name : Sd%s\n", saveDataSpaceName);
            nnt::Exit(1);
        }
        NN_LOG("SaveDataSpace is set: Sd%s\n", saveDataSpaceName);
    }

    nnt::fs::util::InitializeTestLibraryHeap(g_HeapStack, HeapSize);
    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();
    nn::fs::SetEnabledAutoAbort(false);

    nnt::fs::api::LoadAllTests();
    nnt::fs::api::LoadAllSaveDataExtraDataTests();

    nnt::fs::api::SetupSaveData(&g_Parameters, g_SaveDataSpaceId);
    if (::testing::Test::HasFatalFailure())
    {
        NN_LOG("Setup SaveData Failed\n");
        nnt::Exit(1);
    }

    auto ret = RUN_ALL_TESTS();

    nnt::fs::api::CleanupSaveData(g_SaveDataSpaceId, g_Parameters);
    if (::testing::Test::HasFatalFailure())
    {
        NN_LOG("Cleanup SaveData Failed\n");
        nnt::Exit(1);
    }

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(ret);
}
