﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsApi/testFs_Api.h>
#include <nnt/fsApi/testFs_Integration_Api.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

#include <nnt/fsUtil/testFs_util_GlobalNewDeleteChecker.Impl.h>

namespace {
#if !defined(FS_TEST_HOSTROOT)
    const char MountName[] = "TestHost";
#endif
    nnt::fs::util::TemporaryHostDirectory g_HostDirectory;
    const int HeapSize = 128 * 1024;
    char g_HeapStack[HeapSize];
    char g_RootDirPath[256];
}

namespace nnt { namespace fs { namespace api {
    class FsApiHostFileSystem : public FsApiIntegrationTestFileSystem
    {
    public:
        FsApiHostFileSystem() NN_NOEXCEPT
#if !defined(FS_TEST_HOSTROOT)
            : FsApiIntegrationTestFileSystem(MountName)
#endif
        {
#if defined(FS_TEST_HOSTROOT)
            NN_ABORT_UNLESS(nn::fs::MountHostRoot().IsSuccess());
#else
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountHost(MountName, g_HostDirectory.GetPath().c_str()));
#endif
        }
        ~FsApiHostFileSystem() NN_NOEXCEPT
        {
#if defined(FS_TEST_HOSTROOT)
            nn::fs::UnmountHostRoot();
#else
            nn::fs::Unmount(MountName);
#endif
        }
    };

    void GetTestFileSystemInfo(std::unique_ptr<TestFileSystemInfo>* outValue, int index)
    {
        NN_UNUSED(index);
        std::unique_ptr<FsApiHostFileSystem> hostFs(new FsApiHostFileSystem());

        std::unique_ptr<TestFileSystemInfo> info(new TestFileSystemInfo(FsApiTestType::Integration));
        info->fileSystem = std::move(hostFs);
        info->type = FileSystemType_HostFileSystem;
        info->rootDirPath = g_RootDirPath;

        auto& attribute = info->attribute;

        static const int HostFsPathLengthMax      = 260 - 1; // HostFileSystem::WindowsPathLengthMax からヌル文字分を引いた値
        static const int DirectoryPathLengthMax   = 248 - 1; // Windows の CreateDirectory に渡せるパスの最大長（'\0' を含まない）
        attribute.directoryNameLengthMax = DirectoryPathLengthMax;
        attribute.fileNameLengthMax      = HostFsPathLengthMax;
#if defined(FS_TEST_HOSTROOT)
        attribute.directoryPathLengthMax = DirectoryPathLengthMax;
        attribute.filePathLengthMax      = HostFsPathLengthMax;

        attribute.isCommitSupported = false;
#else
        static const int MountNameLength         = (sizeof(MountName) - 1) + 1;     // マウント名に ':' を足した文字数
        static const int HostDirectoryPathLength = std::strlen(g_HostDirectory.GetPath().c_str());  // Hostのマウントベースディレクトリの文字数)
        attribute.directoryPathLengthMax = DirectoryPathLengthMax + MountNameLength - HostDirectoryPathLength;
        attribute.filePathLengthMax      = HostFsPathLengthMax + MountNameLength - HostDirectoryPathLength;
#endif
        attribute.fileSizeMax = 0x0000000220000000LL;
        attribute.storageSize = 0x0000000240000000LL;

        attribute.fileOpenMax      = 128;
        attribute.directoryOpenMax = 64;

        attribute.isSupportedMultiBytePath = false;
        attribute.isSupportedGetFreeSpaceSize = false;
        attribute.isSupportedGetFileTimeStamp = true;
        attribute.isConcatenationFileSystem = false;
        attribute.isSupportedQueryRange = true;

        attribute.fileSizeAlignment = 1;

        *outValue = std::move(info);
    }
}}}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nnt::fs::api::LoadAllTests();

    nnt::fs::util::InitializeTestLibraryHeap(g_HeapStack, HeapSize);
    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();
    nn::fs::SetEnabledAutoAbort(false);

    g_HostDirectory.Create();
#if defined(FS_TEST_HOSTROOT)
    strncpy(g_RootDirPath, g_HostDirectory.GetPath().c_str(), sizeof(g_RootDirPath));
#else
    nn::util::SNPrintf(g_RootDirPath, sizeof(g_RootDirPath), "%s:", MountName);
#endif

    auto ret = RUN_ALL_TESTS();

    g_HostDirectory.Delete();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(ret);
}
