﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <array>

#include <nn/account.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/init/init_Malloc.h>
#include <nn/fs/fs_ApiPrivate.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_MemoryManagement.h>
#include <nn/fs/fs_Priority.h>
#include <nn/fs/fs_PriorityPrivate.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_Rom.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_Utility.h>
#include <nn/fssystem/fs_ThreadPriorityChanger.h>
#include <nn/fs/detail/fs_PriorityUtility.h>
#include <nn/os/os_Thread.h>
#include <nn/svc/svc_Base.h>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>

namespace
{
    struct DefaultValueArgument
    {
        nn::fs::Priority priority;
        nn::fs::PriorityRaw priorityRaw;
    };

    DefaultValueArgument g_MainThreadDefaultValue;

    void GetDefaultValue(DefaultValueArgument* pDefaultValueArgument) NN_NOEXCEPT
    {
        pDefaultValueArgument->priority = nn::fs::GetPriorityOnCurrentThread();
        pDefaultValueArgument->priorityRaw = nn::fs::GetPriorityRawOnCurrentThread();
    }

    const int WorkerThreadCountMax = 16;

    class ITestExecutor
    {
    public:
        virtual ~ITestExecutor() NN_NOEXCEPT {}

    public:
        virtual nn::Result Prepare(int index) NN_NOEXCEPT = 0;
        virtual void CleanUp(int index) NN_NOEXCEPT = 0;
        virtual nn::Result Execute(char* buffer, size_t bufferSize, int index) NN_NOEXCEPT = 0;
    };

    class RomFsTestExecutor : public ITestExecutor
    {
    public:
        RomFsTestExecutor() NN_NOEXCEPT
            : m_CacheBuffer(0, nnt::fs::util::DeleterBuffer)
        {
            size_t sizeRomCache;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&sizeRomCache));
            m_CacheBuffer = nnt::fs::util::AllocateBuffer(sizeRomCache);

            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountRom("rom", m_CacheBuffer.get(), sizeRomCache));

            {
                nn::fs::FileHandle file;
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&file, "rom:/file.dat", nn::fs::OpenMode_Read));
                NN_UTIL_SCOPE_EXIT
                {
                    nn::fs::CloseFile(file);
                };
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&m_FileSize, file));
            }
        }

        virtual ~RomFsTestExecutor() NN_NOEXCEPT NN_OVERRIDE
        {
            nn::fs::Unmount("rom");
        }

    public:
        virtual nn::Result Prepare(int index) NN_NOEXCEPT NN_OVERRIDE
        {
            m_Offsets[index] = m_FileSize / WorkerThreadCountMax * index;

            NN_RESULT_DO(
                nn::fs::OpenFile(
                    &m_Files[index],
                    "rom:/file.dat",
                    nn::fs::OpenMode_Read
                )
            );

            NN_RESULT_SUCCESS;
        }

        virtual void CleanUp(int index) NN_NOEXCEPT NN_OVERRIDE
        {
            nn::fs::CloseFile(m_Files[index]);
        }

        virtual nn::Result Execute(char* buffer, size_t bufferSize, int index) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(
                nn::fs::ReadFile(
                    m_Files[index],
                    m_Offsets[index],
                    buffer,
                    bufferSize
                )
            );

            m_Offsets[index] += bufferSize;
            const auto endOffset = m_FileSize / WorkerThreadCountMax * (index + 1);
            if( m_Offsets[index] >= endOffset )
            {
                m_Offsets[index] = m_FileSize / WorkerThreadCountMax * index;
            }

            NN_RESULT_SUCCESS;
        }

    private:
        int64_t m_FileSize;
        int64_t m_Offsets[WorkerThreadCountMax];
        nn::fs::FileHandle m_Files[WorkerThreadCountMax];
        decltype(nnt::fs::util::AllocateBuffer(0)) m_CacheBuffer;
    };

    class StatisticsArgument
    {
    public:
        void Initialize(
            nn::os::Semaphore* pSemaphore0,
            nn::os::Semaphore* pSemaphore1,
            nn::fs::PriorityRaw priority,
            ITestExecutor* pExecutor,
            size_t accessSize,
            int index
        ) NN_NOEXCEPT
        {
            m_pSemaphore0 = pSemaphore0;
            m_pSemaphore1 = pSemaphore1;
            m_Priority = priority;
            m_Count = 0;
            m_IsContinuous = true;
            m_Result = nn::ResultSuccess();
            m_pExecutor = pExecutor;
            m_AccessSize = accessSize;
            m_Index = index;
        }

        size_t GetAccessSize() NN_NOEXCEPT
        {
            return m_AccessSize;
        }

        nn::Result Prepare() NN_NOEXCEPT
        {
            NN_RESULT_DO(m_pExecutor->Prepare(m_Index));
            NN_RESULT_SUCCESS;
        }

        void CleanUp() NN_NOEXCEPT
        {
            m_pExecutor->CleanUp(m_Index);
        }

        nn::Result Execute(char* pBuffer, size_t size) NN_NOEXCEPT
        {
            NN_RESULT_DO(m_pExecutor->Execute(pBuffer, size, m_Index));
            NN_RESULT_SUCCESS;
        }

        void SetContinuous(bool isContinuous) NN_NOEXCEPT
        {
            m_IsContinuous = isContinuous;
        }

        bool IsContinuous() NN_NOEXCEPT
        {
            return m_IsContinuous;
        }

        void WaitManager() NN_NOEXCEPT
        {
            m_pSemaphore0->Release();
            m_pSemaphore1->Acquire();
        }

        nn::fs::PriorityRaw GetPriority() NN_NOEXCEPT
        {
            return m_Priority;
        }

        void CountUp() NN_NOEXCEPT
        {
            ++m_Count;
        }

        int64_t GetCount() NN_NOEXCEPT
        {
            return m_Count;
        }

        nn::Result GetResult() NN_NOEXCEPT
        {
            return m_Result;
        }

        void SetResult(nn::Result result) NN_NOEXCEPT
        {
            m_Result = result;
        }

    private:
        ITestExecutor* m_pExecutor;
        size_t m_AccessSize;
        int m_Index;
        nn::os::Semaphore* m_pSemaphore0;
        nn::os::Semaphore* m_pSemaphore1;
        nn::fs::PriorityRaw m_Priority;
        std::atomic<int64_t> m_Count;
        volatile bool m_IsContinuous;
        nn::Result m_Result;
    };

    void StatisticsThread(void* pArgument) NN_NOEXCEPT
    {
        auto* pStatisticsArgument = reinterpret_cast<StatisticsArgument*>(pArgument);
        nn::fs::SetPriorityRawOnCurrentThread(pStatisticsArgument->GetPriority());

        {
            auto result = pStatisticsArgument->Prepare();
            if( result.IsFailure() )
            {
                pStatisticsArgument->SetResult(result);
                pStatisticsArgument->WaitManager();
                return;
            }
        }
        NN_UTIL_SCOPE_EXIT
        {
            pStatisticsArgument->CleanUp();
        };

        auto workBuffer = nnt::fs::util::AllocateBuffer(pStatisticsArgument->GetAccessSize());

        pStatisticsArgument->WaitManager();

        while( pStatisticsArgument->IsContinuous() )
        {
            {
                nn::Result result = pStatisticsArgument->Execute(workBuffer.get(), pStatisticsArgument->GetAccessSize());
                if( result.IsFailure() )
                {
                    pStatisticsArgument->SetResult(result);
                    return;
                }
            }

            pStatisticsArgument->CountUp();
        }
    }

    class StatisticsTestCase
    {
    public:
        struct ThreadParameter
        {
            int coreNumber;
            int threadRelativePriority;
            nn::fs::PriorityRaw priority;
        };

    public:
        StatisticsTestCase(
            int threadCount,
            size_t accessSize,
            int mesurementTimeSecond,
            std::function<void(int64_t*) NN_NOEXCEPT> evaluateResults,
            std::array<ThreadParameter, WorkerThreadCountMax> parameters
        ) NN_NOEXCEPT
            : m_ThreadCount(threadCount),
            m_AccessSize(accessSize),
            m_MesurementTimeSecond(mesurementTimeSecond),
            m_EvaluateResults(evaluateResults),
            m_Parameters(parameters)
        {
        }

    public:
        int GetThreadCount() const NN_NOEXCEPT
        {
            return m_ThreadCount;
        }

        size_t GetAccessSize() const NN_NOEXCEPT
        {
            return m_AccessSize;
        }

        int GetMesurementTimeSecond() const NN_NOEXCEPT
        {
            return m_MesurementTimeSecond;
        }

        void EvaluateResults(int64_t* pResults) const NN_NOEXCEPT
        {
            m_EvaluateResults(pResults);
        }

        const ThreadParameter& GetParameter(int index) const NN_NOEXCEPT
        {
            return m_Parameters[index];
        }

    private:
        int m_ThreadCount;
        size_t m_AccessSize;
        int m_MesurementTimeSecond;
        std::function<void(int64_t*) NN_NOEXCEPT> m_EvaluateResults;
        std::array<ThreadParameter, WorkerThreadCountMax> m_Parameters;
    };

    void RunStatisticsTest(ITestExecutor* pExecutor, const StatisticsTestCase* pTestCase, size_t testCasesCount) NN_NOEXCEPT
    {
        auto getPriorityName = [](nn::fs::PriorityRaw priority) NN_NOEXCEPT
        {
            switch( priority )
            {
            case nn::fs::PriorityRaw_Background:
                return "bg";
            case nn::fs::PriorityRaw_Low:
                return "lo";
            case nn::fs::PriorityRaw_Normal:
                return "no";
            case nn::fs::PriorityRaw_Realtime:
                return "rt";
            default:
                return "??";
            }
        };

        for( int i = 0; i < testCasesCount; ++i )
        {
            auto testCase = pTestCase[i];

            ASSERT_LE(testCase.GetThreadCount(), WorkerThreadCountMax);

            nn::os::Semaphore semaphore0(0, testCase.GetThreadCount());
            nn::os::Semaphore semaphore1(0, testCase.GetThreadCount());

            StatisticsArgument arguments[WorkerThreadCountMax];
            bool isThreadInitialized[WorkerThreadCountMax] = { false, false, false, false };
            nn::os::ThreadType thread[WorkerThreadCountMax];
            static NN_OS_ALIGNAS_THREAD_STACK char threadBuffer[WorkerThreadCountMax * 16 * 1024];

            NN_UTIL_SCOPE_EXIT
            {
                for( int i = 0; i < WorkerThreadCountMax; ++i )
                {
                    if( isThreadInitialized[i] )
                    {
                        nn::os::DestroyThread(&thread[i]);
                    }
                }
            };
            for( int i = 0; i < WorkerThreadCountMax; ++i )
            {
                NNT_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(
                    &thread[i],
                    StatisticsThread,
                    &arguments[i],
                    threadBuffer + sizeof(threadBuffer) / WorkerThreadCountMax * i,
                    sizeof(threadBuffer) / WorkerThreadCountMax,
                    nn::os::DefaultThreadPriority + testCase.GetParameter(i).threadRelativePriority,
                    testCase.GetParameter(i).coreNumber
                ));
                isThreadInitialized[i] = true;
            }

            for( int i = 0; i < testCase.GetThreadCount(); ++i )
            {
                arguments[i].Initialize(
                    &semaphore0,
                    &semaphore1,
                    testCase.GetParameter(i).priority,
                    pExecutor,
                    testCase.GetAccessSize(),
                    i
                );
                nn::os::StartThread(&thread[i]);
            }
            for( int i = 0; i < testCase.GetThreadCount(); ++i )
            {
                semaphore0.Acquire();
            }

            semaphore1.Release(testCase.GetThreadCount());

            if( testCase.GetMesurementTimeSecond() > 0 )
            {
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(testCase.GetMesurementTimeSecond()));
            }
            else
            {
                nn::Bit64 processId;
                NNT_ASSERT_RESULT_SUCCESS(nn::svc::GetProcessId(&processId, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS));

                for( ;; )
                {
                    NN_LOG("background process %lld is alive\n", processId);
                    nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));
                }
            }

            int64_t results[WorkerThreadCountMax];
            for( int i = 0; i < testCase.GetThreadCount(); ++i )
            {
                results[i] = arguments[i].GetCount();
            }

            for( int i = 0; i < testCase.GetThreadCount(); ++i )
            {
                arguments[i].SetContinuous(false);
            }

            NN_LOG("try %3d s, access size %4d KiB: ", testCase.GetMesurementTimeSecond(), testCase.GetAccessSize() / 1024);
            for( int i = 0; i < testCase.GetThreadCount(); ++i )
            {
                nn::os::WaitThread(&thread[i]);
                NN_LOG("(%s, %lld) ", getPriorityName(testCase.GetParameter(i).priority), results[i]);
            }
            NN_LOG("\n");

            for( int i = 0; i < testCase.GetThreadCount(); ++i )
            {
                NNT_ASSERT_RESULT_SUCCESS(arguments[i].GetResult());
            }

            NNT_FS_ASSERT_NO_FATAL_FAILURE(testCase.EvaluateResults(results));
        }
    } // NOLINT(impl/function_size)
}

#if defined(NNT_FS_APPLICATION) || defined(NNT_FS_SYSTEM_PROCESS)
namespace
{
    const nn::fs::Priority ForPublicValues[]
    {
        nn::fs::Priority_Realtime,
        nn::fs::Priority_Normal,
        nn::fs::Priority_Low,
    };
    const nn::fs::PriorityRaw ForPrivateValues[]
    {
        nn::fs::PriorityRaw_Realtime,
        nn::fs::PriorityRaw_Normal,
        nn::fs::PriorityRaw_Low,
        nn::fs::PriorityRaw_Background,
    };
}

TEST(AccessPriority, CanUseAllValues)
{
    size_t sizeRomCache;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&sizeRomCache));
    auto cacheBuffer = nnt::fs::util::AllocateBuffer(sizeRomCache);

    auto accessFile = [&]() NN_NOEXCEPT
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountRom("rom", cacheBuffer.get(), sizeRomCache));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount("rom");
        };

        nn::fs::FileHandle file;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&file, "rom:/file.dat", nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(file);
        };

        char buffer[64];
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(file, 0, buffer, sizeof(buffer)));
    };

    for( auto value : ForPublicValues )
    {
        nn::fs::SetPriorityOnCurrentThread(value);
        NNT_FS_ASSERT_NO_FATAL_FAILURE(accessFile());
    }

    for( auto value : ForPrivateValues )
    {
        nn::fs::SetPriorityRawOnCurrentThread(value);
        NNT_FS_ASSERT_NO_FATAL_FAILURE(accessFile());
    }
}

void DefaultValueThread(void* pArgument) NN_NOEXCEPT
{
    auto pDefaultValueArgument = reinterpret_cast<DefaultValueArgument*>(pArgument);
    GetDefaultValue(pDefaultValueArgument);
}

TEST(AccessPriority, DefaultValue)
{
    EXPECT_EQ(nn::fs::Priority_Normal, g_MainThreadDefaultValue.priority);
    EXPECT_EQ(nn::fs::PriorityRaw_Normal, g_MainThreadDefaultValue.priorityRaw);

    nn::os::ThreadType thread;
    static NN_OS_ALIGNAS_THREAD_STACK char threadBuffer[16 * 1024];

    DefaultValueArgument argument = {};
    NNT_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(
        &thread,
        DefaultValueThread,
        &argument,
        threadBuffer,
        sizeof(threadBuffer),
        nn::os::DefaultThreadPriority
    ));
    NN_UTIL_SCOPE_EXIT
    {
        nn::os::DestroyThread(&thread);
    };

    nn::os::StartThread(&thread);
    nn::os::WaitThread(&thread);

    EXPECT_EQ(nn::fs::Priority_Normal, argument.priority);
    EXPECT_EQ(nn::fs::PriorityRaw_Normal, argument.priorityRaw);
}

void SetAndGetThread(void* pArgument) NN_NOEXCEPT
{
    bool* pResult = reinterpret_cast<bool*>(pArgument);
    *pResult = false;

    for( auto value : ForPublicValues )
    {
        nn::fs::SetPriorityOnCurrentThread(value);

        nn::os::YieldThread();

        nn::fs::Priority priorityGotten = nn::fs::GetPriorityOnCurrentThread();

        if( priorityGotten != value )
        {
            NN_LOG("unexpected value got.\n");
            return;
        }
    }

    *pResult = true;
}

TEST(AccessPriority, SetAndGet)
{
    nn::os::ThreadType thread;
    static NN_OS_ALIGNAS_THREAD_STACK char threadBuffer[16 * 1024];

    bool argument;
    NNT_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(
        &thread,
        SetAndGetThread,
        &argument,
        threadBuffer,
        sizeof(threadBuffer),
        nn::os::DefaultThreadPriority
    ));
    NN_UTIL_SCOPE_EXIT
    {
        nn::os::DestroyThread(&thread);
    };

    nn::os::StartThread(&thread);
    bool mainThreadArgument = true;
    for( auto value : ForPrivateValues )
    {
        nn::fs::SetPriorityRawOnCurrentThread(value);

        nn::os::YieldThread();

        nn::fs::PriorityRaw priorityGotten = nn::fs::GetPriorityRawOnCurrentThread();

        if( priorityGotten != value )
        {
            NN_LOG("unexpected value got.\n");
            mainThreadArgument = false;
            break;
        }
    }
    nn::os::WaitThread(&thread);

    ASSERT_TRUE(argument && mainThreadArgument);
}

void StatisticsTest(ITestExecutor* pExecutor) NN_NOEXCEPT
{
    StatisticsTestCase testCases[] =
    {
        {
            2,
            2 * 1024 * 1024,
            10,
            [](int64_t* pResult) NN_NOEXCEPT
            {
                EXPECT_GT(pResult[0], pResult[1]);
            },
            {{
                { 1, 1, nn::fs::PriorityRaw_Realtime },
                { 2, 1, nn::fs::PriorityRaw_Normal }
            }}
        },
        {
            2,
            2 * 1024 * 1024,
            5,
            [](int64_t* pResult) NN_NOEXCEPT
            {
                EXPECT_GT(pResult[0], pResult[1]);
            },
            {{
                { 1, 1, nn::fs::PriorityRaw_Realtime },
                { 2, 1, nn::fs::PriorityRaw_Low }
            }}
        },
        {
            2,
            2 * 1024 * 1024,
            5,
            [](int64_t* pResult) NN_NOEXCEPT
            {
                EXPECT_GT(pResult[0], pResult[1]);
            },
            {{
                { 1, 1, nn::fs::PriorityRaw_Realtime },
                { 2, 1, nn::fs::PriorityRaw_Background }
            }}
        },
        {
            2,
            2 * 1024 * 1024,
            5,
            [](int64_t* pResult) NN_NOEXCEPT
            {
                EXPECT_GT(pResult[0], pResult[1]);
            },
            {{
                { 1, 1, nn::fs::PriorityRaw_Normal },
                { 2, 1, nn::fs::PriorityRaw_Low }
            }}
        },
        {
            2,
            2 * 1024 * 1024,
            5,
            [](int64_t* pResult) NN_NOEXCEPT
            {
                EXPECT_GT(pResult[0], pResult[1]);
            },
            {{
                { 1, 1, nn::fs::PriorityRaw_Normal },
                { 2, 1, nn::fs::PriorityRaw_Background }
            }}
        },
        {
            2,
            2 * 1024 * 1024,
            20,
            [](int64_t* pResult) NN_NOEXCEPT
            {
                EXPECT_GT(pResult[0], pResult[1]);
            },
            {{
                { 1, 1, nn::fs::PriorityRaw_Low },
                { 2, 1, nn::fs::PriorityRaw_Background }
            }}
        },
        {
            4,
            2 * 1024 * 1024,
            60,
            [](int64_t* pResult) NN_NOEXCEPT
            {
#if defined(NNT_FS_SYSTEM_PROCESS)
                // クライアントセッション数 2 のため、カウント値の大小は保証されない
#else
                // Realtime は必ず他より大きいことが保証される
                for( int i = 1; i <= 3; i++ )
                {
                    EXPECT_GT(pResult[0], pResult[i]);
                }
#endif
            },
            {{
                { 1, 1, nn::fs::PriorityRaw_Realtime },
                { 1, 1, nn::fs::PriorityRaw_Normal },
                { 1, 1, nn::fs::PriorityRaw_Low },
                { 1, 1, nn::fs::PriorityRaw_Background }
            }}
        },
        {
            4,
            1024,
            30,
            [](int64_t* pResult) NN_NOEXCEPT
            {
#if defined(NNT_FS_SYSTEM_PROCESS)
                // クライアントセッション数 2 のため、カウント値の大小は保証されない
#else
                // Realtime は必ず他より大きいことが保証される
                for( int i = 1; i <= 3; i++ )
                {
                    EXPECT_GT(pResult[0], pResult[i]);
                }
#endif
            },
            {{
                { 1, 1, nn::fs::PriorityRaw_Realtime },
                { 1, 1, nn::fs::PriorityRaw_Low },
                { 2, 1, nn::fs::PriorityRaw_Low },
                { 2, 1, nn::fs::PriorityRaw_Low }
            }}
        },
        {
            8,
            1024,
            30,
            [](int64_t* pResult) NN_NOEXCEPT
            {
#if defined(NNT_FS_SYSTEM_PROCESS)
                // クライアントセッション数 2 のため、カウント値の大小は保証されない
#else
                // Realtime は必ず他より大きいことが保証される
                for( int i = 1; i <= 7; i++ )
                {
                    EXPECT_GT(pResult[0], pResult[i]);
                }
#endif
            },
            {{
                { 1, 1, nn::fs::PriorityRaw_Realtime },
                { 1, 1, nn::fs::PriorityRaw_Low },
                { 1, 1, nn::fs::PriorityRaw_Low },
                { 1, 1, nn::fs::PriorityRaw_Low },
                { 2, 1, nn::fs::PriorityRaw_Low },
                { 2, 1, nn::fs::PriorityRaw_Low },
                { 2, 1, nn::fs::PriorityRaw_Low },
                { 2, 1, nn::fs::PriorityRaw_Low }
            }}
        },
        {
            16,
            1024,
            30,
            [](int64_t* pResult) NN_NOEXCEPT
            {
#if defined(NNT_FS_SYSTEM_PROCESS)
                // クライアントセッション数 2 のため、カウント値の大小は保証されない
#else
                // Realtime は必ず他より大きいことが保証される
                for( int i = 1; i <= 15; i++ )
                {
                    EXPECT_GT(pResult[0], pResult[i]);
                }
#endif
            },
            {{
                { 1, 1, nn::fs::PriorityRaw_Realtime },
                { 1, 1, nn::fs::PriorityRaw_Low },
                { 1, 1, nn::fs::PriorityRaw_Low },
                { 1, 1, nn::fs::PriorityRaw_Low },
                { 1, 1, nn::fs::PriorityRaw_Low },
                { 1, 1, nn::fs::PriorityRaw_Low },
                { 1, 1, nn::fs::PriorityRaw_Low },
                { 1, 1, nn::fs::PriorityRaw_Low },
                { 2, 1, nn::fs::PriorityRaw_Low },
                { 2, 1, nn::fs::PriorityRaw_Low },
                { 2, 1, nn::fs::PriorityRaw_Low },
                { 2, 1, nn::fs::PriorityRaw_Low },
                { 2, 1, nn::fs::PriorityRaw_Low },
                { 2, 1, nn::fs::PriorityRaw_Low },
                { 2, 1, nn::fs::PriorityRaw_Low },
                { 2, 1, nn::fs::PriorityRaw_Low }
            }}
        }
    };
    static const auto TestCaseCount = sizeof(testCases) / sizeof(testCases[0]);

    RunStatisticsTest(pExecutor, testCases, TestCaseCount);
} // NOLINT(impl/function_size)

TEST(AccessPriority, StatisticsRomReadHeavy)
{
    RomFsTestExecutor executor;
    StatisticsTest(&executor);
}

class BisFsTestExecutor : public ITestExecutor
{
public:
    explicit BisFsTestExecutor(int64_t fileSize) NN_NOEXCEPT
        : m_FileSize(fileSize)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountBis("bis", nn::fs::BisPartitionId::User));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nnt::fs::util::DeleteFileOrDirectoryIfExists("bis:/StatisticsTest"));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::CreateDirectory("bis:/StatisticsTest"));
    }

    virtual ~BisFsTestExecutor() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::fs::Unmount("bis");
    }

public:
    virtual nn::Result Prepare(int index) NN_NOEXCEPT NN_OVERRIDE
    {
        m_Offsets[index] = 0;

        char path[256];
        nn::util::SNPrintf(path, sizeof(path), "bis:/StatisticsTest/file%d.dat", index);

        NN_RESULT_DO(nnt::fs::util::DeleteFileOrDirectoryIfExists(path));
        NN_RESULT_DO(nn::fs::CreateFile(path, m_FileSize));

        NN_RESULT_DO(
            nn::fs::OpenFile(
                &m_Files[index],
                path,
                nn::fs::OpenMode_Write
            )
        );

        NN_RESULT_SUCCESS;
    }

    virtual void CleanUp(int index) NN_NOEXCEPT NN_OVERRIDE
    {
        nn::fs::CloseFile(m_Files[index]);
    }

    virtual nn::Result Execute(char* buffer, size_t bufferSize, int index) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_DO(
            nn::fs::WriteFile(
                m_Files[index],
                m_Offsets[index],
                buffer,
                bufferSize,
                nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)
            )
        );

        m_Offsets[index] += bufferSize;
        if( m_Offsets[index] >= m_FileSize )
        {
            m_Offsets[index] = 0;
        }

        NN_RESULT_SUCCESS;
    }

private:
    int64_t m_FileSize;
    int64_t m_Offsets[WorkerThreadCountMax];
    nn::fs::FileHandle m_Files[WorkerThreadCountMax];
};

TEST(AccessPriority, StatisticsBisWriteHeavy)
{
    BisFsTestExecutor executor(32 * 1024 * 1024);
    StatisticsTest(&executor);
}

class SaveDataExistenceCheckTestExecutor : public ITestExecutor
{
public:
    explicit SaveDataExistenceCheckTestExecutor() NN_NOEXCEPT
        : m_Mutex(true)
    {
        int userCount = 0;
        auto result = nn::account::ListAllUsers(&userCount, m_Uids, nn::account::UserCountMax);
        NN_ABORT_UNLESS(result.IsSuccess() && userCount == nn::account::UserCountMax);
    }

    virtual ~SaveDataExistenceCheckTestExecutor() NN_NOEXCEPT NN_OVERRIDE {}

public:
    virtual nn::Result Prepare(int index) NN_NOEXCEPT NN_OVERRIDE
    {
        if( index >= OpenCountLimit )
        {
            NN_RESULT_SUCCESS;
        }

        std::lock_guard<nn::os::Mutex> locker(m_Mutex);

        CleanUp(index);

        auto id = GetApplicationId(index);
        NN_RESULT_DO(nn::fs::CreateSaveData(
            id,
            GetFsUid(index),
            id.value,
            64 * 1024,
            64 * 1024,
            0
        ));

        NN_RESULT_SUCCESS;
    }

    virtual void CleanUp(int index) NN_NOEXCEPT NN_OVERRIDE
    {
        if( index >= OpenCountLimit )
        {
            return;
        }

        std::lock_guard<nn::os::Mutex> locker(m_Mutex);

        nn::util::optional<nn::fs::SaveDataInfo> info;
        nnt::fs::util::FindSaveData(&info, nn::fs::SaveDataSpaceId::User, [=](const nn::fs::SaveDataInfo& info_) NN_NOEXCEPT
            {
                return info_.applicationId == GetApplicationId(index)
                    && info_.saveDataUserId == GetFsUid(index);
            });
        if( info )
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::DeleteSaveData(info->saveDataId));
        }
    }

    virtual nn::Result Execute(char* buffer, size_t bufferSize, int index) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(buffer);
        NN_UNUSED(bufferSize);

        if( index >= OpenCountLimit )
        {
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            NN_RESULT_SUCCESS;
        }

        NN_RESULT_THROW_UNLESS(
            nn::fs::IsSaveDataExisting(GetApplicationId(index), GetFsUid(index)),
            nn::fs::ResultNotFound()
        );
        NN_RESULT_SUCCESS;
    }

private:
    static const int OpenCountLimit = 10; // fsp 側の仕様
    static const nn::ncm::ApplicationId NcmApplicationIds[2];

private:
    const nn::ncm::ApplicationId GetApplicationId(int index) const NN_NOEXCEPT
    {
        return NcmApplicationIds[index / nn::account::UserCountMax];
    }

    const nn::account::Uid GetUid(int index) const NN_NOEXCEPT
    {
        return m_Uids[index % nn::account::UserCountMax];
    }

    nn::fs::UserId GetFsUid(int index) const NN_NOEXCEPT
    {
        return nn::fs::ConvertAccountUidToFsUserId(GetUid(index));
    }

private:
    nn::os::Mutex m_Mutex;
    nn::account::Uid m_Uids[nn::account::UserCountMax];
};

const nn::ncm::ApplicationId SaveDataExistenceCheckTestExecutor::NcmApplicationIds[] = { { 0x0005000C10000000 }, { 0x0005000C10000001 } };

TEST(AccessPriority, StatisticsSaveDataExistenceCheckHeavy)
{
    SaveDataExistenceCheckTestExecutor executor;
    StatisticsTest(&executor);
}

#if defined(NNT_FS_SYSTEM_PROCESS)
TEST(AccessPriority, BrokenTlsValueNotKillFsService)
{
    {
        size_t sizeRomCache;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&sizeRomCache));
    }

    auto value = nn::sf::GetFsInlineContext();
    nn::sf::SetFsInlineContext(0xFF);
    NN_UTIL_SCOPE_EXIT
    {
        nn::sf::SetFsInlineContext(value);
    };

    // fs プロセス側では適当な優先度に置き換えられて実行される
    {
        size_t sizeRomCache;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&sizeRomCache));
    }
}
#endif // defined(NNT_FS_SYSTEM_PROCESS)

// 色々な組み合わせで defer させたり、
// クライアントのセッションを食い潰したりしても問題なく動作することを確認する
TEST(AccessPriority, NotHungUpOnMultiThreadAccess1Heavy)
{
    RomFsTestExecutor executor;
    StatisticsTestCase testCases[] =
    {
        {
            4,
            4 * 1024 * 1024,
            10,
            [](int64_t* pResult) NN_NOEXCEPT
            {
                for( int i = 0; i < 4; ++i )
                {
                    EXPECT_LT(0, pResult[i]);
                }
            },
            {{
                { 1, 1, nn::fs::PriorityRaw_Realtime },
                { 2, 1, nn::fs::PriorityRaw_Realtime },
                { 3, 1, nn::fs::PriorityRaw_Realtime },
                { 1, 1, nn::fs::PriorityRaw_Realtime }
            }}
        },
        {
            4,
            4 * 1024 * 1024,
            10,
            [](int64_t* pResult) NN_NOEXCEPT
            {
                for( int i = 0; i < 4; ++i )
                {
                    EXPECT_LT(0, pResult[i]);
                }
            },
            {{
                { 1, 1, nn::fs::PriorityRaw_Normal },
                { 2, 1, nn::fs::PriorityRaw_Normal },
                { 3, 1, nn::fs::PriorityRaw_Normal },
                { 1, 1, nn::fs::PriorityRaw_Normal }
            }}
        },
#if defined(NNT_FS_APPLICATION)
        {
            5,
            4 * 1024 * 1024,
            30,
            [](int64_t* pResult) NN_NOEXCEPT
            {
                for( int i = 0; i < 5; ++i )
                {
                    EXPECT_LT(0, pResult[i]);
                }
            },
            {{
                // 実際にはありえない状況かもしれないが、状況を作るために Realtime のスレッド優先度を落とす
                { 1, 2, nn::fs::PriorityRaw_Realtime },
                { 2, 2, nn::fs::PriorityRaw_Realtime },
                { 1, 1, nn::fs::PriorityRaw_Normal },
                { 2, 1, nn::fs::PriorityRaw_Normal },
                { 3, 1, nn::fs::PriorityRaw_Normal }
            }}
        },
#endif // defined(NNT_FS_APPLICATION)
    };
    static const auto TestCaseCount = sizeof(testCases) / sizeof(testCases[0]);

    RunStatisticsTest(&executor, testCases, TestCaseCount);
}

TEST(AccessPriority, NotHungUpOnMultiThreadAccess2Heavy)
{
    RomFsTestExecutor executor;
    StatisticsTestCase testCases[] =
    {
        {
            4,
            4 * 1024 * 1024,
            30,
            [](int64_t* pResult) NN_NOEXCEPT
            {
                for( int i = 0; i < 4; ++i )
                {
                    EXPECT_LT(0, pResult[i]);
                }
            },
            {{
                { 1, 1, nn::fs::PriorityRaw_Background },
                { 2, 1, nn::fs::PriorityRaw_Background },
                { 3, 1, nn::fs::PriorityRaw_Background },
                { 1, 1, nn::fs::PriorityRaw_Background }
            }}
        }
    };
    static const auto TestCaseCount = sizeof(testCases) / sizeof(testCases[0]);

    RunStatisticsTest(&executor, testCases, TestCaseCount);
}

// Unit テストではあるが、数が少ないのでこちらに記述
TEST(AccessPriority, TlsValue)
{
    {
        nn::Bit8 priority;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::detail::ConvertFsPriorityToTlsIoPriority(&priority, nn::fs::PriorityRaw_Normal));
        ASSERT_EQ(0, priority);
    }

    {
        nn::fs::PriorityRaw priority;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::detail::ConvertTlsIoPriorityToFsPriority(&priority, 0));
        ASSERT_EQ(nn::fs::PriorityRaw_Normal, priority);
    }
}

#endif // defined(NNT_FS_APPLICATION) || defined(NNT_FS_SYSTEM_PROCESS)

#if defined(NNT_FS_BACKGROUND_SYSTEM_PROCESS)
void RunBackgroundAccessTest(nn::fs::PriorityRaw priority) NN_NOEXCEPT
{
    RomFsTestExecutor executor;
    StatisticsTestCase testCases[] =
    {
        {
            2,
            1 * 1024 * 1024,
            -1,
            [](int64_t* pResult) NN_NOEXCEPT
            {
                for( int i = 0; i < 4; ++i )
                {
                    EXPECT_LT(0, pResult[i]);
                }
            },
            {{
                { 3, 3, priority },
                { 3, 3, priority }
            }}
        },
    };
    static const auto TestCaseCount = sizeof(testCases) / sizeof(testCases[0]);

    // nn::fssystem::ScopedThreadPriorityChanger changePriority(-2, nn::fssystem::ScopedThreadPriorityChanger::Mode::Relative);

    RunStatisticsTest(&executor, testCases, TestCaseCount);
}

#if defined(NNT_FS_BACKGROUND_SYSTEM_PROCESS_REALTIME)
TEST(AccessPriority, BackgroundAccessRomFsRealtime)
{
    RunBackgroundAccessTest(nn::fs::PriorityRaw_Realtime);
}
#endif

#if defined(NNT_FS_BACKGROUND_SYSTEM_PROCESS_NORMAL)
TEST(AccessPriority, BackgroundAccessRomFsNormal)
{
    RunBackgroundAccessTest(nn::fs::PriorityRaw_Normal);
}
#endif

#if defined(NNT_FS_BACKGROUND_SYSTEM_PROCESS_BACKGROUND)
TEST(AccessPriority, BackgroundAccessRomFsBackground)
{
    RunBackgroundAccessTest(nn::fs::PriorityRaw_Background);
}
#endif

#endif // NNT_FS_BACKGROUND_SYSTEM_PROCESS

TEST(AccessPriority, SessionCount)
{
    //・DEVMENU → HOMEMENU の増分 : nfc が使用するようになる (+1)
    //・ライブラリアプレット (+2)
    //・ソフトウェアキーボード(+2)
    //・アプリ (+3)
    static const int SessionCountMax = 8;
    nn::fs::detail::DisableSessionForRealtimeOnlyAndChangeSessionCount(SessionCountMax);
    NN_UTIL_SCOPE_EXIT
    {
        nnt::fs::util::RevertSessionSettingToDefault();
    };

    size_t sizeRomCache;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&sizeRomCache));
}

extern "C" void nninitStartup()
{
#if defined(NNT_FS_BACKGROUND_SYSTEM_PROCESS)
    static char s_Buffer[16 * 1024 * 1024];
#else
    static char s_Buffer[128 * 1024 * 1024];
#endif
    nn::init::InitializeAllocator(s_Buffer, sizeof(s_Buffer));
}

extern "C" void nnMain()
{
#if !defined(NNT_FS_APPLICATION)
    nn::fs::InitializeWithMultiSessionForSystem();
#endif // !defined(NNT_FS_APPLICATION)

    GetDefaultValue(&g_MainThreadDefaultValue);

    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

#if !defined(NNT_FS_BACKGROUND_SYSTEM_PROCESS)
    nn::account::InitializeForSystemService();
#endif

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();
    nn::fs::SetEnabledAutoAbort(false);

    auto testResult = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(testResult);
}
