﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <cstring>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForApplications.h>
#include <nn/fs.h>
#include <nn/fs/fs_AccessFailureDetection.h>
#include <nn/fs/fs_AccessLogPrivate.h>
#include <nn/fs/fs_ApplicationSaveDataManagement.h>
#include <nn/fs/fs_CacheStorageWithIndex.h>
#include <nn/fs/fs_CloudBackupWorkStorage.h>
#include <nn/fs/fs_ContentStorage.h>
#include <nn/fs/fs_Debug.h>
#include <nn/fs/fs_DebugPrivate.h>
#include <nn/fs/fs_DeviceSaveData.h>
#include <nn/fs/fs_IEventNotifier.h>
#include <nn/fs/fs_GameCard.h>
#include <nn/fs/fs_GameCardForInspection.h>
#include <nn/fs/fs_ImageDirectory.h>
#include <nn/fs/fs_MmcPrivate.h>
#include <nn/fs/fs_RegisteredUpdatePartition.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_RightsId.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataManagementPrivate.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SaveDataTransfer.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/fs/fs_SpeedEmulation.h>
#include <nn/fs/fs_SystemData.h>
#include <nn/fs/fs_SystemDataUpdateEvent.h>
#include <nn/fs/fs_SystemDataUpdateEventPrivate.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_SystemBcatSaveData.h>
#include <nn/fs/fs_SaveDataTransferVersion2.h>
#include <nn/fs/fs_Utility.h>
#include <nn/fssrv/sf/fssrv_IFileSystemProxy.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/ns/ns_ApplicationControlDataApi.h>
#include <nn/oe.h>
#include <nn/os/os_Argument.h>
#include <nn/sf/sf_Types.h>
#include <nn/util/util_FormatString.h>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>

using namespace nn::fs;
using namespace nnt::fs::util;

namespace {
    const size_t SaveDataSize = 1024 * 1024;
    const size_t SaveDataJournalSize = 1024 * 1024;
    const nn::Bit64 TestApplicationId = 0x0100000000010000;
    const nn::fs::SystemSaveDataId TestSystemSaveDataId = 0x800000000000000e;
    const nn::fs::SystemSaveDataId TestExistSystemSaveDataId = 0x8000000000004000;
    const nn::fs::SystemBcatSaveDataId TestSystemBcatSaveDataId = 0x8000000000004010;
    const nn::fs::SystemBcatSaveDataId TestExistSystemBcatSaveDataId = 0x8000000000004011;
    nn::fs::UserId g_TestUserId = nn::fs::InvalidUserId;
}

//! 権限: CreateSaveData（予め実行しておくこと）
TEST(SaveDataAccessControlTest, PrepareTestSaveData)
{
    nn::ncm::ApplicationId appId = { TestApplicationId };
    nn::fs::UserId userId = {{ 0x0, 0x1 }};
    NNT_EXPECT_RESULT_SUCCESS(CreateSaveData(appId, userId, TestApplicationId + 1, SaveDataSize, SaveDataJournalSize, 0));
    nn::ncm::ApplicationId pdAppId = { TestApplicationId + 0x10 };
    NNT_EXPECT_RESULT_SUCCESS(CreateSaveData(pdAppId, userId, TestApplicationId + 0x10, SaveDataSize, SaveDataJournalSize, 0));

    NNT_EXPECT_RESULT_SUCCESS(CreateDeviceSaveData(appId, TestApplicationId, SaveDataSize, SaveDataJournalSize, 0));
}

//! 権限: SaveDataBackUp, SystemSaveDataManagement（予め実行しておくこと）
TEST(SaveDataAccessControlTest, PrepareTestSystemSaveDataOnSd)
{
    NNT_EXPECT_RESULT_SUCCESS(CreateSystemSaveData(nn::fs::SaveDataSpaceId::SdSystem, TestExistSystemSaveDataId, TestApplicationId + 2, SaveDataSize, SaveDataJournalSize, 0));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteSaveData(TestExistSystemSaveDataId));
}

//! 権限: SaveDataBackUp, SystemSaveDataManagement（予め実行しておくこと）
TEST(SaveDataAccessControlTest, PrepareTestSystemSaveData)
{
    NNT_EXPECT_RESULT_SUCCESS(CreateSystemSaveData(TestExistSystemSaveDataId, TestApplicationId + 2, SaveDataSize, SaveDataJournalSize, 0));
    NNT_EXPECT_RESULT_SUCCESS(CreateSystemSaveData(TestExistSystemSaveDataId, g_TestUserId, TestApplicationId + 2, SaveDataSize, SaveDataJournalSize, 0));
}

//! 権限: SaveDataBackUp, SystemSaveDataManagement（予め実行しておくこと）
TEST(SaveDataAccessControlTest, PrepareTestSystemBcatSaveData)
{
    NNT_EXPECT_RESULT_SUCCESS(CreateSystemBcatSaveData(TestExistSystemBcatSaveDataId, SaveDataSize, 0));
}

//! 権限: SaveDataBackUp, SystemSaveDataManagement
TEST(SaveDataAccessControlTest, DeleteAllTestSaveData)
{
    Vector<nn::fs::SaveDataInfo> infos;
    FindSaveData(&infos, SaveDataSpaceId::User, [&](const SaveDataInfo& i) NN_NOEXCEPT
    {
        return i.applicationId.value == TestApplicationId || i.applicationId.value == TestApplicationId + 1 || i.applicationId.value == TestApplicationId + 2 || i.applicationId.value == TestApplicationId + 0x10;
    });
    for (auto& info : infos)
    {
        NNT_EXPECT_RESULT_SUCCESS(DeleteSaveData(info.saveDataId));
    }

    FindSaveData(&infos, SaveDataSpaceId::System, [&](const SaveDataInfo& i) NN_NOEXCEPT
    {
        return (i.systemSaveDataId == TestExistSystemSaveDataId) || (i.systemSaveDataId == TestExistSystemBcatSaveDataId);
    });

    for (auto& info : infos)
    {
        NNT_EXPECT_RESULT_SUCCESS(DeleteSaveData(info.saveDataId));
    }
}

//! 権限: Debug のみ (= アプリ権限)
TEST(SaveDataAccessControlTest, Application)
{
    // 自身のセーブデータを開ける、は FsSaveData サンプルで確認済

    // 他人のセーブデータを開けない
    nn::ncm::ApplicationId appId = { TestApplicationId };
    nn::fs::UserId userId = {{ 0x0, 0x1 }};
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, (MountSaveData("save", appId, userId)));

    // 本体セーブデータを開けない
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, (MountDeviceSaveData("save")));

    // アプリ権限で既存の（他人の）システムセーブデータを開けない
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, MountSystemSaveData("save", TestExistSystemSaveDataId));
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, MountSystemSaveData("save", TestExistSystemSaveDataId, g_TestUserId));

    // アプリ権限で（他人の）システムセーブデータを作成できない
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, CreateSystemSaveData(TestSystemSaveDataId, SaveDataSize, SaveDataJournalSize, 0));
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, CreateSystemSaveData(TestSystemSaveDataId, TestApplicationId + 1, SaveDataSize, SaveDataJournalSize, 0));
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, CreateSystemSaveData(TestSystemSaveDataId, g_TestUserId, TestApplicationId + 1, SaveDataSize, SaveDataJournalSize, 0));

    // アプリ権限で既存の（他人の）システムセーブデータを消せない
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, DeleteSaveData(0x80000000000000a0));
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, DeleteSystemSaveData(SaveDataSpaceId::System, TestExistSystemSaveDataId, g_TestUserId));
    {
        nn::fs::SaveDataId saveDataIds[] = { 0x80000000000000a0, 0x80000000000000a1 };
        NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, RegisterSaveDataAtomicDeletion(saveDataIds, 2));
    }

    // アプリ権限で既存のシステムBCATセーブデータを開けない
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, MountSystemBcatSaveData("save", TestExistSystemSaveDataId));

    // アプリ権限でシステムBCATセーブデータを作成できない
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, CreateSystemBcatSaveData(TestSystemBcatSaveDataId, SaveDataSize, 0));

    // アプリ権限で既存のシステムBCATセーブデータを消せない
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, DeleteSaveData(TestExistSystemSaveDataId));

    // アプリ権限でセーブデータの列挙ができない
    {
        std::unique_ptr<SaveDataIterator> pIterator;
        nn::fs::SaveDataSpaceId spaceIds[] =
        {
            SaveDataSpaceId::System,
            SaveDataSpaceId::User,
            SaveDataSpaceId::SdSystem,
            SaveDataSpaceId::Temporary,
            SaveDataSpaceId::SdUser,
            SaveDataSpaceId::ProperSystem,
            SaveDataSpaceId::SafeMode,
        };

        for (auto& spaceId : spaceIds)
        {
            NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, OpenSaveDataIterator(&pIterator, spaceId));

            using nn::util::nullopt;
            NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, OpenSaveDataIterator(&pIterator, spaceId, nn::fs::SaveDataFilter::Make(nullopt, nullopt, nullopt, nullopt, nullopt, nn::fs::SaveDataEnumerateOption_AllRank)));
        }
    }

    // アプリ権限でセーブデータの拡張データが参照及び設定できない
    {
        nn::Bit64 ownerId = 0;
        NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, GetSaveDataOwnerId(&ownerId, 0x80000000000000a0));
        nn::time::PosixTime timeStamp = { 1 };
        NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, SetSaveDataTimeStamp(SaveDataSpaceId::System, 0x80000000000000a0, timeStamp));
        uint32_t flags = 0;
        NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, SetSaveDataFlags(0x80000000000000a0, SaveDataSpaceId::System, flags));
        SaveDataCommitId commitId = 1;
        NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, SetSaveDataCommitId(SaveDataSpaceId::System, 0x80000000000000a0, commitId));
        NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, GetSaveDataCommitId(&commitId, SaveDataSpaceId::System, 0x80000000000000a0));

        SaveDataExtraData extraData;
        NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, nn::fs::detail::ReadSaveDataFileSystemExtraData(&extraData, SaveDataSpaceId::System, 0x80000000000000a0));
        memset(&extraData, 0xFF, sizeof(extraData));
        NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, nn::fs::detail::WriteSaveDataFileSystemExtraData(SaveDataSpaceId::System, 0x80000000000000a0, extraData));

        SaveDataExtraData extraDataMask;
        memset(&extraDataMask, 0, sizeof(extraDataMask));
        extraDataMask.attribute.programId.value = 0x1;
        NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, nn::fs::detail::WriteSaveDataFileSystemExtraData(SaveDataSpaceId::System, 0x80000000000000a0, extraData, extraDataMask));
        memset(&extraDataMask, 0, sizeof(extraDataMask));
        extraDataMask.reserved[sizeof(extraDataMask.reserved) - 1] = 0x1;
        NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, nn::fs::detail::WriteSaveDataFileSystemExtraData(SaveDataSpaceId::System, 0x80000000000000a0, extraData, extraDataMask));
    }

    // アプリ権限でセーブデータを拡張できない
    {
        NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, ExtendSaveData(
            SaveDataSpaceId::User,
            0x80000000000000a0,
            32 * 1024 * 1024,
            32 * 1024 * 1024));
        NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, ExtendSaveData(
            SaveDataSpaceId::System,
            0x80000000000000a0,
            32 * 1024 * 1024,
            32 * 1024 * 1024));
    }

    // アプリ権限でアップデートパーティションの登録・マウントができない
    {
        NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, RegisterUpdatePartition());
        NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, MountRegisteredUpdatePartition("upp"));
    }
}

//! 権限: Debug + SaveDataOwnerId (TestApplicationId + 1) リスト
TEST(SaveDataAccessControlTest, OthersSaveData)
{
    nn::ncm::ApplicationId appId = { TestApplicationId };
    nn::fs::UserId userId = {{ 0x0, 0x1 }};

    // 他人のセーブデータが存在するかの確認
    ASSERT_TRUE(IsSaveDataExisting(appId, userId));

    // 他人のセーブデータを開けて書ける
    NNT_EXPECT_RESULT_SUCCESS(MountSaveData("save", appId, userId));
    DeleteFile("save:/SimpleTest.file");
    NNT_EXPECT_RESULT_SUCCESS(CreateFile("save:/SimpleTest.file", 0));
    DeleteFile("save:/SimpleTest.file");
    Unmount("save");

    // 他人のセーブデータを開けるが書けない
    NNT_EXPECT_RESULT_SUCCESS(MountSaveDataReadOnly("save", appId, userId));
    NNT_EXPECT_RESULT_FAILURE(ResultUnsupportedOperation, CreateFile("save:/SimpleTest.file", 0));
    Unmount("save");

    // アクセス権を設定していない既存のセーブデータを開けない
    nn::ncm::ApplicationId pdAppId = { TestApplicationId + 0x10 };
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, (MountSaveData("save", pdAppId, userId)));

    // 他人のセーブデータが存在しないことの確認
    nn::ncm::ApplicationId noAppId = { TestApplicationId - 1 };
    ASSERT_FALSE(IsSaveDataExisting(noAppId, userId));
}

//! 権限: Debug + DeviceSaveData
TEST(SaveDataAccessControlTest, DeviceSaveData)
{
    // 本体セーブデータを開ける
    NNT_EXPECT_RESULT_SUCCESS(MountDeviceSaveData("save"));
    Unmount("save");
}

//! 権限: Debug + SystemSaveData
TEST(SaveDataAccessControlTest, SystemSaveData)
{
    // システムセーブデータを作成・マウント・拡張データ参照・削除できる
    {
        NNT_EXPECT_RESULT_SUCCESS(CreateSystemSaveData(TestSystemSaveDataId, SaveDataSize, SaveDataJournalSize, 0));
        NNT_EXPECT_RESULT_SUCCESS(MountSystemSaveData("save", TestSystemSaveDataId));
        {
            nn::Bit64 ownerId = 0;
            NNT_EXPECT_RESULT_SUCCESS(GetSaveDataOwnerId(&ownerId, TestSystemSaveDataId));

            // ExtraData::Flags の書き込み許可は廃止
            uint32_t flags = 0;
            NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, SetSaveDataFlags(TestSystemSaveDataId, SaveDataSpaceId::System, flags));

            // ExtraData の設定はできない
            nn::time::PosixTime timeStamp = { 1 };
            NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, SetSaveDataTimeStamp(SaveDataSpaceId::System, TestSystemSaveDataId, timeStamp));
            SaveDataCommitId commitId = 1;
            NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, SetSaveDataCommitId(SaveDataSpaceId::System, TestSystemSaveDataId, commitId));
            NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, GetSaveDataCommitId(&commitId, SaveDataSpaceId::System, TestSystemSaveDataId));
        }
        Unmount("save");
        NNT_EXPECT_RESULT_SUCCESS(DeleteSaveData(TestSystemSaveDataId));

        NNT_EXPECT_RESULT_SUCCESS(CreateSystemSaveData(TestSystemSaveDataId, g_TestUserId, SaveDataSize, SaveDataJournalSize, 0));
        NNT_EXPECT_RESULT_SUCCESS(MountSystemSaveData("save", TestSystemSaveDataId, g_TestUserId));
        Unmount("save");
        NNT_EXPECT_RESULT_SUCCESS(DeleteSystemSaveData(SaveDataSpaceId::System, TestSystemSaveDataId, g_TestUserId));
    }

    // システムセーブデータを作成 (w/ ownerId)・マウント・拡張データ参照・削除できる
    {
        NNT_EXPECT_RESULT_SUCCESS(CreateSystemSaveData(TestSystemSaveDataId, TestApplicationId, SaveDataSize, SaveDataJournalSize, 0));
        NNT_EXPECT_RESULT_SUCCESS(MountSystemSaveData("save", TestSystemSaveDataId));
        {
            nn::Bit64 ownerId = 0;
            NNT_EXPECT_RESULT_SUCCESS(GetSaveDataOwnerId(&ownerId, TestSystemSaveDataId));

            // ExtraData::Flags の書き込み許可は廃止
            uint32_t flags = 0;
            NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, SetSaveDataFlags(TestSystemSaveDataId, SaveDataSpaceId::System, flags));

            // ExtraData の設定はできない
            nn::time::PosixTime timeStamp = { 1 };
            NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, SetSaveDataTimeStamp(SaveDataSpaceId::System, TestSystemSaveDataId, timeStamp));
            SaveDataCommitId commitId = 1;
            NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, SetSaveDataCommitId(SaveDataSpaceId::System, TestSystemSaveDataId, commitId));
            NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, GetSaveDataCommitId(&commitId, SaveDataSpaceId::System, TestSystemSaveDataId));
        }
        Unmount("save");
        NNT_EXPECT_RESULT_SUCCESS(DeleteSaveData(TestSystemSaveDataId));

        NNT_EXPECT_RESULT_SUCCESS(CreateSystemSaveData(TestSystemSaveDataId, g_TestUserId, TestApplicationId, SaveDataSize, SaveDataJournalSize, 0));
        NNT_EXPECT_RESULT_SUCCESS(MountSystemSaveData("save", TestSystemSaveDataId, g_TestUserId));
        Unmount("save");
        NNT_EXPECT_RESULT_SUCCESS(DeleteSystemSaveData(SaveDataSpaceId::System, TestSystemSaveDataId, g_TestUserId));
    }

    // 他人のシステムセーブデータを作成・マウント・拡張データ設定できない
    {
        NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, CreateSystemSaveData(TestSystemSaveDataId, TestApplicationId + 1, SaveDataSize, SaveDataJournalSize, 0));
        NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, MountSystemSaveData("save", TestExistSystemSaveDataId));
        {
            nn::Bit64 ownerId = 0;
            NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, GetSaveDataOwnerId(&ownerId, TestExistSystemSaveDataId));
            uint32_t flags = 0;
            NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, SetSaveDataFlags(TestExistSystemSaveDataId, SaveDataSpaceId::System, flags));
            nn::time::PosixTime timeStamp = { 1 };
            NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, SetSaveDataTimeStamp(SaveDataSpaceId::System, TestExistSystemSaveDataId, timeStamp));
            SaveDataCommitId commitId = 1;
            NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, SetSaveDataCommitId(SaveDataSpaceId::System, TestExistSystemSaveDataId, commitId));
            NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, GetSaveDataCommitId(&commitId, SaveDataSpaceId::System, TestExistSystemSaveDataId));
        }
        // TODO: 壊れていても消せるように権限チェックしていないが、壊れていなければチェックした方がいいかも
//        NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, DeleteSaveData(TestExistSystemSaveDataId));

        NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, CreateSystemSaveData(TestSystemSaveDataId, g_TestUserId, TestApplicationId + 1, SaveDataSize, SaveDataJournalSize, 0));
        NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, MountSystemSaveData("save", TestExistSystemSaveDataId, g_TestUserId));
        // TODO: 壊れていても消せるように権限チェックしていないが、壊れていなければチェックした方がいいかも
//        NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, DeleteSystemSaveData(SaveDataSpaceId::System, TestExistSystemSaveDataId, g_TestUserId));
    }
}

//! 権限: Debug + SystemSaveData + SaveDataOwnerId (TestApplicationId + 1) リスト
TEST(SaveDataAccessControlTest, OthersSystemSaveData)
{
    // 指定した OwnerId に限り、他人のシステムセーブデータを作成・マウントできる
    {
        NNT_EXPECT_RESULT_SUCCESS(CreateSystemSaveData(TestSystemSaveDataId, TestApplicationId + 1, SaveDataSize, SaveDataJournalSize, 0));
        NNT_EXPECT_RESULT_SUCCESS(MountSystemSaveData("save", TestSystemSaveDataId));
        {
            nn::Bit64 ownerId = 0;
            NNT_EXPECT_RESULT_SUCCESS(GetSaveDataOwnerId(&ownerId, TestSystemSaveDataId));

            // ExtraData::Flags の書き込み許可は廃止
            uint32_t flags = 0;
            NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, SetSaveDataFlags(TestSystemSaveDataId, SaveDataSpaceId::System, flags));

            // ExtraData の設定はできない
            nn::time::PosixTime timeStamp = { 1 };
            NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, SetSaveDataTimeStamp(SaveDataSpaceId::System, TestSystemSaveDataId, timeStamp));
            SaveDataCommitId commitId = 1;
            NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, SetSaveDataCommitId(SaveDataSpaceId::System, TestSystemSaveDataId, commitId));
            NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, GetSaveDataCommitId(&commitId, SaveDataSpaceId::System, TestSystemSaveDataId));
        }
        Unmount("save");
        NNT_EXPECT_RESULT_SUCCESS(DeleteSaveData(TestSystemSaveDataId));

        NNT_EXPECT_RESULT_SUCCESS(CreateSystemSaveData(TestSystemSaveDataId, g_TestUserId, TestApplicationId + 1, SaveDataSize, SaveDataJournalSize, 0));
        NNT_EXPECT_RESULT_SUCCESS(MountSystemSaveData("save", TestSystemSaveDataId, g_TestUserId));
        Unmount("save");
        NNT_EXPECT_RESULT_SUCCESS(DeleteSystemSaveData(SaveDataSpaceId::System, TestSystemSaveDataId, g_TestUserId));
    }
    {
        NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, CreateSystemSaveData(TestSystemSaveDataId, TestApplicationId + 2, SaveDataSize, SaveDataJournalSize, 0));
        NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, MountSystemSaveData("save", TestExistSystemSaveDataId));
        {
            nn::Bit64 ownerId = 0;
            NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, GetSaveDataOwnerId(&ownerId, TestExistSystemSaveDataId));
            uint32_t flags = 0;
            NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, SetSaveDataFlags(TestExistSystemSaveDataId, SaveDataSpaceId::System, flags));
            nn::time::PosixTime timeStamp = { 1 };
            NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, SetSaveDataTimeStamp(SaveDataSpaceId::System, TestExistSystemSaveDataId, timeStamp));
            SaveDataCommitId commitId = 1;
            NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, SetSaveDataCommitId(SaveDataSpaceId::System, TestExistSystemSaveDataId, commitId));
            NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, GetSaveDataCommitId(&commitId, SaveDataSpaceId::System, TestExistSystemSaveDataId));
        }

        NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, CreateSystemSaveData(TestSystemSaveDataId, g_TestUserId, TestApplicationId + 2, SaveDataSize, SaveDataJournalSize, 0));
        NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, MountSystemSaveData("save", TestExistSystemSaveDataId, g_TestUserId));
        // TODO: 壊れていても消せるように権限チェックしていないが、壊れていなければチェックした方がいいかも
//        NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, DeleteSystemSaveData(SaveDataSpaceId::System, TestExistSystemSaveDataId, g_TestUserId));
    }
}

//! 権限: Debug + SystemSaveData
TEST(SaveDataAccessControlTest, SystemBcatSaveData)
{
    // システムBCATセーブデータを作成できる
    NNT_EXPECT_RESULT_SUCCESS(CreateSystemBcatSaveData(TestSystemBcatSaveDataId, SaveDataSize, 0));
    // BCATプロセスでないのでシステムBCATセーブデータのマウントはできない
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, MountSystemBcatSaveData("save", TestSystemBcatSaveDataId));
    // システムBCATセーブデータを削除できる
    NNT_EXPECT_RESULT_SUCCESS(DeleteSaveData(TestSystemBcatSaveDataId));
}

//! 権限: Debug + SaveDataBackUp
TEST(SaveDataAccessControlTest, SaveDataBackup)
{
    // 他人のセーブデータを作成・マウント・列挙・拡張データ参照/設定、削除できる
    {
        nn::ncm::ApplicationId appId = { TestApplicationId + 1 };
        nn::fs::UserId userId = {{ 0x0, 0x1 }};
        NNT_EXPECT_RESULT_SUCCESS(CreateSaveData(appId, userId, TestApplicationId + 1, SaveDataSize, SaveDataJournalSize, 0));
        NNT_EXPECT_RESULT_SUCCESS(MountSaveData("save", appId, userId));
        Unmount("save");

        nn::util::optional<SaveDataInfo> info;
        {
            FindSaveData(&info, SaveDataSpaceId::User, [&](const SaveDataInfo& i) NN_NOEXCEPT
            {
                return i.applicationId.value == appId.value && i.saveDataType == SaveDataType::Account && i.saveDataUserId == userId;
            });
        }
        EXPECT_NE(info, nn::util::nullopt);

        {
            nn::Bit64 ownerId = 0;
            NNT_EXPECT_RESULT_SUCCESS(GetSaveDataOwnerId(&ownerId, info->saveDataId));
            nn::time::PosixTime timeStamp = { 1 };
            NNT_EXPECT_RESULT_SUCCESS(SetSaveDataTimeStamp(SaveDataSpaceId::User, info->saveDataId, timeStamp));
            uint32_t flags = 0;
            NNT_EXPECT_RESULT_SUCCESS(SetSaveDataFlags(info->saveDataId, SaveDataSpaceId::User, flags));
            SaveDataCommitId commitId = 1;
            NNT_EXPECT_RESULT_SUCCESS(SetSaveDataCommitId(SaveDataSpaceId::User, info->saveDataId, commitId));
            NNT_EXPECT_RESULT_SUCCESS(GetSaveDataCommitId(&commitId, SaveDataSpaceId::User, info->saveDataId));
        }

        NNT_EXPECT_RESULT_SUCCESS(DeleteSaveData(info->saveDataId));
    }

    // 他人のシステムセーブデータを作成（CreateOthersSystemSaveData テストで実行済み）・マウント・列挙・拡張データ参照及び設定・削除できる
    {
        NNT_EXPECT_RESULT_SUCCESS(MountSystemSaveData("save", TestExistSystemSaveDataId));
        Unmount("save");

        NNT_EXPECT_RESULT_SUCCESS(MountSystemSaveData("save", TestExistSystemSaveDataId, g_TestUserId));
        Unmount("save");

        nn::util::optional<SaveDataInfo> info;
        {
            FindSaveData(&info, SaveDataSpaceId::System, [&](const SaveDataInfo& i) NN_NOEXCEPT
            {
                return i.systemSaveDataId == TestExistSystemSaveDataId && i.saveDataType == SaveDataType::System;
            });
        }
        EXPECT_NE(info, nn::util::nullopt);

        {
            nn::Bit64 ownerId = 0;
            NNT_EXPECT_RESULT_SUCCESS(GetSaveDataOwnerId(&ownerId, TestExistSystemSaveDataId));
            nn::time::PosixTime timeStamp = { 1 };
            NNT_EXPECT_RESULT_SUCCESS(SetSaveDataTimeStamp(SaveDataSpaceId::System, TestExistSystemSaveDataId, timeStamp));
            uint32_t flags = 0;
            NNT_EXPECT_RESULT_SUCCESS(SetSaveDataFlags(TestExistSystemSaveDataId, SaveDataSpaceId::System, flags));
            SaveDataCommitId commitId = 1;
            NNT_EXPECT_RESULT_SUCCESS(SetSaveDataCommitId(SaveDataSpaceId::System, TestExistSystemSaveDataId, commitId));
            NNT_EXPECT_RESULT_SUCCESS(GetSaveDataCommitId(&commitId, SaveDataSpaceId::System, TestExistSystemSaveDataId));
        }

        NNT_EXPECT_RESULT_SUCCESS(DeleteSaveData(TestExistSystemSaveDataId));
        NNT_EXPECT_RESULT_SUCCESS(DeleteSystemSaveData(SaveDataSpaceId::System, TestExistSystemSaveDataId, g_TestUserId));
    }
}

typedef ::testing::TestWithParam<ContentStorageId> MountContentStorageAccessControlTest;

//! アプリ権限で MountContentStorage できないこと
TEST_P(MountContentStorageAccessControlTest, ContentStorage)
{
    auto contentStorageId = GetParam();
    NNT_ASSERT_RESULT_FAILURE(ResultPermissionDenied, MountContentStorage(contentStorageId));
}

ContentStorageId contentStorageIds[] = {
    ContentStorageId::System,
    ContentStorageId::User,
    ContentStorageId::SdCard,
};

INSTANTIATE_TEST_CASE_P(ContentStorages,
                        MountContentStorageAccessControlTest,
                        ::testing::ValuesIn(contentStorageIds));

typedef ::testing::TestWithParam<ImageDirectoryId> MountImageDirectoryAccessControlTest;

//! アプリ権限で MountImageDirectory できないこと
TEST_P(MountImageDirectoryAccessControlTest, ImageDirectory)
{
    auto imageDirectoryId = GetParam();
    NNT_ASSERT_RESULT_FAILURE(ResultPermissionDenied, MountImageDirectory("image", imageDirectoryId));
}

ImageDirectoryId imageDirectoryIds[] = {
    ImageDirectoryId::Nand,
    ImageDirectoryId::SdCard,
};

INSTANTIATE_TEST_CASE_P(ImageDirectories,
                        MountImageDirectoryAccessControlTest,
                        ::testing::ValuesIn(imageDirectoryIds));

typedef ::testing::TestWithParam<GameCardPartition> MountGameCardPartitionDeniedDeathTest;

//! アプリ権限で MountGameCardPartition できないこと
TEST_P(MountGameCardPartitionDeniedDeathTest, GameCard)
{
    auto partition = GetParam();
    GameCardHandle handle;
    NNT_ASSERT_RESULT_SUCCESS(GetGameCardHandle(&handle));
    NNT_ASSERT_RESULT_FAILURE(ResultPermissionDenied, MountGameCardPartition("gc", handle, partition));
}

const GameCardPartition gameCardPartitions[] = {
    GameCardPartition::Update,
    GameCardPartition::Normal,
    GameCardPartition::Secure,
};

INSTANTIATE_TEST_CASE_P(GameCardPartitions,
                        MountGameCardPartitionDeniedDeathTest,
                        ::testing::ValuesIn(gameCardPartitions));

//! アプリ権限で Public でないシステムデータをマウントできないこと
//! アプリ権限で Public なシステムデータをマウントできること
TEST(MountSystemDataAccessControlTest, Application)
{
    struct
    {
        nn::ncm::SystemDataId id;
        bool isPublic;
    } systemDataList[] =
    {
        {{0x0100000000000806}, true}, // NgWord
        {{0x0100000000000809}, false} // FirmwareVersion
    };

    for( auto systemData : systemDataList )
    {
        if (systemData.isPublic)
        {
            NNT_ASSERT_RESULT_SUCCESS(MountSystemData("systemData", systemData.id));
            Unmount("systemData");
        }
        else
        {
            NNT_ASSERT_RESULT_FAILURE(ResultPermissionDenied, MountSystemData("systemData", systemData.id));
        }
    }
}

//! アプリ権限で SetGlobalAccessLogMode できないこと
TEST(SettingsControlAccessControlTest, Application)
{
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, SetGlobalAccessLogMode(AccessLogMode::AccessLogMode_Off));
}

//! アプリ権限で Debug API が実行できること
TEST(DebugApiAccessControlTest, Application)
{
    NNT_EXPECT_RESULT_SUCCESS(MountHostRoot());
    UnmountHostRoot();
    NNT_EXPECT_RESULT_SUCCESS(MountSdCardForDebug("sd"));
    Unmount("sd");
    NNT_EXPECT_RESULT_FAILURE(ResultTargetNotFound, CorruptSaveDataForDebug(0xFFFFFFFF));
#if 0 // 自動テストでは無効化
    NNT_EXPECT_RESULT_SUCCESS(MountSaveDataForDebug("save"));
    Unmount("save");
    SetSaveDataRootPath("path");
#endif
    NNT_EXPECT_RESULT_SUCCESS(CreatePaddingFile(1024));
    NNT_EXPECT_RESULT_SUCCESS(DeleteAllPaddingFiles());
}

//! アプリ権限で FormatSdCard できないこと
TEST(FormatSdCardAccessControlTest, Application)
{
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, FormatSdCard());
}

//! アプリ権限で RightsId 関係の機能が使えないこと
TEST(RightsIdAccessControlTest, Application)
{
    RightsId rightsId = { {0} };
    nn::spl::AccessKey accessKey = { {0} };
    nn::ncm::ProgramId programId = { 0x0005000c10000000ULL };
    nn::ncm::StorageId storageId = nn::ncm::StorageId::BuildInUser;

    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, GetRightsId(&rightsId, programId, storageId));
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, RegisterExternalKey(rightsId, accessKey));
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, UnregisterAllExternalKey());
}

//! アプリ権限で SetEncryptionSeed() できないこと
TEST(SetEncryptionSeedAccessControlTest, Application)
{
    EncryptionSeed seed = { {0} };
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, SetSdCardEncryptionSeed(seed));
}

namespace nn { namespace fs { namespace detail {
    // TORIAEZU:
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> GetFileSystemProxyServiceObject() NN_NOEXCEPT;
}}}


//! アプリ権限で OpenSaveDataTransferManager() できないこと
TEST(OpenSaveDataTransferManagerTest, Application)
{
    nn::sf::SharedPointer<nn::fssrv::sf::ISaveDataTransferManager> manager;

    // SaveDataTransferManager のコンストラクタ相当
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = nn::fs::detail::GetFileSystemProxyServiceObject();
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, fileSystemProxy->OpenSaveDataTransferManager(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::ISaveDataTransferManager>>(&manager)));
}

//! アプリ権限で OpenSaveDataTransferManagerVersion2() できないこと
TEST(OpenSaveDataTransferManagerVersion2Test, Application)
{
    nn::sf::SharedPointer<nn::fssrv::sf::ISaveDataTransferManagerWithDivision> manager;

    // SaveDataTransferManagerVersion2 のコンストラクタ相当
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = nn::fs::detail::GetFileSystemProxyServiceObject();
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, fileSystemProxy->OpenSaveDataTransferManagerVersion2(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::ISaveDataTransferManagerWithDivision>>(&manager)));
}

//! アプリ権限で OpenSaveDataExportProhibiter() できないこと
TEST(OpenSaveDataExportProhibiter, Application)
{
    std::unique_ptr<SaveDataExportProhibiter> prohibiter;
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, OpenSaveDataExportProhibiter(&prohibiter, TestApplicationId0));
}

//! アプリ権限で ListApplicationAccessibleSaveDataOwnerId() できないこと
TEST(ListApplicationAccessibleSaveDataOwnerId, Application)
{
    nn::ncm::ApplicationId appId = { TestApplicationId };
    nn::ncm::ApplicationId ids[4] = {};
    int idCount = 0;
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, ListApplicationAccessibleSaveDataOwnerId(&idCount, ids, appId, 0, 0, 4));
}

//! アプリ権限で OverrideSaveDataTransferTokenSignVerificationKey() できないこと
TEST(OverrideSaveDataTransferTokenSignVerificationKeyTest, Application)
{
    char key[256];
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, OverrideSaveDataTransferTokenSignVerificationKey(key, sizeof(key)));
}

//! アプリ権限で QuerySaveDataInternalStorageTotalSize() できないこと
TEST(QuerySaveDataInternalStorageTotalSize, Application)
{
    int64_t size;
    SaveDataId dummyId = 0;
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, QuerySaveDataInternalStorageTotalSize(&size, SaveDataSpaceId::User, dummyId));
}

//! アプリ権限で eMMC, GC 操作系の一部の API が実行できないこと
TEST(UnpermittedDeviceOperatorTest, Application)
{
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, EraseMmc(MmcPartition::UserData));
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, SuspendMmcPatrol());
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, ResumeMmcPatrol());

    GameCardHandle gcHandle = {};
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, EraseGameCard(GameCardSize::Size1GB, 1024));
    {
        auto fileSystemProxy = nn::fs::detail::GetFileSystemProxyServiceObject();
        nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator> deviceOperator;
        NNT_EXPECT_RESULT_SUCCESS(fileSystemProxy->OpenDeviceOperator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator>>(&deviceOperator)));
        NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, deviceOperator->FinalizeGameCardDriver());
    }
    {
        char cert[GameCardDeviceCertificateSize];
        NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, GetGameCardDeviceCertificate(cert, sizeof(cert), gcHandle));
    }
    {
        nn::gc::RmaInformation rmaInfo;
        char fwBuffer[GameCardAsicFwSize];
        NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, GetGameCardAsicInfo(&rmaInfo, fwBuffer, sizeof(fwBuffer)));
    }
    {
        char buffer[nn::gc::GcPageSize];
        char header[nn::gc::GcPageSize];
        NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, WriteToGameCard(0, buffer, sizeof(buffer)));
        NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, GetGameCardDeviceIdForProdCard(buffer, sizeof(buffer), header, sizeof(header)));
        NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, EraseAndWriteParamDirectly(buffer, sizeof(buffer)));
        NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, ReadParamDirectly(buffer, sizeof(buffer)));
    }
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, ForceEraseGameCard());

    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, SetSpeedEmulationMode(SpeedEmulationMode::None));
}

//! アプリ権限で OpenSdCardDetectionEventNotifier できないこと
TEST(OpenSdCardDetectionEventNotifierTest, Application)
{
    std::unique_ptr<nn::fs::IEventNotifier> deviceDetectionEventNotifier;
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, OpenSdCardDetectionEventNotifier(&deviceDetectionEventNotifier));
}

//! アプリ権限で OpenGameCardDetectionEventNotifier できないこと
TEST(OpenGameCardDetectionEventNotifierTest, Application)
{
    std::unique_ptr<nn::fs::IEventNotifier> deviceDetectionEventNotifier;
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, OpenGameCardDetectionEventNotifier(&deviceDetectionEventNotifier));
}

//! アプリ権限で OpenSystemDataUpdateEventNotifier できないこと
TEST(OpenSystemDataUpdateEventNotifierTest, Application)
{
    std::unique_ptr<nn::fs::IEventNotifier> eventNotifier;
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, OpenSystemDataUpdateEventNotifier(&eventNotifier));
}

//! アプリ権限で NotifySystemDataUpdateEvent できないこと
TEST(NotifySystemDataUpdateEventTest, Application)
{
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, NotifySystemDataUpdateEvent());
}

//! アプリ権限でアクセスエラー検知関連の API が実行できないこと
TEST(AccessFailureResolition, Application)
{
    std::unique_ptr<nn::fs::AccessFailureResolver> resolver;
    std::unique_ptr<nn::fs::IEventNotifier> eventNotifier;
    nn::Bit64 tmpProcessId = 128; // no effect
    nn::os::SystemEvent event;

    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, OpenAccessFailureDetectionEventNotifier(&eventNotifier, tmpProcessId));
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, OpenAccessFailureResolver(&resolver, tmpProcessId));
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, GetAccessFailureDetectionEvent(&event));
    {
        auto fileSystemProxy = nn::fs::detail::GetFileSystemProxyServiceObject();
        bool isAccessFailureDetected = false;
        NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, fileSystemProxy->IsAccessFailureDetected(&isAccessFailureDetected, tmpProcessId));
        NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, fileSystemProxy->ResolveAccessFailure(tmpProcessId));
        NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, fileSystemProxy->AbandonAccessFailure(tmpProcessId));
    }
}

//! アプリ権限で OpenSaveDataInternalStorageFileSystem できないこと
TEST(OpenSaveDataInternalStorageFileSystemTest, Application)
{
    std::unique_ptr<fsa::IFileSystem> fileSystem;
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, OpenSaveDataInternalStorageFileSystem(&fileSystem, nn::fs::SaveDataSpaceId::System, TestSystemSaveDataId));
}

//! アプリ権限で SetSdCardAccessibility できないこと
TEST(SetSdCardAccessibilityTest, Application)
{
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, detail::SetSdCardAccessibility(true));
}

//! アプリ権限で VerifySaveData できないこと
TEST(VerifySaveDataTest, Application)
{
    bool outIsValid;
    char buffer[1024];
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, VerifySaveData(&outIsValid, TestSystemSaveDataId, buffer, sizeof(buffer)));
}

//! 権限: Debug のみ (= アプリ権限)ではシステムセーブデータを CorruptSaveDataForDebug できないこと
TEST(CorruptSaveDataForDebugTest, Application)
{
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, CorruptSaveDataForDebug(TestExistSystemSaveDataId));
}

TEST(EnsureApplicationCacheStorageTest, Application)
{
    int64_t outRequiredSize;
    const nn::ncm::ApplicationId applicationId = {TestApplicationId};
    nn::ns::ApplicationControlProperty property = {};
    property.saveDataOwnerId = applicationId.value;
    property.cacheStorageSize = 1 * 1024 * 1024;
    property.cacheStorageJournalSize = 1 * 1024 * 1024;
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, nn::fs::EnsureApplicationCacheStorage(&outRequiredSize, applicationId, property));
}

TEST(CreateApplicationCacheStorageTest, Application)
{
    int64_t outRequiredSize;
    nn::fs::CacheStorageTargetMedia outTargetMedia;
    const nn::ncm::ApplicationId applicationId = {TestApplicationId};
    nn::ns::ApplicationControlProperty property = {};
    property.saveDataOwnerId = applicationId.value;
    property.cacheStorageSize = 1 * 1024 * 1024;
    property.cacheStorageJournalSize = 1 * 1024 * 1024;
    property.cacheStorageDataAndJournalSizeMax = 4 * 1024 * 1024;
    NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, nn::fs::CreateApplicationCacheStorage(&outRequiredSize, &outTargetMedia, applicationId, property, 0, property.cacheStorageSize, property.cacheStorageJournalSize));
}

//! アプリ権限で CacheStorage の操作ができること
TEST(CacheStorageTest, Application)
{
    const int64_t Size = 1024 * 1024;

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateCacheStorage(0, Size, Size));

    // 列挙確認
    nn::fs::CacheStorageListHandle handle;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenCacheStorageList(&handle));

    nn::fs::CacheStorageInfo info;
    int num = 0;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadCacheStorageList(&num, &info, handle, 1));

    nn::fs::CloseCacheStorageList(handle);

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountCacheStorage("cache", 0));
    Unmount("cache");

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteCacheStorage(0));
}

//! アプリ権限で MountCloudBackupWorkStorage できないこと
TEST(MountCloudBackupWorkStorage, Application)
{
    NNT_ASSERT_RESULT_FAILURE(ResultPermissionDenied, MountCloudBackupWorkStorage("CbWork", CloudBackupWorkStorageId::Nand));
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nn::fs::SetEnabledAutoAbort(false);
    nnt::fs::util::ResetAllocateCount();

    nn::fs::SetEnabledAutoAbort(false);

    nn::oe::Initialize();
    nn::account::Initialize();

    {
        nn::account::Uid user = nn::account::InvalidUid;
        int userCount = 0;
        auto result = nn::account::ListAllUsers(&userCount, &user, 1);
        NN_ASSERT(result.IsSuccess() && userCount > 0);
        g_TestUserId = nn::fs::ConvertAccountUidToFsUserId(user);
    }

    auto testResult = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(testResult);
}
