﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>

#include <nn/fs.h>
#include <nn/nn_Abort.h>
#include <nn/fs/fs_Application.h>
#include <nn/fs/fs_Code.h>
#include <nn/fs/fs_Logo.h>
#include <fsa/fs_MountUtility.h>
#include <nn/fssrv/sf/fssrv_IFileSystemProxy.h>

#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util_function.h>

namespace nn { namespace fs { namespace detail {

    MountName GetMountName(const char* path) NN_NOEXCEPT;

}}}

namespace nnt { namespace fs {

    namespace {

        void MakeTooLongPath(::std::add_pointer<decltype(nnt::fs::util::AllocateBuffer(0))>::type outBuffer) NN_NOEXCEPT
        {
            const auto BufferSize = sizeof(nn::fssrv::sf::FspPath::str) + 1;
            *outBuffer = nnt::fs::util::AllocateBuffer(BufferSize);
            std::memset(outBuffer->get(), 'a', BufferSize - 1);
            outBuffer->get()[BufferSize - 1] = '\0';
        }

    }

}}

class GetMountName : public ::testing::Test
{
};

TEST_F(GetMountName, HasNoDelimiter)
{
    nn::fs::detail::GetMountName("a");
}

TEST_F(GetMountName, MountNameTooLong)
{
    const auto BufferLength = nn::fs::MountName::LENGTH + 3;

    auto buffer = nnt::fs::util::AllocateBuffer(BufferLength);
    std::memset(buffer.get(), 'a', BufferLength - 2);
    buffer.get()[BufferLength - 2] = ':';
    buffer.get()[BufferLength - 1] = 0;

    nn::fs::detail::GetMountName(buffer.get());
}

class GetSubPath : public ::testing::Test
{
protected:
    static const char* InvalidPath;
    static const char* ValidPath;

protected:
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHost("testfs", "C:\\"));
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::fs::Unmount("testfs");
    }
};

const char* GetSubPath::InvalidPath = "testfs:InvalidPathFormat";
const char* GetSubPath::ValidPath = "testfs:/ValidPathFormat";

TEST_F(GetSubPath, AbortViaCreateFile)
{
    nn::fs::CreateFile(InvalidPath, 0);
}

TEST_F(GetSubPath, AbortViaDeleteFile)
{
    nn::fs::DeleteFile(InvalidPath);
}

TEST_F(GetSubPath, AbortViaCreateDirectory)
{
    nn::fs::CreateDirectory(InvalidPath);
}

TEST_F(GetSubPath, AbortViaDeleteDirectory)
{
    nn::fs::DeleteDirectory(InvalidPath);
}

TEST_F(GetSubPath, AbortViaDeleteDirectoryRecursively)
{
    nn::fs::DeleteDirectoryRecursively(InvalidPath);
}

TEST_F(GetSubPath, AbortViaCleanDirectoryRecursively)
{
    nn::fs::CleanDirectoryRecursively(InvalidPath);
}

TEST_F(GetSubPath, AbortViaRenameFileArg1)
{
    nn::fs::RenameFile(InvalidPath, ValidPath);
}

TEST_F(GetSubPath, AbortViaRenameFileArg2)
{
    nn::fs::RenameFile(ValidPath, InvalidPath);
}

TEST_F(GetSubPath, AbortViaRenameDirectoryArg1)
{
    nn::fs::RenameDirectory(InvalidPath, ValidPath);
}

TEST_F(GetSubPath, AbortViaRenameDirectoryArg2)
{
    nn::fs::RenameDirectory(ValidPath, InvalidPath);
}

TEST_F(GetSubPath, AbortViaGetEntryType)
{
    nn::fs::DirectoryEntryType type;
    nn::fs::GetEntryType(&type, InvalidPath);
}

TEST_F(GetSubPath, AbortViaOpenFile)
{
    nn::fs::FileHandle handle;
    nn::fs::OpenFile(&handle, InvalidPath, nn::fs::OpenMode_Read);
}

TEST_F(GetSubPath, AbortViaOpenDirectory)
{
    nn::fs::DirectoryHandle handle;
    nn::fs::OpenDirectory(&handle, InvalidPath, nn::fs::OpenDirectoryMode_All);
}

class MountApplicationPackage : public ::testing::Test
{
};

TEST_F(MountApplicationPackage, PathIsTooLong)
{
    decltype(nnt::fs::util::AllocateBuffer(0)) path(nullptr, nnt::fs::util::DeleterBuffer);
    nnt::fs::MakeTooLongPath(&path);
    nn::fs::MountApplicationPackage("aa", path.get());
}

class MountCode : public ::testing::Test
{
};

TEST_F(MountCode, PathIsTooLong)
{
    nn::ncm::ProgramId id;
    id.value = 0;

    decltype(nnt::fs::util::AllocateBuffer(0)) path(nullptr, nnt::fs::util::DeleterBuffer);
    nnt::fs::MakeTooLongPath(&path);
    nn::fs::MountCode("aa", path.get(), id);
}

class MountLogo : public ::testing::Test
{
};

TEST_F(MountLogo, PathIsTooLong)
{
    nn::ncm::ProgramId id;
    id.value = 0;

    decltype(nnt::fs::util::AllocateBuffer(0)) path(nullptr, nnt::fs::util::DeleterBuffer);
    nnt::fs::MakeTooLongPath(&path);
    nn::fs::MountLogo("aa", path.get(), id);
}
