﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <memory>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/fs.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/fs/fs_MmcPrivate.h>
#include <nn/fs/fs_ErrorInfoPrivate.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/util/util_ScopeExit.h>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

using namespace nn::fs;

//! @pre SD カードが接続されている
TEST(SdCard, Inserted)
{
    EXPECT_TRUE(IsSdCardInserted());
}


//! @pre SD カードが接続されている
TEST(SdCard, GetSpeed)
{
    SdCardSpeedMode speed;

    // Activate のために MountSdCardForDebug() が必要
    NNT_EXPECT_RESULT_SUCCESS(MountSdCardForDebug("sd"));
    {
        NNT_EXPECT_RESULT_SUCCESS(GetSdCardSpeedMode(&speed));
        NN_LOG("speed: %d\n", speed);
        EXPECT_NE(SdCardSpeedMode_Unknown, speed);
    }
    Unmount("sd");
}

TEST(SdCardDeathTest, NullptrArgument)
{
    // Activate のために MountSdCardForDebug() が必要
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSdCardForDebug("sd"));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount("sd");
    };

    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultNullptrArgument,
        nn::fs::GetSdCardSpeedMode(nullptr));

    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultNullptrArgument,
        nn::fs::GetSdCardCid(nullptr, 0));

    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultNullptrArgument,
        nn::fs::GetSdCardUserAreaSize(nullptr));

    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultNullptrArgument,
        nn::fs::GetSdCardProtectedAreaSize(nullptr));

    nn::fs::StorageErrorInfo errorInfo;
    size_t logSize;
    char logBuffer[256];
    const size_t logBufferSize = sizeof(logBuffer);
    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultNullptrArgument,
        nn::fs::GetAndClearSdCardErrorInfo(nullptr, &logSize, logBuffer, logBufferSize));
    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultNullptrArgument,
        nn::fs::GetAndClearSdCardErrorInfo(&errorInfo, nullptr, logBuffer, logBufferSize));
    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultNullptrArgument,
        nn::fs::GetAndClearSdCardErrorInfo(&errorInfo, &logSize, nullptr, 0));

    // DeathTest（実機のみを対象としているテストのためほぼ無意味）
    {
        nn::fs::SetEnabledAutoAbort(true);
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::SetEnabledAutoAbort(false);
        };

        EXPECT_DEATH_IF_SUPPORTED(
            nn::fs::GetSdCardSpeedMode(nullptr), "");

        EXPECT_DEATH_IF_SUPPORTED(
            nn::fs::GetSdCardCid(nullptr, 0), "");

        EXPECT_DEATH_IF_SUPPORTED(
            nn::fs::GetSdCardUserAreaSize(nullptr), "");

        EXPECT_DEATH_IF_SUPPORTED(
            nn::fs::GetSdCardProtectedAreaSize(nullptr), "");

        EXPECT_DEATH_IF_SUPPORTED(
            nn::fs::GetAndClearSdCardErrorInfo(nullptr, &logSize, logBuffer, logBufferSize), "");
        EXPECT_DEATH_IF_SUPPORTED(
            nn::fs::GetAndClearSdCardErrorInfo(&errorInfo, nullptr, logBuffer, logBufferSize), "");
        EXPECT_DEATH_IF_SUPPORTED(
            nn::fs::GetAndClearSdCardErrorInfo(&errorInfo, &logSize, nullptr, 0), "");
    }
}

TEST(Mmc, GetSpeed)
{
    MmcSpeedMode speed;

    NNT_EXPECT_RESULT_SUCCESS(GetMmcSpeedMode(&speed));
    NN_LOG("speed: %d\n", speed);
    EXPECT_EQ(MmcSpeedMode_Hs400, speed);
}

TEST(Mmc, GetCid)
{
    uint8_t cid[nn::fs::MmcCidSize];
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetMmcCid(cid, sizeof(cid)));

    nnt::fs::util::DumpBuffer(cid, nn::fs::MmcCidSize);
}


#if 0 // mmc を消去してしまうので無効化
TEST(Mmc, EraseAll)
{
    NNT_EXPECT_RESULT_SUCCESS(EraseMmc(MmcPartition::UserData));
    NNT_EXPECT_RESULT_SUCCESS(EraseMmc(MmcPartition::BootPartition1));
    NNT_EXPECT_RESULT_SUCCESS(EraseMmc(MmcPartition::BootPartition2));
}
#endif

TEST(Mmc, GetPartitionSize)
{
    int64_t mmcPartitionSize = 0;

    NNT_EXPECT_RESULT_SUCCESS(GetMmcPartitionSize(&mmcPartitionSize, MmcPartition::UserData));
    // SDEV, EDEV, PROD 想定, http://spdlybra.nintendo.co.jp/confluence/display/ISWD/eMMC+NAND
    EXPECT_TRUE((mmcPartitionSize == 31268536320) || (mmcPartitionSize == 62537072640));
    NN_LOG("User Data Size: %d (MB)\n", mmcPartitionSize / 1024 / 1024);

    NNT_EXPECT_RESULT_SUCCESS(GetMmcPartitionSize(&mmcPartitionSize, MmcPartition::BootPartition1));
    EXPECT_EQ(mmcPartitionSize, 4 * 1024 * 1024);
    NN_LOG("Boot Partition 1 Size: %d (MB)\n", mmcPartitionSize / 1024 / 1024);

    NNT_EXPECT_RESULT_SUCCESS(GetMmcPartitionSize(&mmcPartitionSize, MmcPartition::BootPartition2));
    EXPECT_EQ(mmcPartitionSize, 4 * 1024 * 1024);
    NN_LOG("Boot Partition 2 Size: %d (MB)\n", mmcPartitionSize / 1024 / 1024);
}

TEST(MmcDeathTest, NullptrArgument)
{
    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultNullptrArgument,
        nn::fs::GetMmcSpeedMode(nullptr));

    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultNullptrArgument,
        nn::fs::GetMmcCid(nullptr, 0));

    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultNullptrArgument,
        nn::fs::GetMmcPartitionSize(nullptr, nn::fs::MmcPartition::UserData));

    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultNullptrArgument,
        nn::fs::GetMmcPatrolCount(nullptr));

    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultNullptrArgument,
        nn::fs::GetMmcExtendedCsd(nullptr, 0));

    nn::fs::StorageErrorInfo errorInfo;
    size_t logSize;
    char logBuffer[256];
    const size_t logBufferSize = sizeof(logBuffer);
    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultNullptrArgument,
        nn::fs::GetAndClearMmcErrorInfo(nullptr, &logSize, logBuffer, logBufferSize));
    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultNullptrArgument,
        nn::fs::GetAndClearMmcErrorInfo(&errorInfo, nullptr, logBuffer, logBufferSize));
    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultNullptrArgument,
        nn::fs::GetAndClearMmcErrorInfo(&errorInfo, &logSize, nullptr, 0));

    // DeathTest（実機のみを対象としているテストのためほぼ無意味）
    {
        nn::fs::SetEnabledAutoAbort(true);
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::SetEnabledAutoAbort(false);
        };

        EXPECT_DEATH_IF_SUPPORTED(
            nn::fs::GetMmcSpeedMode(nullptr), "");

        EXPECT_DEATH_IF_SUPPORTED(
            nn::fs::GetMmcCid(nullptr, 0), "");

        EXPECT_DEATH_IF_SUPPORTED(
            nn::fs::GetMmcPartitionSize(nullptr, nn::fs::MmcPartition::UserData), "");

        EXPECT_DEATH_IF_SUPPORTED(
            nn::fs::GetMmcPatrolCount(nullptr), "");

        EXPECT_DEATH_IF_SUPPORTED(
            nn::fs::GetMmcExtendedCsd(nullptr, 0), "");

        EXPECT_DEATH_IF_SUPPORTED(
            nn::fs::GetAndClearMmcErrorInfo(nullptr, &logSize, logBuffer, logBufferSize), "");
        EXPECT_DEATH_IF_SUPPORTED(
            nn::fs::GetAndClearMmcErrorInfo(&errorInfo, nullptr, logBuffer, logBufferSize), "");
        EXPECT_DEATH_IF_SUPPORTED(
            nn::fs::GetAndClearMmcErrorInfo(&errorInfo, &logSize, nullptr, 0), "");
    }
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);
    nnt::fs::util::ResetAllocateCount();

    SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nn::fs::SetEnabledAutoAbort(false);

    auto testResult = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(testResult);
}
