﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

#if defined(NN_BUILD_CONFIG_OS_WIN32)
#include <nn/fssystem/fs_PathOnExecutionDirectory.h>
#endif

namespace {
    void DeleteUserSaveData(nn::fs::UserId user) NN_NOEXCEPT
    {
        const nn::fs::SaveDataSpaceId SpaceIds[] = {nn::fs::SaveDataSpaceId::User, nn::fs::SaveDataSpaceId::System};
        for(auto spaceId : SpaceIds)
        {
            std::unique_ptr<nn::fs::SaveDataIterator> iter;
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::OpenSaveDataIterator(&iter, spaceId));

            while(NN_STATIC_CONDITION(true))
            {
                int64_t count;
                nn::fs::SaveDataInfo info;
                NNT_EXPECT_RESULT_SUCCESS(iter->ReadSaveDataInfo(&count, &info, 1));
                if( count == 0 )
                {
                    break;
                }

                auto saveId = info.saveDataId;
                if (info.saveDataUserId == user)
                {
                    char saveDataPath[5 + 1 + 3 + 1 + 6 + 1 + 4 + 1 + 16 + 1];
                    char saveDataMetaPath[5 + 1 + 3 + 1 + 4 + 1 + 8 + 1 + 16 + 1];
#if defined(NN_BUILD_CONFIG_OS_WIN32)
                    nn::fssystem::PathOnExecutionDirectory pathOnExe("");
                    NNT_EXPECT_RESULT_SUCCESS(nn::fs::MountHost("save", pathOnExe.Get()));
                    if(spaceId == nn::fs::SaveDataSpaceId::System)
                    {
                        nn::util::SNPrintf(saveDataPath, sizeof(saveDataPath), "save:/bis/system/save/%016llx", saveId);
                    }
                    else
                    {
                        nn::util::SNPrintf(saveDataPath, sizeof(saveDataPath), "save:/bis/user/save/%016llx", saveId);
                    }
                    nn::util::SNPrintf(saveDataMetaPath, sizeof(saveDataMetaPath), "save:/bis/user/saveMeta/%016llx", saveId);
#else
                    NNT_EXPECT_RESULT_SUCCESS(nn::fs::MountBis("save", spaceId == nn::fs::SaveDataSpaceId::User ? nn::fs::BisPartitionId::User : nn::fs::BisPartitionId::System));
                    nn::util::SNPrintf(saveDataPath, sizeof(saveDataPath), "save:/save/%016llx", saveId);
                    nn::util::SNPrintf(saveDataMetaPath, sizeof(saveDataMetaPath), "save:/saveMeta/%016llx", saveId);
#endif
                    nn::fs::DirectoryEntryType type;
                    NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetEntryType(&type, saveDataPath));
                    NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetEntryType(&type, saveDataMetaPath));

                    NN_LOG("delete save data (%x, %llx).\n", spaceId, saveId);
                    NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteSaveData(saveId));

                    // 実体がちゃんと消えていることの確認
                    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultPathNotFound, nn::fs::GetEntryType(&type, saveDataPath));
                    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultPathNotFound, nn::fs::GetEntryType(&type, saveDataMetaPath));

                    nn::fs::Unmount("save");

                    // 削除後は要再オープン
                    NNT_EXPECT_RESULT_SUCCESS(nn::fs::OpenSaveDataIterator(&iter, spaceId));
                }
            }
        }
    }

    nn::Bit64 GetSaveDataOwnerId() NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_WIN32)
        return 0x0;
#else
        return 0x0005000C10000000;
#endif
    }

    const size_t ThumbnailFileSize = 256 * 1024;
}

#if defined(NN_BUILD_CONFIG_OS_WIN32)
const nn::ncm::ApplicationId ApplicationId = { 0ULL };
#else
const nn::ncm::ApplicationId ApplicationId = { 0x0005000c10000000ULL };
#endif
const size_t SaveDataSize = 16 * 1024 * 1024;
const size_t SaveDataJournalSize = 16 * 1024 * 1024;
const uint64_t SaveDataOwnerId = ApplicationId.value;

TEST(SaveDataThumbnail, Basic)
{
    // セーブデータ作成
    nn::fs::UserId userId1 = {{ 0x00ull, 0x01ull }};
    DeleteUserSaveData(userId1);
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(ApplicationId, userId1, SaveDataOwnerId, SaveDataSize, SaveDataJournalSize, 0));

    auto headerSize = ThumbnailFileSize / 2;
    auto bodySize = headerSize;

    auto header = nnt::fs::util::AllocateBuffer(headerSize);
    nnt::fs::util::FillBufferWithRandomValue(header.get(), headerSize);

    auto body = nnt::fs::util::AllocateBuffer(bodySize);
    nnt::fs::util::FillBufferWithRandomValue(body.get(), bodySize);

    auto verifyHeader = nnt::fs::util::AllocateBuffer(headerSize);
    auto verifyBody = nnt::fs::util::AllocateBuffer(bodySize);

    // Read/Write
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteSaveDataThumbnailFile(GetSaveDataOwnerId(), userId1, header.get(), headerSize, body.get(), bodySize));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadSaveDataThumbnailFile(GetSaveDataOwnerId(), userId1, verifyHeader.get(), headerSize, verifyBody.get(), bodySize));
    NNT_FS_UTIL_EXPECT_MEMCMPEQ(header.get(), verifyHeader.get(), headerSize);
    NNT_FS_UTIL_EXPECT_MEMCMPEQ(body.get(), verifyBody.get(), bodySize);

    // header のみ Read/Write
    nnt::fs::util::FillBufferWithRandomValue(header.get(), headerSize);
    memset(verifyHeader.get(), 0x0, headerSize);
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteSaveDataThumbnailFileHeader(GetSaveDataOwnerId(), userId1, header.get(), headerSize));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadSaveDataThumbnailFileHeader(GetSaveDataOwnerId(), userId1, verifyHeader.get(), headerSize));
    NNT_FS_UTIL_EXPECT_MEMCMPEQ(header.get(), verifyHeader.get(), headerSize);

    // 再度 body も確認
    nnt::fs::util::InvalidateVariable(verifyHeader.get(), static_cast<int>(headerSize));
    nnt::fs::util::InvalidateVariable(verifyBody.get(), static_cast<int>(bodySize));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadSaveDataThumbnailFile(GetSaveDataOwnerId(), userId1, verifyHeader.get(), headerSize, verifyBody.get(), bodySize));
    NNT_FS_UTIL_EXPECT_MEMCMPEQ(header.get(), verifyHeader.get(), headerSize);
    NNT_FS_UTIL_EXPECT_MEMCMPEQ(body.get(), verifyBody.get(), bodySize);

    // Delete でサムネイルも同時に削除
    DeleteUserSaveData(userId1);

    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, nn::fs::WriteSaveDataThumbnailFile(GetSaveDataOwnerId(), userId1, header.get(), headerSize, body.get(), bodySize));
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, nn::fs::WriteSaveDataThumbnailFileHeader(GetSaveDataOwnerId(), userId1, header.get(), headerSize));
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, nn::fs::ReadSaveDataThumbnailFile(GetSaveDataOwnerId(), userId1, verifyHeader.get(), headerSize, verifyBody.get(), bodySize));
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, nn::fs::ReadSaveDataThumbnailFileHeader(GetSaveDataOwnerId(), userId1, verifyHeader.get(), headerSize));
}

TEST(SaveDataThumbnail, RestrictedFileExtention)
{
    nn::fs::UserId userId1 = {{ 0x00ull, 0x01ull }};
    DeleteUserSaveData(userId1);
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(ApplicationId, userId1, SaveDataOwnerId, SaveDataSize, SaveDataJournalSize, 0));

    NN_UTIL_SCOPE_EXIT
    {
        DeleteUserSaveData(userId1);
    };

    auto headerSize = ThumbnailFileSize + 1;

    auto header = nnt::fs::util::AllocateBuffer(headerSize);
    nnt::fs::util::FillBufferWithRandomValue(header.get(), headerSize);

    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultFileExtensionWithoutOpenModeAllowAppend,
        nn::fs::WriteSaveDataThumbnailFileHeader(GetSaveDataOwnerId(), userId1, header.get(), headerSize)
    );
}

TEST(SaveDataThumbnail, TargetNotFound)
{
    nn::fs::UserId userId1 = {{ 0x00ull, 0x01ull }};
    DeleteUserSaveData(userId1);
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(ApplicationId, userId1, SaveDataOwnerId, SaveDataSize, SaveDataJournalSize, 0));

    nn::fs::UserId userId2 = {{ 0x00ull, 0x02ull }};

    auto headerSize = ThumbnailFileSize / 2;
    auto bodySize = headerSize;

    auto header = nnt::fs::util::AllocateBuffer(headerSize);
    nnt::fs::util::FillBufferWithRandomValue(header.get(), headerSize);

    auto body = nnt::fs::util::AllocateBuffer(bodySize);
    nnt::fs::util::FillBufferWithRandomValue(body.get(), bodySize);

    auto buffer = nnt::fs::util::AllocateBuffer(ThumbnailFileSize);
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, nn::fs::WriteSaveDataThumbnailFile(GetSaveDataOwnerId(), userId2, header.get(), headerSize, body.get(), bodySize));
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, nn::fs::ReadSaveDataThumbnailFile(GetSaveDataOwnerId(), userId2, header.get(), headerSize, body.get(), bodySize));
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, nn::fs::WriteSaveDataThumbnailFileHeader(GetSaveDataOwnerId(), userId2, header.get(), headerSize));
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, nn::fs::ReadSaveDataThumbnailFileHeader(GetSaveDataOwnerId(), userId2, header.get(), headerSize));

    DeleteUserSaveData(userId1);
}

// サムネイル欠落状態の復旧
TEST(SaveDataThumbnail, RecoverMissingThumbnail)
{
    nn::fs::UserId userId1 = { { 0x00ull, 0x01ull } };
    DeleteUserSaveData(userId1);
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(ApplicationId, userId1, SaveDataOwnerId, SaveDataSize, SaveDataJournalSize, 0));

    nn::util::optional<nn::fs::SaveDataInfo> info;
    nnt::fs::util::FindSaveData(&info, nn::fs::SaveDataSpaceId::User, [&](nn::fs::SaveDataInfo info_) {
        return info_.applicationId == ApplicationId && info_.saveDataType == nn::fs::SaveDataType::Account && info_.saveDataUserId == userId1; });

    NNT_ASSERT_RESULT_SUCCESS(MountBis("bis", nn::fs::BisPartitionId::User));

    char path[128];
    nn::util::SNPrintf(path, sizeof(path), "bis:/saveMeta/%016llx/%08x.meta", info->saveDataId, 1);


    auto headerSize = ThumbnailFileSize / 2;
    auto bodySize = headerSize;

    auto header = nnt::fs::util::AllocateBuffer(headerSize);
    nnt::fs::util::FillBufferWithRandomValue(header.get(), headerSize);

    auto body = nnt::fs::util::AllocateBuffer(bodySize);
    nnt::fs::util::FillBufferWithRandomValue(body.get(), bodySize);

    auto buffer = nnt::fs::util::AllocateBuffer(ThumbnailFileSize);

    nn::fs::DirectoryEntryType type;

    {
        // 欠落させる
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteFile(path));
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultPathNotFound, nn::fs::GetEntryType(&type, path));

        // サムネイルへの write で復旧
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteSaveDataThumbnailFile(GetSaveDataOwnerId(), userId1, header.get(), headerSize, body.get(), bodySize));
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetEntryType(&type, path));
    }

    {
        // 欠落させる
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteFile(path));
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultPathNotFound, nn::fs::GetEntryType(&type, path));

        // サムネイルの read で復旧
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadSaveDataThumbnailFile(GetSaveDataOwnerId(), userId1, header.get(), headerSize, body.get(), bodySize));
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetEntryType(&type, path));
    }

    nn::fs::Unmount("bis");
    DeleteUserSaveData(userId1);
}


TEST(SaveDataThumbnail, ReadBeforeWrite)
{
    nn::fs::UserId userId1 = {{ 0x00ull, 0x01ull }};
    DeleteUserSaveData(userId1);
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(ApplicationId, userId1, SaveDataOwnerId, SaveDataSize, SaveDataJournalSize, 0));

    auto headerSize = ThumbnailFileSize / 2;
    auto bodySize = headerSize;

    auto header = nnt::fs::util::AllocateBuffer(headerSize);
    nnt::fs::util::FillBufferWithRandomValue(header.get(), headerSize);

    auto body = nnt::fs::util::AllocateBuffer(bodySize);
    nnt::fs::util::FillBufferWithRandomValue(body.get(), bodySize);

    NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadSaveDataThumbnailFile(GetSaveDataOwnerId(), userId1, header.get(), headerSize, body.get(), bodySize));
    // 作成直後、未書き込み時のデータは 0 埋めである
    EXPECT_TRUE(nnt::fs::util::IsFilledWithValue(header.get(), headerSize, 0));
    EXPECT_TRUE(nnt::fs::util::IsFilledWithValue(body.get(), bodySize, 0));

    NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadSaveDataThumbnailFileHeader(GetSaveDataOwnerId(), userId1, header.get(), headerSize));
    EXPECT_TRUE(nnt::fs::util::IsFilledWithValue(header.get(), headerSize, 0)); // 作成直後、未書き込み時のデータは 0 埋めである

    // header だけ書き込み
    nnt::fs::util::FillBufferWithRandomValue(header.get(), headerSize);
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::WriteSaveDataThumbnailFileHeader(GetSaveDataOwnerId(), userId1, header.get(), headerSize));

    // body は 0 埋めのまま成功
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadSaveDataThumbnailFile(GetSaveDataOwnerId(), userId1, header.get(), headerSize, body.get(), bodySize));
    EXPECT_TRUE(nnt::fs::util::IsFilledWithValue(body.get(), bodySize, 0));

    DeleteUserSaveData(userId1);
}

TEST(SaveDataThumbnail, CorruptData)
{
    nn::fs::UserId userId1 = {{ 0x00ull, 0x01ull }};
    DeleteUserSaveData(userId1);

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(ApplicationId, userId1, SaveDataOwnerId, SaveDataSize, SaveDataJournalSize, 0));

    auto headerSize = ThumbnailFileSize / 2;
    auto bodySize = headerSize;

    auto header = nnt::fs::util::AllocateBuffer(headerSize);
    nnt::fs::util::FillBufferWith32BitCount(header.get(), headerSize, 0);

    auto body = nnt::fs::util::AllocateBuffer(bodySize);
    nnt::fs::util::FillBufferWith32BitCount(body.get(), bodySize, 0);

    // 書き込み
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::WriteSaveDataThumbnailFile(GetSaveDataOwnerId(), userId1, header.get(), headerSize, body.get(), bodySize));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadSaveDataThumbnailFile(GetSaveDataOwnerId(), userId1, header.get(), headerSize, body.get(), bodySize));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadSaveDataThumbnailFileHeader(GetSaveDataOwnerId(), userId1, header.get(), headerSize));

    // 破壊・検知、破壊検知時は0埋めが返る
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CorruptSaveDataThumbnailFileForDebug(GetSaveDataOwnerId(), userId1));
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultDataCorrupted, nn::fs::ReadSaveDataThumbnailFileHeader(GetSaveDataOwnerId(), userId1, header.get(), headerSize));
    EXPECT_TRUE(nnt::fs::util::IsFilledWithValue(header.get(), headerSize, 0));
    nnt::fs::util::FillBufferWith32BitCount(header.get(), headerSize, 0);
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultDataCorrupted, nn::fs::ReadSaveDataThumbnailFile(GetSaveDataOwnerId(), userId1, header.get(), headerSize, body.get(), bodySize));
    EXPECT_TRUE(nnt::fs::util::IsFilledWithValue(header.get(), headerSize, 0));
    EXPECT_TRUE(nnt::fs::util::IsFilledWithValue(body.get(), bodySize, 0));

    // 再書き込み
    nnt::fs::util::FillBufferWith32BitCount(header.get(), headerSize, 0);
    nnt::fs::util::FillBufferWith32BitCount(body.get(), bodySize, 0);

    NNT_EXPECT_RESULT_SUCCESS(nn::fs::WriteSaveDataThumbnailFile(GetSaveDataOwnerId(), userId1, header.get(), headerSize, body.get(), bodySize));
    nnt::fs::util::InvalidateVariable(header.get(), static_cast<int>(headerSize));
    nnt::fs::util::InvalidateVariable(body.get(), static_cast<int>(bodySize));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadSaveDataThumbnailFile(GetSaveDataOwnerId(), userId1, header.get(), headerSize, body.get(), bodySize));
    EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount(header.get(), headerSize, 0));
    EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount(body.get(), bodySize, 0));

    DeleteUserSaveData(userId1);
}

TEST(SaveDataThumbnail, RawReadWrite)
{
    nn::fs::UserId userId1 = { { 0x00ull, 0x01ull } };
    DeleteUserSaveData(userId1);

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(ApplicationId, userId1, SaveDataOwnerId, SaveDataSize, SaveDataJournalSize, 0));

    std::unique_ptr<nn::fs::fsa::IFile> file;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenSaveDataThumbnailFile(&file, ApplicationId.value, userId1));

    // get size
    int64_t size;
    NNT_EXPECT_RESULT_SUCCESS(file->GetSize(&size));
    EXPECT_EQ(nn::fs::detail::ThumbnailFileSize, size); // 256KB + Hash * 3

    auto buffer = nnt::fs::util::AllocateBuffer(static_cast<size_t>(size));
    nnt::fs::util::FillBufferWith32BitCount(buffer.get(), static_cast<size_t>(size), 0);

    // write, read, verify
    NNT_EXPECT_RESULT_SUCCESS(file->Write(0, buffer.get(), static_cast<size_t>(size), nn::fs::WriteOption()));
    nnt::fs::util::InvalidateVariable(buffer.get(), static_cast<int>(size));

    size_t readSize;
    NNT_EXPECT_RESULT_SUCCESS(file->Read(&readSize, 0, buffer.get(), static_cast<size_t>(size), nn::fs::ReadOption()));
    EXPECT_EQ(size, readSize);
    EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount(buffer.get(), static_cast<size_t>(size), 0));

    file.reset();

    DeleteUserSaveData(userId1);
}

// TODO
// TEST(SaveDataThumbnail, FileDataCorrupted)
// {
// }

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nn::fs::SetEnabledAutoAbort(false);
    nnt::fs::util::ResetAllocateCount();

    auto result = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(result);
}
