﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForApplications.h>
#include <nn/crypto/crypto_HmacSha256Generator.h>
#include <nn/fs/fs_SaveDataManagementPrivate.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fssystem/save/fs_SaveDataFileSystemCore.h>
#include <nn/fssystem/save/fs_JournalIntegritySaveDataStorage.h>
#include <nn/fssystem/save/fs_JournalIntegritySaveDataFileSystem.h>

#if defined(NN_BUILD_CONFIG_OS_WIN)
    #include <nn/fs/fs_Bis.h>
#endif // defined(NN_BUILD_CONFIG_OS_WIN)

#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/gtest/gtest.h>

namespace
{
    class SaveDataDumpRestore : public ::testing::Test
    {
    public:
        SaveDataDumpRestore() NN_NOEXCEPT
            : m_Mt(nnt::fs::util::GetRandomSeed())
        {
        }

        virtual ~SaveDataDumpRestore() NN_NOEXCEPT NN_OVERRIDE {}

        static void SetUpTestCase() NN_NOEXCEPT
        {
#if defined(NN_BUILD_CONFIG_SPEC_NX)
            nn::account::Uid uid;

            // アカウントライブラリを初期化
            nn::account::Initialize();

            nn::Result result;
            uid = nn::account::InvalidUid;
            int userCount = 0;
            result = nn::account::ListAllUsers(&userCount, &uid, 1);
            NN_ASSERT(result.IsSuccess() && userCount > 0);

            g_UserId._data[0] = uid._data[0];
            g_UserId._data[1] = uid._data[1];
#endif // defined(NN_BUILD_CONFIG_SPEC_NX)
        }

    protected:
        static const nn::fs::SystemSaveDataId TestSaveDataIds[2];
        static const size_t BufferSize = 512 * 1024 * 1024;
        static const int64_t FileSizeWithMulipleFiles = 256 * 1024;
        static const int64_t MarginSize = 16 * 1024;
        static const int64_t SaveDataDataSizeLarge = 1 * 1024 * 1024;
        static const int64_t SaveDataDataSizeSmall = 256 * 1024;

        static const uint8_t MarkerOnMaximumFile[4];

        static const int64_t HashSize = nn::crypto::Sha256Generator::HashSize;
        static const size_t SaveDataMetaIntegrityBlockSize = 16 * 1024;

        NN_STATIC_ASSERT(FileSizeWithMulipleFiles <= BufferSize);

    protected:
        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            DeleteSaveData();
        }

        void CreateSaveData(int64_t saveDataDataSize, bool isSystem) NN_NOEXCEPT;
        void DeleteSaveData() NN_NOEXCEPT;
        bool GetSaveDataId(nn::fs::SaveDataId* pOutId, nn::ncm::ApplicationId applicationId) NN_NOEXCEPT;
        void PrepareSaveDataWithMulipleFiles(bool isSystem) NN_NOEXCEPT;
        void PrepareSaveDataWithMaximumFile(int64_t entryDataSize) NN_NOEXCEPT;
        void ModifySaveDataWithMulipleFiles() NN_NOEXCEPT;
        void CreateFileWithMulipleFiles() NN_NOEXCEPT;
        void DeleteFileWithMulipleFiles() NN_NOEXCEPT;
        void ShuffleSaveDataWithMultipleFiles() NN_NOEXCEPT;
        void CreateFileWith8BitCount(const char* name, int64_t size) NN_NOEXCEPT;
        void CreateFileWithRandomValue(const char* name, int64_t size, bool isSystem) NN_NOEXCEPT;
        void DeleteFile(const char* name) NN_NOEXCEPT;
        void DumpAndRecovery(bool isZeroFilled, bool isSystem) NN_NOEXCEPT;
        void CheckSaveDataWithMulipleFiles(bool isSilent, bool isSystem) NN_NOEXCEPT;
        static bool CheckBufferWithMaximumFile(const char* buffer, size_t size) NN_NOEXCEPT;
        void CheckSaveDataWithMaximumFile() NN_NOEXCEPT;
        void CheckZeroFill() NN_NOEXCEPT;
        void CheckL1Hash(int data, int seed, bool isSystem) NN_NOEXCEPT;
        void DuplicateSaveData(int64_t saveDataDataSize) NN_NOEXCEPT;
        void InternalStorageCommitCheck(int64_t saveDataDataSize) NN_NOEXCEPT;
        void ReadExtraDara(int saveData) NN_NOEXCEPT;

        static char* GetBuffer() NN_NOEXCEPT
        {
            return g_Buffer;
        }

        int64_t GetFileCountWithMulipleFiles() const NN_NOEXCEPT
        {
            return static_cast<int>(m_SaveDataDataSize / (FileSizeWithMulipleFiles + MarginSize));
        }

        nn::fs::UserId GetUserId() const NN_NOEXCEPT
        {
            return g_UserId;
        }

        nn::fs::SaveDataId GetSaveDataId(int index) const NN_NOEXCEPT
        {
            return g_TestApplicationSaveDataIds[index];
        }

    private:
        void CheckSaveDataEntries(bool isSilent, bool isSystem) NN_NOEXCEPT;
        void GenerateExistingFilePathWithMulipleFiles(const char* mountName, char* path, size_t size) NN_NOEXCEPT;
        void GenerateNotExistingFilePathWithMulipleFiles(const char* mountName, char* path, size_t size) NN_NOEXCEPT;

    private:
        static char g_Buffer[BufferSize];
        int64_t m_SaveDataDataSize;
        std::mt19937 m_Mt;
        int64_t m_FileCountWithMulipleFiles;
        static nn::fs::UserId g_UserId;
        static nn::fs::SaveDataId g_TestApplicationSaveDataIds[2];
    };

    nn::fs::UserId SaveDataDumpRestore::g_UserId;
    nn::fs::SaveDataId SaveDataDumpRestore::g_TestApplicationSaveDataIds[2];

    const nn::ncm::ApplicationId ApplicationIds[2] =
    {
        {0x0005000c10000000},
        {0x0005000c10000001},
    };
    const int64_t OwnerIds[2] =
    {
        0x0005000c10000000,
        0x0005000c10000001,
    };

    NN_DEFINE_STATIC_CONSTANT(const size_t SaveDataDumpRestore::BufferSize);
    NN_DEFINE_STATIC_CONSTANT(const int64_t SaveDataDumpRestore::SaveDataDataSizeLarge);
    NN_DEFINE_STATIC_CONSTANT(const int64_t SaveDataDumpRestore::SaveDataDataSizeSmall);

    const nn::fs::SystemSaveDataId SaveDataDumpRestore::TestSaveDataIds[2] =
    {
        0x8000000000001000,
        0x8000000000001001,
    };

    const uint8_t SaveDataDumpRestore::MarkerOnMaximumFile[4] = { 0xFE, 0xAE, 0x0E, 0xDC };

    char SaveDataDumpRestore::g_Buffer[SaveDataDumpRestore::BufferSize];

    int g_DumpAndRecoveryHeavyTryCount = 1;
    int64_t g_SaveDataDataSizeHeavy = 1 * 1024 * 1024;

    void SaveDataDumpRestore::CreateSaveData(int64_t saveDataDataSize, bool isSystem) NN_NOEXCEPT
    {
        const auto SaveDataJournalSize = saveDataDataSize;

        if(isSystem)
        {
            for( auto id : TestSaveDataIds )
            {
                (void)nn::fs::DeleteSaveData(nn::fs::SaveDataSpaceId::System, id);
                NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSystemSaveData(
                    nn::fs::SaveDataSpaceId::System,
                    id,
                    nnt::fs::util::UserSaveDataApplicationId,
                    saveDataDataSize,
                    SaveDataJournalSize,
                    0
                ));
            }
        }
        else
        {
            for(int i = 0; i < 2; ++i)
            {
                nn::fs::SaveDataId id;
                if(GetSaveDataId(&id, ApplicationIds[i]))
                {
                    nn::fs::DeleteSaveData(id);
                }
                NNT_ASSERT_RESULT_SUCCESS(
                    nn::fs::CreateSaveData(ApplicationIds[i], GetUserId(), OwnerIds[0], saveDataDataSize, SaveDataJournalSize, 0));
                GetSaveDataId(&g_TestApplicationSaveDataIds[i], ApplicationIds[i]);
            }
        }

        m_SaveDataDataSize = saveDataDataSize;
    }

    void SaveDataDumpRestore::DeleteSaveData() NN_NOEXCEPT
    {
        for( auto id : TestSaveDataIds )
        {
            (void)nn::fs::DeleteSaveData(nn::fs::SaveDataSpaceId::System, id);
        }
    }

    bool SaveDataDumpRestore::GetSaveDataId(nn::fs::SaveDataId* pOutId, nn::ncm::ApplicationId applicationId) NN_NOEXCEPT
    {
        while(true)
        {
            std::unique_ptr<nn::fs::SaveDataIterator> iter;
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::OpenSaveDataIterator(&iter, nn::fs::SaveDataSpaceId::User));

            int64_t count = 0;
            nn::fs::SaveDataInfo info;
            while(true)
            {
                NNT_EXPECT_RESULT_SUCCESS(iter->ReadSaveDataInfo(&count, &info, 1));
                if (count == 0)
                {
                    return false;
                }
                if (info.applicationId.value == applicationId.value)
                {
                    *pOutId = info.saveDataId;
                    return true;
                }
            }
        }
    }

    void SaveDataDumpRestore::PrepareSaveDataWithMulipleFiles(bool isSystem) NN_NOEXCEPT
    {
        ASSERT_LT(0, m_SaveDataDataSize);
        ASSERT_GE(std::max(SaveDataDataSizeLarge, g_SaveDataDataSizeHeavy), m_SaveDataDataSize);

        static const char* MountName = "save";
        if(isSystem)
        {
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fs::MountSystemSaveData(MountName, nn::fs::SaveDataSpaceId::System, TestSaveDataIds[0])
            );
        }
        else
        {
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fs::MountSaveData(MountName, ApplicationIds[0], GetUserId())
            );
        }
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(MountName);
        };

        m_FileCountWithMulipleFiles = GetFileCountWithMulipleFiles();

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateDirectory("save:/test"));
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateDirectory("save:/test/test"));
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateDirectory("save:/test/test/test"));
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile("save:/test/test/test/test.bin", FileSizeWithMulipleFiles));
        for( int i = 0; i < m_FileCountWithMulipleFiles - 1; ++i )
        {
            char path[256];
            nn::util::SNPrintf(path, sizeof(path), "%s:/test/test/test%d.bin", MountName, i);
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile(path, FileSizeWithMulipleFiles));

            {
                nn::fs::FileHandle file;
                NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&file, path, nn::fs::OpenMode_Write));
                NN_UTIL_SCOPE_EXIT
                {
                    nn::fs::CloseFile(file);
                };
                nnt::fs::util::FillBufferWithRandomValue(GetBuffer(), FileSizeWithMulipleFiles);
                NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteFile(file, 0, GetBuffer(), FileSizeWithMulipleFiles, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
            }
        }
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CommitSaveData(MountName));
    }

    void SaveDataDumpRestore::PrepareSaveDataWithMaximumFile(int64_t entryDataSize) NN_NOEXCEPT
    {
        ASSERT_LT(0, m_SaveDataDataSize);
        ASSERT_GE(std::max(SaveDataDataSizeLarge, g_SaveDataDataSizeHeavy), m_SaveDataDataSize);

        static const char* MountName = "save";
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fs::MountSystemSaveData(MountName, nn::fs::SaveDataSpaceId::System, TestSaveDataIds[0])
        );
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(MountName);
        };

        const auto FileSize = m_SaveDataDataSize - entryDataSize;
        ASSERT_LE(static_cast<size_t>(FileSize), BufferSize);

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile("save:/f", FileSize));

        {
            nn::fs::FileHandle file;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&file, "save:/f", nn::fs::OpenMode_Write));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(file);
            };
            std::memcpy(GetBuffer(), MarkerOnMaximumFile, sizeof(MarkerOnMaximumFile));
            nnt::fs::util::FillBufferWith32BitCount(GetBuffer() + sizeof(MarkerOnMaximumFile), static_cast<size_t>(FileSize) - sizeof(MarkerOnMaximumFile), 0);
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteFile(file, 0, GetBuffer(), static_cast<size_t>(FileSize), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
        }

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CommitSaveData(MountName));
    }

    void SaveDataDumpRestore::ModifySaveDataWithMulipleFiles() NN_NOEXCEPT
    {
        static const char* MountName = "save";
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fs::MountSystemSaveData(MountName, nn::fs::SaveDataSpaceId::System, TestSaveDataIds[0])
        );
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(MountName);
        };

        char path[256];
        GenerateExistingFilePathWithMulipleFiles(MountName, path, sizeof(path));

        {
            nn::fs::FileHandle file;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&file, path, nn::fs::OpenMode_Write));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(file);
            };

            int64_t fileSize;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, file));

            if( fileSize > 0 )
            {
                auto offset = std::uniform_int_distribution<int64_t>(0, fileSize - 1)(m_Mt);
                auto size = std::uniform_int_distribution<int64_t>(0, fileSize - offset)(m_Mt);

                nnt::fs::util::FillBufferWithRandomValue(GetBuffer(), static_cast<size_t>(size));
                NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteFile(file, offset, GetBuffer(), static_cast<size_t>(size), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
            }
        }

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CommitSaveData(MountName));
    }

    void SaveDataDumpRestore::CreateFileWithMulipleFiles() NN_NOEXCEPT
    {
        if( m_FileCountWithMulipleFiles == GetFileCountWithMulipleFiles() )
        {
            return;
        }

        static const char* MountName = "save";
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fs::MountSystemSaveData(MountName, nn::fs::SaveDataSpaceId::System, TestSaveDataIds[0])
        );
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(MountName);
        };

        char path[256];
        GenerateNotExistingFilePathWithMulipleFiles(MountName, path, sizeof(path));

        int64_t freeSize;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFreeSpaceSize(&freeSize, "save:/"));

        int64_t fileSizeMax = freeSize;
        for( ; ; )
        {
            auto result = nn::fs::CreateFile(path, fileSizeMax);
            if( result.IsSuccess() )
            {
                break;
            }
            else if( nn::fs::ResultUsableSpaceNotEnough::Includes(result) )
            {
                fileSizeMax -= 16 * 1024;
                ASSERT_LE(0, fileSizeMax);
            }
            else
            {
                NNT_ASSERT_RESULT_SUCCESS(result);
            }
        }
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteFile(path));

        fileSizeMax = std::min(fileSizeMax, FileSizeWithMulipleFiles * 10);
        auto fileSize = std::uniform_int_distribution<int64_t>(0, fileSizeMax)(m_Mt);
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile(path, fileSize));
        ++m_FileCountWithMulipleFiles;

        {
            nn::fs::FileHandle file;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&file, path, nn::fs::OpenMode_Write));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(file);
            };
            nnt::fs::util::FillBufferWithRandomValue(GetBuffer(), static_cast<size_t>(fileSize));
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteFile(file, 0, GetBuffer(), static_cast<size_t>(fileSize), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
        }

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CommitSaveData(MountName));
    }

    void SaveDataDumpRestore::DeleteFileWithMulipleFiles() NN_NOEXCEPT
    {
        if( m_FileCountWithMulipleFiles == 2 )
        {
            return;
        }

        static const char* MountName = "save";
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fs::MountSystemSaveData(MountName, nn::fs::SaveDataSpaceId::System, TestSaveDataIds[0])
        );
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(MountName);
        };

        char path[256];
        GenerateExistingFilePathWithMulipleFiles(MountName, path, sizeof(path));
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteFile(path));
        --m_FileCountWithMulipleFiles;

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CommitSaveData(MountName));
    }

    void SaveDataDumpRestore::ShuffleSaveDataWithMultipleFiles() NN_NOEXCEPT
    {
        enum class Operation
        {
            Modify,
            Create,
            Delete,
        };

        auto value = std::uniform_int_distribution<int>(0, 2)(m_Mt);
        Operation operation = static_cast<Operation>(value);

        const auto operationCount = operation == Operation::Modify ? 10 : 100;
        for( int j = 0; j < operationCount; ++j )
        {
            switch( operation )
            {
            case Operation::Modify:
                NNT_FS_ASSERT_NO_FATAL_FAILURE(ModifySaveDataWithMulipleFiles());
                break;

            case Operation::Create:
                NNT_FS_ASSERT_NO_FATAL_FAILURE(CreateFileWithMulipleFiles());
                break;

            case Operation::Delete:
                NNT_FS_ASSERT_NO_FATAL_FAILURE(DeleteFileWithMulipleFiles());
                break;

            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
    }

    void SaveDataDumpRestore::CreateFileWith8BitCount(const char* name, int64_t size) NN_NOEXCEPT
    {
        ASSERT_LT(0, m_SaveDataDataSize);
        ASSERT_GE(m_SaveDataDataSize, size);

        static const char* MountName = "save";
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fs::MountSystemSaveData(MountName, nn::fs::SaveDataSpaceId::System, TestSaveDataIds[0])
        );
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(MountName);
        };

        char path[256];
        nn::util::SNPrintf(path, sizeof(path), "%s:/%s", MountName, name);
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile(path, size));

        {
            nn::fs::FileHandle file;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&file, path, nn::fs::OpenMode_Write));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(file);
            };
            nnt::fs::util::FillBufferWith8BitCount(GetBuffer(), static_cast<size_t>(size), 0);
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteFile(file, 0, GetBuffer(), static_cast<size_t>(size), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
        }
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CommitSaveData(MountName));
    }

    void SaveDataDumpRestore::CreateFileWithRandomValue(const char* name, int64_t size, bool isSystem) NN_NOEXCEPT
    {
        ASSERT_LT(0, m_SaveDataDataSize);
        ASSERT_GE(m_SaveDataDataSize, size);

        static const char* MountName = "save";
        if(isSystem)
        {
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fs::MountSystemSaveData(MountName, nn::fs::SaveDataSpaceId::System, TestSaveDataIds[0])
            );
        }
        else
        {
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fs::MountSaveData(MountName, ApplicationIds[0], GetUserId())
            );
        }
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(MountName);
        };

        char path[256];
        nn::util::SNPrintf(path, sizeof(path), "%s:/%s", MountName, name);
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile(path, size));

        {
            nn::fs::FileHandle file;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&file, path, nn::fs::OpenMode_Write));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(file);
            };
            nnt::fs::util::FillBufferWithRandomValue(GetBuffer(), static_cast<size_t>(size));
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteFile(file, 0, GetBuffer(), static_cast<size_t>(size), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
        }
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CommitSaveData(MountName));
    }

    void SaveDataDumpRestore::DeleteFile(const char* name) NN_NOEXCEPT
    {
        static const char* MountName = "save";
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fs::MountSystemSaveData(MountName, nn::fs::SaveDataSpaceId::System, TestSaveDataIds[0])
        );
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(MountName);
        };

        char path[256];
        nn::util::SNPrintf(path, sizeof(path), "%s:/%s", MountName, name);
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteFile(path));
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CommitSaveData(MountName));
    }

    void SaveDataDumpRestore::DumpAndRecovery(bool isZeroFilled, bool isSystem) NN_NOEXCEPT
    {
        static const char* MountNames[2] = { "test1", "test2" };
        static const char* RootPath[2] = { "test1:/", "test2:/" };

        if(isSystem)
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataInternalStorage(MountNames[0], nn::fs::SaveDataSpaceId::System, TestSaveDataIds[0]));
        }
        else
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataInternalStorage(MountNames[0], nn::fs::SaveDataSpaceId::User, g_TestApplicationSaveDataIds[0]));
        }
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(MountNames[0]);
        };
        if(isSystem)
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataInternalStorage(MountNames[1], nn::fs::SaveDataSpaceId::System, TestSaveDataIds[1]));
        }
        else
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataInternalStorage(MountNames[1], nn::fs::SaveDataSpaceId::User, g_TestApplicationSaveDataIds[1]));
        }
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(MountNames[1]);
        };

        static const char* Paths[] =
        {
            nn::fssystem::save::SaveDataFileSystemCore::InternalStorageFileNameAllocationTableControlArea,
            nn::fssystem::save::SaveDataFileSystemCore::InternalStorageFileNameAllocationTableMeta,
            nn::fssystem::save::SaveDataFileSystemCore::InternalStorageFileNameAllocationTableData,
        };

        static const char* PathsZeroFill[] =
        {
            nn::fssystem::save::SaveDataFileSystemCore::InternalStorageFileNameAllocationTableControlArea,
            nn::fssystem::save::SaveDataFileSystemCore::InternalStorageFileNameAllocationTableMeta,
            nn::fssystem::save::SaveDataFileSystemCore::InternalStorageFileNameAllocationTableDataWithZeroFree,
        };

        nnt::fs::util::String filePath;
        for( auto path : isZeroFilled ? PathsZeroFill : Paths )
        {
            nn::fs::FileHandle handles[2];
            filePath = RootPath[0];
            filePath += path;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&handles[0], filePath.c_str(), nn::fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(handles[0]);
            };

            filePath = RootPath[1];
            filePath += path;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&handles[1], filePath.c_str(), nn::fs::OpenMode_Write));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(handles[1]);
            };

            int64_t size;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&size, handles[0]));
            ASSERT_LT(size, static_cast<int64_t>(BufferSize));

            size_t sizeRead;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(&sizeRead, handles[0], 0, GetBuffer(), static_cast<size_t>(size), nn::fs::ReadOption::MakeValue(0)));
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteFile(handles[1], 0, GetBuffer(), static_cast<size_t>(size), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
        }

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CommitSaveData(MountNames[1]));
    }

    void SaveDataDumpRestore::CheckSaveDataWithMulipleFiles(bool isSilent, bool isSystem) NN_NOEXCEPT
    {
        NNT_FS_ASSERT_NO_FATAL_FAILURE(CheckSaveDataEntries(isSilent, isSystem));
    }

    bool SaveDataDumpRestore::CheckBufferWithMaximumFile(const char* buffer, size_t size) NN_NOEXCEPT
    {
        return (std::memcmp(MarkerOnMaximumFile, buffer, sizeof(MarkerOnMaximumFile)) == 0)
            && nnt::fs::util::IsFilledWith32BitCount(buffer + sizeof(MarkerOnMaximumFile), static_cast<size_t>(size) - sizeof(MarkerOnMaximumFile), 0);
    }

    void SaveDataDumpRestore::CheckSaveDataWithMaximumFile() NN_NOEXCEPT
    {
        static const char* MountName = "save";
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fs::MountSystemSaveData(MountName, nn::fs::SaveDataSpaceId::System, TestSaveDataIds[0])
        );
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(MountName);
        };

        {
            nn::fs::FileHandle file;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&file, "save:/f", nn::fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(file);
            };

            int64_t size;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&size, file));

            size_t readSize;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(&readSize, file, 0, GetBuffer(), static_cast<size_t>(size), nn::fs::ReadOption::MakeValue(0)));
            ASSERT_TRUE(CheckBufferWithMaximumFile(GetBuffer(), readSize));
        }
    }

    void SaveDataDumpRestore::CheckZeroFill() NN_NOEXCEPT
    {
        static const char* MountNames = "test";

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataInternalStorage(MountNames, nn::fs::SaveDataSpaceId::System, TestSaveDataIds[0]));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(MountNames);
        };

        {
            char path[nn::fs::EntryNameLengthMax];
            nn::util::SNPrintf(path, sizeof(path), "test:/%s", nn::fssystem::save::SaveDataFileSystemCore::InternalStorageFileNameAllocationTableDataWithZeroFree);

            nn::fs::FileHandle handles;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&handles, path, nn::fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(handles);
            };

            int64_t size;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&size, handles));
            ASSERT_LT(size, static_cast<int64_t>(BufferSize));

            size_t sizeRead;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(&sizeRead, handles, 0, GetBuffer(), static_cast<size_t>(size), nn::fs::ReadOption::MakeValue(0)));
            ASSERT_EQ(size, static_cast<int64_t>(sizeRead));

            for(int i = 0; i < static_cast<int>(sizeRead) - 256; ++i)
            {
                if(GetBuffer()[i] == 0x00 && GetBuffer()[i + 1] == 0x01)
                {
                    ASSERT_FALSE(nnt::fs::util::IsFilledWith8BitCount(&GetBuffer()[i], 256, 0));
                }
            }
        }

        {
            char path[nn::fs::EntryNameLengthMax];
            nn::util::SNPrintf(path, sizeof(path), "test:/%s", nn::fssystem::save::JournalIntegritySaveDataStorage::InternalStorageFileNameIntegrityWithZeroFree);

            nn::fs::FileHandle handles;
            NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidOperationForOpenMode, nn::fs::OpenFile(&handles, path, nn::fs::OpenMode_Write));
            NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidOperationForOpenMode, nn::fs::OpenFile(&handles, path, nn::fs::OpenMode_Read | nn::fs::OpenMode_AllowAppend));
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&handles, path, nn::fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(handles);
            };

            int64_t size;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&size, handles));
            ASSERT_LT(size, static_cast<int64_t>(BufferSize));

            size_t sizeRead;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(&sizeRead, handles, 0, &GetBuffer()[SaveDataDataSizeLarge], static_cast<size_t>(size), nn::fs::ReadOption::MakeValue(0)));
            ASSERT_EQ(size, static_cast<int64_t>(sizeRead));

            for(auto i = 0; i < sizeRead / HashSize; ++i)
            {
                if(nnt::fs::util::IsFilledWithValue(&GetBuffer()[SaveDataDataSizeLarge + i * HashSize], HashSize, 0))
                {
                    ASSERT_TRUE(nnt::fs::util::IsFilledWithValue(&GetBuffer()[i * SaveDataMetaIntegrityBlockSize], SaveDataMetaIntegrityBlockSize, 0));
                }
            }
        }
    }

    void SaveDataDumpRestore::CheckL1Hash(int data, int seed, bool isSystem) NN_NOEXCEPT
    {
        static const char* MountNames[] = { "test1", "test2" };

        if(isSystem)
        {
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fs::MountSaveDataInternalStorage(MountNames[0], nn::fs::SaveDataSpaceId::System, TestSaveDataIds[0])
            );
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fs::MountSaveDataInternalStorage(MountNames[1], nn::fs::SaveDataSpaceId::System, TestSaveDataIds[1])
            );
        }
        else
        {
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fs::MountSaveDataInternalStorage(MountNames[0], nn::fs::SaveDataSpaceId::User, GetSaveDataId(0))
            );
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fs::MountSaveDataInternalStorage(MountNames[1], nn::fs::SaveDataSpaceId::User, GetSaveDataId(1))
            );
        }
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(MountNames[0]);
            nn::fs::Unmount(MountNames[1]);
        };

        int64_t dataSize;
        {
            char path[nn::fs::EntryNameLengthMax];
            nn::util::SNPrintf(path, sizeof(path), "%s:/%s", MountNames[data], nn::fssystem::save::SaveDataFileSystemCore::InternalStorageFileNameAllocationTableDataWithZeroFree);

            nn::fs::FileHandle handles;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&handles, path, nn::fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(handles);
            };

            NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&dataSize, handles));
            ASSERT_LT(dataSize, static_cast<int64_t>(BufferSize));

            size_t sizeRead;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(&sizeRead, handles, 0, GetBuffer(), static_cast<size_t>(dataSize), nn::fs::ReadOption::MakeValue(0)));
            ASSERT_EQ(dataSize, static_cast<int64_t>(sizeRead));
        }

        int64_t hashSize;
        {
            char path[nn::fs::EntryNameLengthMax];
            nn::util::SNPrintf(path, sizeof(path), "%s:/%s", MountNames[data], nn::fssystem::save::JournalIntegritySaveDataStorage::InternalStorageFileNameIntegrityWithZeroFree);

            nn::fs::FileHandle handles;
            NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidOperationForOpenMode, nn::fs::OpenFile(&handles, path, nn::fs::OpenMode_Write));
            NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidOperationForOpenMode, nn::fs::OpenFile(&handles, path, nn::fs::OpenMode_Read | nn::fs::OpenMode_AllowAppend));
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&handles, path, nn::fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(handles);
            };

            NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&hashSize, handles));
            ASSERT_LT(hashSize, static_cast<int64_t>(BufferSize));

            size_t sizeRead;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(&sizeRead, handles, 0, &GetBuffer()[dataSize], static_cast<size_t>(hashSize), nn::fs::ReadOption::MakeValue(0)));
            ASSERT_EQ(hashSize, static_cast<int64_t>(sizeRead));
        }


        {
            char path[nn::fs::EntryNameLengthMax];
            nn::util::SNPrintf(path, sizeof(path), "%s:/%s", MountNames[seed], nn::fssystem::save::JournalIntegritySaveDataFileSystem::InternalStorageFileNameIntegritySeed);

            nn::fs::FileHandle handles;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&handles, path, nn::fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(handles);
            };

            int64_t seedSize;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&seedSize, handles));
            ASSERT_EQ(seedSize, 32);

            nn::fs::HashSalt salt;
            size_t sizeRead;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(&sizeRead, handles, 0, &salt, sizeof(salt), nn::fs::ReadOption::MakeValue(0)));
            ASSERT_EQ(seedSize, static_cast<int64_t>(sizeRead));

            const char L3Key[] = "HierarchicalIntegrityVerificationStorage::L3";
            nn::fs::HashSalt mac;
            nn::crypto::GenerateHmacSha256Mac(
                mac.value, sizeof(mac),
                salt.value, sizeof(salt),
                L3Key, sizeof(L3Key)
            );

            for(int i = 0; i < static_cast<int>(dataSize / SaveDataMetaIntegrityBlockSize); ++i)
            {
                nn::crypto::Sha256Generator generator;
                generator.Initialize();
                generator.Update(mac.value, sizeof(mac));
                generator.Update(&GetBuffer()[i * SaveDataMetaIntegrityBlockSize], SaveDataMetaIntegrityBlockSize);
                nn::fssystem::save::IntegrityVerificationStorage::BlockHash blockHash;
                generator.GetHash(&blockHash, sizeof(blockHash));
                blockHash.hash[31] |= 0x80;

                ASSERT_TRUE(std::memcmp(blockHash.hash, &GetBuffer()[dataSize + i * HashSize], sizeof(blockHash)) == 0);
            }
        }
    }

    void SaveDataDumpRestore::DuplicateSaveData(int64_t saveDataDataSize) NN_NOEXCEPT
    {
        const auto SaveDataJournalSize = saveDataDataSize;
        static const char* MountNames[] = { "test1", "test2" };

        {
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataInternalStorage(MountNames[0], nn::fs::SaveDataSpaceId::User, GetSaveDataId(0)));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::Unmount(MountNames[0]);
            };

            nn::fs::HashSalt salt;
            {
                char path[nn::fs::EntryNameLengthMax];
                nn::util::SNPrintf(path, sizeof(path), "%s:/%s", MountNames[0], nn::fssystem::save::JournalIntegritySaveDataFileSystem::InternalStorageFileNameIntegritySeed);

                nn::fs::FileHandle handles;
                NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&handles, path, nn::fs::OpenMode_Read));
                NN_UTIL_SCOPE_EXIT
                {
                    nn::fs::CloseFile(handles);
                };

                int64_t seedSize;
                NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&seedSize, handles));
                ASSERT_EQ(seedSize, 32);

                size_t sizeRead;
                NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(&sizeRead, handles, 0, &salt, sizeof(salt), nn::fs::ReadOption::MakeValue(0)));
                ASSERT_EQ(seedSize, static_cast<int64_t>(sizeRead));
            }

            nn::fs::DeleteSaveData(GetSaveDataId(1));
            nn::fs::CreateSaveData(ApplicationIds[1], GetUserId(), OwnerIds[0], saveDataDataSize, SaveDataJournalSize, salt, 0);
            GetSaveDataId(&g_TestApplicationSaveDataIds[1], ApplicationIds[1]);
        }

        DumpAndRecovery(false, false);
    }

    void SaveDataDumpRestore::InternalStorageCommitCheck(int64_t saveDataDataSize) NN_NOEXCEPT
    {
        const auto SaveDataJournalSize = saveDataDataSize;
        static const char* MountNames[] = { "test1", "test2" };
        char path[nn::fs::EntryNameLengthMax];

        nn::fs::SaveDataId id;
        if(GetSaveDataId(&id, ApplicationIds[0]))
        {
            nn::fs::DeleteSaveData(id);
        }
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fs::CreateSaveData(ApplicationIds[0], GetUserId(), OwnerIds[0], saveDataDataSize, SaveDataJournalSize, 0));
        GetSaveDataId(&g_TestApplicationSaveDataIds[0], ApplicationIds[0]);

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveData(MountNames[0], ApplicationIds[0], GetUserId()));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(MountNames[0]);
        };

        nn::util::SNPrintf(path, sizeof(path), "%s:/test.bin", MountNames[0]);
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile(path, 256));

        {
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataInternalStorage(MountNames[1], nn::fs::SaveDataSpaceId::User, GetSaveDataId(0)));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::Unmount(MountNames[1]);
            };

            nn::util::SNPrintf(path, sizeof(path), "%s:/%s", MountNames[1], nn::fssystem::save::JournalIntegritySaveDataStorage::InternalStorageFileNameIntegrityWithZeroFree);

            {
                nn::fs::FileHandle handles;
                NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&handles, path, nn::fs::OpenMode_Read));
                NN_UTIL_SCOPE_EXIT
                {
                    nn::fs::CloseFile(handles);
                };

                int64_t size;
                NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&size, handles));
                ASSERT_LT(size, static_cast<int64_t>(BufferSize));

                size_t sizeRead;
                NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(&sizeRead, handles, 0, GetBuffer(), static_cast<size_t>(size), nn::fs::ReadOption::MakeValue(0)));
                ASSERT_EQ(size, static_cast<int64_t>(sizeRead));

                for(auto i = 2; i < sizeRead / HashSize; ++i)
                {
                    ASSERT_TRUE(nnt::fs::util::IsFilledWithValue(&GetBuffer()[i * HashSize], HashSize, 0));
                }
            }
        }

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CommitSaveData(MountNames[0]));

        {
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataInternalStorage(MountNames[1], nn::fs::SaveDataSpaceId::User, GetSaveDataId(0)));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::Unmount(MountNames[1]);
            };

            nn::util::SNPrintf(path, sizeof(path), "%s:/%s", MountNames[1], nn::fssystem::save::JournalIntegritySaveDataStorage::InternalStorageFileNameIntegrityWithZeroFree);

            nn::fs::FileHandle handles;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&handles, path, nn::fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(handles);
            };

            int64_t size;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&size, handles));
            ASSERT_LT(size, static_cast<int64_t>(BufferSize));

            size_t sizeRead;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(&sizeRead, handles, 0, GetBuffer(), static_cast<size_t>(size), nn::fs::ReadOption::MakeValue(0)));
            ASSERT_EQ(size, static_cast<int64_t>(sizeRead));

            ASSERT_FALSE(nnt::fs::util::IsFilledWithValue(&GetBuffer()[2 * HashSize], HashSize, 0));
        }
    }

    void SaveDataDumpRestore::GenerateExistingFilePathWithMulipleFiles(const char* mountName, char* path, size_t size) NN_NOEXCEPT
    {
        ASSERT_LT(1, m_FileCountWithMulipleFiles);

        auto generateFileIndex = std::uniform_int_distribution<int64_t>(0, GetFileCountWithMulipleFiles() - 1);

        for( ; ; )
        {
            auto index = generateFileIndex(m_Mt);
            nn::util::SNPrintf(path, size, "%s:/test/test/test%lld.bin", mountName, index);

            nn::fs::DirectoryEntryType type;
            auto result = nn::fs::GetEntryType(&type, path);
            if( result.IsSuccess() )
            {
                break;
            }
        }
    }

    void SaveDataDumpRestore::GenerateNotExistingFilePathWithMulipleFiles(const char* mountName, char* path, size_t size) NN_NOEXCEPT
    {
        ASSERT_LT(m_FileCountWithMulipleFiles, GetFileCountWithMulipleFiles());

        auto generateFileIndex = std::uniform_int_distribution<int64_t>(0, GetFileCountWithMulipleFiles() - 1);

        for( ; ; )
        {
            auto index = generateFileIndex(m_Mt);
            nn::util::SNPrintf(path, size, "%s:/test/test/test%lld.bin", mountName, index);

            nn::fs::DirectoryEntryType type;
            auto result = nn::fs::GetEntryType(&type, path);
            if( result.IsFailure() )
            {
                break;
            }
        }
    }

    void SaveDataDumpRestore::CheckSaveDataEntries(bool isSilent, bool isSystem) NN_NOEXCEPT
    {
        static const char* MountNames[2] = { "save1", "save2" };
        static const char* RootPath[2] = { "save1:/", "save2:/" };

        if(isSystem)
        {
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fs::MountSystemSaveData(MountNames[0], nn::fs::SaveDataSpaceId::System, TestSaveDataIds[0])
            );
        }
        else
        {
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fs::MountSaveData(MountNames[0], ApplicationIds[0], GetUserId())
            );
        }
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(MountNames[0]);
        };
        if(isSystem)
        {
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fs::MountSystemSaveData(MountNames[1], nn::fs::SaveDataSpaceId::System, TestSaveDataIds[1])
            );
        }
        else
        {
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fs::MountSaveData(MountNames[1], ApplicationIds[1], GetUserId())
            );
        }
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(MountNames[1]);
        };

        if( isSilent )
        {
            NNT_FS_ASSERT_NO_FATAL_FAILURE(nnt::fs::util::CompareDirectoryRecursiveSilently(RootPath[0], RootPath[1]));
        }
        else
        {
            NNT_FS_ASSERT_NO_FATAL_FAILURE(nnt::fs::util::CompareDirectoryRecursive(RootPath[0], RootPath[1]));
        }
    }

    void SaveDataDumpRestore::ReadExtraDara(int saveData) NN_NOEXCEPT
    {
        // 拡張データ読み出し
        nn::Bit64 tmpOwnerId = 0;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetSaveDataOwnerId(&tmpOwnerId, GetSaveDataId(saveData)));

        uint32_t tmpFlags = 0;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetSaveDataFlags(&tmpFlags, GetSaveDataId(saveData)));

        nn::time::PosixTime saveDataTimeStamp = { -1 };
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetSaveDataTimeStamp(&saveDataTimeStamp, GetSaveDataId(saveData)));
    }

#if defined(NN_BUILD_CONFIG_OS_WIN)
    void DumpFile(const char* fileName, const char* buffer, size_t size) NN_NOEXCEPT
    {
        char path[256];
        nn::util::SNPrintf(path, sizeof(path), "%s:/%s", nn::fs::GetBisMountName(nn::fs::BisPartitionId::User), fileName);
        (void) nn::fs::DeleteFile(path);
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile(path, static_cast<int64_t>(size)));

        nn::fs::FileHandle file;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&file, path, nn::fs::OpenMode_Write));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(file);
        };

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteFile(file, 0, buffer, size, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
    }
#endif // defined(NN_BUILD_CONFIG_OS_WIN)
}

TEST_F(SaveDataDumpRestore, DumpDirectoryRecursively)
{
    NNT_FS_ASSERT_NO_FATAL_FAILURE(CreateSaveData(SaveDataDataSizeLarge, true));
    NNT_FS_ASSERT_NO_FATAL_FAILURE(PrepareSaveDataWithMulipleFiles(true));

    static const char* MountName = "test";
    static const char* RootPath = "test:/";

    for( auto id : TestSaveDataIds )
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataInternalStorage(MountName, nn::fs::SaveDataSpaceId::System, id));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(MountName);
        };

        NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::DumpDirectoryRecursive(RootPath));
    }
}

TEST_F(SaveDataDumpRestore, DumpAndRecovery)
{
    const bool ZeroFill[] = { false, true };

    NNT_FS_ASSERT_NO_FATAL_FAILURE(CreateSaveData(SaveDataDataSizeLarge, true));
    NNT_FS_ASSERT_NO_FATAL_FAILURE(PrepareSaveDataWithMulipleFiles(true));

    for(auto bZero : ZeroFill)
    {
        NNT_FS_ASSERT_NO_FATAL_FAILURE(DumpAndRecovery(bZero, true));
        NNT_FS_ASSERT_NO_FATAL_FAILURE(CheckSaveDataWithMulipleFiles(false, true));
    }
}

TEST_F(SaveDataDumpRestore, DumpAndRecoveryWithExtension)
{
    const bool ZeroFill[] = { false, true };

    NNT_FS_ASSERT_NO_FATAL_FAILURE(CreateSaveData(SaveDataDataSizeLarge, true));
    NNT_FS_ASSERT_NO_FATAL_FAILURE(PrepareSaveDataWithMulipleFiles(true));

    static const int64_t ExtendedDataSize = SaveDataDataSizeLarge + 1024 * 1024;
    static const int64_t ExtendedJournalSize = ExtendedDataSize;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::ExtendSaveData(nn::fs::SaveDataSpaceId::System, TestSaveDataIds[1], ExtendedDataSize, ExtendedJournalSize));

    for(auto isZeroFilled : ZeroFill)
    {
        NNT_FS_ASSERT_NO_FATAL_FAILURE(DumpAndRecovery(isZeroFilled, true));
        NNT_FS_ASSERT_NO_FATAL_FAILURE(CheckSaveDataWithMulipleFiles(false, true));
    }
}

TEST_F(SaveDataDumpRestore, DumpAndRecoveryHeavy)
{
    const bool ZeroFill[] = { false, true };

    NNT_FS_ASSERT_NO_FATAL_FAILURE(CreateSaveData(g_SaveDataDataSizeHeavy, true));
    NNT_FS_ASSERT_NO_FATAL_FAILURE(PrepareSaveDataWithMulipleFiles(true));

    for(auto isZeroFilled : ZeroFill)
    {
        const auto TryCount = g_DumpAndRecoveryHeavyTryCount;
        for( int i = 0; i < TryCount; ++i )
        {
            {
                static const char* MountName = "save";
                NNT_ASSERT_RESULT_SUCCESS(
                    nn::fs::MountSystemSaveData(MountName, nn::fs::SaveDataSpaceId::System, TestSaveDataIds[0])
                );
                NN_UTIL_SCOPE_EXIT
                {
                    nn::fs::Unmount(MountName);
                };

                int64_t freeSize;
                NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFreeSpaceSize(&freeSize, "save:/"));

                NN_LOG("%d/%d (space used %d%%)\n", i + 1, TryCount, 100 - static_cast<int>(freeSize * 100 / g_SaveDataDataSizeHeavy));
            }

            NNT_FS_ASSERT_NO_FATAL_FAILURE(ShuffleSaveDataWithMultipleFiles());
            NNT_FS_ASSERT_NO_FATAL_FAILURE(DumpAndRecovery(isZeroFilled, true));
            NNT_FS_ASSERT_NO_FATAL_FAILURE(CheckSaveDataWithMulipleFiles(true, true));
        }
    }
}

TEST_F(SaveDataDumpRestore, ZeroFillFile)
{
    NNT_FS_ASSERT_NO_FATAL_FAILURE(CreateSaveData(SaveDataDataSizeLarge, true));
    NNT_FS_ASSERT_NO_FATAL_FAILURE(CreateFileWith8BitCount("test.bin", SaveDataDataSizeLarge - MarginSize * 2));
    NNT_FS_ASSERT_NO_FATAL_FAILURE(DeleteFile("test.bin"));

    NNT_FS_ASSERT_NO_FATAL_FAILURE(CheckZeroFill());
}

TEST_F(SaveDataDumpRestore, L1HashFileSystem)
{
    NNT_FS_ASSERT_NO_FATAL_FAILURE(CreateSaveData(SaveDataDataSizeLarge, true));
    NNT_FS_ASSERT_NO_FATAL_FAILURE(CreateFileWithRandomValue("test.bin", SaveDataDataSizeLarge - MarginSize * 2, true));
    NNT_FS_ASSERT_NO_FATAL_FAILURE(CheckL1Hash(0, 0, true));
}

#if defined(NN_BUILD_CONFIG_SPEC_NX)
TEST_F(SaveDataDumpRestore, L1HashFile)
{
    NNT_FS_ASSERT_NO_FATAL_FAILURE(CreateSaveData(SaveDataDataSizeLarge, false));
    NNT_FS_ASSERT_NO_FATAL_FAILURE(CreateFileWithRandomValue("test.bin", SaveDataDataSizeLarge - MarginSize * 2, false));
    NNT_FS_ASSERT_NO_FATAL_FAILURE(CheckL1Hash(0, 0, false));
}

TEST_F(SaveDataDumpRestore, DuplicateSaveDataL1HashCheck)
{
    NNT_FS_ASSERT_NO_FATAL_FAILURE(CreateSaveData(SaveDataDataSizeLarge, false));
    NNT_FS_ASSERT_NO_FATAL_FAILURE(CreateFileWithRandomValue("test.bin", SaveDataDataSizeLarge - MarginSize * 2, false));
    NNT_FS_ASSERT_NO_FATAL_FAILURE(DuplicateSaveData(SaveDataDataSizeLarge));
    NNT_FS_ASSERT_NO_FATAL_FAILURE(CheckL1Hash(1, 0, false));
}
#endif

#if defined(NN_BUILD_CONFIG_SPEC_NX)
TEST_F(SaveDataDumpRestore, InternalStorageCommitCheck)
{
    NNT_FS_ASSERT_NO_FATAL_FAILURE(InternalStorageCommitCheck(SaveDataDataSizeLarge));
}
#endif

TEST_F(SaveDataDumpRestore, DumpHeavy)
{
    static const int64_t SameDataDataSize = 1024 * 1024;

    // セーブデータのデータ部分以外のサイズ
    static const int64_t EntryDataSize = 2 * 16 * 1024;

    NNT_FS_ASSERT_NO_FATAL_FAILURE(CreateSaveData(SameDataDataSize, true));
    NNT_FS_ASSERT_NO_FATAL_FAILURE(PrepareSaveDataWithMaximumFile(EntryDataSize));
    NNT_FS_ASSERT_NO_FATAL_FAILURE(CheckSaveDataWithMaximumFile());

    auto checkDumpData = [=]() NN_NOEXCEPT
    {
        static const char* MountName = "test";

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataInternalStorage(MountName, nn::fs::SaveDataSpaceId::System, TestSaveDataIds[0]));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(MountName);
        };

        {
            nn::fs::FileHandle file;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&file, "test:/AllocationTableData", nn::fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(file);
            };

            size_t sizeRead;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(&sizeRead, file, EntryDataSize, GetBuffer(), BufferSize, nn::fs::ReadOption::MakeValue(0)));
            ASSERT_EQ(SameDataDataSize - EntryDataSize, sizeRead);
            ASSERT_TRUE(CheckBufferWithMaximumFile(GetBuffer(), sizeRead));
        }
    };
    NNT_FS_ASSERT_NO_FATAL_FAILURE(checkDumpData());

    // 同じ内容で適当に書き換え
    static const auto LoopCount = 20;
    for( auto i = 0; i < LoopCount; ++i )
    {
        static const char* MountName = "save";
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fs::MountSystemSaveData(MountName, nn::fs::SaveDataSpaceId::System, TestSaveDataIds[0])
        );
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(MountName);
        };

        {
            static const size_t Size = 7 * 16 * 1024;
            const int64_t offset = 16 * 1024 * (i + 1);

            nn::fs::FileHandle file;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&file, "save:/f", nn::fs::OpenMode_Read|nn::fs::OpenMode_Write));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(file);
            };
            size_t sizeRead;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(&sizeRead, file, offset, GetBuffer(), Size, nn::fs::ReadOption::MakeValue(0)));
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteFile(file, offset, GetBuffer(), Size, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
        }

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CommitSaveData(MountName));
    }

    // セーブデータ上、ダンプ上では崩れていない
    NNT_FS_ASSERT_NO_FATAL_FAILURE(CheckSaveDataWithMaximumFile());
    NNT_FS_ASSERT_NO_FATAL_FAILURE(checkDumpData());
}

TEST_F(SaveDataDumpRestore, DumpWithCreateAndExtendFile)
{
    static const int64_t FileSizeFirst = 16 * 1024;
    static const int64_t FileSizeSecond = 32 * 1024;
    static const int64_t FileSizeThird = 48 * 1024;
    static const int64_t File2Size = 48 * 1024;
    static const int64_t File3Size = 0;
    static const int64_t File3Count = 171 - 3;

    // セーブデータのデータ部分以外のサイズ
    static const int64_t EntryDataSize = 2 * 16 * 1024;
    static const int64_t EntryData2Size = 16 * 1024;

    NNT_FS_ASSERT_NO_FATAL_FAILURE(CreateSaveData(SaveDataDataSizeSmall, true));

    {
        static const char* MountName = "save";
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fs::MountSystemSaveData(MountName, nn::fs::SaveDataSpaceId::System, TestSaveDataIds[0])
        );
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(MountName);
        };

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile("save:/f1", FileSizeFirst));

        {
            nn::fs::FileHandle file;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&file, "save:/f1", nn::fs::OpenMode_Write));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(file);
            };
            nnt::fs::util::FillBufferWith32BitCount(GetBuffer(), FileSizeFirst, 0);
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteFile(file, 0, GetBuffer(), FileSizeFirst, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
        }

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile("save:/f2", File2Size));
        {
            nn::fs::FileHandle file;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&file, "save:/f2", nn::fs::OpenMode_Write));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(file);
            };
            nnt::fs::util::FillBufferWith32BitCount(GetBuffer(), FileSizeSecond, 0);
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteFile(file, 0, GetBuffer(), File2Size, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
        }

        {
            nn::fs::FileHandle file;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&file, "save:/f1", nn::fs::OpenMode_Write));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(file);
            };
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::SetFileSize(file, FileSizeSecond));
            nnt::fs::util::FillBufferWith32BitCount(GetBuffer(), FileSizeSecond, 0);
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteFile(file, 0, GetBuffer(), FileSizeSecond, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
        }

        {
            char path[256];
            for( int i = 0; i < File3Count; ++i )
            {
                nn::util::SNPrintf(path, sizeof(path), "save:/f%d", 3 + i);
                NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile(path, File3Size));
            }
        }

        {
            nn::fs::FileHandle file;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&file, "save:/f1", nn::fs::OpenMode_Write));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(file);
            };
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::SetFileSize(file, FileSizeThird));
            nnt::fs::util::FillBufferWith32BitCount(GetBuffer(), FileSizeThird, 0);
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteFile(file, 0, GetBuffer(), FileSizeThird, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
        }

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CommitSaveData(MountName));
    }

    {
        static const char* MountName = "test";

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataInternalStorage(MountName, nn::fs::SaveDataSpaceId::System, TestSaveDataIds[0]));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(MountName);
        };

        {
            nn::fs::FileHandle file;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&file, "test:/AllocationTableData", nn::fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(file);
            };

            size_t sizeRead;

            NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(&sizeRead, file, EntryDataSize, GetBuffer(), FileSizeThird, nn::fs::ReadOption::MakeValue(0)));
            ASSERT_FALSE(nnt::fs::util::IsFilledWith32BitCount(GetBuffer(), FileSizeThird, 0));

            {
                static const auto Offset = EntryDataSize + FileSizeFirst + File2Size;
                static const auto Size = FileSizeThird - FileSizeFirst;
                NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(&sizeRead, file, Offset, GetBuffer() + FileSizeFirst, Size, nn::fs::ReadOption::MakeValue(0)));
                ASSERT_FALSE(nnt::fs::util::IsFilledWith32BitCount(GetBuffer(), FileSizeThird, 0));
            }

            {
                static const auto Offset = EntryDataSize + FileSizeFirst + File2Size + (FileSizeSecond - FileSizeFirst) + EntryData2Size;
                static const auto Size = FileSizeThird - FileSizeSecond;
                NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(&sizeRead, file, Offset, GetBuffer() + FileSizeSecond, Size, nn::fs::ReadOption::MakeValue(0)));
                ASSERT_TRUE(nnt::fs::util::IsFilledWith32BitCount(GetBuffer(), FileSizeThird, 0));
            }
        }
    }
} // NOLINT(impl/function_size)

TEST_F(SaveDataDumpRestore, NonExistentStorage)
{
    NNT_FS_ASSERT_NO_FATAL_FAILURE(CreateSaveData(SaveDataDataSizeSmall, true));

    static const char* MountName = "test";

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataInternalStorage(MountName, nn::fs::SaveDataSpaceId::System, TestSaveDataIds[0]));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(MountName);
    };

    {
        nn::fs::FileHandle file;
        auto result = nn::fs::OpenFile(&file, "test:/NonExistentStorage", nn::fs::OpenMode_Read | nn::fs::OpenMode_Write);
        if( result.IsSuccess() )
        {
            nn::fs::CloseFile(file);
        }
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultPathNotFound, result);
    }

    {
        nn::fs::DirectoryHandle directory;
        auto result = nn::fs::OpenDirectory(&directory, "test:/NonExistentStorage", nn::fs::OpenDirectoryMode_All);
        if( result.IsSuccess() )
        {
            nn::fs::CloseDirectory(directory);
        }
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultPathNotFound, result);
    }
}

TEST_F(SaveDataDumpRestore, CannotMountMulti)
{
    NNT_FS_ASSERT_NO_FATAL_FAILURE(CreateSaveData(SaveDataDataSizeSmall, true));

    static const char* MountName1 = "test1";
    static const char* MountName2 = "test2";

    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSystemSaveData(MountName1, nn::fs::SaveDataSpaceId::System, TestSaveDataIds[0]));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(MountName1);
        };

        // 同じセーブデータの2重マウントは失敗する
        auto result = nn::fs::MountSystemSaveData(MountName2, nn::fs::SaveDataSpaceId::System, TestSaveDataIds[0]);
        if( result.IsSuccess() )
        {
            nn::fs::Unmount(MountName2);
        }
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultTargetLocked, result);
    }

    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataInternalStorage(MountName1, nn::fs::SaveDataSpaceId::System, TestSaveDataIds[0]));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(MountName1);
        };

        // 同じセーブデータの2重マウントは失敗する
        auto result = nn::fs::MountSaveDataInternalStorage(MountName2, nn::fs::SaveDataSpaceId::System, TestSaveDataIds[0]);
        if( result.IsSuccess() )
        {
            nn::fs::Unmount(MountName2);
        }
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultTargetLocked, result);
    }
}

#if defined(NN_BUILD_CONFIG_SPEC_NX)
TEST_F(SaveDataDumpRestore, MountMultiInternalStorage)
{
    NNT_FS_ASSERT_NO_FATAL_FAILURE(CreateSaveData(SaveDataDataSizeSmall, false));

    static const char* MountName1 = "test1";
    static const char* MountName2 = "test2";

    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveData(MountName1, ApplicationIds[0], GetUserId()));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(MountName1);
        };

        // 同じセーブデータでも通常マウントと InternalStorage のマウントは成功する
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataInternalStorage(MountName2, nn::fs::SaveDataSpaceId::User, GetSaveDataId(0)));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::Unmount(MountName2);
            };
        }

        {
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataInternalStorage(MountName2, nn::fs::SaveDataSpaceId::User, GetSaveDataId(0)));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::Unmount(MountName2);
            };
        }

        // 同じセーブデータの2重マウントは失敗する
        auto result = nn::fs::MountSaveData(MountName2, ApplicationIds[0], GetUserId());
        if( result.IsSuccess() )
        {
            nn::fs::Unmount(MountName2);
        }
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultTargetLocked, result);
    }

    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataInternalStorage(MountName1, nn::fs::SaveDataSpaceId::User, GetSaveDataId(0)));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(MountName1);
        };

        // 同じセーブデータでも通常マウントと InternalStorage のマウントは成功する
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveData(MountName2, ApplicationIds[0], GetUserId()));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::Unmount(MountName2);
            };
        }

        {
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveData(MountName2, ApplicationIds[0], GetUserId()));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::Unmount(MountName2);
            };
        }

        // 同じセーブデータの2重マウントは失敗する
        auto result = nn::fs::MountSaveDataInternalStorage(MountName2, nn::fs::SaveDataSpaceId::User, GetSaveDataId(0));
        if( result.IsSuccess() )
        {
            nn::fs::Unmount(MountName2);
        }
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultTargetLocked, result);
    }
}

TEST_F(SaveDataDumpRestore, MountMulti)
{
    NNT_FS_ASSERT_NO_FATAL_FAILURE(CreateSaveData(SaveDataDataSizeLarge, false));
    NNT_FS_ASSERT_NO_FATAL_FAILURE(PrepareSaveDataWithMulipleFiles(false));

    static const char* MountName = "test";

    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveData(MountName, ApplicationIds[0], GetUserId()));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(MountName);
        };

        // 通常マウントしたままでも InternalStorage を読める
        NNT_FS_ASSERT_NO_FATAL_FAILURE(DumpAndRecovery(false, false));
    }
    NNT_FS_ASSERT_NO_FATAL_FAILURE(CheckSaveDataWithMulipleFiles(false, false));
}

TEST_F(SaveDataDumpRestore, MountMultiAccessSaveData)
{
    NNT_FS_ASSERT_NO_FATAL_FAILURE(CreateSaveData(SaveDataDataSizeLarge, false));

    static const char* MountName1 = "test1";
    static const char* MountName2 = "test2";
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveData(MountName1, ApplicationIds[0], GetUserId()));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(MountName1);
    };

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile("test1:/test1.bin", SaveDataDataSizeSmall));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CommitSaveData(MountName1));

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataInternalStorage(MountName2, nn::fs::SaveDataSpaceId::User, GetSaveDataId(0)));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(MountName2);
    };

    nnt::fs::util::String filePath("test2:/");
    filePath += nn::fssystem::save::SaveDataFileSystemCore::InternalStorageFileNameAllocationTableData;
    nn::fs::FileHandle file;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&file, filePath.c_str(), nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(file);
    };
    int64_t size;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&size, file));
    size_t readSize;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(&readSize, file, 0, GetBuffer(), static_cast<size_t>(size), nn::fs::ReadOption::MakeValue(0)));

    // 2重マウント中に通常マウント側のアクセス発生
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile("test1:/test2.bin", SaveDataDataSizeSmall));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CommitSaveData(MountName1));

    // 以後 InternalStorage のアクセスは失敗する
    auto result = nn::fs::ReadFile(&readSize, file, 0, GetBuffer(), static_cast<size_t>(size), nn::fs::ReadOption::MakeValue(0));

    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultTargetLocked, result);
}

TEST_F(SaveDataDumpRestore, MountMultiReadExtraData)
{
    NNT_FS_ASSERT_NO_FATAL_FAILURE(CreateSaveData(SaveDataDataSizeSmall, false));

    static const char* MountName1 = "test1";
    static const char* MountName2 = "test2";

    // 通常マウント -> Internal マウントでの拡張データ読み出し
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveData(MountName1, ApplicationIds[0], GetUserId()));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(MountName1);
        };

        // 拡張データ読み出し
        ReadExtraDara(0);

        // 同じセーブデータでも通常マウントと InternalStorage のマウントは成功する
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataInternalStorage(MountName2, nn::fs::SaveDataSpaceId::User, GetSaveDataId(0)));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::Unmount(MountName2);
            };

            // 拡張データ読み出し
            ReadExtraDara(0);
        }

        {
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataInternalStorage(MountName2, nn::fs::SaveDataSpaceId::User, GetSaveDataId(0)));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::Unmount(MountName2);
            };

            // 拡張データ読み出し
            ReadExtraDara(0);
        }

        // 同じセーブデータの2重マウントは失敗する
        auto result = nn::fs::MountSaveData(MountName2, ApplicationIds[0], GetUserId());
        if( result.IsSuccess() )
        {
            nn::fs::Unmount(MountName2);
        }
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultTargetLocked, result);

        // 拡張データ読み出し
        ReadExtraDara(0);
    }

    // Internal マウント -> 通常マウントでの拡張データ読み出し
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataInternalStorage(MountName1, nn::fs::SaveDataSpaceId::User, GetSaveDataId(0)));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(MountName1);
        };

        // 拡張データ読み出し
        ReadExtraDara(0);

        // 同じセーブデータでも通常マウントと InternalStorage のマウントは成功する
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveData(MountName2, ApplicationIds[0], GetUserId()));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::Unmount(MountName2);
            };

            // 拡張データ読み出し
            ReadExtraDara(0);
        }

        {
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveData(MountName2, ApplicationIds[0], GetUserId()));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::Unmount(MountName2);
            };

            // 拡張データ読み出し
            ReadExtraDara(0);
        }

        // 同じセーブデータの2重マウントは失敗する
        auto result = nn::fs::MountSaveDataInternalStorage(MountName2, nn::fs::SaveDataSpaceId::User, GetSaveDataId(0));
        if( result.IsSuccess() )
        {
            nn::fs::Unmount(MountName2);
        }
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultTargetLocked, result);

        // 拡張データ読み出し
        ReadExtraDara(0);
    }
} // NOLINT(impl/function_size)
#endif // defined(NN_BUILD_CONFIG_SPEC_NX)

#if defined(NN_BUILD_CONFIG_OS_WIN)
TEST_F(SaveDataDumpRestore, DumpToLocalFilesHeavy)
{
    static const int64_t SameDataDataSize = 1024 * 1024;

    NNT_FS_ASSERT_NO_FATAL_FAILURE(CreateSaveData(SameDataDataSize, true));
    NNT_FS_ASSERT_NO_FATAL_FAILURE(PrepareSaveDataWithMulipleFiles(true));

    nn::fs::MountBis(nn::fs::GetBisMountName(nn::fs::BisPartitionId::User), nn::fs::BisPartitionId::User);
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(nn::fs::GetBisMountName(nn::fs::BisPartitionId::User));
    };

    auto dumpToFile = [&](int geration) NN_NOEXCEPT
    {
        static const char* MountName = "test";
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataInternalStorage(MountName, nn::fs::SaveDataSpaceId::System, TestSaveDataIds[0]));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(MountName);
        };

        const char* Paths[] =
        {
            nn::fssystem::save::SaveDataFileSystemCore::InternalStorageFileNameAllocationTableControlArea,
            nn::fssystem::save::SaveDataFileSystemCore::InternalStorageFileNameAllocationTableMeta,
            nn::fssystem::save::SaveDataFileSystemCore::InternalStorageFileNameAllocationTableData,
        };
        nnt::fs::util::String path;
        for( auto entryName : Paths )
        {
            path = MountName;
            path += ":/";
            path += entryName;

            nn::fs::FileHandle file;

            NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&file, path.c_str(), nn::fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(file);
            };

            int64_t size;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&size, file));
            ASSERT_LT(size, static_cast<int64_t>(BufferSize));

            size_t readSize;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(&readSize, file, 0, GetBuffer(), static_cast<size_t>(size), nn::fs::ReadOption::MakeValue(0)));

            char suffix[128];
            nn::util::SNPrintf(suffix, sizeof(suffix), "%d", geration);

            path = entryName;
            path += "_";
            path += suffix;
            path += ".bin";
            NNT_FS_ASSERT_NO_FATAL_FAILURE(DumpFile(path.c_str(), GetBuffer(), static_cast<size_t>(size)));
        }
    };

    auto copySaveData = [&](int geration) NN_NOEXCEPT
    {
        size_t size = 0;
        NNT_FS_ASSERT_NO_FATAL_FAILURE(nnt::fs::util::SaveDataFileSystemMounter::ReadFile(&size, nn::fs::SaveDataSpaceId::System, TestSaveDataIds[0], 0, GetBuffer(), BufferSize));

        char suffix[128];
        nn::util::SNPrintf(suffix, sizeof(suffix), "%d", geration);

        nnt::fs::util::String path;
        path = "Raw";
        path += "_";
        path += suffix;
        path += ".bin";
        NNT_FS_ASSERT_NO_FATAL_FAILURE(DumpFile(path.c_str(), GetBuffer(), static_cast<size_t>(size)));
    };

    NNT_FS_ASSERT_NO_FATAL_FAILURE(copySaveData(0));
    NNT_FS_ASSERT_NO_FATAL_FAILURE(dumpToFile(0));

    NNT_FS_ASSERT_NO_FATAL_FAILURE(ModifySaveDataWithMulipleFiles());
    NNT_FS_ASSERT_NO_FATAL_FAILURE(copySaveData(1));
    NNT_FS_ASSERT_NO_FATAL_FAILURE(dumpToFile(1));
}
#endif // defined(NN_BUILD_CONFIG_OS_WIN)

extern "C" void nnMain()
{
    int     argc = nn::os::GetHostArgc();
    char**  argv = nn::os::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    if( argc >= 2 )
    {
        g_DumpAndRecoveryHeavyTryCount = std::atoi(argv[1]);

        if( argc >= 3 )
        {
            g_SaveDataDataSizeHeavy = std::strtoll(argv[2], nullptr, 10);
        }
    }

    nnt::fs::util::ResetAllocateCount();
    nnt::fs::util::InitializeSaveDataTestHelper();

    nn::fs::SetEnabledAutoAbort(false);
    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);

    auto ret = RUN_ALL_TESTS();

    nnt::fs::util::FinalizeSaveDataTestHelper();

    if( nnt::fs::util::CheckMemoryLeak() )
    {
        nnt::Exit(1);
    }

    nnt::Exit(ret);
}
